import apprise
from flask import Flask, request, jsonify
from flask_cors import CORS
import threading
import queue

# Memory where we store loaded configurations
cache = {}

# Create an Apprise instance
apobj = apprise.Apprise()
apobj.asset.app_desc = "DataTalk"
apobj.asset.app_id = "DataTalk"
apobj.asset.app_url = "https://datatalk.eu"

# Web server instance
app = Flask(__name__)
CORS(app)

# Create a queue to hold requests
request_queue = queue.Queue()

# Worker function to process requests from the queue
def process_requests():
    while True:
        try:
            # Get the next request data from the queue (blocking)
            data = request_queue.get()
            if data is None:  # Exit signal
                break

            url = data['url']
            apobj.clear()
            apobj.add(url, tag='test')

            attach = data.get('attach')

            # Send notification
            apobj.notify(
                body=data['body'],
                title=data['title'],
                tag='test',
                attach=attach
            )

            print("Processed notification:", data)
        except Exception as e:
            print(f"Error processing request: {e}")
        finally:
            # Indicate task is done
            request_queue.task_done()

# Start the worker thread
worker_thread = threading.Thread(target=process_requests, daemon=True)
worker_thread.start()

@app.route('/notify', methods=['POST'])
def handle_post_request():
    try:
        data = request.json
        # Add request data to the queue
        request_queue.put(data)
        return jsonify({"status": "success", "message": "Notification queued!"}), 200
    except Exception as e:
        return jsonify({"status": "error", "message": str(e)}), 500

@app.route('/stop', methods=['POST'])
def stop_worker():
    # Add None to the queue to signal the worker to exit
    request_queue.put(None)
    worker_thread.join()
    return jsonify({"status": "success", "message": "Worker stopped!"}), 200

# Run the Flask app
if __name__ == '__main__':
    app.run(debug=True, port=4601)
