"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Server = void 0;
const array_flatten_1 = __importDefault(require("array-flatten"));
const es6_1 = __importDefault(require("fast-deep-equal/es6"));
const dns_equal_1 = __importDefault(require("dns-equal"));
const multicast_dns_1 = __importDefault(require("multicast-dns"));
function makeIsDuplicateRecord(a) {
    return (b) => {
        return a.type === b.type && a.name === b.name && (0, es6_1.default)(a.data, b.data);
    };
}
function makeUnique() {
    const set = [];
    return (obj) => {
        if (~set.indexOf(obj))
            return false;
        set.push(obj);
        return true;
    };
}
class Server {
    constructor(opts, errorCallback) {
        this.registry = {};
        this.mdns = (0, multicast_dns_1.default)(opts);
        this.mdns.setMaxListeners(0);
        this.mdns.on('query', this.respondToQuery.bind(this));
        this.errorCallback =
            errorCallback !== null && errorCallback !== void 0 ? errorCallback : function (err) {
                throw err;
            };
    }
    register(records) {
        // Register a record
        const shouldRegister = (record) => {
            let subRegistry = this.registry[record.type];
            if (!subRegistry) {
                subRegistry = this.registry[record.type] = [];
            }
            else if (subRegistry.some(makeIsDuplicateRecord(record))) {
                return;
            }
            subRegistry.push(record);
        };
        if (Array.isArray(records)) {
            // Multiple records
            records.forEach(shouldRegister);
        }
        else {
            // Single record
            shouldRegister(records);
        }
    }
    unregister(records) {
        // Unregister a record
        const shouldUnregister = (record) => {
            let type = record.type;
            if (!(type in this.registry)) {
                return;
            }
            this.registry[type] = this.registry[type].filter((i) => i.name !== record.name);
        };
        if (Array.isArray(records)) {
            // Multiple records
            records.forEach(shouldUnregister);
        }
        else {
            // Single record
            shouldUnregister(records);
        }
    }
    respondToQuery(query) {
        let self = this;
        query.questions.forEach((question) => {
            const type = question.type;
            const name = question.name;
            // generate the answers section
            const answers = type === 'ANY'
                ? array_flatten_1.default.depth(Object.keys(self.registry).map(self.recordsFor.bind(self, name)), 1)
                : self.recordsFor(name, type);
            if (answers.length === 0)
                return;
            // generate the additionals section
            let additionals = [];
            if (type !== 'ANY') {
                answers.forEach((answer) => {
                    if (answer.type !== 'PTR')
                        return;
                    additionals = additionals
                        .concat(self.recordsFor(answer.data, 'SRV'))
                        .concat(self.recordsFor(answer.data, 'TXT'));
                });
                // to populate the A and AAAA records, we need to get a set of unique
                // targets from the SRV record
                additionals
                    .filter(function (record) {
                    return record.type === 'SRV';
                })
                    .map(function (record) {
                    return record.data.target;
                })
                    .filter(makeUnique())
                    .forEach(function (target) {
                    additionals = additionals.concat(self.recordsFor(target, 'A')).concat(self.recordsFor(target, 'AAAA'));
                });
            }
            self.mdns.respond({ answers, additionals }, (err) => {
                if (err) {
                    this.errorCallback(err);
                }
            });
        });
    }
    recordsFor(name, type) {
        if (!(type in this.registry)) {
            return [];
        }
        return this.registry[type].filter((record) => {
            const _name = ~name.indexOf('.') ? record.name : record.name.split('.')[0];
            return (0, dns_equal_1.default)(_name, name);
        });
    }
}
exports.Server = Server;
exports.default = Server;
//# sourceMappingURL=mdns-server.js.map