"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BasePath = void 0;
const fs_routes_1 = require("fs-routes");
const openapi_default_setter_1 = require("openapi-default-setter");
const openapi_request_coercer_1 = require("openapi-request-coercer");
const openapi_request_validator_1 = require("openapi-request-validator");
const openapi_response_validator_1 = require("openapi-response-validator");
const openapi_schema_validator_1 = require("openapi-schema-validator");
const openapi_security_handler_1 = require("openapi-security-handler");
const BasePath_1 = require("./src/BasePath");
exports.BasePath = BasePath_1.default;
const types_1 = require("./src/types");
const util_1 = require("./src/util");
class OpenAPIFramework {
    constructor(args = {}) {
        this.args = args;
        this.name = args.name;
        this.featureType = args.featureType;
        this.loggingPrefix = args.name ? `${this.name}: ` : '';
        this.logger = args.logger ? args.logger : new types_1.ConsoleDebugAdapterLogger();
        // monkey patch for node v6:
        if (!this.logger.debug) {
            this.logger.debug = this.logger.info;
        }
        [
            { name: 'apiDoc', required: true },
            { name: 'errorTransformer', type: 'function' },
            { name: 'externalSchemas', type: 'object' },
            { name: 'featureType', required: true },
            { name: 'name', required: true },
            { name: 'pathSecurity', class: Array, className: 'Array' },
            { name: 'securityHandlers', type: 'object' },
        ].forEach((arg) => {
            if (arg.required && !(arg.name in args)) {
                throw new Error(`${this.loggingPrefix}args.${arg.name} is required`);
            }
            if (arg.type && arg.name in args && typeof args[arg.name] !== arg.type) {
                throw new Error(`${this.loggingPrefix}args.${arg.name} must be a ${arg.type} when given`);
            }
            if (arg.class &&
                arg.name in args &&
                !(args[arg.name] instanceof arg.class)) {
                throw new Error(`${this.loggingPrefix}args.${arg.name} must be an instance of ${arg.className} when given`);
            }
        });
        if (!args.paths && !args.operations) {
            throw new Error(`${this.loggingPrefix}args.paths and args.operations must not both be empty`);
        }
        this.enableObjectCoercion = !!args.enableObjectCoercion;
        this.originalApiDoc = (0, util_1.handleYaml)((0, util_1.handleFilePath)(args.apiDoc));
        this.apiDoc = (0, util_1.copy)(this.originalApiDoc);
        this.basePaths = this.apiDoc.openapi
            ? (0, util_1.getBasePathsFromServers)(this.apiDoc.servers)
            : [
                new BasePath_1.default({
                    url: (this.apiDoc.basePath || '').replace(/\/$/, ''),
                }),
            ];
        this.validateApiDoc =
            'validateApiDoc' in args ? !!args.validateApiDoc : true;
        this.validator = new openapi_schema_validator_1.default({
            version: this.apiDoc.openapi ||
                this.apiDoc.swagger,
            extensions: this.apiDoc[`x-${this.name}-schema-extension`],
        });
        this.customFormats = args.customFormats;
        this.customKeywords = args.customKeywords;
        this.dependencies = args.dependencies;
        this.errorTransformer = args.errorTransformer;
        this.externalSchemas = args.externalSchemas;
        this.features = args.features;
        this.operations = args.operations;
        this.paths = args.paths;
        this.pathsIgnore = args.pathsIgnore;
        this.pathSecurity = Array.isArray(args.pathSecurity)
            ? args.pathSecurity
            : [];
        this.routesGlob = args.routesGlob;
        this.routesIndexFileRegExp = args.routesIndexFileRegExp;
        this.securityHandlers = args.securityHandlers;
        this.pathSecurity.forEach(util_1.assertRegExpAndSecurity.bind(null, this));
        if (this.validateApiDoc) {
            const apiDocValidation = this.validator.validate(this.apiDoc);
            if (apiDocValidation.errors.length) {
                this.logger.error(`${this.loggingPrefix}Validating schema before populating paths`);
                this.logger.error(`${this.loggingPrefix}validation errors`, JSON.stringify(apiDocValidation.errors, null, '  '));
                throw new Error(`${this.loggingPrefix}args.apiDoc was invalid.  See the output.`);
            }
        }
    }
    async initialize(visitor) {
        const securitySchemes = this.apiDoc.openapi
            ? (this.apiDoc.components || {}).securitySchemes
            : this.apiDoc.securityDefinitions;
        const apiSecurityMiddleware = this.securityHandlers && this.apiDoc.security && securitySchemes
            ? new openapi_security_handler_1.default({
                securityDefinitions: securitySchemes,
                securityHandlers: this.securityHandlers,
                operationSecurity: this.apiDoc.security,
                loggingKey: `${this.name}-security`,
            })
            : null;
        let paths = [];
        let routes = [];
        const routesCheckMap = {};
        if (this.paths) {
            paths = [].concat(this.paths);
            this.logger.debug(`${this.loggingPrefix}paths=`, paths);
            for (let pathItem of paths) {
                if ((0, util_1.byString)(pathItem)) {
                    pathItem = (0, util_1.toAbsolutePath)(pathItem);
                    if (!(0, util_1.byDirectory)(pathItem)) {
                        throw new Error(`${this.loggingPrefix}args.paths contained a value that was not a path to a directory`);
                    }
                    routes = routes.concat(await Promise.all((0, fs_routes_1.default)(pathItem, {
                        glob: this.routesGlob,
                        indexFileRegExp: this.routesIndexFileRegExp,
                    })
                        .filter((fsRoutesItem) => {
                        return this.pathsIgnore
                            ? !this.pathsIgnore.test(fsRoutesItem.route)
                            : true;
                    })
                        .map(async (fsRoutesItem) => {
                        routesCheckMap[fsRoutesItem.route] = true;
                        // There are two cases to distinguish:
                        // - file is a CommonJS script, and `module.export` appears
                        //   as `default` property
                        // - file is a ECMAScript module, and `export default` appears
                        //   at top-level
                        const imported = await import(`file://${fsRoutesItem.path}`);
                        return {
                            path: fsRoutesItem.route,
                            module: imported.default ?? imported,
                        };
                    })));
                }
                else {
                    if (!pathItem.path || !pathItem.module) {
                        throw new Error(`${this.loggingPrefix}args.paths must consist of strings or valid route specifications`);
                    }
                    routes.push(pathItem);
                }
            }
            routes = routes.sort(util_1.byRoute);
        }
        if (this.operations) {
            const apiDocPaths = this.apiDoc.paths;
            Object.keys(apiDocPaths).forEach((apiDocPathUrl) => {
                const pathDoc = apiDocPaths[apiDocPathUrl];
                const route = {
                    path: apiDocPathUrl,
                    module: Object.keys(pathDoc)
                        .filter(util_1.byMethods)
                        .reduce((acc, method) => {
                        const methodDoc = pathDoc[util_1.METHOD_ALIASES[method]];
                        const operationId = methodDoc.operationId;
                        if (operationId && operationId in this.operations) {
                            const operation = this.operations[operationId];
                            acc[util_1.METHOD_ALIASES[method]] = (() => {
                                /**
                                 * We have two options:
                                 *
                                 * 1. the middleware gets bound + dependency injected, this may be breaking.
                                 * 2. we pick the last middleware as the operation handler. This means we cannot support
                                 *    _after_ middlewares (though not a common express pattern)
                                 */
                                const innerFunction = Array.isArray(operation)
                                    ? operation.map((middleware) => middleware.bind({ dependencies: this.dependencies }))
                                    : operation.bind({ dependencies: this.dependencies });
                                innerFunction.apiDoc = methodDoc;
                                return innerFunction;
                            })();
                        }
                        else if (operationId === undefined) {
                            this.logger.warn(`${this.loggingPrefix}path ${apiDocPathUrl}, operation ${method} is missing an operationId`);
                        }
                        else {
                            this.logger.warn(`${this.loggingPrefix}Operation ${operationId} not found in the operations parameter`);
                        }
                        return acc;
                    }, {}),
                };
                if (routesCheckMap[route.path]) {
                    this.logger.warn(`${this.loggingPrefix}Overriding path ${route.path} with handlers from operations`);
                    const routeIndex = routes.findIndex((r) => r.path === route.path);
                    routes[routeIndex] = {
                        ...routes[routeIndex],
                        ...route,
                        module: {
                            ...((routes[routeIndex] || {}).module || {}),
                            ...(route.module || {}),
                        },
                    };
                }
                else {
                    routes.push(route);
                }
            });
        }
        this.logger.debug(`${this.loggingPrefix}routes=`, routes);
        // Check for duplicate routes
        const dups = routes.filter((v, i, o) => {
            if (i > 0 && v.path === o[i - 1].path) {
                return v.path;
            }
        });
        if (dups.length > 0) {
            throw new Error(`${this.loggingPrefix}args.paths produced duplicate urls for "${dups[0].path}"`);
        }
        const getApiDoc = () => {
            return (0, util_1.copy)(this.apiDoc);
        };
        routes.forEach((routeItem) => {
            const route = routeItem.path;
            this.logger.debug(`${this.loggingPrefix}setting up`, route);
            const pathModule = (0, util_1.injectDependencies)(routeItem.module.default || routeItem.module, this.dependencies);
            // express path params start with :paramName
            // openapi path params use {paramName}
            const openapiPath = route;
            // Do not make modifications to this.
            const originalPathItem = this.originalApiDoc.paths[openapiPath] || {};
            const pathDoc = this.apiDoc.paths[openapiPath] || {};
            const pathParameters = pathDoc.parameters || [];
            // push all parameters defined in the path module to the path parameter list
            if (Array.isArray(pathModule.parameters)) {
                [].push.apply(pathParameters, pathModule.parameters);
            }
            pathDoc.parameters = pathParameters;
            this.apiDoc.paths[openapiPath] = pathDoc;
            const methodsProcessed = {};
            new Set(Object.keys(pathModule).concat(Object.keys(pathDoc)).filter(util_1.byMethods)).forEach((methodAlias) => {
                const methodName = util_1.METHOD_ALIASES[methodAlias];
                if (methodName in methodsProcessed) {
                    this.logger.warn(`${this.loggingPrefix}${openapiPath}.${methodAlias} has already been defined as ${openapiPath}.${methodsProcessed[methodName]}. Ignoring the 2nd definition...`);
                    return;
                }
                methodsProcessed[methodName] = methodAlias;
                // operationHandler may be an array or a function.
                const operationHandler = pathModule[methodAlias] ||
                    routeItem.module[(pathDoc?.[methodAlias] || {}).operationId];
                const operationDoc = (0, util_1.handleYaml)((0, util_1.getMethodDoc)(operationHandler)) || pathDoc[methodName];
                // consumes is defined as property of each operation or entire document
                // in Swagger 2.0. For OpenAPI 3.0 consumes mime types are defined as the
                // key value(s) for each operation requestBody.content object.
                const consumes = operationDoc && Array.isArray(operationDoc.consumes)
                    ? operationDoc.consumes
                    : operationDoc &&
                        operationDoc.requestBody &&
                        (operationDoc.requestBody.content ||
                            operationDoc.requestBody.$ref)
                        ? Object.keys((0, util_1.resolveRequestBodyRefs)(this, operationDoc.requestBody, this.apiDoc).content)
                        : Array.isArray(this.apiDoc.consumes)
                            ? this.apiDoc.consumes
                            : [];
                const operationContext = {
                    additionalFeatures: (0, util_1.getAdditionalFeatures)(this, this.logger, this.originalApiDoc, originalPathItem, pathModule, operationDoc),
                    allowsFeatures: (0, util_1.allowsFeatures)(this, this.apiDoc, pathModule, pathDoc, operationDoc),
                    apiDoc: this.apiDoc,
                    basePaths: this.basePaths,
                    consumes,
                    features: {},
                    methodName,
                    methodParameters: [],
                    operationDoc,
                    operationHandler,
                    path: openapiPath,
                };
                if (operationDoc) {
                    pathDoc[methodName] = operationDoc;
                    if (operationDoc.tags) {
                        (0, util_1.sortOperationDocTags)(operationDoc);
                        operationDoc.tags.forEach(util_1.addOperationTagToApiDoc.bind(null, this.apiDoc));
                    }
                    if (operationContext.allowsFeatures) {
                        // add features
                        if (operationDoc.responses &&
                            (0, util_1.allowsResponseValidationFeature)(this, this.apiDoc, pathModule, pathDoc, operationDoc)) {
                            const ResponseValidatorClass = this.features?.responseValidator || openapi_response_validator_1.default;
                            // add response validation feature
                            // it's invalid for a method doc to not have responses, but the post
                            // validation will pick it up, so this is almost always going to be added.
                            const responseValidator = new ResponseValidatorClass({
                                loggingKey: `${this.name}-response-validation`,
                                components: this.apiDoc.components,
                                definitions: this.apiDoc.definitions,
                                externalSchemas: this.externalSchemas,
                                errorTransformer: this.errorTransformer,
                                responses: (0, util_1.resolveResponseRefs)(this, operationDoc.responses, this.apiDoc, route),
                                customFormats: this.customFormats,
                            });
                            operationContext.features.responseValidator = responseValidator;
                        }
                        const methodParameters = (0, util_1.withNoDuplicates)((0, util_1.resolveParameterRefs)(this, Array.isArray(operationDoc.parameters)
                            ? pathParameters.concat(operationDoc.parameters)
                            : pathParameters, this.apiDoc));
                        operationContext.methodParameters = methodParameters;
                        const requestBody = (0, util_1.resolveRequestBodyRefs)(this, operationDoc.requestBody, this.apiDoc);
                        if (methodParameters.length || operationDoc.requestBody) {
                            // defaults, coercion, and parameter validation middleware
                            if ((0, util_1.allowsValidationFeature)(this, this.apiDoc, pathModule, pathDoc, operationDoc)) {
                                const RequestValidatorClass = this.features?.requestValidator || openapi_request_validator_1.default;
                                const requestValidator = new RequestValidatorClass({
                                    errorTransformer: this.errorTransformer,
                                    logger: this.logger,
                                    parameters: methodParameters,
                                    schemas: this.apiDoc.definitions,
                                    componentSchemas: this.apiDoc.components // v3
                                        ? this.apiDoc.components.schemas
                                        : undefined,
                                    externalSchemas: this.externalSchemas,
                                    customFormats: this.customFormats,
                                    customKeywords: this.customKeywords,
                                    requestBody,
                                });
                                operationContext.features.requestValidator = requestValidator;
                                this.logger.debug(`${this.loggingPrefix}request validator on for`, methodName, openapiPath);
                            }
                            if ((0, util_1.allowsCoercionFeature)(this, this.apiDoc, pathModule, pathDoc, operationDoc)) {
                                const CoercerClass = this.features?.coercer || openapi_request_coercer_1.default;
                                const coercer = new CoercerClass({
                                    extensionBase: `x-${this.name}-coercion`,
                                    loggingKey: `${this.name}-coercion`,
                                    parameters: methodParameters,
                                    requestBody,
                                    enableObjectCoercion: this.enableObjectCoercion,
                                });
                                operationContext.features.coercer = coercer;
                            }
                            // no point in default feature if we don't have any parameters with defaults.
                            if (methodParameters.filter(util_1.byDefault).length &&
                                (0, util_1.allowsDefaultsFeature)(this, this.apiDoc, pathModule, pathDoc, operationDoc)) {
                                const DefaultSetterClass = this.features?.defaultSetter || openapi_default_setter_1.default;
                                const defaultSetter = new DefaultSetterClass({
                                    parameters: methodParameters,
                                });
                                operationContext.features.defaultSetter = defaultSetter;
                            }
                        }
                        let securityFeature;
                        let securityDefinition;
                        if (this.securityHandlers && securitySchemes) {
                            if (operationDoc.security) {
                                securityDefinition = operationDoc.security;
                            }
                            else if (this.pathSecurity.length) {
                                securityDefinition = (0, util_1.getSecurityDefinitionByPath)(openapiPath, this.pathSecurity);
                            }
                        }
                        if (securityDefinition) {
                            pathDoc[methodName].security = securityDefinition;
                            const SecurityHandlerClass = this.features?.securityHandler || openapi_security_handler_1.default;
                            securityFeature = new SecurityHandlerClass({
                                securityDefinitions: securitySchemes,
                                securityHandlers: this.securityHandlers,
                                operationSecurity: securityDefinition,
                                loggingKey: `${this.name}-security`,
                            });
                        }
                        else if (apiSecurityMiddleware) {
                            securityFeature = apiSecurityMiddleware;
                        }
                        if (securityFeature) {
                            operationContext.features.securityHandler = securityFeature;
                        }
                    }
                }
                if (visitor.visitOperation) {
                    visitor.visitOperation(operationContext);
                }
            });
            if (visitor.visitPath) {
                visitor.visitPath({
                    basePaths: this.basePaths,
                    getApiDoc,
                    getPathDoc: () => (0, util_1.copy)(pathDoc),
                });
            }
        });
        (0, util_1.sortApiDocTags)(this.apiDoc);
        if (this.validateApiDoc) {
            const apiDocValidation = this.validator.validate(this.apiDoc);
            if (apiDocValidation.errors.length) {
                this.logger.error(`${this.loggingPrefix}Validating schema after populating paths`);
                this.logger.error(`${this.loggingPrefix}validation errors`, JSON.stringify(apiDocValidation.errors, null, '  '));
                throw new Error(`${this.loggingPrefix}args.apiDoc was invalid after populating paths.  See the output.`);
            }
        }
        if (visitor.visitApi) {
            visitor.visitApi({
                basePaths: this.basePaths,
                getApiDoc,
            });
        }
    }
}
exports.default = OpenAPIFramework;
//# sourceMappingURL=index.js.map