"use strict";
/**
 * @module node-opcua-xml2json
 * node -> see if https://github.com/isaacs/sax-js could be used instead
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Xml2Json = exports.ReaderState = exports.ReaderStateBase = void 0;
const node_opcua_assert_1 = require("node-opcua-assert");
const lts_1 = require("./thirdparties/parser/lts");
/**
 * @static
 * @private

 * @param parser {map<ReaderState|options>}
 * @return {map}
 */
function _coerceParser(parser) {
    for (const name of Object.keys(parser)) {
        if (parser[name] && !(parser[name] instanceof ReaderState)) {
            // this is to prevent recursion
            const tmp = parser[name];
            delete parser[name];
            parser[name] = new ReaderState(tmp);
        }
    }
    return parser;
}
class ReaderStateBase {
}
exports.ReaderStateBase = ReaderStateBase;
/**
 * @private
 */
class ReaderState extends ReaderStateBase {
    _init;
    _finish;
    _startElement;
    _endElement;
    parser;
    attrs;
    chunks = [];
    text = "";
    name = "";
    level = -1;
    currentLevel = -1;
    engine;
    parent;
    root;
    data;
    constructor(options) {
        super();
        // ensure options object has only expected properties
        options.parser = options.parser || {};
        if (!(options instanceof ReaderStateBase)) {
            this._init = options.init;
            this._finish = options.finish;
            this._startElement = options.startElement;
            this._endElement = options.endElement;
        }
        this.parser = _coerceParser(options.parser);
    }
    /**
     * @protected
     */
    _on_init(elementName, attrs, parent, level, engine) {
        this.name = elementName;
        this.parent = parent;
        this.engine = engine;
        this.data = {};
        this.level = level;
        this.currentLevel = this.level;
        this.attrs = attrs;
        (0, node_opcua_assert_1.assert)(this.attrs);
        if (this._init) {
            this._init(elementName, attrs, parent, engine);
        }
    }
    /**
     * @protected
     */
    _on_finish() {
        if (this._finish) {
            this._finish();
        }
    }
    /**
     * @protected
     */
    _on_startElement(level, elementName, attrs) {
        this.currentLevel = level;
        this.chunks = [];
        this.text = "";
        if (this._startElement) {
            this._startElement(elementName, attrs);
        }
        if (this.engine && Object.prototype.hasOwnProperty.call(this.parser, elementName)) {
            this.engine._promote(this.parser[elementName], level, elementName, attrs);
        }
    }
    /**
     * @protected
     */
    _on_endElement2(level, elementName) {
        if (this._endElement) {
            this._endElement(elementName);
        }
    }
    /**

     * @protected
     */
    _on_endElement(level, elementName) {
        (0, node_opcua_assert_1.assert)(this.attrs);
        this.chunks = this.chunks || [];
        if (this.level > level) {
            // we end a child element of this node
            this._on_endElement2(level, elementName);
        }
        else if (this.level === level) {
            // we received the end event of this node
            // we need to finish
            this.text = this.chunks.join("");
            this.chunks = [];
            // this is the end
            this._on_finish();
            if (this.parent &&
                this.parent.parser &&
                Object.prototype.hasOwnProperty.call(this.parent.parser, elementName)) {
                this.engine._demote(this, level, elementName);
            }
        }
    }
    /**
     * @param text {String} the text found inside the element
     * @protected
     */
    _on_text(text) {
        this.chunks = this.chunks || [];
        text = text.trim();
        if (text.length === 0) {
            return;
        }
        this.chunks.push(text);
    }
}
exports.ReaderState = ReaderState;
const regexp = /(([^:]+):)?(.*)/;
function resolve_namespace(name) {
    const m = name.match(regexp);
    if (!m) {
        throw new Error("Invalid match");
    }
    return {
        ns: m[2],
        tag: m[3]
    };
}
/**
 *
 * @example
 *  var parser = new Xml2Json({
 *       parser: {
 *           'person': {
 *               init: function(name,attrs) {
 *                   this.parent.root.obj = {};
 *                   this.obj =  this.parent.root.obj;
 *                   this.obj['name'] = attrs['name'];
 *               },
 *               parser: {
 *                   'address': {
 *                       finish: function(){
 *                           this.parent.obj['address'] = this.text;
 *                       }
 *                   }
 *               }
 *           }
 *       }
 *   });
 *
 * var xml_string =  "<employees>" +
 * "  <person name='John'>" +
 * "     <address>Paris</address>" +
 * "   </person>" +
 * "</employees>";
 *
 * parser.parseString(xml_string, function() {
 *       parser.obj.should.eql({name: 'John',address: 'Paris'});
 *       done();
 *   });
 */
class Xml2Json {
    currentLevel = 0;
    state_stack = [];
    current_state = null;
    constructor(options) {
        const state = options instanceof ReaderStateBase ? options : new ReaderState(options);
        state.root = this;
        this.state_stack = [];
        this.current_state = null;
        this._promote(state, 0);
    }
    parseString(xml_text) {
        return this.__parseInternal(xml_text);
    }
    /**
     * @private
     * @internal
     */
    _promote(new_state, level, name, attr) {
        attr = attr || {};
        this.state_stack.push({
            backup: {},
            state: this.current_state
        });
        const parent = this.current_state;
        this.current_state = new_state;
        this.current_state._on_init(name || "???", attr, parent, level, this);
    }
    /**
     * @private
     * @internal
     */
    _demote(cur_state, level, elementName) {
        ///  assert(this.current_state === cur_state);
        const { state, backup } = this.state_stack.pop();
        this.current_state = state;
        if (this.current_state) {
            this.current_state._on_endElement2(level, elementName);
        }
    }
    /**
     * @private
     * @internal
     */
    __parseInternal(data) {
        const parser = new lts_1.SaxLtx();
        this.currentLevel = 0;
        parser.on("startElement", (name, attrs) => {
            const tag_ns = resolve_namespace(name);
            this.currentLevel += 1;
            if (this.current_state) {
                this.current_state._on_startElement(this.currentLevel, tag_ns.tag, attrs);
            }
        });
        parser.on("endElement", (name) => {
            const tag_ns = resolve_namespace(name);
            if (this.current_state) {
                this.current_state._on_endElement(this.currentLevel, tag_ns.tag);
            }
            this.currentLevel -= 1;
            if (this.currentLevel === 0) {
                parser.emit("close");
            }
        });
        parser.on("text", (text) => {
            text = text.trim();
            if (text.length === 0) {
                return;
            }
            if (this.current_state) {
                this.current_state._on_text(text);
            }
        });
        parser.write(data);
        parser.end("");
        return this.current_state._pojo;
        /*
        return await new Promise((resolve) => {
            parser.once("close", () => {
                resolve((this.current_state! as any)._pojo);
            });
            //parser.write(data);
            parser.end(data);
        })*/
    }
}
exports.Xml2Json = Xml2Json;
//# sourceMappingURL=xml2json.js.map