"use strict";
/**
 * @module node-opcua-service-discovery
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BonjourHolder = void 0;
exports.acquireMulticastDNS = acquireMulticastDNS;
exports._announceServerOnMulticastSubnet = _announceServerOnMulticastSubnet;
const chalk_1 = __importDefault(require("chalk"));
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_object_registry_1 = require("node-opcua-object-registry");
const sterfive_bonjour_service_1 = require("sterfive-bonjour-service");
const announcement_to_service_config_1 = require("./announcement_to_service_config");
const tools_1 = require("./tools");
const debugLog = (0, node_opcua_debug_1.make_debugLog)("Bonjour");
const doDebug = (0, node_opcua_debug_1.checkDebugFlag)("Bonjour");
const errorLog = (0, node_opcua_debug_1.make_errorLog)("Bonjour");
const warningLog = (0, node_opcua_debug_1.make_warningLog)("Bonjour");
const registry = new node_opcua_object_registry_1.ObjectRegistry();
async function releaseMulticastDNS(bonjour) {
    await new Promise((resolve) => {
        bonjour.unpublishAll(() => {
            resolve();
        });
    });
    await new Promise((resolve) => {
        bonjour.destroy(() => {
            resolve();
        });
    });
    registry.unregister(bonjour);
}
function acquireMulticastDNS() {
    const bonjour = new sterfive_bonjour_service_1.Bonjour();
    registry.register(bonjour);
    return bonjour;
}
async function _announceServerOnMulticastSubnet(multicastDNS, serviceConfig) {
    return new Promise((resolve, reject) => {
        const port = serviceConfig.port;
        (0, node_opcua_assert_1.assert)(typeof port === "number");
        (0, node_opcua_assert_1.assert)(multicastDNS, "bonjour must have been initialized?");
        let timer;
        debugLog(chalk_1.default.cyan("  announceServerOnMulticastSubnet", (0, tools_1.serviceToString)(serviceConfig)));
        // waitServiceUp(serviceConfig, () => {
        //     // istanbul ignore next
        //     if (doDebug) {
        //         debugLog(chalk.cyan("  announcedOnMulticastSubnet done ", serviceToString(serviceConfig)));
        //     }
        //     //    resolve(service);
        // });
        const service = multicastDNS.publish({ ...serviceConfig, probe: false });
        function onError(err) {
            if (timer) {
                clearTimeout(timer);
                timer = undefined;
            }
            errorLog(" error during announcement ", err.message);
            service.removeListener("up", onUp);
            service.removeListener("error", onError);
            reject(err);
        }
        service.on("error", onError);
        const onUp = () => {
            if (timer) {
                clearTimeout(timer);
                timer = undefined;
            }
            debugLog("_announceServerOnMulticastSubnet: bonjour UP received ! ", (0, tools_1.serviceToString)(serviceConfig));
            service.removeListener("error", onError);
            service.removeListener("up", onUp);
            resolve(service);
        };
        service.on("up", onUp);
        // set a timer to ensure that "up" or "error" event is raised within a reasonable time period
        timer = setTimeout(() => {
            timer = undefined;
            service.removeListener("error", onError);
            service.removeListener("up", onUp);
            const err = new Error("Timeout waiting for bonjour to announce service " + serviceConfig.name);
            errorLog(err.message);
            reject(err);
        }, 10_000);
        service.start();
    });
}
class BonjourHolder {
    serviceConfig;
    #_multicastDNS;
    #_service;
    #pendingAnnouncement = false;
    /**
     *
     * @param announcement
     * @returns
     */
    async announcedOnMulticastSubnet(announcement) {
        debugLog(chalk_1.default.yellow("\n\nentering announcedOnMulticastSubnet"));
        const serviceConfig = (0, announcement_to_service_config_1.announcementToServiceConfig)(announcement);
        if (this.#_service && this.serviceConfig) {
            // verify that Announcement has changed
            if ((0, tools_1.isSameService)(serviceConfig, this.serviceConfig)) {
                debugLog(" Announcement ignored as it has been already made", announcement.name);
                debugLog("exiting announcedOnMulticastSubnet-2", false);
                return false; // nothing changed
            }
        }
        (0, node_opcua_assert_1.assert)(!this.#_multicastDNS, "already called ?");
        this.#_multicastDNS = acquireMulticastDNS();
        this.#pendingAnnouncement = true;
        this.serviceConfig = serviceConfig;
        this.#_service = await _announceServerOnMulticastSubnet(this.#_multicastDNS, serviceConfig);
        this.#pendingAnnouncement = false;
        debugLog(chalk_1.default.yellow("exiting announcedOnMulticastSubnet-3", true));
        return true;
    }
    isStarted() {
        return !!this.#_multicastDNS;
    }
    /**
     * @private
     */
    async stopAnnouncedOnMulticastSubnet() {
        if (this.#pendingAnnouncement) {
            debugLog(chalk_1.default.bgWhite.redBright("stopAnnnouncedOnMulticastSubnet is pending : let's wait a little bit and try again"));
            // wait until announcement is done
            await new Promise((resolve) => setTimeout(resolve, 500));
            return this.stopAnnouncedOnMulticastSubnet();
        }
        debugLog(chalk_1.default.green("\n\nentering stop_announcedOnMulticastSubnet = ", this.serviceConfig ? (0, tools_1.serviceToString)(this.serviceConfig) : "<null>"));
        if (!this.#_service) {
            debugLog(chalk_1.default.green("leaving stop_announcedOnMulticastSubnet = no service"));
            return;
        }
        // due to a wrong declaration of Service.stop in the d.ts file we
        // need to use a workaround here
        const that_service = this.#_service;
        const that_multicastDNS = this.#_multicastDNS;
        this.#_service = undefined;
        this.#_multicastDNS = undefined;
        this.serviceConfig = undefined;
        if (that_multicastDNS && that_service.stop) {
            await new Promise((resolve) => {
                that_service.stop((err) => {
                    debugLog(chalk_1.default.green("service stopped err=", err));
                    err && warningLog(err.message);
                    that_multicastDNS.unpublishAll(() => {
                        resolve();
                    });
                });
            });
            await releaseMulticastDNS(that_multicastDNS);
        }
        debugLog(chalk_1.default.green("leaving stop_announcedOnMulticastSubnet = done"));
        debugLog(chalk_1.default.green("leaving stop_announcedOnMulticastSubnet stop announcement completed"));
    }
}
exports.BonjourHolder = BonjourHolder;
//# sourceMappingURL=bonjourHolder.js.map