"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BonjourHolder = exports.serviceToString = void 0;
exports.acquireBonjour = acquireBonjour;
exports.releaseBonjour = releaseBonjour;
exports.acquireBonjour2 = acquireBonjour2;
exports.releaseBonjour2 = releaseBonjour2;
exports.announcementToServiceConfig = announcementToServiceConfig;
exports.isSameService = isSameService;
exports._announceServerOnMulticastSubnet = _announceServerOnMulticastSubnet;
/**
 * @module node-opcua-service-discovery
 */
// tslint:disable:no-console
const util_1 = require("util");
const chalk_1 = __importDefault(require("chalk"));
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_object_registry_1 = require("node-opcua-object-registry");
const sterfive_bonjour_service_1 = require("sterfive-bonjour-service");
const debugLog = (0, node_opcua_debug_1.make_debugLog)(__filename);
const doDebug = (0, node_opcua_debug_1.checkDebugFlag)(__filename);
const errorLog = (0, node_opcua_debug_1.make_errorLog)(__filename);
let gBonjour;
let gBonjourRefCount = 0;
const registry = new node_opcua_object_registry_1.ObjectRegistry();
function errorCallback(err) {
    errorLog(" ERROR received from Bonjour", err.message);
}
function acquireBonjour() {
    const bonjour = new sterfive_bonjour_service_1.Bonjour();
    registry.unregister(bonjour);
    return bonjour;
}
function releaseBonjour(bonjour, callback) {
    bonjour.unpublishAll(() => {
        bonjour.destroy(callback);
        registry.unregister(bonjour);
    });
}
function acquireBonjour2() {
    if (gBonjourRefCount === 0) {
        // will start the Bonjour service
        debugLog("Starting Bonjour");
        gBonjour = new sterfive_bonjour_service_1.Bonjour(undefined, errorCallback);
        registry.register(gBonjour);
    }
    gBonjourRefCount++;
    return gBonjour;
}
function releaseBonjour2(bonjour) {
    gBonjourRefCount--;
    (0, node_opcua_assert_1.assert)(gBonjourRefCount >= 0);
    if (gBonjourRefCount === 0) {
        if (!gBonjour) {
            throw new Error("internal error");
        }
        const tmp = gBonjour;
        gBonjour = undefined;
        // will stop the Bonjour service
        tmp.unpublishAll(() => {
            tmp.destroy();
            registry.unregister(tmp);
            debugLog("Releasing Bonjour");
        });
    }
}
function announcementToServiceConfig(announcement) {
    const serviceConfig = {
        name: announcement.name,
        port: announcement.port,
        protocol: "tcp",
        host: announcement.host,
        txt: {
            caps: announcement.capabilities.sort().join(","),
            path: announcement.path
        },
        type: "opcua-tcp"
    };
    return serviceConfig;
}
function isSameService(a, b) {
    if (!a && !b) {
        return true;
    }
    if (!a || !b) {
        return false;
    }
    return a.port === b.port && a.txt?.path === b.txt?.path && a.name === b.name && a.txt?.caps === b.txt?.caps;
}
const serviceToString = (service) => {
    return ("host" +
        service.host +
        " type=" +
        service.type +
        service.name +
        " on port " +
        service.port +
        " txt " +
        JSON.stringify(service.txt));
};
exports.serviceToString = serviceToString;
// function waitServiceUp(serviceConfig: ServiceConfig, callback: () => void) {
//     const multicastDNS = new Bonjour();
//     const browser = multicastDNS.find({
//         protocol: "tcp",
//         type: "opcua-tcp"
//     });
//     const onUp = (service: Service) => {
//         if (doDebug) {
//             debugLog(chalk.cyan("    waitServiceUp is up with  ", serviceToString(service)));
//         }
//         if (isSameService(service, serviceConfig)) {
//             browser.removeAllListeners("up");
//             multicastDNS.destroy();
//             callback();
//         }
//     };
//     browser.on("up", onUp);
// }
// function waitServiceDown(serviceConfig: ServiceConfig, callback: () => void) {
//     const multicastDNS = new Bonjour();
//     const browser = multicastDNS.find({
//         protocol: "tcp",
//         type: "opcua-tcp"
//     });
//     const onDown = (service: Service) => {
//         if (doDebug) {
//             debugLog(chalk.cyan("    waitServiceDown down with  ", serviceToString(service)));
//         }
//         if (isSameService(service, serviceConfig)) {
//             browser.removeAllListeners("down");
//             multicastDNS.destroy();
//             callback();
//         }
//     };
//     browser.on("down", onDown);
// }
async function _announceServerOnMulticastSubnet(multicastDNS, serviceConfig) {
    return new Promise((resolve, reject) => {
        const port = serviceConfig.port;
        (0, node_opcua_assert_1.assert)(typeof port === "number");
        (0, node_opcua_assert_1.assert)(multicastDNS, "bonjour must have been initialized?");
        debugLog(chalk_1.default.cyan("  announceServerOnMulticastSubnet", (0, exports.serviceToString)(serviceConfig)));
        // waitServiceUp(serviceConfig, () => {
        //     // istanbul ignore next
        //     if (doDebug) {
        //         debugLog(chalk.cyan("  announcedOnMulticastSubnet done ", serviceToString(serviceConfig)));
        //     }
        //     //    resolve(service);
        // });
        const service = multicastDNS.publish({ ...serviceConfig, probe: false });
        service.on("error", (err) => {
            debugLog("bonjour ERROR received ! ", err.message);
            debugLog("params = ", serviceConfig);
        });
        service.on("up", () => {
            debugLog("_announceServerOnMulticastSubnet: bonjour UP received ! ", (0, exports.serviceToString)(serviceConfig));
            resolve(service);
        });
        service.start();
    });
}
class BonjourHolder {
    constructor() {
        this.pendingAnnouncement = false;
    }
    /**
     *
     * @param announcement
     * @returns
     */
    async announcedOnMulticastSubnet(announcement) {
        debugLog(chalk_1.default.yellow("\n\nentering announcedOnMulticastSubnet"));
        const serviceConfig = announcementToServiceConfig(announcement);
        if (this._service && this.serviceConfig) {
            // verify that Announcement has changed
            if (isSameService(serviceConfig, this.serviceConfig)) {
                debugLog(" Announcement ignored as it has been already made", announcement.name);
                debugLog("exiting announcedOnMulticastSubnet-2", false);
                return false; // nothing changed
            }
        }
        (0, node_opcua_assert_1.assert)(!this._multicastDNS, "already called ?");
        this._multicastDNS = acquireBonjour();
        this.pendingAnnouncement = true;
        this.serviceConfig = serviceConfig;
        this._service = await _announceServerOnMulticastSubnet(this._multicastDNS, serviceConfig);
        this.pendingAnnouncement = false;
        debugLog(chalk_1.default.yellow("exiting announcedOnMulticastSubnet-3", true));
        return true;
    }
    isStarted() {
        return !!this._multicastDNS;
    }
    /**
     *
     * @param announcement
     * @param callback
     * @private
     */
    announcedOnMulticastSubnetWithCallback(announcement, callback) {
        callback(new Error("Internal Error"));
    }
    /**
     * @private
     */
    async stopAnnouncedOnMulticastSubnet() {
        if (this.pendingAnnouncement) {
            debugLog(chalk_1.default.bgWhite.redBright("stopAnnnouncedOnMulticastSubnet is pending : let's wait a little bit and try again"));
            // wait until announcement is done
            await new Promise((resolve) => setTimeout(resolve, 500));
            return this.stopAnnouncedOnMulticastSubnet();
        }
        debugLog(chalk_1.default.green("\n\nentering stop_announcedOnMulticastSubnet = ", this.serviceConfig ? (0, exports.serviceToString)(this.serviceConfig) : "<null>"));
        if (!this._service) {
            debugLog(chalk_1.default.green("leaving stop_announcedOnMulticastSubnet = no service"));
            return;
        }
        // due to a wrong declaration of Service.stop in the d.ts file we
        // need to use a workaround here
        const that_service = this._service;
        const that_multicastDNS = this._multicastDNS;
        this._service = undefined;
        this._multicastDNS = undefined;
        this.serviceConfig = undefined;
        const proxy = (callback) => {
            if (that_multicastDNS && that_service.stop) {
                // waitServiceDown(that_service, () => {
                //     debugLog(chalk.green("stop_announcedOnMulticastSubnet, ", serviceToString(that_service)));
                // });
                that_service.stop((err) => {
                    debugLog(chalk_1.default.green("service stopped err=", err));
                    that_multicastDNS.unpublishAll(() => {
                        releaseBonjour(that_multicastDNS, () => {
                            callback();
                        });
                    });
                });
                return;
            }
            else {
                callback();
            }
        };
        const stop = (0, util_1.promisify)(proxy);
        await stop.call(this);
        debugLog(chalk_1.default.green("leaving stop_announcedOnMulticastSubnet = done"));
        debugLog(chalk_1.default.green("leaving stop_announcedOnMulticastSubnet stop announcement completed"));
    }
    stopAnnouncedOnMulticastSubnetWithCallback(callback) {
        callback(new Error("Internal Error"));
    }
}
exports.BonjourHolder = BonjourHolder;
BonjourHolder.prototype.announcedOnMulticastSubnetWithCallback = (0, util_1.callbackify)(BonjourHolder.prototype.announcedOnMulticastSubnet);
BonjourHolder.prototype.stopAnnouncedOnMulticastSubnetWithCallback = (0, util_1.callbackify)(BonjourHolder.prototype.stopAnnouncedOnMulticastSubnet);
//# sourceMappingURL=bonjour.js.map