"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateFilter = validateFilter;
/**
 * @module node-opcua-server
 */
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_data_model_2 = require("node-opcua-data-model");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_service_filter_1 = require("node-opcua-service-filter");
const node_opcua_service_subscription_1 = require("node-opcua-service-subscription");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_basic_types_1 = require("node-opcua-basic-types");
const node_opcua_debug_1 = require("node-opcua-debug");
const warningLog = (0, node_opcua_debug_1.make_warningLog)(__filename);
function isNumberDataType(node) {
    const n = node.dataType;
    if (n.namespace === 0 && n.identifierType === node_opcua_nodeid_1.NodeIdType.NUMERIC && n.value < 22) {
        switch (n.value) {
            case node_opcua_basic_types_1.DataType.Float:
            case node_opcua_basic_types_1.DataType.Double:
            case node_opcua_basic_types_1.DataType.Byte:
            case node_opcua_basic_types_1.DataType.SByte:
            case node_opcua_basic_types_1.DataType.Int16:
            case node_opcua_basic_types_1.DataType.Int32:
            case node_opcua_basic_types_1.DataType.Int64:
            case node_opcua_basic_types_1.DataType.UInt16:
            case node_opcua_basic_types_1.DataType.UInt32:
            case node_opcua_basic_types_1.DataType.UInt64:
                return true;
            default:
                return false;
        }
    }
    const dataType = node.addressSpace.findDataType(node.dataType);
    const dataTypeNumber = node.addressSpace.findDataType("Number");
    return dataType.isSubtypeOf(dataTypeNumber);
}
function __validateDataChangeFilter(filter, itemToMonitor, node) {
    (0, node_opcua_assert_1.assert)(itemToMonitor.attributeId === node_opcua_data_model_1.AttributeIds.Value);
    if (node.nodeClass !== node_opcua_data_model_2.NodeClass.Variable) {
        return node_opcua_status_code_1.StatusCodes.BadNodeIdInvalid;
    }
    if (filter.deadbandType !== node_opcua_service_subscription_1.DeadbandType.None) {
        // if node is not Numerical=> DataChangeFilter
        (0, node_opcua_assert_1.assert)(node.dataType instanceof node_opcua_nodeid_1.NodeId);
        if (!isNumberDataType(node)) {
            return node_opcua_status_code_1.StatusCodes.BadFilterNotAllowed;
        }
    }
    if (filter.deadbandType === node_opcua_service_subscription_1.DeadbandType.Percent) {
        if (filter.deadbandValue < 0 || filter.deadbandValue > 100) {
            return node_opcua_status_code_1.StatusCodes.BadDeadbandFilterInvalid;
        }
        // node must also have a valid euRange
        if (!node.getChildByName("EURange", 0)) {
            warningLog(" node has no euRange ! Dead band Percent cannot be used on node " + node.nodeId.toString());
            return node_opcua_status_code_1.StatusCodes.BadMonitoredItemFilterUnsupported;
        }
    }
    return node_opcua_status_code_1.StatusCodes.Good;
}
function validateFilter(filter, itemToMonitor, node) {
    // handle filter information
    if (filter && filter instanceof node_opcua_service_filter_1.EventFilter && itemToMonitor.attributeId !== node_opcua_data_model_1.AttributeIds.EventNotifier) {
        // invalid filter on Event
        return node_opcua_status_code_1.StatusCodes.BadFilterNotAllowed;
    }
    if (filter && filter instanceof node_opcua_service_filter_1.DataChangeFilter && itemToMonitor.attributeId !== node_opcua_data_model_1.AttributeIds.Value) {
        // invalid DataChange filter on non Value Attribute
        return node_opcua_status_code_1.StatusCodes.BadFilterNotAllowed;
    }
    if (filter && itemToMonitor.attributeId !== node_opcua_data_model_1.AttributeIds.EventNotifier && itemToMonitor.attributeId !== node_opcua_data_model_1.AttributeIds.Value) {
        return node_opcua_status_code_1.StatusCodes.BadFilterNotAllowed;
    }
    if (filter instanceof node_opcua_service_filter_1.DataChangeFilter) {
        return __validateDataChangeFilter(filter, itemToMonitor, node);
    }
    return node_opcua_status_code_1.StatusCodes.Good;
}
//# sourceMappingURL=validate_filter.js.map