"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerEngine = void 0;
exports.setNextSubscriptionId = setNextSubscriptionId;
/**
 * @module node-opcua-server
 */
const events_1 = require("events");
const util_1 = require("util");
const async_1 = __importDefault(require("async"));
const chalk_1 = __importDefault(require("chalk"));
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_binary_stream_1 = require("node-opcua-binary-stream");
const node_opcua_address_space_1 = require("node-opcua-address-space");
const nodeJS_1 = require("node-opcua-address-space/nodeJS");
const node_opcua_common_1 = require("node-opcua-common");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_constants_1 = require("node-opcua-constants");
const node_opcua_date_time_1 = require("node-opcua-date-time");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_nodesets_1 = require("node-opcua-nodesets");
const node_opcua_object_registry_1 = require("node-opcua-object-registry");
const node_opcua_service_call_1 = require("node-opcua-service-call");
const node_opcua_service_subscription_1 = require("node-opcua-service-subscription");
const node_opcua_service_endpoints_1 = require("node-opcua-service-endpoints");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_types_1 = require("node-opcua-types");
const node_opcua_variant_1 = require("node-opcua-variant");
const history_server_capabilities_1 = require("./history_server_capabilities");
const monitored_item_1 = require("./monitored_item");
const server_capabilities_1 = require("./server_capabilities");
const server_publish_engine_1 = require("./server_publish_engine");
const server_publish_engine_for_orphan_subscriptions_1 = require("./server_publish_engine_for_orphan_subscriptions");
const server_session_1 = require("./server_session");
const server_subscription_1 = require("./server_subscription");
const sessions_compatible_for_transfer_1 = require("./sessions_compatible_for_transfer");
const addressSpace_accessor_1 = require("./addressSpace_accessor");
const debugLog = (0, node_opcua_debug_1.make_debugLog)(__filename);
const errorLog = (0, node_opcua_debug_1.make_errorLog)(__filename);
const warningLog = (0, node_opcua_debug_1.make_warningLog)(__filename);
const doDebug = (0, node_opcua_debug_1.checkDebugFlag)(__filename);
function upperCaseFirst(str) {
    return str.slice(0, 1).toUpperCase() + str.slice(1);
}
async function shutdownAndDisposeAddressSpace() {
    if (this.addressSpace) {
        await this.addressSpace.shutdown();
        this.addressSpace.dispose();
        delete this.addressSpace;
    }
}
function setSubscriptionDurable(inputArguments, context, callback) {
    // see https://reference.opcfoundation.org/v104/Core/docs/Part5/9.3/
    // https://reference.opcfoundation.org/v104/Core/docs/Part4/6.8/
    (0, node_opcua_assert_1.assert)(typeof callback === "function");
    const data = _getSubscription.call(this, inputArguments, context);
    if (data.statusCode)
        return callback(null, { statusCode: data.statusCode });
    const { subscription } = data;
    const lifetimeInHours = inputArguments[1].value;
    if (subscription.monitoredItemCount > 0) {
        // This is returned when a Subscription already contains MonitoredItems.
        return callback(null, { statusCode: node_opcua_status_code_1.StatusCodes.BadInvalidState });
    }
    /**
     * MonitoredItems are used to monitor Variable Values for data changes and event notifier
     * Objects for new Events. Subscriptions are used to combine data changes and events of
     * the assigned MonitoredItems to an optimized stream of network messages. A reliable
     * delivery is ensured as long as the lifetime of the Subscription and the queues in the
     * MonitoredItems are long enough for a network interruption between OPC UA Client and
     * Server. All queues that ensure reliable delivery are normally kept in memory and a
     * Server restart would delete them.
     * There are use cases where OPC UA Clients have no permanent network connection to the
     * OPC UA Server or where reliable delivery of data changes and events is necessary
     * even if the OPC UA Server is restarted or the network connection is interrupted
     * for a longer time.
     * To ensure this reliable delivery, the OPC UA Server must store collected data and
     * events in non-volatile memory until the OPC UA Client has confirmed reception.
     * It is possible that there will be data lost if the Server is not shut down gracefully
     * or in case of power failure. But the OPC UA Server should store the queues frequently
     * even if the Server is not shut down.
     * The Method SetSubscriptionDurable defined in OPC 10000-5 is used to set a Subscription
     * into this durable mode and to allow much longer lifetimes and queue sizes than for normal
     * Subscriptions. The Method shall be called before the MonitoredItems are created in the
     * durable Subscription. The Server shall verify that the Method is called within the
     * Session context of the Session that owns the Subscription.
     *
     * A value of 0 for the parameter lifetimeInHours requests the highest lifetime supported by the Server.
     */
    const highestLifetimeInHours = 24 * 100;
    const revisedLifetimeInHours = lifetimeInHours === 0 ? highestLifetimeInHours : Math.max(1, Math.min(lifetimeInHours, highestLifetimeInHours));
    // also adjust subscription life time
    const currentLifeTimeInHours = (subscription.lifeTimeCount * subscription.publishingInterval) / (1000 * 60 * 60);
    if (currentLifeTimeInHours < revisedLifetimeInHours) {
        const requestedLifetimeCount = Math.ceil((revisedLifetimeInHours * (1000 * 60 * 60)) / subscription.publishingInterval);
        subscription.modify({
            requestedMaxKeepAliveCount: subscription.maxKeepAliveCount,
            requestedPublishingInterval: subscription.publishingInterval,
            maxNotificationsPerPublish: subscription.maxNotificationsPerPublish,
            priority: subscription.priority,
            requestedLifetimeCount
        });
    }
    const callMethodResult = new node_opcua_service_call_1.CallMethodResult({
        statusCode: node_opcua_status_code_1.StatusCodes.Good,
        outputArguments: [{ dataType: node_opcua_variant_1.DataType.UInt32, arrayType: node_opcua_variant_1.VariantArrayType.Scalar, value: revisedLifetimeInHours }]
    });
    callback(null, callMethodResult);
}
function requestServerStateChange(inputArguments, context, callback) {
    (0, node_opcua_assert_1.assert)(Array.isArray(inputArguments));
    (0, node_opcua_assert_1.assert)(typeof callback === "function");
    (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(context, "session"), " expecting a session id in the context object");
    const session = context.session;
    if (!session) {
        return callback(null, { statusCode: node_opcua_status_code_1.StatusCodes.BadInternalError });
    }
    return callback(null, { statusCode: node_opcua_status_code_1.StatusCodes.BadNotImplemented });
}
function _getSubscription(inputArguments, context) {
    (0, node_opcua_assert_1.assert)(Array.isArray(inputArguments));
    (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(context, "session"), " expecting a session id in the context object");
    const session = context.session;
    if (!session) {
        return { statusCode: node_opcua_status_code_1.StatusCodes.BadInternalError };
    }
    const subscriptionId = inputArguments[0].value;
    const subscription = session.getSubscription(subscriptionId);
    if (!subscription) {
        // subscription may belongs to a different session  that ours
        if (this.findSubscription(subscriptionId)) {
            // if yes, then access to  Subscription data should be denied
            return { statusCode: node_opcua_status_code_1.StatusCodes.BadUserAccessDenied };
        }
        return { statusCode: node_opcua_status_code_1.StatusCodes.BadSubscriptionIdInvalid };
    }
    return { subscription };
}
function resendData(inputArguments, context, callback) {
    (0, node_opcua_assert_1.assert)(typeof callback === "function");
    const data = _getSubscription.call(this, inputArguments, context);
    if (data.statusCode)
        return callback(null, { statusCode: data.statusCode });
    const { subscription } = data;
    subscription
        .resendInitialValues()
        .then(() => {
        callback(null, { statusCode: node_opcua_status_code_1.StatusCodes.Good });
    })
        .catch((err) => callback(err));
}
// binding methods
function getMonitoredItemsId(inputArguments, context, callback) {
    (0, node_opcua_assert_1.assert)(typeof callback === "function");
    const data = _getSubscription.call(this, inputArguments, context);
    if (data.statusCode)
        return callback(null, { statusCode: data.statusCode });
    const { subscription } = data;
    const result = subscription.getMonitoredItems();
    (0, node_opcua_assert_1.assert)(result.statusCode);
    (0, node_opcua_assert_1.assert)(result.serverHandles.length === result.clientHandles.length);
    const callMethodResult = new node_opcua_service_call_1.CallMethodResult({
        statusCode: result.statusCode,
        outputArguments: [
            { dataType: node_opcua_variant_1.DataType.UInt32, arrayType: node_opcua_variant_1.VariantArrayType.Array, value: result.serverHandles },
            { dataType: node_opcua_variant_1.DataType.UInt32, arrayType: node_opcua_variant_1.VariantArrayType.Array, value: result.clientHandles }
        ]
    });
    callback(null, callMethodResult);
}
function __bindVariable(self, nodeId, options) {
    options = options || {};
    const variable = self.addressSpace.findNode(nodeId);
    if (variable && variable.bindVariable) {
        variable.bindVariable(options, true);
        (0, node_opcua_assert_1.assert)(typeof variable.asyncRefresh === "function");
        (0, node_opcua_assert_1.assert)(typeof variable.refreshFunc === "function");
    }
    else {
        warningLog("Warning: cannot bind object with id ", nodeId.toString(), " please check your nodeset.xml file or add this node programmatically");
    }
}
// note OPCUA 1.03 part 4 page 76
// The Server-assigned identifier for the Subscription (see 7.14 for IntegerId definition). This identifier shall
// be unique for the entire Server, not just for the Session, in order to allow the Subscription to be transferred
// to another Session using the TransferSubscriptions service.
// After Server start-up the generation of subscriptionIds should start from a random IntegerId or continue from
// the point before the restart.
let next_subscriptionId = Math.ceil(Math.random() * 1000000);
function setNextSubscriptionId(n) {
    next_subscriptionId = Math.max(n, 1);
}
function _get_next_subscriptionId() {
    debugLog(" next_subscriptionId = ", next_subscriptionId);
    return next_subscriptionId++;
}
/**
 *
 */
class ServerEngine extends events_1.EventEmitter {
    static registry = new node_opcua_object_registry_1.ObjectRegistry();
    isAuditing;
    serverDiagnosticsSummary;
    serverDiagnosticsEnabled;
    serverCapabilities;
    historyServerCapabilities;
    serverConfiguration;
    clientDescription;
    addressSpace;
    addressSpaceAccessor = null;
    // pseudo private
    _internalState;
    _sessions;
    _closedSessions;
    _orphanPublishEngine;
    _shutdownTasks;
    _applicationUri;
    _expectedShutdownTime;
    _serverStatus;
    _globalCounter = { totalMonitoredItemCount: 0 };
    constructor(options) {
        super();
        options = options || { applicationUri: "" };
        options.buildInfo = options.buildInfo || {};
        ServerEngine.registry.register(this);
        this._sessions = {};
        this._closedSessions = {};
        this._orphanPublishEngine = undefined; // will be constructed on demand
        this.isAuditing = typeof options.isAuditing === "boolean" ? options.isAuditing : false;
        options.buildInfo.buildDate = options.buildInfo.buildDate || new Date();
        // ---------------------------------------------------- ServerStatusDataType
        this._serverStatus = new node_opcua_common_1.ServerStatusDataType({
            buildInfo: options.buildInfo,
            currentTime: new Date(),
            secondsTillShutdown: 0,
            shutdownReason: { text: "" },
            startTime: new Date(),
            state: node_opcua_common_1.ServerState.NoConfiguration
        });
        // --------------------------------------------------- ServerCapabilities
        options.serverCapabilities = options.serverCapabilities || {};
        options.serverConfiguration = options.serverConfiguration || {
            supportedPrivateKeyFormat: ["PEM"]
        };
        // https://profiles.opcfoundation.org/profile
        options.serverCapabilities.serverProfileArray = options.serverCapabilities.serverProfileArray || [
            "http://opcfoundation.org/UA-Profile/Server/Standard", // Standard UA Server Profile",
            "http://opcfoundation.org/UA-Profile/Server/DataAccess",
            "http://opcfoundation.org/UA-Profile/Server/Events",
            "http://opcfoundation.org/UA-Profile/Client/HistoricalAccess",
            "http://opcfoundation.org/UA-Profile/Server/Methods",
            "http://opcfoundation.org/UA-Profile/Server/StandardEventSubscription",
            "http://opcfoundation.org/UA-Profile/Transport/uatcp-uasc-uabinary",
            "http://opcfoundation.org/UA-Profile/Server/FileAccess",
            "http://opcfoundation.org/UA-Profile/Server/StateMachine"
            // "http://opcfoundation.org/UA-Profile/Transport/wss-uajson",
            // "http://opcfoundation.org/UA-Profile/Transport/wss-uasc-uabinary"
            // "http://opcfoundation.org/UA-Profile/Server/DurableSubscription"
            // "http://opcfoundation.org/UA-Profile/Server/ReverseConnect",
            // "http://opcfoundation.org/UAProfile/Server/NodeManagement",
            //  "Embedded UA Server Profile",
            // "Micro Embedded Device Server Profile",
            // "Nano Embedded Device Server Profile"
        ];
        options.serverCapabilities.localeIdArray = options.serverCapabilities.localeIdArray || ["en-EN", "fr-FR"];
        this.serverCapabilities = new server_capabilities_1.ServerCapabilities(options.serverCapabilities);
        // to do when spec is clear about what goes here!
        // spec 1.04 says (in Part 4 7.33 SignedSoftwareCertificate
        // Note: Details on SoftwareCertificates need to be defined in a future version.
        this.serverCapabilities.softwareCertificates = [
        // new SignedSoftwareCertificate({})
        ];
        // make sure minSupportedSampleRate matches MonitoredItem.minimumSamplingInterval
        this.serverCapabilities.__defineGetter__("minSupportedSampleRate", () => {
            return options.serverCapabilities?.minSupportedSampleRate || monitored_item_1.MonitoredItem.minimumSamplingInterval;
        });
        this.serverConfiguration = options.serverConfiguration;
        this.historyServerCapabilities = new history_server_capabilities_1.HistoryServerCapabilities(options.historyServerCapabilities);
        // --------------------------------------------------- serverDiagnosticsSummary extension Object
        this.serverDiagnosticsSummary = new node_opcua_common_1.ServerDiagnosticsSummaryDataType();
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(this.serverDiagnosticsSummary, "currentSessionCount"));
        // note spelling is different for serverDiagnosticsSummary.currentSubscriptionCount
        //      and sessionDiagnostics.currentSubscriptionsCount ( with an s)
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(this.serverDiagnosticsSummary, "currentSubscriptionCount"));
        this.serverDiagnosticsSummary.__defineGetter__("currentSubscriptionCount", () => {
            // currentSubscriptionCount returns the total number of subscriptions
            // that are currently active on all sessions
            let counter = 0;
            Object.values(this._sessions).forEach((session) => {
                counter += session.currentSubscriptionCount;
            });
            // we also need to add the orphan subscriptions
            counter += this._orphanPublishEngine ? this._orphanPublishEngine.subscriptions.length : 0;
            return counter;
        });
        this._internalState = "creating";
        this.setServerState(node_opcua_common_1.ServerState.NoConfiguration);
        this.addressSpace = null;
        this._shutdownTasks = [];
        this._applicationUri = "";
        if (typeof options.applicationUri === "function") {
            this.__defineGetter__("_applicationUri", options.applicationUri);
        }
        else {
            this._applicationUri = options.applicationUri || "<unset _applicationUri>";
        }
        options.serverDiagnosticsEnabled = Object.prototype.hasOwnProperty.call(options, "serverDiagnosticsEnable")
            ? options.serverDiagnosticsEnabled
            : true;
        this.serverDiagnosticsEnabled = options.serverDiagnosticsEnabled;
    }
    isStarted() {
        return !!this._serverStatus;
    }
    dispose() {
        this.addressSpace = null;
        (0, node_opcua_assert_1.assert)(Object.keys(this._sessions).length === 0, "ServerEngine#_sessions not empty");
        this._sessions = {};
        // todo fix me
        this._closedSessions = {};
        (0, node_opcua_assert_1.assert)(Object.keys(this._closedSessions).length === 0, "ServerEngine#_closedSessions not empty");
        this._closedSessions = {};
        if (this._orphanPublishEngine) {
            this._orphanPublishEngine.dispose();
            this._orphanPublishEngine = undefined;
        }
        this._shutdownTasks = [];
        this._serverStatus = null;
        this._internalState = "disposed";
        this.removeAllListeners();
        ServerEngine.registry.unregister(this);
    }
    get startTime() {
        return this._serverStatus.startTime;
    }
    get currentTime() {
        return this._serverStatus.currentTime;
    }
    get buildInfo() {
        return this._serverStatus.buildInfo;
    }
    /**
     * register a function that will be called when the server will perform its shut down.
     */
    registerShutdownTask(task) {
        (0, node_opcua_assert_1.assert)(typeof task === "function");
        this._shutdownTasks.push(task);
    }
    /**
     */
    async shutdown() {
        debugLog("ServerEngine#shutdown");
        this._internalState = "shutdown";
        this.setServerState(node_opcua_common_1.ServerState.Shutdown);
        // delete any existing sessions
        const tokens = Object.keys(this._sessions).map((key) => {
            const session = this._sessions[key];
            return session.authenticationToken;
        });
        // delete and close any orphan subscriptions
        if (this._orphanPublishEngine) {
            this._orphanPublishEngine.shutdown();
        }
        for (const token of tokens) {
            this.closeSession(token, true, "Terminated");
        }
        // all sessions must have been terminated
        (0, node_opcua_assert_1.assert)(this.currentSessionCount === 0);
        // all subscriptions must have been terminated
        (0, node_opcua_assert_1.assert)(this.currentSubscriptionCount === 0, "all subscriptions must have been terminated");
        this._shutdownTasks.push(shutdownAndDisposeAddressSpace);
        // perform registerShutdownTask
        for (const task of this._shutdownTasks) {
            await task.call(this);
        }
        this.setServerState(node_opcua_common_1.ServerState.Invalid);
        this.dispose();
    }
    /**
     * the number of active sessions
     */
    get currentSessionCount() {
        return this.serverDiagnosticsSummary.currentSessionCount;
    }
    /**
     * the cumulated number of sessions that have been opened since this object exists
     */
    get cumulatedSessionCount() {
        return this.serverDiagnosticsSummary.cumulatedSessionCount;
    }
    /**
     * the number of active subscriptions.
     */
    get currentSubscriptionCount() {
        return this.serverDiagnosticsSummary.currentSubscriptionCount;
    }
    /**
     * the cumulated number of subscriptions that have been created since this object exists
     */
    get cumulatedSubscriptionCount() {
        return this.serverDiagnosticsSummary.cumulatedSubscriptionCount;
    }
    get rejectedSessionCount() {
        return this.serverDiagnosticsSummary.rejectedSessionCount;
    }
    get rejectedRequestsCount() {
        return this.serverDiagnosticsSummary.rejectedRequestsCount;
    }
    get sessionAbortCount() {
        return this.serverDiagnosticsSummary.sessionAbortCount;
    }
    get sessionTimeoutCount() {
        return this.serverDiagnosticsSummary.sessionTimeoutCount;
    }
    get publishingIntervalCount() {
        return this.serverDiagnosticsSummary.publishingIntervalCount;
    }
    incrementSessionTimeoutCount() {
        if (this.serverDiagnosticsSummary && this.serverDiagnosticsEnabled) {
            // The requests include all Services defined in Part 4 of the OPC UA Specification, also requests to create sessions. This number includes the securityRejectedRequestsCount.
            this.serverDiagnosticsSummary.sessionTimeoutCount += 1;
        }
    }
    incrementSessionAbortCount() {
        if (this.serverDiagnosticsSummary && this.serverDiagnosticsEnabled) {
            // The requests include all Services defined in Part 4 of the OPC UA Specification, also requests to create sessions. This number includes the securityRejectedRequestsCount.
            this.serverDiagnosticsSummary.sessionAbortCount += 1;
        }
    }
    incrementRejectedRequestsCount() {
        if (this.serverDiagnosticsSummary && this.serverDiagnosticsEnabled) {
            // The requests include all Services defined in Part 4 of the OPC UA Specification, also requests to create sessions. This number includes the securityRejectedRequestsCount.
            this.serverDiagnosticsSummary.rejectedRequestsCount += 1;
        }
    }
    /**
     * increment rejected session count (also increment rejected requests count)
     */
    incrementRejectedSessionCount() {
        if (this.serverDiagnosticsSummary && this.serverDiagnosticsEnabled) {
            // The requests include all Services defined in Part 4 of the OPC UA Specification, also requests to create sessions. This number includes the securityRejectedRequestsCount.
            this.serverDiagnosticsSummary.rejectedSessionCount += 1;
        }
        this.incrementRejectedRequestsCount();
    }
    incrementSecurityRejectedRequestsCount() {
        if (this.serverDiagnosticsSummary && this.serverDiagnosticsEnabled) {
            // The requests include all Services defined in Part 4 of the OPC UA Specification, also requests to create sessions. This number includes the securityRejectedRequestsCount.
            this.serverDiagnosticsSummary.securityRejectedRequestsCount += 1;
        }
        this.incrementRejectedRequestsCount();
    }
    /**
     * increment rejected session count (also increment rejected requests count)
     */
    incrementSecurityRejectedSessionCount() {
        if (this.serverDiagnosticsSummary && this.serverDiagnosticsEnabled) {
            // The requests include all Services defined in Part 4 of the OPC UA Specification, also requests to create sessions. This number includes the securityRejectedRequestsCount.
            this.serverDiagnosticsSummary.securityRejectedSessionCount += 1;
        }
        this.incrementSecurityRejectedRequestsCount();
    }
    setShutdownTime(date) {
        this._expectedShutdownTime = date;
    }
    setShutdownReason(reason) {
        this.addressSpace?.rootFolder.objects.server.serverStatus.shutdownReason.setValueFromSource({
            dataType: node_opcua_variant_1.DataType.LocalizedText,
            value: (0, node_opcua_data_model_1.coerceLocalizedText)(reason)
        });
    }
    /**
     * @return the approximate number of seconds until the server will be shut down. The
     * value is only relevant once the state changes into SHUTDOWN.
     */
    secondsTillShutdown() {
        if (!this._expectedShutdownTime) {
            return 0;
        }
        // ToDo: implement a correct solution here
        const now = Date.now();
        return Math.max(0, Math.ceil((this._expectedShutdownTime.getTime() - now) / 1000));
    }
    /**
     * the name of the server
     */
    get serverName() {
        return this._serverStatus.buildInfo.productName;
    }
    /**
     * the server urn
     */
    get serverNameUrn() {
        return this._applicationUri;
    }
    /**
     * the urn of the server namespace
     */
    get serverNamespaceUrn() {
        return this._applicationUri; // "urn:" + engine.serverName;
    }
    get serverStatus() {
        return this._serverStatus;
    }
    setServerState(serverState) {
        (0, node_opcua_assert_1.assert)(serverState !== null && serverState !== undefined);
        this.addressSpace?.rootFolder?.objects?.server?.serverStatus?.state?.setValueFromSource({
            dataType: node_opcua_variant_1.DataType.Int32,
            value: serverState
        });
    }
    getServerDiagnosticsEnabledFlag() {
        const server = this.addressSpace.rootFolder.objects.server;
        const serverDiagnostics = server.getComponentByName("ServerDiagnostics");
        if (!serverDiagnostics) {
            return false;
        }
        return serverDiagnostics.readValue().value.value;
    }
    /**
     *
     */
    initialize(options, callback) {
        (0, node_opcua_assert_1.assert)(!this.addressSpace); // check that 'initialize' has not been already called
        this._internalState = "initializing";
        options = options || {};
        (0, node_opcua_assert_1.assert)(typeof callback === "function");
        options.nodeset_filename = options.nodeset_filename || node_opcua_nodesets_1.nodesets.standard;
        const startTime = new Date();
        debugLog("Loading ", options.nodeset_filename, "...");
        this.addressSpace = node_opcua_address_space_1.AddressSpace.create();
        this.addressSpaceAccessor = new addressSpace_accessor_1.AddressSpaceAccessor(this.addressSpace);
        if (!options.skipOwnNamespace) {
            // register namespace 1 (our namespace);
            const serverNamespace = this.addressSpace.registerNamespace(this.serverNamespaceUrn);
            (0, node_opcua_assert_1.assert)(serverNamespace.index === 1);
        }
        // eslint-disable-next-line max-statements
        (0, nodeJS_1.generateAddressSpace)(this.addressSpace, options.nodeset_filename)
            .catch((err) => {
            console.log(err.message);
            callback(err);
        })
            .then(() => {
            /* istanbul ignore next */
            if (!this.addressSpace) {
                throw new Error("Internal error");
            }
            const addressSpace = this.addressSpace;
            const endTime = new Date();
            debugLog("Loading ", options.nodeset_filename, " done : ", endTime.getTime() - startTime.getTime(), " ms");
            const bindVariableIfPresent = (nodeId, opts) => {
                (0, node_opcua_assert_1.assert)(!nodeId.isEmpty());
                const obj = addressSpace.findNode(nodeId);
                if (obj) {
                    __bindVariable(this, nodeId, opts);
                }
                return obj;
            };
            // -------------------------------------------- install default get/put handler
            const server_NamespaceArray_Id = (0, node_opcua_nodeid_1.makeNodeId)(node_opcua_constants_1.VariableIds.Server_NamespaceArray); // ns=0;i=2255
            bindVariableIfPresent(server_NamespaceArray_Id, {
                get() {
                    return new node_opcua_variant_1.Variant({
                        arrayType: node_opcua_variant_1.VariantArrayType.Array,
                        dataType: node_opcua_variant_1.DataType.String,
                        value: addressSpace.getNamespaceArray().map((x) => x.namespaceUri)
                    });
                },
                set: null // read only
            });
            const server_NameUrn_var = new node_opcua_variant_1.Variant({
                arrayType: node_opcua_variant_1.VariantArrayType.Array,
                dataType: node_opcua_variant_1.DataType.String,
                value: [
                    this.serverNameUrn // this is us !
                ]
            });
            const server_ServerArray_Id = (0, node_opcua_nodeid_1.makeNodeId)(node_opcua_constants_1.VariableIds.Server_ServerArray); // ns=0;i=2254
            bindVariableIfPresent(server_ServerArray_Id, {
                get() {
                    return server_NameUrn_var;
                },
                set: null // read only
            });
            // fix DefaultUserRolePermissions and DefaultUserRolePermissions
            // of namespaces
            const namespaces = (0, node_opcua_nodeid_1.makeNodeId)(node_opcua_constants_1.ObjectIds.Server_Namespaces);
            const namespacesNode = addressSpace.findNode(namespaces);
            if (namespacesNode) {
                for (const ns of namespacesNode.getComponents()) {
                    const defaultUserRolePermissions = ns.getChildByName("DefaultUserRolePermissions");
                    if (defaultUserRolePermissions) {
                        defaultUserRolePermissions.setValueFromSource({ dataType: node_opcua_variant_1.DataType.Null });
                    }
                    const defaultRolePermissions = ns.getChildByName("DefaultRolePermissions");
                    if (defaultRolePermissions) {
                        defaultRolePermissions.setValueFromSource({ dataType: node_opcua_variant_1.DataType.Null });
                    }
                }
            }
            const bindStandardScalar = (id, dataType, func, setter_func) => {
                (0, node_opcua_assert_1.assert)(typeof id === "number", "expecting id to be a number");
                (0, node_opcua_assert_1.assert)(typeof func === "function");
                (0, node_opcua_assert_1.assert)(typeof setter_func === "function" || !setter_func);
                (0, node_opcua_assert_1.assert)(dataType !== null); // check invalid dataType
                let setter_func2 = null;
                if (setter_func) {
                    setter_func2 = (variant) => {
                        const variable2 = !!variant.value;
                        setter_func(variable2);
                        return node_opcua_status_code_1.StatusCodes.Good;
                    };
                }
                const nodeId = (0, node_opcua_nodeid_1.makeNodeId)(id);
                // make sur the provided function returns a valid value for the variant type
                // This test may not be exhaustive but it will detect obvious mistakes.
                /* istanbul ignore next */
                if (!(0, node_opcua_variant_1.isValidVariant)(node_opcua_variant_1.VariantArrayType.Scalar, dataType, func())) {
                    errorLog("func", func());
                    throw new Error("bindStandardScalar : func doesn't provide an value of type " + node_opcua_variant_1.DataType[dataType]);
                }
                return bindVariableIfPresent(nodeId, {
                    get() {
                        return new node_opcua_variant_1.Variant({
                            arrayType: node_opcua_variant_1.VariantArrayType.Scalar,
                            dataType,
                            value: func()
                        });
                    },
                    set: setter_func2
                });
            };
            const bindStandardArray = (id, variantDataType, dataType, func) => {
                (0, node_opcua_assert_1.assert)(typeof func === "function");
                (0, node_opcua_assert_1.assert)(variantDataType !== null); // check invalid dataType
                const nodeId = (0, node_opcua_nodeid_1.makeNodeId)(id);
                // make sur the provided function returns a valid value for the variant type
                // This test may not be exhaustive but it will detect obvious mistakes.
                (0, node_opcua_assert_1.assert)((0, node_opcua_variant_1.isValidVariant)(node_opcua_variant_1.VariantArrayType.Array, variantDataType, func()));
                bindVariableIfPresent(nodeId, {
                    get() {
                        const value = func();
                        (0, node_opcua_assert_1.assert)(Array.isArray(value));
                        return new node_opcua_variant_1.Variant({
                            arrayType: node_opcua_variant_1.VariantArrayType.Array,
                            dataType: variantDataType,
                            value
                        });
                    },
                    set: null // read only
                });
            };
            bindStandardScalar(node_opcua_constants_1.VariableIds.Server_EstimatedReturnTime, node_opcua_variant_1.DataType.DateTime, () => (0, node_opcua_date_time_1.getMinOPCUADate)());
            // TimeZoneDataType
            const timeZoneDataType = addressSpace.findDataType((0, node_opcua_nodeid_1.resolveNodeId)(node_opcua_constants_1.DataTypeIds.TimeZoneDataType));
            const timeZone = new node_opcua_types_1.TimeZoneDataType({
                daylightSavingInOffset: /* boolean*/ false,
                offset: /* int16 */ 0
            });
            bindStandardScalar(node_opcua_constants_1.VariableIds.Server_LocalTime, node_opcua_variant_1.DataType.ExtensionObject, () => {
                return timeZone;
            });
            bindStandardScalar(node_opcua_constants_1.VariableIds.Server_ServiceLevel, node_opcua_variant_1.DataType.Byte, () => {
                return 255;
            });
            bindStandardScalar(node_opcua_constants_1.VariableIds.Server_Auditing, node_opcua_variant_1.DataType.Boolean, () => {
                return this.isAuditing;
            });
            // eslint-disable-next-line @typescript-eslint/no-this-alias
            const engine = this;
            const makeNotReadableIfEnabledFlagIsFalse = (variable) => {
                const originalIsReadable = variable.isReadable;
                variable.isUserReadable = checkReadableFlag;
                function checkReadableFlag(context) {
                    const isEnabled = engine.serverDiagnosticsEnabled;
                    return originalIsReadable.call(this, context) && isEnabled;
                }
                for (const c of variable.getAggregates()) {
                    if (c.nodeClass === node_opcua_data_model_1.NodeClass.Variable) {
                        makeNotReadableIfEnabledFlagIsFalse(c);
                    }
                }
            };
            const bindServerDiagnostics = () => {
                bindStandardScalar(node_opcua_constants_1.VariableIds.Server_ServerDiagnostics_EnabledFlag, node_opcua_variant_1.DataType.Boolean, () => {
                    return this.serverDiagnosticsEnabled;
                }, (newFlag) => {
                    this.serverDiagnosticsEnabled = newFlag;
                });
                const nodeId = (0, node_opcua_nodeid_1.makeNodeId)(node_opcua_constants_1.VariableIds.Server_ServerDiagnostics_ServerDiagnosticsSummary);
                const serverDiagnosticsSummaryNode = addressSpace.findNode(nodeId);
                if (serverDiagnosticsSummaryNode) {
                    serverDiagnosticsSummaryNode.bindExtensionObject(this.serverDiagnosticsSummary);
                    this.serverDiagnosticsSummary = serverDiagnosticsSummaryNode.$extensionObject;
                    makeNotReadableIfEnabledFlagIsFalse(serverDiagnosticsSummaryNode);
                }
            };
            const bindServerStatus = () => {
                const serverStatusNode = addressSpace.findNode((0, node_opcua_nodeid_1.makeNodeId)(node_opcua_constants_1.VariableIds.Server_ServerStatus));
                if (!serverStatusNode) {
                    return;
                }
                if (serverStatusNode) {
                    serverStatusNode.bindExtensionObject(this._serverStatus);
                    serverStatusNode.minimumSamplingInterval = 1000;
                }
                const currentTimeNode = addressSpace.findNode((0, node_opcua_nodeid_1.makeNodeId)(node_opcua_constants_1.VariableIds.Server_ServerStatus_CurrentTime));
                if (currentTimeNode) {
                    currentTimeNode.minimumSamplingInterval = 1000;
                }
                const secondsTillShutdown = addressSpace.findNode((0, node_opcua_nodeid_1.makeNodeId)(node_opcua_constants_1.VariableIds.Server_ServerStatus_SecondsTillShutdown));
                if (secondsTillShutdown) {
                    secondsTillShutdown.minimumSamplingInterval = 1000;
                }
                (0, node_opcua_assert_1.assert)(serverStatusNode.$extensionObject);
                serverStatusNode.$extensionObject = new Proxy(serverStatusNode.$extensionObject, {
                    get(target, prop) {
                        if (prop === "currentTime") {
                            serverStatusNode.currentTime.touchValue();
                            return new Date();
                        }
                        else if (prop === "secondsTillShutdown") {
                            serverStatusNode.secondsTillShutdown.touchValue();
                            return engine.secondsTillShutdown();
                        }
                        return target[prop];
                    }
                });
                this._serverStatus = serverStatusNode.$extensionObject;
            };
            const bindServerCapabilities = () => {
                bindStandardArray(node_opcua_constants_1.VariableIds.Server_ServerCapabilities_ServerProfileArray, node_opcua_variant_1.DataType.String, node_opcua_variant_1.DataType.String, () => {
                    return this.serverCapabilities.serverProfileArray;
                });
                bindStandardArray(node_opcua_constants_1.VariableIds.Server_ServerCapabilities_LocaleIdArray, node_opcua_variant_1.DataType.String, "LocaleId", () => {
                    return this.serverCapabilities.localeIdArray;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.Server_ServerCapabilities_MinSupportedSampleRate, node_opcua_variant_1.DataType.Double, () => {
                    return Math.max(this.serverCapabilities.minSupportedSampleRate, server_capabilities_1.defaultServerCapabilities.minSupportedSampleRate);
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.Server_ServerCapabilities_MaxBrowseContinuationPoints, node_opcua_variant_1.DataType.UInt16, () => {
                    return this.serverCapabilities.maxBrowseContinuationPoints;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.Server_ServerCapabilities_MaxQueryContinuationPoints, node_opcua_variant_1.DataType.UInt16, () => {
                    return this.serverCapabilities.maxQueryContinuationPoints;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.Server_ServerCapabilities_MaxHistoryContinuationPoints, node_opcua_variant_1.DataType.UInt16, () => {
                    return this.serverCapabilities.maxHistoryContinuationPoints;
                });
                // new in 1.05
                bindStandardScalar(node_opcua_constants_1.VariableIds.Server_ServerCapabilities_MaxSessions, node_opcua_variant_1.DataType.UInt32, () => {
                    return this.serverCapabilities.maxSessions;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.Server_ServerCapabilities_MaxSubscriptions, node_opcua_variant_1.DataType.UInt32, () => {
                    return this.serverCapabilities.maxSubscriptions;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.Server_ServerCapabilities_MaxMonitoredItems, node_opcua_variant_1.DataType.UInt32, () => {
                    return this.serverCapabilities.maxMonitoredItems;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.Server_ServerCapabilities_MaxSubscriptionsPerSession, node_opcua_variant_1.DataType.UInt32, () => {
                    return this.serverCapabilities.maxSubscriptionsPerSession;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.Server_ServerCapabilities_MaxSelectClauseParameters, node_opcua_variant_1.DataType.UInt32, () => {
                    return this.serverCapabilities.maxSelectClauseParameters;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.Server_ServerCapabilities_MaxWhereClauseParameters, node_opcua_variant_1.DataType.UInt32, () => {
                    return this.serverCapabilities.maxWhereClauseParameters;
                });
                //bindStandardArray(VariableIds.Server_ServerCapabilities_ConformanceUnits, DataType.QualifiedName, () => {
                //    return this.serverCapabilities.conformanceUnits;
                //});
                bindStandardScalar(node_opcua_constants_1.VariableIds.Server_ServerCapabilities_MaxMonitoredItemsPerSubscription, node_opcua_variant_1.DataType.UInt32, () => {
                    return this.serverCapabilities.maxMonitoredItemsPerSubscription;
                });
                // added by DI : Server-specific period of time in milliseconds until the Server will revoke a lock.
                // TODO bindStandardScalar(VariableIds.Server_ServerCapabilities_MaxInactiveLockTime,
                // TODO     DataType.UInt16, function () {
                // TODO         return self.serverCapabilities.maxInactiveLockTime;
                // TODO });
                bindStandardArray(node_opcua_constants_1.VariableIds.Server_ServerCapabilities_SoftwareCertificates, node_opcua_variant_1.DataType.ExtensionObject, "SoftwareCertificates", () => {
                    return this.serverCapabilities.softwareCertificates;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.Server_ServerCapabilities_MaxArrayLength, node_opcua_variant_1.DataType.UInt32, () => {
                    return Math.min(this.serverCapabilities.maxArrayLength, node_opcua_variant_1.Variant.maxArrayLength);
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.Server_ServerCapabilities_MaxStringLength, node_opcua_variant_1.DataType.UInt32, () => {
                    return Math.min(this.serverCapabilities.maxStringLength, node_opcua_binary_stream_1.BinaryStream.maxStringLength);
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.Server_ServerCapabilities_MaxByteStringLength, node_opcua_variant_1.DataType.UInt32, () => {
                    return Math.min(this.serverCapabilities.maxByteStringLength, node_opcua_binary_stream_1.BinaryStream.maxByteStringLength);
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.Server_ServerCapabilities_MaxMonitoredItemsQueueSize, node_opcua_variant_1.DataType.UInt32, () => {
                    return Math.max(1, this.serverCapabilities.maxMonitoredItemsQueueSize);
                });
                const bindOperationLimits = (operationLimits) => {
                    (0, node_opcua_assert_1.assert)(operationLimits !== null && typeof operationLimits === "object");
                    const keys = Object.keys(operationLimits);
                    keys.forEach((key) => {
                        const uid = "Server_ServerCapabilities_OperationLimits_" + upperCaseFirst(key);
                        const nodeId = (0, node_opcua_nodeid_1.makeNodeId)(node_opcua_constants_1.VariableIds[uid]);
                        (0, node_opcua_assert_1.assert)(!nodeId.isEmpty());
                        bindStandardScalar(node_opcua_constants_1.VariableIds[uid], node_opcua_variant_1.DataType.UInt32, () => {
                            return operationLimits[key];
                        });
                    });
                };
                bindOperationLimits(this.serverCapabilities.operationLimits);
                // i=2399 [ProgramStateMachineType_ProgramDiagnostics];
                function fix_ProgramStateMachineType_ProgramDiagnostics() {
                    const nodeId = (0, node_opcua_nodeid_1.coerceNodeId)("i=2399"); // ProgramStateMachineType_ProgramDiagnostics
                    const variable = addressSpace.findNode(nodeId);
                    if (variable) {
                        variable.$extensionObject = new node_opcua_types_1.ProgramDiagnosticDataType({});
                        //  variable.setValueFromSource({
                        //     dataType: DataType.ExtensionObject,
                        //     //     value: new ProgramDiagnostic2DataType()
                        //     value: new ProgramDiagnosticDataType({})
                        // });
                    }
                }
                fix_ProgramStateMachineType_ProgramDiagnostics();
            };
            const bindHistoryServerCapabilities = () => {
                bindStandardScalar(node_opcua_constants_1.VariableIds.HistoryServerCapabilities_MaxReturnDataValues, node_opcua_variant_1.DataType.UInt32, () => {
                    return this.historyServerCapabilities.maxReturnDataValues;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.HistoryServerCapabilities_MaxReturnEventValues, node_opcua_variant_1.DataType.UInt32, () => {
                    return this.historyServerCapabilities.maxReturnEventValues;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.HistoryServerCapabilities_AccessHistoryDataCapability, node_opcua_variant_1.DataType.Boolean, () => {
                    return this.historyServerCapabilities.accessHistoryDataCapability;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.HistoryServerCapabilities_AccessHistoryEventsCapability, node_opcua_variant_1.DataType.Boolean, () => {
                    return this.historyServerCapabilities.accessHistoryEventsCapability;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.HistoryServerCapabilities_InsertDataCapability, node_opcua_variant_1.DataType.Boolean, () => {
                    return this.historyServerCapabilities.insertDataCapability;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.HistoryServerCapabilities_ReplaceDataCapability, node_opcua_variant_1.DataType.Boolean, () => {
                    return this.historyServerCapabilities.replaceDataCapability;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.HistoryServerCapabilities_UpdateDataCapability, node_opcua_variant_1.DataType.Boolean, () => {
                    return this.historyServerCapabilities.updateDataCapability;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.HistoryServerCapabilities_InsertEventCapability, node_opcua_variant_1.DataType.Boolean, () => {
                    return this.historyServerCapabilities.insertEventCapability;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.HistoryServerCapabilities_ReplaceEventCapability, node_opcua_variant_1.DataType.Boolean, () => {
                    return this.historyServerCapabilities.replaceEventCapability;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.HistoryServerCapabilities_UpdateEventCapability, node_opcua_variant_1.DataType.Boolean, () => {
                    return this.historyServerCapabilities.updateEventCapability;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.HistoryServerCapabilities_DeleteEventCapability, node_opcua_variant_1.DataType.Boolean, () => {
                    return this.historyServerCapabilities.deleteEventCapability;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.HistoryServerCapabilities_DeleteRawCapability, node_opcua_variant_1.DataType.Boolean, () => {
                    return this.historyServerCapabilities.deleteRawCapability;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.HistoryServerCapabilities_DeleteAtTimeCapability, node_opcua_variant_1.DataType.Boolean, () => {
                    return this.historyServerCapabilities.deleteAtTimeCapability;
                });
                bindStandardScalar(node_opcua_constants_1.VariableIds.HistoryServerCapabilities_InsertAnnotationCapability, node_opcua_variant_1.DataType.Boolean, () => {
                    return this.historyServerCapabilities.insertAnnotationCapability;
                });
            };
            function r(a, defaultValue) {
                if (a === undefined)
                    return defaultValue;
                if (typeof a === "function") {
                    return a();
                }
                return a;
            }
            const bindServerConfigurationBasic = () => {
                bindStandardArray(node_opcua_constants_1.VariableIds.ServerConfiguration_ServerCapabilities, node_opcua_variant_1.DataType.String, node_opcua_variant_1.DataType.String, () => r(this.serverConfiguration.serverCapabilities, ["NA"]));
                bindStandardScalar(node_opcua_constants_1.VariableIds.ServerConfiguration_ApplicationType, node_opcua_variant_1.DataType.Int32, () => r(this.serverConfiguration.applicationType, node_opcua_types_1.ApplicationType.Server));
                bindStandardScalar(node_opcua_constants_1.VariableIds.ServerConfiguration_ApplicationUri, node_opcua_variant_1.DataType.String, () => r(this.serverConfiguration.applicationUri, ""));
                bindStandardScalar(node_opcua_constants_1.VariableIds.ServerConfiguration_ProductUri, node_opcua_variant_1.DataType.String, () => r(this.serverConfiguration.productUri, ""));
                bindStandardScalar(node_opcua_constants_1.VariableIds.ServerConfiguration_HasSecureElement, node_opcua_variant_1.DataType.Boolean, () => r(this.serverConfiguration.hasSecureElement, false));
                bindStandardScalar(node_opcua_constants_1.VariableIds.ServerConfiguration_MulticastDnsEnabled, node_opcua_variant_1.DataType.Boolean, () => r(this.serverConfiguration.multicastDnsEnabled, false));
                bindStandardArray(node_opcua_constants_1.VariableIds.ServerConfiguration_SupportedPrivateKeyFormats, node_opcua_variant_1.DataType.String, node_opcua_variant_1.DataType.String, () => r(this.serverConfiguration.supportedPrivateKeyFormat, ["PEM"]));
            };
            bindServerDiagnostics();
            bindServerStatus();
            bindServerCapabilities();
            bindServerConfigurationBasic();
            bindHistoryServerCapabilities();
            const bindExtraStuff = () => {
                // mainly for compliance
                /*
            // The version number for the data type description. i=104
            bindStandardScalar(VariableIds.DataTypeDescriptionType_DataTypeVersion, DataType.String, () => {
                return "0";
            });

            const namingRuleDataTypeNode = addressSpace.findDataType(resolveNodeId(DataTypeIds.NamingRuleType))! as UADataType;

            if (namingRuleDataTypeNode) {
                const namingRuleType = (namingRuleDataTypeNode as any)._getEnumerationInfo().nameIndex; // getEnumeration("NamingRuleType");
                if (!namingRuleType) {
                    throw new Error("Cannot find Enumeration definition for NamingRuleType");
                }
                // i=111
                bindStandardScalar(VariableIds.ModellingRuleType_NamingRule, DataType.Int32, () => {
                    return 0;
                });

                // i=112
                bindStandardScalar(VariableIds.ModellingRule_Mandatory_NamingRule, DataType.Int32, () => {
                    return namingRuleType.Mandatory ? namingRuleType.Mandatory.value : 0;
                });

                // i=113
                bindStandardScalar(VariableIds.ModellingRule_Optional_NamingRule, DataType.Int32, () => {
                    return namingRuleType.Optional ? namingRuleType.Optional.value : 0;
                });
                // i=114
                bindStandardScalar(VariableIds.ModellingRule_ExposesItsArray_NamingRule, DataType.Int32, () => {
                    return namingRuleType.ExposesItsArray ? namingRuleType.ExposesItsArray.value : 0;
                });
                bindStandardScalar(VariableIds.ModellingRule_MandatoryPlaceholder_NamingRule, DataType.Int32, () => {
                    return namingRuleType.MandatoryPlaceholder ? namingRuleType.MandatoryPlaceholder.value : 0;
                });
            }
*/
            };
            bindExtraStuff();
            this.__internal_bindMethod((0, node_opcua_nodeid_1.makeNodeId)(node_opcua_constants_1.MethodIds.Server_GetMonitoredItems), getMonitoredItemsId.bind(this));
            this.__internal_bindMethod((0, node_opcua_nodeid_1.makeNodeId)(node_opcua_constants_1.MethodIds.Server_SetSubscriptionDurable), setSubscriptionDurable.bind(this));
            this.__internal_bindMethod((0, node_opcua_nodeid_1.makeNodeId)(node_opcua_constants_1.MethodIds.Server_ResendData), resendData.bind(this));
            this.__internal_bindMethod((0, node_opcua_nodeid_1.makeNodeId)(node_opcua_constants_1.MethodIds.Server_RequestServerStateChange), requestServerStateChange.bind(this));
            // fix getMonitoredItems.outputArguments arrayDimensions
            const fixGetMonitoredItemArgs = () => {
                const objects = this.addressSpace.rootFolder?.objects;
                if (!objects || !objects.server) {
                    return;
                }
                const getMonitoredItemsMethod = objects.server.getMethodByName("GetMonitoredItems");
                if (!getMonitoredItemsMethod) {
                    return;
                }
                const outputArguments = getMonitoredItemsMethod.outputArguments;
                const dataValue = outputArguments.readValue();
                if (!dataValue.value?.value) {
                    // value is null or undefined , meaning no arguments necessary
                    return;
                }
                (0, node_opcua_assert_1.assert)(dataValue.value.value[0].arrayDimensions.length === 1 && dataValue.value.value[0].arrayDimensions[0] === 0);
                (0, node_opcua_assert_1.assert)(dataValue.value.value[1].arrayDimensions.length === 1 && dataValue.value.value[1].arrayDimensions[0] === 0);
            };
            fixGetMonitoredItemArgs();
            const prepareServerDiagnostics = () => {
                const addressSpace1 = this.addressSpace;
                if (!addressSpace1.rootFolder.objects) {
                    return;
                }
                const server = addressSpace1.rootFolder.objects.server;
                if (!server) {
                    return;
                }
                // create SessionsDiagnosticsSummary
                const serverDiagnosticsNode = server.getComponentByName("ServerDiagnostics");
                if (!serverDiagnosticsNode) {
                    return;
                }
                if (true) {
                    // set serverDiagnosticsNode enabledFlag writeable for admin user only
                    // TO DO ...
                    serverDiagnosticsNode.enabledFlag.userAccessLevel = (0, node_opcua_data_model_1.makeAccessLevelFlag)("CurrentRead");
                    serverDiagnosticsNode.enabledFlag.accessLevel = (0, node_opcua_data_model_1.makeAccessLevelFlag)("CurrentRead");
                }
                // A Server may not expose the SamplingIntervalDiagnosticsArray if it does not use fixed sampling rates.
                // because we are not using fixed sampling rate, we need to remove the optional SamplingIntervalDiagnosticsArray
                // component
                const samplingIntervalDiagnosticsArray = serverDiagnosticsNode.getComponentByName("SamplingIntervalDiagnosticsArray");
                if (samplingIntervalDiagnosticsArray) {
                    addressSpace.deleteNode(samplingIntervalDiagnosticsArray);
                    const s = serverDiagnosticsNode.getComponents();
                }
                const subscriptionDiagnosticsArrayNode = serverDiagnosticsNode.getComponentByName("SubscriptionDiagnosticsArray");
                (0, node_opcua_assert_1.assert)(subscriptionDiagnosticsArrayNode.nodeClass === node_opcua_data_model_1.NodeClass.Variable);
                (0, node_opcua_address_space_1.bindExtObjArrayNode)(subscriptionDiagnosticsArrayNode, "SubscriptionDiagnosticsType", "subscriptionId");
                makeNotReadableIfEnabledFlagIsFalse(subscriptionDiagnosticsArrayNode);
                const sessionsDiagnosticsSummary = serverDiagnosticsNode.getComponentByName("SessionsDiagnosticsSummary");
                const sessionDiagnosticsArray = sessionsDiagnosticsSummary.getComponentByName("SessionDiagnosticsArray");
                (0, node_opcua_assert_1.assert)(sessionDiagnosticsArray.nodeClass === node_opcua_data_model_1.NodeClass.Variable);
                (0, node_opcua_address_space_1.bindExtObjArrayNode)(sessionDiagnosticsArray, "SessionDiagnosticsVariableType", "sessionId");
                const varType = addressSpace.findVariableType("SessionSecurityDiagnosticsType");
                if (!varType) {
                    debugLog("Warning cannot find SessionSecurityDiagnosticsType variable Type");
                }
                else {
                    const sessionSecurityDiagnosticsArray = sessionsDiagnosticsSummary.getComponentByName("SessionSecurityDiagnosticsArray");
                    (0, node_opcua_assert_1.assert)(sessionSecurityDiagnosticsArray.nodeClass === node_opcua_data_model_1.NodeClass.Variable);
                    (0, node_opcua_address_space_1.bindExtObjArrayNode)(sessionSecurityDiagnosticsArray, "SessionSecurityDiagnosticsType", "sessionId");
                    (0, node_opcua_address_space_1.ensureObjectIsSecure)(sessionSecurityDiagnosticsArray);
                }
            };
            prepareServerDiagnostics();
            this._internalState = "initialized";
            this.setServerState(node_opcua_common_1.ServerState.Running);
            setImmediate(() => callback());
        });
    }
    async browseWithAutomaticExpansion(nodesToBrowse, context) {
        // do expansion first
        for (const browseDescription of nodesToBrowse) {
            const nodeId = (0, node_opcua_nodeid_1.resolveNodeId)(browseDescription.nodeId);
            const node = this.addressSpace.findNode(nodeId);
            if (node) {
                if (node.onFirstBrowseAction) {
                    try {
                        await node.onFirstBrowseAction();
                        node.onFirstBrowseAction = undefined;
                    }
                    catch (err) {
                        if (util_1.types.isNativeError(err)) {
                            errorLog("onFirstBrowseAction method has failed", err.message);
                        }
                        errorLog(err);
                    }
                    (0, node_opcua_assert_1.assert)(node.onFirstBrowseAction === undefined, "expansion can only be made once");
                }
            }
        }
        return await this.browse(context, nodesToBrowse);
    }
    async browse(context, nodesToBrowse) {
        return this.addressSpaceAccessor.browse(context, nodesToBrowse);
    }
    async read(context, readRequest) {
        return this.addressSpaceAccessor.read(context, readRequest);
    }
    async write(context, nodesToWrite) {
        return await this.addressSpaceAccessor.write(context, nodesToWrite);
    }
    async call(context, methodsToCall) {
        return await this.addressSpaceAccessor.call(context, methodsToCall);
    }
    async historyRead(context, historyReadRequest) {
        return this.addressSpaceAccessor.historyRead(context, historyReadRequest);
    }
    getOldestInactiveSession() {
        // search screwed or closed session first
        let tmp = Object.values(this._sessions).filter((session1) => session1.status === "screwed" || session1.status === "disposed" || session1.status === "closed");
        if (tmp.length === 0) {
            // if none available, tap into the session that are not yet activated
            tmp = Object.values(this._sessions).filter((session1) => session1.status === "new");
        }
        if (tmp.length === 0)
            return null;
        let session = tmp[0];
        for (let i = 1; i < tmp.length; i++) {
            const c = tmp[i];
            if (session.creationDate.getTime() < c.creationDate.getTime()) {
                session = c;
            }
        }
        return session;
    }
    /**
     * create a new server session object.
     */
    createSession(options) {
        options = options || {};
        options.server = options.server || {};
        debugLog("createSession : increasing serverDiagnosticsSummary cumulatedSessionCount/currentSessionCount ");
        this.serverDiagnosticsSummary.cumulatedSessionCount += 1;
        this.serverDiagnosticsSummary.currentSessionCount += 1;
        this.clientDescription = options.clientDescription || new node_opcua_service_endpoints_1.ApplicationDescription({});
        const sessionTimeout = options.sessionTimeout || 1000;
        (0, node_opcua_assert_1.assert)(typeof sessionTimeout === "number");
        const session = new server_session_1.ServerSession(this, options.server.userManager, sessionTimeout);
        debugLog("createSession :sessionTimeout = ", session.sessionTimeout);
        const key = session.authenticationToken.toString();
        this._sessions[key] = session;
        // see spec OPC Unified Architecture,  Part 2 page 26 Release 1.02
        // TODO : When a Session is created, the Server adds an entry for the Client
        //        in its SessionDiagnosticsArray Variable
        session.on("new_subscription", (subscription) => {
            this.serverDiagnosticsSummary.cumulatedSubscriptionCount += 1;
            // add the subscription diagnostics in our subscriptions diagnostics array
            // note currentSubscriptionCount is handled directly with a special getter
        });
        session.on("subscription_terminated", (subscription) => {
            // remove the subscription diagnostics in our subscriptions diagnostics array
            // note currentSubscriptionCount is handled directly with a special getter
        });
        // OPC Unified Architecture, Part 4 23 Release 1.03
        // Sessions are terminated by the Server automatically if the Client fails to issue a Service request on the
        // Session within the timeout period negotiated by the Server in the CreateSession Service response.
        // This protects the Server against Client failures and against situations where a failed underlying
        // connection cannot be re-established. Clients shall be prepared to submit requests in a timely manner
        // prevent the Session from closing automatically. Clients may explicitly terminate sessions using the
        // CloseSession Service.
        session.on("timeout", () => {
            // the session hasn't been active for a while , probably because the client has disconnected abruptly
            // it is now time to close the session completely
            this.serverDiagnosticsSummary.sessionTimeoutCount += 1;
            session.sessionName = session.sessionName || "";
            const channel = session.channel;
            errorLog(chalk_1.default.cyan("Server: closing SESSION "), session.status, chalk_1.default.yellow(session.sessionName), chalk_1.default.yellow(session.nodeId.toString()), chalk_1.default.cyan(" because of timeout = "), session.sessionTimeout, chalk_1.default.cyan(" has expired without a keep alive"), chalk_1.default.bgCyan("channel = "), channel?.remoteAddress, " port = ", channel?.remotePort);
            // If a Server terminates a Session for any other reason, Subscriptions  associated with the Session,
            // are not deleted. => deleteSubscription= false
            this.closeSession(session.authenticationToken, /*deleteSubscription=*/ false, /* reason =*/ "Timeout");
            this.incrementSessionTimeoutCount();
        });
        return session;
    }
    /**
     * @param authenticationToken
     * @param deleteSubscriptions {Boolean} : true if session's subscription shall be deleted
     * @param {String} [reason = "CloseSession"] the reason for closing the session (
     *                 shall be "Timeout", "Terminated" or "CloseSession")
     *
     *
     * what the specs say:
     * -------------------
     *
     * If a Client invokes the CloseSession Service then all Subscriptions associated with the Session are also deleted
     * if the deleteSubscriptions flag is set to TRUE. If a Server terminates a Session for any other reason,
     * Subscriptions associated with the Session, are not deleted. Each Subscription has its own lifetime to protect
     * against data loss in the case of a Session termination. In these cases, the Subscription can be reassigned to
     * another Client before its lifetime expires.
     */
    closeSession(authenticationToken, deleteSubscriptions, reason) {
        reason = reason || "CloseSession";
        (0, node_opcua_assert_1.assert)(typeof reason === "string");
        (0, node_opcua_assert_1.assert)(reason === "Timeout" || reason === "Terminated" || reason === "CloseSession" || reason === "Forcing");
        debugLog("ServerEngine.closeSession ", authenticationToken.toString(), deleteSubscriptions);
        const session = this.getSession(authenticationToken);
        // istanbul ignore next
        if (!session) {
            throw new Error("cannot find session with this authenticationToken " + authenticationToken.toString());
        }
        if (!deleteSubscriptions) {
            // Live Subscriptions will not be deleted, but transferred to the orphanPublishEngine
            // until they time out or until a other session transfer them back to it.
            if (!this._orphanPublishEngine) {
                this._orphanPublishEngine = new server_publish_engine_for_orphan_subscriptions_1.ServerSidePublishEngineForOrphanSubscription({ maxPublishRequestInQueue: 0 });
            }
            debugLog("transferring remaining live subscription to orphanPublishEngine !");
            server_publish_engine_1.ServerSidePublishEngine.transferSubscriptionsToOrphan(session.publishEngine, this._orphanPublishEngine);
        }
        session.close(deleteSubscriptions, reason);
        (0, node_opcua_assert_1.assert)(session.status === "closed");
        debugLog(" engine.serverDiagnosticsSummary.currentSessionCount -= 1;");
        this.serverDiagnosticsSummary.currentSessionCount -= 1;
        // xx //TODO make sure _closedSessions gets cleaned at some point
        // xx self._closedSessions[key] = session;
        // remove sessionDiagnostics from server.ServerDiagnostics.SessionsDiagnosticsSummary.SessionDiagnosticsSummary
        delete this._sessions[authenticationToken.toString()];
        session.dispose();
    }
    findSubscription(subscriptionId) {
        const subscriptions = [];
        Object.values(this._sessions).map((session) => {
            if (subscriptions.length) {
                return;
            }
            const subscription = session.publishEngine.getSubscriptionById(subscriptionId);
            if (subscription) {
                subscriptions.push(subscription);
            }
        });
        if (subscriptions.length) {
            (0, node_opcua_assert_1.assert)(subscriptions.length === 1);
            return subscriptions[0];
        }
        return this.findOrphanSubscription(subscriptionId);
    }
    findOrphanSubscription(subscriptionId) {
        if (!this._orphanPublishEngine) {
            return null;
        }
        return this._orphanPublishEngine.getSubscriptionById(subscriptionId);
    }
    deleteOrphanSubscription(subscription) {
        if (!this._orphanPublishEngine) {
            return node_opcua_status_code_1.StatusCodes.BadInternalError;
        }
        (0, node_opcua_assert_1.assert)(this.findSubscription(subscription.id));
        const c = this._orphanPublishEngine.subscriptionCount;
        subscription.terminate();
        subscription.dispose();
        (0, node_opcua_assert_1.assert)(this._orphanPublishEngine.subscriptionCount === c - 1);
        return node_opcua_status_code_1.StatusCodes.Good;
    }
    /**
     * @param session           {ServerSession}  - the new session that will own the subscription
     * @param subscriptionId    {IntegerId}      - the subscription Id to transfer
     * @param sendInitialValues {Boolean}        - true if initial values will be resent.
     * @return                  {TransferResult}
     */
    async transferSubscription(session, subscriptionId, sendInitialValues) {
        if (subscriptionId <= 0) {
            return new node_opcua_service_subscription_1.TransferResult({ statusCode: node_opcua_status_code_1.StatusCodes.BadSubscriptionIdInvalid });
        }
        const subscription = this.findSubscription(subscriptionId);
        if (!subscription) {
            return new node_opcua_service_subscription_1.TransferResult({ statusCode: node_opcua_status_code_1.StatusCodes.BadSubscriptionIdInvalid });
        }
        // check that session have same userIdentity
        if (!(0, sessions_compatible_for_transfer_1.sessionsCompatibleForTransfer)(subscription.$session, session)) {
            return new node_opcua_service_subscription_1.TransferResult({ statusCode: node_opcua_status_code_1.StatusCodes.BadUserAccessDenied });
        }
        // update diagnostics
        subscription.subscriptionDiagnostics.transferRequestCount++;
        // now check that new session has sufficient right
        // if (session.authenticationToken.toString() !== subscription.authenticationToken.toString()) {
        //     warningLog("ServerEngine#transferSubscription => BadUserAccessDenied");
        //     return new TransferResult({ statusCode: StatusCodes.BadUserAccessDenied });
        // }
        if (session.publishEngine === subscription.publishEngine) {
            // subscription is already in this session !!
            return new node_opcua_service_subscription_1.TransferResult({ statusCode: node_opcua_status_code_1.StatusCodes.BadNothingToDo });
        }
        if (session === subscription.$session) {
            // subscription is already in this session !!
            return new node_opcua_service_subscription_1.TransferResult({ statusCode: node_opcua_status_code_1.StatusCodes.BadNothingToDo });
        }
        // The number of times the subscription has been transferred to an alternate client.
        subscription.subscriptionDiagnostics.transferredToAltClientCount++;
        // The number of times the subscription has been transferred to an alternate session for the same client.
        subscription.subscriptionDiagnostics.transferredToSameClientCount++;
        const nbSubscriptionBefore = session.publishEngine.subscriptionCount;
        if (subscription.$session) {
            subscription.$session._unexposeSubscriptionDiagnostics(subscription);
        }
        subscription.$session = session;
        await server_publish_engine_1.ServerSidePublishEngine.transferSubscription(subscription, session.publishEngine, sendInitialValues);
        session._exposeSubscriptionDiagnostics(subscription);
        (0, node_opcua_assert_1.assert)(subscription.publishEngine === session.publishEngine);
        // assert(session.publishEngine.subscriptionCount === nbSubscriptionBefore + 1);
        const result = new node_opcua_service_subscription_1.TransferResult({
            availableSequenceNumbers: subscription.getAvailableSequenceNumbers(),
            statusCode: node_opcua_status_code_1.StatusCodes.Good
        });
        // istanbul ignore next
        if (doDebug) {
            debugLog("TransferResult", result.toString());
        }
        return result;
    }
    /**
     * retrieve a session by its authenticationToken.
     *
     * @param authenticationToken
     * @param activeOnly
     * @return {ServerSession}
     */
    getSession(authenticationToken, activeOnly) {
        if (!authenticationToken ||
            (authenticationToken.identifierType && authenticationToken.identifierType !== node_opcua_nodeid_1.NodeIdType.BYTESTRING)) {
            return null; // wrong type !
        }
        const key = authenticationToken.toString();
        let session = this._sessions[key];
        if (!activeOnly && !session) {
            session = this._closedSessions[key];
        }
        return session;
    }
    async translateBrowsePaths(browsePaths) {
        const browsePathResults = [];
        for (const browsePath of browsePaths) {
            const result = await this.translateBrowsePath(browsePath);
            browsePathResults.push(result);
        }
        return browsePathResults;
    }
    async translateBrowsePath(browsePath) {
        return this.addressSpace.browsePath(browsePath);
    }
    /**
     *
     * performs a call to ```asyncRefresh``` on all variable nodes that provide an async refresh func.
     *
     * @param nodesToRefresh {Array<ReadValueId|HistoryReadValueId>}  an array containing the node to consider
     * Each element of the array shall be of the form { nodeId: <xxx>, attributeIds: <value> }.
     * @param maxAge {number}  the maximum age of the value to be read, in milliseconds.
     * @param callback
     *
     */
    refreshValues(nodesToRefresh, maxAge, 
    /**
     * @param err
     * @param dataValues an array containing value read
     * The array length matches the number of  nodeIds that are candidate for an
     * async refresh (i.e: nodes that are of type Variable with asyncRefresh func }
     */
    callback) {
        const referenceTime = (0, node_opcua_date_time_1.getCurrentClock)();
        maxAge && referenceTime.timestamp.setTime(referenceTime.timestamp.getTime() - maxAge);
        (0, node_opcua_assert_1.assert)(typeof callback === "function");
        const nodeMap = {};
        for (const nodeToRefresh of nodesToRefresh) {
            // only consider node  for which the caller wants to read the Value attribute
            // assuming that Value is requested if attributeId is missing,
            if (nodeToRefresh instanceof node_opcua_types_1.ReadValueId && nodeToRefresh.attributeId !== node_opcua_data_model_1.AttributeIds.Value) {
                continue;
            }
            // ... and that are valid object and instances of Variables ...
            const uaNode = this.addressSpace.findNode(nodeToRefresh.nodeId);
            if (!uaNode || !(uaNode.nodeClass === node_opcua_data_model_1.NodeClass.Variable)) {
                continue;
            }
            // ... and that have been declared as asynchronously updating
            if (typeof uaNode.refreshFunc !== "function") {
                continue;
            }
            const key = uaNode.nodeId.toString();
            if (nodeMap[key]) {
                continue;
            }
            nodeMap[key] = uaNode;
        }
        const uaVariableArray = Object.values(nodeMap);
        if (uaVariableArray.length === 0) {
            // nothing to do
            return callback(null, []);
        }
        // perform all asyncRefresh in parallel
        async_1.default.map(uaVariableArray, (uaVariable, inner_callback) => {
            try {
                uaVariable.asyncRefresh(referenceTime, (err, dataValue) => {
                    inner_callback(err, dataValue);
                });
            }
            catch (err) {
                const _err = err;
                errorLog("asyncRefresh internal error", _err.message);
                inner_callback(_err);
            }
        }, (err, arrResult) => {
            callback(err || null, arrResult);
        });
    }
    _exposeSubscriptionDiagnostics(subscription) {
        try {
            debugLog("ServerEngine#_exposeSubscriptionDiagnostics", subscription.subscriptionId);
            const subscriptionDiagnosticsArray = this._getServerSubscriptionDiagnosticsArrayNode();
            const subscriptionDiagnostics = subscription.subscriptionDiagnostics;
            (0, node_opcua_assert_1.assert)(subscriptionDiagnostics.$subscription === subscription);
            (0, node_opcua_assert_1.assert)(subscriptionDiagnostics instanceof node_opcua_common_1.SubscriptionDiagnosticsDataType);
            if (subscriptionDiagnostics && subscriptionDiagnosticsArray) {
                (0, node_opcua_address_space_1.addElement)(subscriptionDiagnostics, subscriptionDiagnosticsArray);
            }
        }
        catch (err) {
            errorLog("_exposeSubscriptionDiagnostics err", err);
        }
    }
    _unexposeSubscriptionDiagnostics(subscription) {
        const serverSubscriptionDiagnosticsArray = this._getServerSubscriptionDiagnosticsArrayNode();
        const subscriptionDiagnostics = subscription.subscriptionDiagnostics;
        (0, node_opcua_assert_1.assert)(subscriptionDiagnostics instanceof node_opcua_common_1.SubscriptionDiagnosticsDataType);
        if (subscriptionDiagnostics && serverSubscriptionDiagnosticsArray) {
            const node = serverSubscriptionDiagnosticsArray[subscription.id];
            (0, node_opcua_address_space_1.removeElement)(serverSubscriptionDiagnosticsArray, (a) => a.subscriptionId === subscription.id);
            /*assert(
                !(subscriptionDiagnosticsArray as any)[subscription.id],
                " subscription node must have been removed from subscriptionDiagnosticsArray"
            );
            */
        }
        debugLog("ServerEngine#_unexposeSubscriptionDiagnostics", subscription.subscriptionId);
    }
    /**
     * create a new subscription
     * @return {Subscription}
     */
    _createSubscriptionOnSession(session, request) {
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(request, "requestedPublishingInterval")); // Duration
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(request, "requestedLifetimeCount")); // Counter
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(request, "requestedMaxKeepAliveCount")); // Counter
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(request, "maxNotificationsPerPublish")); // Counter
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(request, "publishingEnabled")); // Boolean
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(request, "priority")); // Byte
        // adjust publishing parameters
        const publishingInterval = request.requestedPublishingInterval || 0;
        const maxKeepAliveCount = request.requestedMaxKeepAliveCount || 0;
        const lifeTimeCount = request.requestedLifetimeCount || 0;
        const subscription = new server_subscription_1.Subscription({
            id: _get_next_subscriptionId(),
            lifeTimeCount,
            maxKeepAliveCount,
            maxNotificationsPerPublish: request.maxNotificationsPerPublish,
            priority: request.priority || 0,
            publishEngine: session.publishEngine, //
            publishingEnabled: request.publishingEnabled,
            publishingInterval,
            // -------------------
            sessionId: node_opcua_nodeid_1.NodeId.nullNodeId,
            globalCounter: this._globalCounter,
            serverCapabilities: this.serverCapabilities // shared
        });
        // add subscriptionDiagnostics
        this._exposeSubscriptionDiagnostics(subscription);
        (0, node_opcua_assert_1.assert)(subscription.publishEngine === session.publishEngine);
        session.publishEngine.add_subscription(subscription);
        // eslint-disable-next-line @typescript-eslint/no-this-alias
        const engine = this;
        subscription.once("terminated", function () {
            engine._unexposeSubscriptionDiagnostics(this);
        });
        return subscription;
    }
    /**
     */
    __internal_bindMethod(nodeId, func) {
        (0, node_opcua_assert_1.assert)(typeof func === "function");
        (0, node_opcua_assert_1.assert)(nodeId instanceof node_opcua_nodeid_1.NodeId);
        const methodNode = this.addressSpace.findNode(nodeId);
        if (!methodNode) {
            return;
        }
        // istanbul ignore else
        if (methodNode && methodNode.bindMethod) {
            methodNode.bindMethod(func);
        }
        else {
            warningLog(chalk_1.default.yellow("WARNING:  cannot bind a method with id ") +
                chalk_1.default.cyan(nodeId.toString()) +
                chalk_1.default.yellow(". please check your nodeset.xml file or add this node programmatically"));
            warningLog((0, node_opcua_debug_1.traceFromThisProjectOnly)());
        }
    }
    _getServerSubscriptionDiagnosticsArrayNode() {
        // istanbul ignore next
        if (!this.addressSpace) {
            doDebug && debugLog("ServerEngine#_getServerSubscriptionDiagnosticsArray : no addressSpace");
            return null; // no addressSpace
        }
        const subscriptionDiagnosticsType = this.addressSpace.findVariableType("SubscriptionDiagnosticsType");
        if (!subscriptionDiagnosticsType) {
            doDebug &&
                debugLog("ServerEngine#_getServerSubscriptionDiagnosticsArray " + ": cannot find SubscriptionDiagnosticsType");
        }
        // SubscriptionDiagnosticsArray = i=2290
        const subscriptionDiagnosticsArrayNode = this.addressSpace.findNode((0, node_opcua_nodeid_1.makeNodeId)(node_opcua_constants_1.VariableIds.Server_ServerDiagnostics_SubscriptionDiagnosticsArray));
        return subscriptionDiagnosticsArrayNode;
    }
}
exports.ServerEngine = ServerEngine;
//# sourceMappingURL=server_engine.js.map