"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.RegisterServerManager = void 0;
/**
 * @module node-opcua-server
 */
// tslint:disable:no-console
const events_1 = require("events");
const chalk_1 = __importDefault(require("chalk"));
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_client_1 = require("node-opcua-client");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_hostname_1 = require("node-opcua-hostname");
const node_opcua_secure_channel_1 = require("node-opcua-secure-channel");
const node_opcua_service_discovery_1 = require("node-opcua-service-discovery");
const node_opcua_types_1 = require("node-opcua-types");
const web_1 = require("node-opcua-crypto/web");
const i_register_server_manager_1 = require("./i_register_server_manager");
const doDebug = (0, node_opcua_debug_1.checkDebugFlag)("REGISTER_LDS");
const debugLog = (0, node_opcua_debug_1.make_debugLog)("REGISTER_LDS");
const warningLog = (0, node_opcua_debug_1.make_warningLog)("REGISTER_LDS");
const g_DefaultRegistrationServerTimeout = 8 * 60 * 1000; // 8 minutes
function securityPolicyLevel(securityPolicy) {
    switch (securityPolicy) {
        case node_opcua_secure_channel_1.SecurityPolicy.None:
            return 0;
        case node_opcua_secure_channel_1.SecurityPolicy.Basic128:
            return 1;
        case node_opcua_secure_channel_1.SecurityPolicy.Basic128Rsa15:
            return 2;
        case node_opcua_secure_channel_1.SecurityPolicy.Basic192:
            return 3;
        case node_opcua_secure_channel_1.SecurityPolicy.Basic192Rsa15:
            return 4;
        case node_opcua_secure_channel_1.SecurityPolicy.Basic256:
            return 5;
        case node_opcua_secure_channel_1.SecurityPolicy.Basic256Rsa15:
            return 6;
        case node_opcua_secure_channel_1.SecurityPolicy.Aes128_Sha256_RsaOaep:
            return 7;
        case node_opcua_secure_channel_1.SecurityPolicy.Basic256Sha256:
            return 8;
        case node_opcua_secure_channel_1.SecurityPolicy.Aes256_Sha256_RsaPss:
            return 9;
        default:
            return 0;
    }
}
function sortEndpointBySecurityLevel(endpoints) {
    endpoints.sort((a, b) => {
        if (a.securityMode === b.securityMode) {
            if (a.securityPolicyUri === b.securityPolicyUri) {
                const sa = a.securityLevel;
                const sb = b.securityLevel;
                return sa < sb ? 1 : sa > sb ? -1 : 0;
            }
            else {
                const sa = securityPolicyLevel(a.securityPolicyUri);
                const sb = securityPolicyLevel(b.securityPolicyUri);
                return sa < sb ? 1 : sa > sb ? -1 : 0;
            }
        }
        else {
            return a.securityMode < b.securityMode ? 1 : 0;
        }
    });
    return endpoints;
}
function findSecureEndpoint(endpoints) {
    // we only care about binary tcp transport endpoint
    endpoints = endpoints.filter((e) => {
        return e.transportProfileUri === "http://opcfoundation.org/UA-Profile/Transport/uatcp-uasc-uabinary";
    });
    endpoints = endpoints.filter((e) => {
        return e.securityMode === node_opcua_secure_channel_1.MessageSecurityMode.SignAndEncrypt;
    });
    if (endpoints.length === 0) {
        endpoints = endpoints.filter((e) => {
            return e.securityMode === node_opcua_secure_channel_1.MessageSecurityMode.Sign;
        });
    }
    if (endpoints.length === 0) {
        endpoints = endpoints.filter((e) => {
            return e.securityMode === node_opcua_secure_channel_1.MessageSecurityMode.None;
        });
    }
    endpoints = sortEndpointBySecurityLevel(endpoints);
    return endpoints[0];
}
function constructRegisteredServer(server, isOnline) {
    const discoveryUrls = server.getDiscoveryUrls();
    (0, node_opcua_assert_1.assert)(!isOnline || discoveryUrls.length >= 1, "expecting some discoveryUrls if we go online .... ");
    const info = (0, web_1.exploreCertificate)(server.getCertificate());
    const commonName = info.tbsCertificate.subject.commonName;
    const serverUri = info.tbsCertificate.extensions?.subjectAltName.uniformResourceIdentifier[0];
    // istanbul ignore next
    if (serverUri !== server.serverInfo.applicationUri) {
        warningLog(chalk_1.default.yellow("Warning certificate uniformResourceIdentifier doesn't match serverInfo.applicationUri"), "\n subjectKeyIdentifier      : ", info.tbsCertificate.extensions?.subjectKeyIdentifier, "\n subjectAltName            : ", info.tbsCertificate.extensions?.subjectAltName, "\n commonName                : ", info.tbsCertificate.subject.commonName, "\n serverInfo.applicationUri : ", server.serverInfo.applicationUri);
    }
    // istanbul ignore next
    if (!server.serverInfo.applicationName.text) {
        debugLog("warning: application name is missing");
    }
    // The globally unique identifier for the Server instance. The serverUri matches
    // the applicationUri from the ApplicationDescription defined in 7.1.
    const s = {
        serverUri: server.serverInfo.applicationUri,
        // The globally unique identifier for the Server product.
        productUri: server.serverInfo.productUri,
        serverNames: [
            {
                locale: "en-US",
                text: server.serverInfo.applicationName.text
            }
        ],
        serverType: server.serverType,
        discoveryUrls,
        gatewayServerUri: null,
        isOnline,
        semaphoreFilePath: null
    };
    return s;
}
function constructRegisterServerRequest(serverB, isOnline) {
    const server = constructRegisteredServer(serverB, isOnline);
    return new node_opcua_service_discovery_1.RegisterServerRequest({
        server
    });
}
function constructRegisterServer2Request(serverB, isOnline) {
    const server = constructRegisteredServer(serverB, isOnline);
    return new node_opcua_service_discovery_1.RegisterServer2Request({
        discoveryConfiguration: [
            new node_opcua_types_1.MdnsDiscoveryConfiguration({
                mdnsServerName: serverB.serverInfo.applicationUri,
                serverCapabilities: serverB.capabilitiesForMDNS
            })
        ],
        server
    });
}
const no_retry_connectivity_strategy = {
    initialDelay: 1000,
    maxDelay: 2000,
    maxRetry: 1, // NO RETRY !!!
    randomisationFactor: 0
};
const infinite_connectivity_strategy = {
    initialDelay: 2000,
    maxDelay: 50000,
    maxRetry: 10000000,
    randomisationFactor: 0
};
const pause = async (duration) => await new Promise((resolve) => setTimeout(resolve, duration));
async function sendRegisterServerRequest(server, client, isOnline) {
    // try to send a RegisterServer2Request
    const request = constructRegisterServer2Request(server, isOnline);
    await new Promise((resolve, reject) => {
        client.performMessageTransaction(request, (err, response) => {
            if (!err) {
                // RegisterServerResponse
                debugLog("RegisterServerManager#_registerServer sendRegisterServer2Request has succeeded (isOnline", isOnline, ")");
                return resolve();
            }
            debugLog("RegisterServerManager#_registerServer sendRegisterServer2Request has failed " + "(isOnline", isOnline, ")");
            debugLog("RegisterServerManager#_registerServer" + " falling back to using sendRegisterServerRequest instead");
            // fall back to
            const request1 = constructRegisterServerRequest(server, isOnline);
            client.performMessageTransaction(request1, (err1, response1) => {
                if (!err1) {
                    debugLog("RegisterServerManager#_registerServer sendRegisterServerRequest " + "has succeeded (isOnline", isOnline, ")");
                    return resolve();
                }
                debugLog("RegisterServerManager#_registerServer sendRegisterServerRequest " + "has failed (isOnline", isOnline, ")");
                reject(err1);
            });
        });
    });
}
let g_registeringClientCounter = 0;
/**
 * RegisterServerManager is responsible to Register an opcua server on a LDS or LDS-ME server
 * This class takes in charge :
 * - the initial registration of a server
 * - the regular registration renewal (every 8 minutes or so ...)
 * - dealing with cases where LDS is not up and running when server starts.
 * ( in this case the connection will be continuously attempted using the infinite
 * back-off strategy
 * - the un-registration of the server ( during shutdown for instance)
 *
 * Events:
 *
 * Emitted when the server is trying to registered the LDS
 * but when the connection to the lds has failed
 * serverRegistrationPending is sent when the backoff signal of the
 * connection process is rained
 * @event serverRegistrationPending
 *
 * emitted when the server is successfully registered to the LDS
 * @event serverRegistered
 *
 * emitted when the server has successfully renewed its registration to the LDS
 * @event serverRegistrationRenewed
 *
 * emitted when the server is successfully unregistered to the LDS
 * ( for instance during shutdown)
 * @event serverUnregistered
 *
 *
 * (LDS => Local Discovery Server)
 */
class RegisterServerManager extends events_1.EventEmitter {
    discoveryServerEndpointUrl;
    timeout;
    server;
    _registrationTimerId;
    state = i_register_server_manager_1.RegisterServerManagerStatus.INACTIVE;
    _registration_client = null;
    selectedEndpoint;
    _serverEndpoints = [];
    getState() {
        return this.state;
    }
    constructor(options) {
        super();
        this.server = options.server;
        this.#_setState(i_register_server_manager_1.RegisterServerManagerStatus.INACTIVE);
        this.timeout = g_DefaultRegistrationServerTimeout;
        this.discoveryServerEndpointUrl = options.discoveryServerEndpointUrl || "opc.tcp://localhost:4840";
        (0, node_opcua_assert_1.assert)(typeof this.discoveryServerEndpointUrl === "string");
        this._registrationTimerId = null;
    }
    dispose() {
        this.server = null;
        debugLog("RegisterServerManager#dispose", this.state.toString());
        if (this._registrationTimerId) {
            clearTimeout(this._registrationTimerId);
            this._registrationTimerId = null;
        }
        (0, node_opcua_assert_1.assert)(this._registrationTimerId === null, "stop has not been called");
        this.removeAllListeners();
    }
    #_emitEvent(eventName) {
        setImmediate(() => {
            debugLog("emiting event", eventName);
            this.emit(eventName);
        });
    }
    #_setState(status) {
        const previousState = this.state || i_register_server_manager_1.RegisterServerManagerStatus.INACTIVE;
        debugLog("RegisterServerManager#setState : ", i_register_server_manager_1.RegisterServerManagerStatus[previousState], " => ", i_register_server_manager_1.RegisterServerManagerStatus[status]);
        this.state = status;
    }
    /**
     * The start method initiates the registration process in a non-blocking way.
     * It immediately returns while the actual work is performed in a background task.
     */
    async start() {
        debugLog("RegisterServerManager#start");
        if (this.state !== i_register_server_manager_1.RegisterServerManagerStatus.INACTIVE) {
            throw new Error("RegisterServer process already started: " + i_register_server_manager_1.RegisterServerManagerStatus[this.state]);
        }
        this.discoveryServerEndpointUrl = (0, node_opcua_hostname_1.resolveFullyQualifiedDomainName)(this.discoveryServerEndpointUrl);
        // Immediately set the state to INITIALIZING and run the process in the background.
        this.#_setState(i_register_server_manager_1.RegisterServerManagerStatus.INITIALIZING);
        // This method is called without await to ensure it is non-blocking.
        // The catch block handles any synchronous errors.
        this.#_runRegistrationProcess().catch((err) => {
            warningLog("Synchronous error in #_runRegistrationProcess: ", (err?.message));
        });
    }
    /**
     * Private method to run the entire registration process in the background.
     * It handles the state machine transitions and re-connection logic.
     * @private
     */
    async #_runRegistrationProcess() {
        while (this.getState() !== i_register_server_manager_1.RegisterServerManagerStatus.WAITING && !this.#_isTerminating()) {
            debugLog("RegisterServerManager#_runRegistrationProcess - state =", i_register_server_manager_1.RegisterServerManagerStatus[this.state], "isTerminating =", this.#_isTerminating());
            try {
                if (this.getState() === i_register_server_manager_1.RegisterServerManagerStatus.INACTIVE) {
                    this.#_setState(i_register_server_manager_1.RegisterServerManagerStatus.INITIALIZING);
                }
                await this.#_establish_initial_connection();
                if (this.getState() !== i_register_server_manager_1.RegisterServerManagerStatus.INITIALIZING) {
                    debugLog("RegisterServerManager#_runRegistrationProcess: aborted during initialization");
                    return;
                }
                this.#_setState(i_register_server_manager_1.RegisterServerManagerStatus.INITIALIZED);
                this.#_setState(i_register_server_manager_1.RegisterServerManagerStatus.REGISTERING);
                this.#_emitEvent("serverRegistrationPending");
                await this.#_registerOrUnregisterServer(true);
                if (this.getState() !== i_register_server_manager_1.RegisterServerManagerStatus.REGISTERING) {
                    debugLog("RegisterServerManager#_runRegistrationProcess: aborted during registration");
                    return;
                }
                this.#_setState(i_register_server_manager_1.RegisterServerManagerStatus.REGISTERED);
                this.#_emitEvent("serverRegistered");
                this.#_setState(i_register_server_manager_1.RegisterServerManagerStatus.WAITING);
                this.#_trigger_next();
                return;
            }
            catch (err) {
                debugLog("RegisterServerManager#_runRegistrationProcess - operation failed!", (err.message));
                if (!this.#_isTerminating()) {
                    this.#_setState(i_register_server_manager_1.RegisterServerManagerStatus.INACTIVE);
                    this.#_emitEvent("serverRegistrationFailure");
                    await pause(Math.min(5000, this.timeout));
                }
            }
        }
    }
    #_isTerminating() {
        return this.getState() === i_register_server_manager_1.RegisterServerManagerStatus.UNREGISTERING || this.getState() === i_register_server_manager_1.RegisterServerManagerStatus.DISPOSING;
    }
    /**
     * Establish the initial connection with the Discovery Server to extract best endpoint to use.
     * @private
     */
    async #_establish_initial_connection() {
        if (!this.server) {
            this.#_setState(i_register_server_manager_1.RegisterServerManagerStatus.DISPOSING);
            return;
        }
        if (this.state !== i_register_server_manager_1.RegisterServerManagerStatus.INITIALIZING) {
            debugLog("RegisterServerManager#_establish_initial_connection: aborting due to state change");
            return;
        }
        debugLog("RegisterServerManager#_establish_initial_connection");
        (0, node_opcua_assert_1.assert)(!this._registration_client);
        (0, node_opcua_assert_1.assert)(typeof this.discoveryServerEndpointUrl === "string");
        (0, node_opcua_assert_1.assert)(this.state === i_register_server_manager_1.RegisterServerManagerStatus.INITIALIZING);
        this.selectedEndpoint = undefined;
        const applicationName = (0, node_opcua_client_1.coerceLocalizedText)(this.server.serverInfo.applicationName)?.text || undefined;
        this.server.serverCertificateManager.referenceCounter++;
        const server = this.server;
        const prefix = `Client-${g_registeringClientCounter++}`;
        const action = "initializing";
        const ldsInfo = this.discoveryServerEndpointUrl;
        const serverInfo = this.server?.serverInfo.applicationUri;
        const clientName = `${prefix} for server ${serverInfo} to LDS ${ldsInfo} for ${action}`;
        const registrationClient = node_opcua_client_1.OPCUAClientBase.create({
            clientName,
            applicationName,
            applicationUri: server.serverInfo.applicationUri,
            connectionStrategy: infinite_connectivity_strategy,
            clientCertificateManager: server.serverCertificateManager,
            certificateFile: server.certificateFile,
            privateKeyFile: server.privateKeyFile
        });
        registrationClient.on("backoff", (nbRetry, delay) => {
            if (this.state !== i_register_server_manager_1.RegisterServerManagerStatus.INITIALIZING)
                return; // Ignore event if state has changed
            debugLog("RegisterServerManager - received backoff");
            debugLog(registrationClient.clientName, chalk_1.default.bgWhite.cyan("contacting discovery server backoff "), this.discoveryServerEndpointUrl, " attempt #", nbRetry, " retrying in ", delay / 1000.0, " seconds");
            this.#_emitEvent("serverRegistrationPending");
        });
        // Keep track of the client to allow cancellation during connect()
        this._registration_client = registrationClient;
        try {
            await registrationClient.connect(this.discoveryServerEndpointUrl);
            if (!this._registration_client)
                return;
            // Re-check state after the long-running connect operation
            if (this.state !== i_register_server_manager_1.RegisterServerManagerStatus.INITIALIZING) {
                debugLog("RegisterServerManager#_establish_initial_connection: aborted after connection");
                return;
            }
            const endpoints = await registrationClient.getEndpoints();
            const endpoint = findSecureEndpoint(endpoints);
            if (!endpoint) {
                throw new Error("Cannot find Secure endpoint");
            }
            if (endpoint.serverCertificate) {
                (0, node_opcua_assert_1.assert)(endpoint.serverCertificate);
                this.selectedEndpoint = endpoint;
            }
            else {
                this.selectedEndpoint = undefined;
            }
            this._serverEndpoints = registrationClient._serverEndpoints;
        }
        catch (err) {
            // Do not set state to INACTIVE or rethrow here, let the caller handle it.
            throw err;
        }
        finally {
            if (this._registration_client) {
                const tmp = this._registration_client;
                this._registration_client = null;
                try {
                    await tmp.disconnect();
                }
                catch (err) {
                    warningLog("RegisterServerManager#_establish_initial_connection: error disconnecting client", err);
                }
            }
        }
    }
    #_trigger_next() {
        (0, node_opcua_assert_1.assert)(!this._registrationTimerId);
        (0, node_opcua_assert_1.assert)(this.state === i_register_server_manager_1.RegisterServerManagerStatus.WAITING);
        debugLog("RegisterServerManager#_trigger_next " + ": installing timeout to perform registerServer renewal (timeout =", this.timeout, ")");
        if (this._registrationTimerId)
            clearTimeout(this._registrationTimerId);
        this._registrationTimerId = setTimeout(() => {
            if (!this._registrationTimerId) {
                debugLog("RegisterServerManager => cancelling re registration");
                return;
            }
            this._registrationTimerId = null;
            if (this.#_isTerminating()) {
                debugLog("RegisterServerManager#_trigger_next : cancelling re registration");
                return;
            }
            debugLog("RegisterServerManager#_trigger_next : renewing RegisterServer");
            const after_register = (err) => {
                if (!this.#_isTerminating()) {
                    debugLog("RegisterServerManager#_trigger_next : renewed ! err:", err?.message);
                    this.#_setState(i_register_server_manager_1.RegisterServerManagerStatus.WAITING);
                    this.#_emitEvent("serverRegistrationRenewed");
                    this.#_trigger_next();
                }
            };
            // State transition before the call
            this.#_setState(i_register_server_manager_1.RegisterServerManagerStatus.REGISTERING);
            this.#_emitEvent("serverRegistrationPending");
            this.#_registerOrUnregisterServer(/*isOnline=*/ true)
                .then(() => after_register())
                .catch((err) => after_register(err));
        }, this.timeout);
    }
    async stop() {
        debugLog("RegisterServerManager#stop");
        if (this.#_isTerminating()) {
            debugLog("Already stopping  or stopped...");
            return;
        }
        // make sure we don't have any timer running
        // so a registration renewal won't happen while we are stopping
        if (this._registrationTimerId) {
            clearTimeout(this._registrationTimerId);
            this._registrationTimerId = null;
        }
        // Immediately set state to signal a stop
        this.#_setState(i_register_server_manager_1.RegisterServerManagerStatus.UNREGISTERING);
        // Cancel any pending client connections
        await this.#_cancel_pending_client_if_any();
        if (this.selectedEndpoint) {
            try {
                await this.#_registerOrUnregisterServer(/* isOnline= */ false);
                this.#_setState(i_register_server_manager_1.RegisterServerManagerStatus.UNREGISTERED);
                this.#_emitEvent("serverUnregistered");
            }
            catch (err) {
                warningLog(err);
                warningLog("RegisterServerManager#stop: Unregistration failed.", err.message);
            }
        }
        // Final state transition to INACTIVE
        this.#_setState(i_register_server_manager_1.RegisterServerManagerStatus.DISPOSING);
    }
    /**
     * Handles the actual registration/unregistration request.
     * It is designed to be interruptible by checking the state.
     * @param isOnline - true for registration, false for unregistration
     * @private
     */
    async #_registerOrUnregisterServer(isOnline) {
        const expectedState = isOnline ? i_register_server_manager_1.RegisterServerManagerStatus.REGISTERING : i_register_server_manager_1.RegisterServerManagerStatus.UNREGISTERING;
        if (this.getState() !== expectedState) {
            debugLog("RegisterServerManager#_registerServer: aborting due to state change");
            return;
        }
        debugLog("RegisterServerManager#_registerServer isOnline:", isOnline);
        (0, node_opcua_assert_1.assert)(this.selectedEndpoint, "must have a selected endpoint");
        (0, node_opcua_assert_1.assert)(this.server.serverType !== undefined, " must have a valid server Type");
        const server = this.server;
        const selectedEndpoint = this.selectedEndpoint;
        if (!selectedEndpoint) {
            warningLog("Warning: cannot register server - no endpoint available");
            // Do not rethrow here, let the caller handle it.
            return;
        }
        server.serverCertificateManager.referenceCounter++;
        const applicationName = (0, node_opcua_client_1.coerceLocalizedText)(server.serverInfo.applicationName)?.text || undefined;
        const prefix = `Client-${g_registeringClientCounter++}`;
        const action = isOnline ? "registering" : "unregistering";
        const ldsInfo = this.discoveryServerEndpointUrl;
        const serverInfo = server.serverInfo.applicationUri;
        const clientName = `${prefix} for server ${serverInfo} to LDS ${ldsInfo} for ${action}`;
        const client = node_opcua_client_1.OPCUAClientBase.create({
            clientName,
            applicationName,
            applicationUri: server.serverInfo.applicationUri,
            connectionStrategy: no_retry_connectivity_strategy,
            clientCertificateManager: server.serverCertificateManager,
            securityMode: selectedEndpoint.securityMode,
            securityPolicy: (0, node_opcua_secure_channel_1.coerceSecurityPolicy)(selectedEndpoint.securityPolicyUri),
            serverCertificate: selectedEndpoint.serverCertificate,
            certificateFile: server.certificateFile,
            privateKeyFile: server.privateKeyFile,
        });
        client.on("backoff", (nbRetry, delay) => {
            debugLog(client.clientCertificateManager, "backoff trying to connect to the LDS has failed", nbRetry, delay);
        });
        this._registration_client = client;
        debugLog("lds endpoint uri : ", selectedEndpoint.endpointUrl);
        const state = isOnline ? "UnRegisterServer" : "RegisterServer";
        try {
            await client.connect(selectedEndpoint.endpointUrl);
            // Check state again after connection is established
            if (this.getState() === expectedState) {
                try {
                    await sendRegisterServerRequest(server, client, isOnline);
                }
                catch (err) {
                    if (this.getState() !== expectedState) {
                        warningLog(`${state} '${this.server.serverInfo.applicationUri}' to the LDS has failed during secure connection to the LDS server`);
                        warningLog(chalk_1.default.red("  Error message:"), err.message); // Do not rethrow here, let the caller 
                    }
                }
            }
            else {
                debugLog("RegisterServerManager#_registerServer: aborted ");
            }
        }
        catch (err) {
            if (this.getState() !== expectedState) {
                warningLog(`${state} '${this.server.serverInfo.applicationUri}' cannot connect to LDS at endpoint ${client.clientName}, ${selectedEndpoint.endpointUrl} :`);
                warningLog(chalk_1.default.red("  Error message:"), err.message);
            }
        }
        finally {
            if (this._registration_client) {
                const tmp = this._registration_client;
                this._registration_client = null;
                await tmp.disconnect();
            }
        }
    }
    /**
     * Cancels any pending client connections.
     * This is crucial for a clean shutdown.
     * @private
     */
    async #_cancel_pending_client_if_any() {
        debugLog("RegisterServerManager#_cancel_pending_client_if_any");
        if (this._registration_client) {
            const client = this._registration_client;
            this._registration_client = null;
            debugLog("RegisterServerManager#_cancel_pending_client_if_any "
                + "=> wee need to disconnect_registration_client");
            try {
                await client.disconnect();
            }
            catch (err) {
                warningLog("Error disconnecting registration client:", err.message);
            }
            await this.#_cancel_pending_client_if_any(); // Recursive call to ensure all are handled
        }
        else {
            debugLog("RegisterServerManager#_cancel_pending_client_if_any : done (nothing to do)");
        }
    }
}
exports.RegisterServerManager = RegisterServerManager;
//# sourceMappingURL=register_server_manager.js.map