"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.OPCUAServer = exports.RegisterServerMethod = void 0;
exports.filterDiagnosticInfo = filterDiagnosticInfo;
/* eslint-disable complexity */
/**
 * @module node-opcua-server
 */
const crypto_1 = require("crypto");
const util_1 = require("util");
const async_1 = __importDefault(require("async"));
const chalk_1 = __importDefault(require("chalk"));
const node_opcua_hostname_1 = require("node-opcua-hostname");
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_utils_1 = require("node-opcua-utils");
const node_opcua_address_space_1 = require("node-opcua-address-space");
const node_opcua_certificate_manager_1 = require("node-opcua-certificate-manager");
const node_opcua_common_1 = require("node-opcua-common");
const web_1 = require("node-opcua-crypto/web");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_data_value_1 = require("node-opcua-data-value");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_object_registry_1 = require("node-opcua-object-registry");
const node_opcua_secure_channel_1 = require("node-opcua-secure-channel");
const node_opcua_service_browse_1 = require("node-opcua-service-browse");
const node_opcua_service_call_1 = require("node-opcua-service-call");
const node_opcua_service_endpoints_1 = require("node-opcua-service-endpoints");
const node_opcua_service_history_1 = require("node-opcua-service-history");
const node_opcua_service_node_management_1 = require("node-opcua-service-node-management");
const node_opcua_service_query_1 = require("node-opcua-service-query");
const node_opcua_service_read_1 = require("node-opcua-service-read");
const node_opcua_service_register_node_1 = require("node-opcua-service-register-node");
const node_opcua_service_session_1 = require("node-opcua-service-session");
const node_opcua_service_subscription_1 = require("node-opcua-service-subscription");
const node_opcua_service_translate_browse_path_1 = require("node-opcua-service-translate-browse-path");
const node_opcua_service_write_1 = require("node-opcua-service-write");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_types_1 = require("node-opcua-types");
const node_opcua_variant_1 = require("node-opcua-variant");
const node_opcua_variant_2 = require("node-opcua-variant");
const node_opcua_utils_2 = require("node-opcua-utils");
const base_server_1 = require("./base_server");
const factory_1 = require("./factory");
const monitored_item_1 = require("./monitored_item");
const register_server_manager_1 = require("./register_server_manager");
const register_server_manager_hidden_1 = require("./register_server_manager_hidden");
const register_server_manager_mdns_only_1 = require("./register_server_manager_mdns_only");
const server_end_point_1 = require("./server_end_point");
const server_engine_1 = require("./server_engine");
const user_manager_1 = require("./user_manager");
const user_manager_ua_1 = require("./user_manager_ua");
function isSubscriptionIdInvalid(subscriptionId) {
    return subscriptionId < 0 || subscriptionId >= 0xffffffff;
}
// tslint:disable-next-line:no-var-requires
const thenify_ex_1 = require("thenify-ex");
// tslint:disable-next-line:no-var-requires
const package_info = require("../package.json");
const debugLog = (0, node_opcua_debug_1.make_debugLog)(__filename);
const errorLog = (0, node_opcua_debug_1.make_errorLog)(__filename);
const warningLog = (0, node_opcua_debug_1.make_warningLog)(__filename);
const default_maxConnectionsPerEndpoint = 10;
function g_sendError(channel, message, ResponseClass, statusCode) {
    const response = new node_opcua_types_1.ServiceFault({
        responseHeader: { serviceResult: statusCode }
    });
    return channel.send_response("MSG", response, message);
}
const default_build_info = {
    manufacturerName: "NodeOPCUA : MIT Licence ( see http://node-opcua.github.io/)",
    productName: "NodeOPCUA-Server",
    productUri: null, // << should be same as default_server_info.productUri?
    softwareVersion: package_info.version,
    buildNumber: "0",
    buildDate: new Date(2020, 1, 1)
    // xx buildDate: fs.statSync(package_json_file).mtime
};
const minSessionTimeout = 100; // 100 milliseconds
const defaultSessionTimeout = 1000 * 30; // 30 seconds
const maxSessionTimeout = 1000 * 60 * 50; // 50 minutes
let unnamed_session_count = 0;
function _adjust_session_timeout(sessionTimeout) {
    let revisedSessionTimeout = sessionTimeout || defaultSessionTimeout;
    revisedSessionTimeout = Math.min(revisedSessionTimeout, maxSessionTimeout);
    revisedSessionTimeout = Math.max(revisedSessionTimeout, minSessionTimeout);
    return revisedSessionTimeout;
}
function channel_has_session(channel, session) {
    if (session.channel === channel) {
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(channel.sessionTokens, session.authenticationToken.toString()));
        return true;
    }
    return false;
}
function moveSessionToChannel(session, channel) {
    debugLog("moveSessionToChannel sessionId", session.nodeId, " channelId=", channel.channelId);
    if (session.publishEngine) {
        session.publishEngine.cancelPendingPublishRequestBeforeChannelChange();
    }
    session._detach_channel();
    session._attach_channel(channel);
    (0, node_opcua_assert_1.assert)(session.channel.channelId === channel.channelId);
}
async function _attempt_to_close_some_old_unactivated_session(server) {
    const session = server.engine.getOldestInactiveSession();
    if (session) {
        await server.engine.closeSession(session.authenticationToken, false, "Forcing");
    }
}
function getRequiredEndpointInfo(endpoint) {
    (0, node_opcua_assert_1.assert)(endpoint instanceof node_opcua_types_1.EndpointDescription);
    // https://reference.opcfoundation.org/v104/Core/docs/Part4/5.6.2/
    // https://reference.opcfoundation.org/v105/Core/docs/Part4/5.6.2/
    const e = new node_opcua_types_1.EndpointDescription({
        endpointUrl: endpoint.endpointUrl,
        securityLevel: endpoint.securityLevel,
        securityMode: endpoint.securityMode,
        securityPolicyUri: endpoint.securityPolicyUri,
        server: {
            applicationUri: endpoint.server.applicationUri,
            applicationType: endpoint.server.applicationType,
            applicationName: endpoint.server.applicationName,
            productUri: endpoint.server.productUri
        },
        transportProfileUri: endpoint.transportProfileUri,
        userIdentityTokens: endpoint.userIdentityTokens
    });
    // reduce even further by explicitly setting unwanted members to null
    e.server.applicationName = null;
    // xx e.server.applicationType = null as any;
    e.server.gatewayServerUri = null;
    e.server.discoveryProfileUri = null;
    e.server.discoveryUrls = null;
    e.serverCertificate = null;
    return e;
}
// serverUri  String This value is only specified if the EndpointDescription has a gatewayServerUri.
//            This value is the applicationUri from the EndpointDescription which is the applicationUri for the
//            underlying Server. The type EndpointDescription is defined in 7.10.
function _serverEndpointsForCreateSessionResponse(server, endpointUrl, serverUri) {
    serverUri = null; // unused then
    // https://reference.opcfoundation.org/v104/Core/docs/Part4/5.6.2/
    // https://reference.opcfoundation.org/v105/Core/docs/Part4/5.6.2/
    return server
        ._get_endpoints(endpointUrl)
        .filter((e) => !e.restricted) // remove restricted endpoints
        .filter((e) => (0, node_opcua_utils_2.matchUri)(e.endpointUrl, endpointUrl))
        .map(getRequiredEndpointInfo);
}
function adjustSecurityPolicy(channel, userTokenPolicy_securityPolicyUri) {
    // check that userIdentityToken
    let securityPolicy = (0, node_opcua_secure_channel_1.fromURI)(userTokenPolicy_securityPolicyUri);
    // if the security policy is not specified we use the session security policy
    if (securityPolicy === node_opcua_secure_channel_1.SecurityPolicy.Invalid) {
        securityPolicy = (0, node_opcua_secure_channel_1.fromURI)(channel.securityPolicy);
        (0, node_opcua_assert_1.assert)(securityPolicy !== node_opcua_secure_channel_1.SecurityPolicy.Invalid);
    }
    return securityPolicy;
}
function findUserTokenByPolicy(endpoint_description, userTokenType, policyId) {
    (0, node_opcua_assert_1.assert)(endpoint_description instanceof node_opcua_types_1.EndpointDescription);
    const r = endpoint_description.userIdentityTokens.filter((userIdentity) => userIdentity.tokenType === userTokenType && (!policyId || userIdentity.policyId === policyId));
    return r.length === 0 ? null : r[0];
}
function findUserTokenPolicy(endpoint_description, userTokenType) {
    (0, node_opcua_assert_1.assert)(endpoint_description instanceof node_opcua_types_1.EndpointDescription);
    const r = endpoint_description.userIdentityTokens.filter((userIdentity) => {
        (0, node_opcua_assert_1.assert)(userIdentity.tokenType !== undefined);
        return userIdentity.tokenType === userTokenType;
    });
    return r.length === 0 ? null : r[0];
}
function createAnonymousIdentityToken(endpoint_desc) {
    (0, node_opcua_assert_1.assert)(endpoint_desc instanceof node_opcua_types_1.EndpointDescription);
    const userTokenPolicy = findUserTokenPolicy(endpoint_desc, node_opcua_service_endpoints_1.UserTokenType.Anonymous);
    if (!userTokenPolicy) {
        throw new Error("Cannot find ANONYMOUS user token policy in end point description");
    }
    return new node_opcua_service_session_1.AnonymousIdentityToken({ policyId: userTokenPolicy.policyId });
}
function sameIdentityToken(token1, token2) {
    if (token1 instanceof node_opcua_service_session_1.UserNameIdentityToken) {
        if (!(token2 instanceof node_opcua_service_session_1.UserNameIdentityToken)) {
            return false;
        }
        if (token1.userName !== token2.userName) {
            return false;
        }
        if (token1.password.toString("hex") !== token2.password.toString("hex")) {
            // note pasword hash may be different from two request and cannot be verified at this stage
            // we assume that we have a valid password
            // NOT CALLING return false;
        }
        return true;
    }
    else if (token1 instanceof node_opcua_service_session_1.AnonymousIdentityToken) {
        if (!(token2 instanceof node_opcua_service_session_1.AnonymousIdentityToken)) {
            return false;
        }
        if (token1.policyId !== token2.policyId) {
            return false;
        }
        return true;
    }
    (0, node_opcua_assert_1.assert)(false, " Not implemented yet");
    return false;
}
function getTokenType(userIdentityToken) {
    if (userIdentityToken instanceof node_opcua_service_session_1.AnonymousIdentityToken) {
        return node_opcua_service_endpoints_1.UserTokenType.Anonymous;
    }
    else if (userIdentityToken instanceof node_opcua_service_session_1.UserNameIdentityToken) {
        return node_opcua_service_endpoints_1.UserTokenType.UserName;
    }
    else if (userIdentityToken instanceof node_opcua_types_1.IssuedIdentityToken) {
        return node_opcua_service_endpoints_1.UserTokenType.IssuedToken;
    }
    else if (userIdentityToken instanceof node_opcua_service_session_1.X509IdentityToken) {
        return node_opcua_service_endpoints_1.UserTokenType.Certificate;
    }
    return node_opcua_service_endpoints_1.UserTokenType.Invalid;
}
function thumbprint(certificate) {
    return certificate ? certificate.toString("base64") : "";
}
/*=== private
 *
 * perform the read operation on a given node for a monitored item.
 * this method DOES NOT apply to Variable Values attribute
 *
 * @param self
 * @param oldValue
 * @param node
 * @param itemToMonitor
 * @private
 */
function monitoredItem_read_and_record_value(self, context, oldValue, node, itemToMonitor, callback) {
    (0, node_opcua_assert_1.assert)(self instanceof monitored_item_1.MonitoredItem);
    (0, node_opcua_assert_1.assert)(oldValue instanceof node_opcua_data_value_1.DataValue);
    (0, node_opcua_assert_1.assert)(itemToMonitor.attributeId === node_opcua_data_model_1.AttributeIds.Value);
    const dataValue = node.readAttribute(context, itemToMonitor.attributeId, itemToMonitor.indexRange, itemToMonitor.dataEncoding);
    callback(null, dataValue);
}
/*== private
 * this method applies to Variable Values attribute
 * @private
 */
function monitoredItem_read_and_record_value_async(self, context, oldValue, node, itemToMonitor, callback) {
    (0, node_opcua_assert_1.assert)(context instanceof node_opcua_address_space_1.SessionContext);
    (0, node_opcua_assert_1.assert)(itemToMonitor.attributeId === node_opcua_data_model_1.AttributeIds.Value);
    (0, node_opcua_assert_1.assert)(self instanceof monitored_item_1.MonitoredItem);
    (0, node_opcua_assert_1.assert)(oldValue instanceof node_opcua_data_value_1.DataValue);
    // do it asynchronously ( this is only valid for value attributes )
    (0, node_opcua_assert_1.assert)(itemToMonitor.attributeId === node_opcua_data_model_1.AttributeIds.Value);
    node.readValueAsync(context, (err, dataValue) => {
        callback(err, dataValue);
    });
}
function build_scanning_node_function(addressSpace, itemToMonitor) {
    (0, node_opcua_assert_1.assert)(itemToMonitor instanceof node_opcua_service_read_1.ReadValueId);
    const node = addressSpace.findNode(itemToMonitor.nodeId);
    /* istanbul ignore next */
    if (!node) {
        errorLog(" INVALID NODE ID  , ", itemToMonitor.nodeId.toString());
        (0, node_opcua_debug_1.dump)(itemToMonitor);
        return (_sessionContext, _oldData, callback) => {
            callback(null, new node_opcua_data_value_1.DataValue({
                statusCode: node_opcua_status_code_1.StatusCodes.BadNodeIdUnknown,
                value: { dataType: node_opcua_variant_1.DataType.Null, value: 0 }
            }));
        };
    }
    ///// !!monitoredItem.setNode(node);
    if (itemToMonitor.attributeId === node_opcua_data_model_1.AttributeIds.Value) {
        const monitoredItem_read_and_record_value_func = itemToMonitor.attributeId === node_opcua_data_model_1.AttributeIds.Value && typeof node.readValueAsync === "function"
            ? monitoredItem_read_and_record_value_async
            : monitoredItem_read_and_record_value;
        return function func(sessionContext, oldDataValue, callback) {
            (0, node_opcua_assert_1.assert)(this instanceof monitored_item_1.MonitoredItem);
            (0, node_opcua_assert_1.assert)(oldDataValue instanceof node_opcua_data_value_1.DataValue);
            (0, node_opcua_assert_1.assert)(typeof callback === "function");
            monitoredItem_read_and_record_value_func(this, sessionContext, oldDataValue, node, itemToMonitor, callback);
        };
    }
    else {
        // Attributes, other than the  Value  Attribute, are only monitored for a change in value.
        // The filter is not used for these  Attributes. Any change in value for these  Attributes
        // causes a  Notification  to be  generated.
        // only record value when it has changed
        return function func(sessionContext, oldDataValue, callback) {
            (0, node_opcua_assert_1.assert)(this instanceof monitored_item_1.MonitoredItem);
            (0, node_opcua_assert_1.assert)(oldDataValue instanceof node_opcua_data_value_1.DataValue);
            (0, node_opcua_assert_1.assert)(typeof callback === "function");
            const newDataValue = node.readAttribute(sessionContext, itemToMonitor.attributeId);
            callback(null, newDataValue);
        };
    }
}
function prepareMonitoredItem(context, addressSpace, monitoredItem) {
    const itemToMonitor = monitoredItem.itemToMonitor;
    const readNodeFunc = build_scanning_node_function(addressSpace, itemToMonitor);
    monitoredItem.samplingFunc = readNodeFunc;
}
function isMonitoringModeValid(monitoringMode) {
    (0, node_opcua_assert_1.assert)(node_opcua_types_1.MonitoringMode.Invalid !== undefined);
    return monitoringMode !== node_opcua_types_1.MonitoringMode.Invalid && monitoringMode <= node_opcua_types_1.MonitoringMode.Reporting;
}
function _installRegisterServerManager(self) {
    (0, node_opcua_assert_1.assert)(self instanceof OPCUAServer);
    (0, node_opcua_assert_1.assert)(!self.registerServerManager);
    /* istanbul ignore next */
    if (!self.registerServerMethod) {
        throw new Error("Internal Error");
    }
    switch (self.registerServerMethod) {
        case RegisterServerMethod.HIDDEN:
            self.registerServerManager = new register_server_manager_hidden_1.RegisterServerManagerHidden({
                server: self
            });
            break;
        case RegisterServerMethod.MDNS:
            self.registerServerManager = new register_server_manager_mdns_only_1.RegisterServerManagerMDNSONLY({
                server: self
            });
            break;
        case RegisterServerMethod.LDS:
            self.registerServerManager = new register_server_manager_1.RegisterServerManager({
                discoveryServerEndpointUrl: self.discoveryServerEndpointUrl,
                server: self
            });
            break;
        /* istanbul ignore next */
        default:
            throw new Error("Invalid switch");
    }
    self.registerServerManager.on("serverRegistrationPending", () => {
        /**
         * emitted when the server is trying to registered the LDS
         * but when the connection to the lds has failed
         * serverRegistrationPending is sent when the backoff signal of the
         * connection process is raised
         * @event serverRegistrationPending
         */
        debugLog("serverRegistrationPending");
        self.emit("serverRegistrationPending");
    });
    self.registerServerManager.on("serverRegistered", () => {
        /**
         * emitted when the server is successfully registered to the LDS
         * @event serverRegistered
         */
        debugLog("serverRegistered");
        self.emit("serverRegistered");
    });
    self.registerServerManager.on("serverRegistrationRenewed", () => {
        /**
         * emitted when the server has successfully renewed its registration to the LDS
         * @event serverRegistrationRenewed
         */
        debugLog("serverRegistrationRenewed");
        self.emit("serverRegistrationRenewed");
    });
    self.registerServerManager.on("serverUnregistered", () => {
        debugLog("serverUnregistered");
        /**
         * emitted when the server is successfully unregistered to the LDS
         * ( for instance during shutdown)
         * @event serverUnregistered
         */
        self.emit("serverUnregistered");
    });
}
function validate_applicationUri(channel, request) {
    const applicationUri = request.clientDescription.applicationUri;
    const clientCertificate = request.clientCertificate;
    // if session is insecure there is no need to check certificate information
    if (channel.securityMode === node_opcua_secure_channel_1.MessageSecurityMode.None) {
        return true; // assume correct
    }
    if (!clientCertificate || clientCertificate.length === 0) {
        return true; // can't check
    }
    const e = (0, web_1.exploreCertificate)(clientCertificate);
    const uniformResourceIdentifier = e.tbsCertificate.extensions.subjectAltName?.uniformResourceIdentifier ?? null;
    const applicationUriFromCert = uniformResourceIdentifier && uniformResourceIdentifier.length > 0 ? uniformResourceIdentifier[0] : null;
    /* istanbul ignore next */
    if (applicationUriFromCert !== applicationUri) {
        errorLog("BadCertificateUriInvalid!");
        errorLog("applicationUri           = ", applicationUri);
        errorLog("applicationUriFromCert   = ", applicationUriFromCert);
    }
    return applicationUriFromCert === applicationUri;
}
function validate_security_endpoint(server, request, channel) {
    debugLog("validate_security_endpoint = ", request.endpointUrl);
    let endpoints = server._get_endpoints(request.endpointUrl);
    // endpointUrl String The network address that the Client used to access the Session Endpoint.
    //             The HostName portion of the URL should be one of the HostNames for the application that are
    //             specified in the Server’s ApplicationInstanceCertificate (see 7.2). The Server shall raise an
    //             AuditUrlMismatchEventType event if the URL does not match the Server’s HostNames.
    //             AuditUrlMismatchEventType event type is defined in Part 5.
    //             The Server uses this information for diagnostics and to determine the set of
    //             EndpointDescriptions to return in the response.
    // ToDo: check endpointUrl validity and emit an AuditUrlMismatchEventType event if not
    // sometime endpoints have a extra leading "/" that can be ignored
    // don't be too harsh.
    if (endpoints.length === 0 && request.endpointUrl?.endsWith("/")) {
        endpoints = server._get_endpoints(request.endpointUrl.slice(0, -1));
    }
    if (endpoints.length === 0) {
        // we have a UrlMismatch here
        const ua_server = server.engine.addressSpace.rootFolder.objects.server;
        if (!request.endpointUrl?.match(/localhost/i) || OPCUAServer.requestExactEndpointUrl) {
            warningLog("Cannot find suitable endpoints in available endpoints. endpointUri =", request.endpointUrl);
        }
        ua_server.raiseEvent("AuditUrlMismatchEventType", {
            endpointUrl: { dataType: node_opcua_variant_1.DataType.String, value: request.endpointUrl }
        });
        if (OPCUAServer.requestExactEndpointUrl) {
            return { errCode: node_opcua_status_code_1.StatusCodes.BadServiceUnsupported };
        }
        else {
            endpoints = server._get_endpoints(null);
        }
    }
    // ignore restricted endpoints
    endpoints = endpoints.filter((e) => !e.restricted);
    const endpoints_matching_security_mode = endpoints.filter((e) => {
        return e.securityMode === channel.securityMode;
    });
    if (endpoints_matching_security_mode.length === 0) {
        return { errCode: node_opcua_status_code_1.StatusCodes.BadSecurityModeRejected };
    }
    const endpoints_matching_security_policy = endpoints_matching_security_mode.filter((e) => {
        return e.securityPolicyUri === channel.securityPolicy;
    });
    if (endpoints_matching_security_policy.length === 0) {
        return { errCode: node_opcua_status_code_1.StatusCodes.BadSecurityPolicyRejected };
    }
    if (endpoints_matching_security_policy.length !== 1) {
        debugLog("endpoints_matching_security_policy= ", endpoints_matching_security_policy.length);
    }
    return { errCode: node_opcua_status_code_1.StatusCodes.Good, endpoint: endpoints_matching_security_policy[0] };
}
function filterDiagnosticInfo(returnDiagnostics, response) {
    if (node_opcua_data_model_1.RESPONSE_DIAGNOSTICS_MASK_ALL & returnDiagnostics) {
        response.responseHeader.serviceDiagnostics = (0, node_opcua_data_model_1.filterDiagnosticServiceLevel)(returnDiagnostics, response.responseHeader.serviceDiagnostics);
        if (response.diagnosticInfos && response.diagnosticInfos.length > 0) {
            response.diagnosticInfos = response.diagnosticInfos.map((d) => (0, node_opcua_data_model_1.filterDiagnosticOperationLevel)(returnDiagnostics, d));
        }
        else {
            response.diagnosticInfos = [];
        }
        if (response.results) {
            for (const entry of response.results) {
                if (entry.inputArgumentDiagnosticInfos && entry.inputArgumentDiagnosticInfos.length > 0) {
                    entry.inputArgumentDiagnosticInfos = entry.inputArgumentDiagnosticInfos.map((d) => (0, node_opcua_data_model_1.filterDiagnosticOperationLevel)(returnDiagnostics, d));
                }
                else {
                    entry.inputArgumentDiagnosticInfos = [];
                }
            }
        }
    }
}
var RegisterServerMethod;
(function (RegisterServerMethod) {
    RegisterServerMethod[RegisterServerMethod["HIDDEN"] = 1] = "HIDDEN";
    RegisterServerMethod[RegisterServerMethod["MDNS"] = 2] = "MDNS";
    RegisterServerMethod[RegisterServerMethod["LDS"] = 3] = "LDS"; // the server registers itself to the LDS or LDS-ME (Local Discovery Server)
})(RegisterServerMethod || (exports.RegisterServerMethod = RegisterServerMethod = {}));
const g_requestExactEndpointUrl = !!process.env.NODEOPCUA_SERVER_REQUEST_EXACT_ENDPOINT_URL;
/**
 *
 */
class OPCUAServer extends base_server_1.OPCUABaseServer {
    static defaultShutdownTimeout = 100; // 250 ms
    /**
     * if requestExactEndpointUrl is set to true the server will only accept createSession that have a endpointUrl that strictly matches
     * one of the provided endpoint.
     * This mean that if the server expose a endpoint with url such as opc.tcp://MYHOSTNAME:1234, client will not be able to reach the server
     * with the ip address of the server.
     * requestExactEndpointUrl = true => emulates the Prosys Server behavior
     * requestExactEndpointUrl = false => emulates the Unified Automation behavior.
     */
    static requestExactEndpointUrl = g_requestExactEndpointUrl;
    /**
     * total number of bytes written  by the server since startup
     */
    get bytesWritten() {
        return this.endpoints.reduce((accumulated, endpoint) => {
            return accumulated + endpoint.bytesWritten;
        }, 0);
    }
    /**
     * total number of bytes read  by the server since startup
     */
    get bytesRead() {
        return this.endpoints.reduce((accumulated, endpoint) => {
            return accumulated + endpoint.bytesRead;
        }, 0);
    }
    /**
     * Number of transactions processed by the server since startup
     */
    get transactionsCount() {
        return this.endpoints.reduce((accumulated, endpoint) => {
            return accumulated + endpoint.transactionsCount;
        }, 0);
    }
    /**
     * The server build info
     */
    get buildInfo() {
        return this.engine.buildInfo;
    }
    /**
     * the number of connected channel on all existing end points
     */
    get currentChannelCount() {
        // TODO : move to base
        return this.endpoints.reduce((currentValue, endPoint) => {
            return currentValue + endPoint.currentChannelCount;
        }, 0);
    }
    /**
     * The number of active subscriptions from all sessions
     */
    get currentSubscriptionCount() {
        return this.engine ? this.engine.currentSubscriptionCount : 0;
    }
    /**
     * the number of session activation requests that have been rejected
     */
    get rejectedSessionCount() {
        return this.engine ? this.engine.rejectedSessionCount : 0;
    }
    /**
     * the number of request that have been rejected
     */
    get rejectedRequestsCount() {
        return this.engine ? this.engine.rejectedRequestsCount : 0;
    }
    /**
     * the number of sessions that have been aborted
     */
    get sessionAbortCount() {
        return this.engine ? this.engine.sessionAbortCount : 0;
    }
    /**
     * the publishing interval count
     */
    get publishingIntervalCount() {
        return this.engine ? this.engine.publishingIntervalCount : 0;
    }
    /**
     * the number of sessions currently active
     */
    get currentSessionCount() {
        return this.engine ? this.engine.currentSessionCount : 0;
    }
    /**
     * true if the server has been initialized
     *
     */
    get initialized() {
        return this.engine && this.engine.addressSpace !== null;
    }
    /**
     * is the server auditing ?
     */
    get isAuditing() {
        return this.engine ? this.engine.isAuditing : false;
    }
    static registry = new node_opcua_object_registry_1.ObjectRegistry();
    static fallbackSessionName = "Client didn't provide a meaningful sessionName ...";
    /**
     * the maximum number of subscription that can be created per server
     * @deprecated
     */
    static deprecated_MAX_SUBSCRIPTION = 50;
    /**
     * the maximum number of concurrent sessions allowed on the server
     */
    get maxAllowedSessionNumber() {
        return this.engine.serverCapabilities.maxSessions;
    }
    /**
     * the maximum number for concurrent connection per end point
     */
    maxConnectionsPerEndpoint;
    /**
     * false if anonymous connection are not allowed
     */
    allowAnonymous = false;
    /**
     * the user manager
     */
    userManager;
    options;
    objectFactory;
    _delayInit;
    constructor(options) {
        super(options);
        this.allowAnonymous = false;
        options = options || {};
        this.options = options;
        if (options.maxAllowedSessionNumber !== undefined) {
            warningLog("[NODE-OPCUA-W21] maxAllowedSessionNumber property is now deprecated , please use serverCapabilities.maxSessions instead");
            options.serverCapabilities = options.serverCapabilities || {};
            options.serverCapabilities.maxSessions = options.maxAllowedSessionNumber;
        }
        // adjust securityPolicies if any
        if (options.securityPolicies) {
            options.securityPolicies = options.securityPolicies.map(node_opcua_secure_channel_1.coerceSecurityPolicy);
        }
        /**
         * @property maxConnectionsPerEndpoint
         */
        this.maxConnectionsPerEndpoint = options.maxConnectionsPerEndpoint || default_maxConnectionsPerEndpoint;
        // build Info
        const buildInfo = {
            ...default_build_info,
            ...options.buildInfo
        };
        // repair product name
        buildInfo.productUri = buildInfo.productUri || this.serverInfo.productUri;
        this.serverInfo.productUri = this.serverInfo.productUri || buildInfo.productUri;
        this.userManager = (0, user_manager_1.makeUserManager)(options.userManager);
        options.allowAnonymous = options.allowAnonymous === undefined ? true : !!options.allowAnonymous;
        /**
         * @property allowAnonymous
         */
        this.allowAnonymous = options.allowAnonymous;
        this.discoveryServerEndpointUrl = options.discoveryServerEndpointUrl || "opc.tcp://%FQDN%:4840";
        (0, node_opcua_assert_1.assert)(typeof this.discoveryServerEndpointUrl === "string");
        this.serverInfo.applicationType =
            options.serverInfo?.applicationType === undefined ? node_opcua_service_endpoints_1.ApplicationType.Server : options.serverInfo.applicationType;
        this.capabilitiesForMDNS = options.capabilitiesForMDNS || ["NA"];
        this.registerServerMethod = options.registerServerMethod || RegisterServerMethod.HIDDEN;
        _installRegisterServerManager(this);
        if (!options.userCertificateManager) {
            this.userCertificateManager = (0, node_opcua_certificate_manager_1.getDefaultCertificateManager)("UserPKI");
        }
        else {
            this.userCertificateManager = options.userCertificateManager;
        }
        // note: we need to delay initialization of endpoint as certain resources
        // such as %FQDN% might not be ready yet at this stage
        this._delayInit = async () => {
            /* istanbul ignore next */
            if (!options) {
                throw new Error("Internal Error");
            }
            // to check => this.serverInfo.applicationName = this.serverInfo.productName || buildInfo.productName;
            // note: applicationUri is handled in a special way
            this.engine = new server_engine_1.ServerEngine({
                applicationUri: () => this.serverInfo.applicationUri,
                buildInfo,
                isAuditing: options.isAuditing,
                serverCapabilities: options.serverCapabilities,
                serverConfiguration: {
                    serverCapabilities: () => {
                        return this.capabilitiesForMDNS || ["NA"];
                    },
                    supportedPrivateKeyFormat: ["PEM"],
                    applicationType: () => this.serverInfo.applicationType,
                    applicationUri: () => this.serverInfo.applicationUri || "",
                    productUri: () => this.serverInfo.productUri || "",
                    // hasSecureElement: () => false,
                    multicastDnsEnabled: () => this.registerServerMethod === RegisterServerMethod.MDNS
                }
            });
            this.objectFactory = new factory_1.Factory(this.engine);
            const endpointDefinitions = [
                ...(options.endpoints || []),
                ...(options.alternateEndpoints || [])
            ];
            const hostname = (0, node_opcua_hostname_1.getFullyQualifiedDomainName)();
            endpointDefinitions.forEach((endpointDefinition) => {
                endpointDefinition.port = endpointDefinition.port === undefined ? 26543 : endpointDefinition.port;
                endpointDefinition.hostname = endpointDefinition.hostname || hostname;
            });
            if (!options.endpoints) {
                endpointDefinitions.push({
                    port: options.port === undefined ? 26543 : options.port,
                    hostname: options.hostname || hostname,
                    host: options.host,
                    allowAnonymous: options.allowAnonymous,
                    alternateHostname: options.alternateHostname,
                    disableDiscovery: options.disableDiscovery,
                    securityModes: options.securityModes,
                    securityPolicies: options.securityPolicies
                });
            }
            // todo  should self.serverInfo.productUri  match self.engine.buildInfo.productUri ?
            for (const endpointOptions of endpointDefinitions) {
                const endPoint = this.createEndpointDescriptions(options, endpointOptions);
                this.endpoints.push(endPoint);
                endPoint.on("message", (message, channel) => {
                    this.on_request(message, channel);
                });
                // endPoint.on("error", (err: Error) => {
                //     errorLog("OPCUAServer endpoint error", err);
                //     // set serverState to ServerState.Failed;
                //     this.engine.setServerState(ServerState.Failed);
                //     this.shutdown(() => {
                //         /* empty */
                //     });
                // });
            }
        };
    }
    initialize(...args) {
        const done = args[0];
        (0, node_opcua_assert_1.assert)(!this.initialized, "server is already initialized"); // already initialized ?
        this._preInitTask.push(async () => {
            /* istanbul ignore else */
            if (this._delayInit) {
                await this._delayInit();
                this._delayInit = undefined;
            }
        });
        this.performPreInitialization()
            .then(() => {
            OPCUAServer.registry.register(this);
            this.engine.initialize(this.options, () => {
                (0, user_manager_ua_1.bindRoleSet)(this.userManager, this.engine.addressSpace);
                setImmediate(() => {
                    this.emit("post_initialize");
                    done();
                });
            });
        })
            .catch((err) => {
            done(err);
        });
    }
    start(...args) {
        const done = args[0];
        const tasks = [];
        tasks.push((0, util_1.callbackify)(node_opcua_hostname_1.extractFullyQualifiedDomainName));
        if (!this.initialized) {
            tasks.push((callback) => {
                this.initialize(callback);
            });
        }
        tasks.push((callback) => {
            super.start((err) => {
                if (err) {
                    this.shutdown((/*err2*/ err2) => {
                        callback(err);
                    });
                }
                else {
                    // we start the registration process asynchronously
                    // as we want to make server immediately available
                    this.registerServerManager.start().then(() => {
                        /* empty */
                    }).catch((err) => {
                        /* empty */
                    });
                    setImmediate(callback);
                }
            });
        });
        async_1.default.series(tasks, done);
    }
    shutdown(...args) {
        const timeout = args.length === 1 ? OPCUAServer.defaultShutdownTimeout : args[0];
        const callback = (args.length === 1 ? args[0] : args[1]);
        (0, node_opcua_assert_1.assert)(typeof callback === "function");
        debugLog("OPCUAServer#shutdown (timeout = ", timeout, ")");
        /* istanbul ignore next */
        if (!this.engine) {
            return callback();
        }
        (0, node_opcua_assert_1.assert)(this.engine);
        if (!this.engine.isStarted()) {
            // server may have been shot down already  , or may have fail to start !!
            const err = new Error("OPCUAServer#shutdown failure ! server doesn't seems to be started yet");
            return callback(err);
        }
        this.userCertificateManager.dispose();
        this.engine.setServerState(node_opcua_common_1.ServerState.Shutdown);
        const shutdownTime = new Date(Date.now() + timeout);
        this.engine.setShutdownTime(shutdownTime);
        debugLog("OPCUAServer is now un-registering itself from  the discovery server " + this.buildInfo);
        this.registerServerManager.stop()
            .then(() => {
            debugLog("OPCUAServer unregistered from discovery server successfully");
        })
            .catch((err) => {
            debugLog("OPCUAServer unregistered from discovery server with err: ", err.message);
        }).finally(() => {
            setTimeout(async () => {
                await this.engine.shutdown();
                debugLog("OPCUAServer#shutdown: started");
                base_server_1.OPCUABaseServer.prototype.shutdown.call(this, (err1) => {
                    debugLog("OPCUAServer#shutdown: completed");
                    this.dispose();
                    callback(err1);
                });
            }, timeout);
        });
    }
    dispose() {
        for (const endpoint of this.endpoints) {
            endpoint.dispose();
        }
        this.endpoints = [];
        this.removeAllListeners();
        if (this.registerServerManager) {
            this.registerServerManager.dispose();
            this.registerServerManager = undefined;
        }
        OPCUAServer.registry.unregister(this);
        /* istanbul ignore next */
        if (this.engine) {
            this.engine.dispose();
        }
    }
    raiseEvent(eventType, options) {
        /* istanbul ignore next */
        if (!this.engine.addressSpace) {
            errorLog("addressSpace missing");
            return;
        }
        const server = this.engine.addressSpace.findNode("Server");
        /* istanbul ignore next */
        if (!server) {
            // xx throw new Error("OPCUAServer#raiseEvent : cannot find Server object");
            return;
        }
        let eventTypeNode = eventType;
        if (typeof eventType === "string") {
            eventTypeNode = this.engine.addressSpace.findEventType(eventType);
            if (eventTypeNode) {
                return server.raiseEvent(eventTypeNode, options);
            }
            else {
                console.warn(" cannot find event type ", eventType);
            }
        }
        else {
            return server.raiseEvent(eventTypeNode, options);
        }
    }
    /**
     * create and register a new session
     * @private
     */
    createSession(options) {
        /* istanbul ignore next */
        if (!this.engine) {
            throw new Error("Internal Error");
        }
        return this.engine.createSession(options);
    }
    /**
     * retrieve a session by authentication token
     * @private
     */
    getSession(authenticationToken, activeOnly) {
        return this.engine ? this.engine.getSession(authenticationToken, activeOnly) : null;
    }
    /**
     *
     * @param channel
     * @param clientCertificate
     * @param clientNonce
     * @private
     */
    computeServerSignature(channel, clientCertificate, clientNonce) {
        return (0, node_opcua_secure_channel_1.computeSignature)(clientCertificate, clientNonce, this.getPrivateKey(), channel.securityPolicy);
    }
    /**
     *
     * @param session
     * @param channel
     * @param clientSignature
     */
    verifyClientSignature(session, channel, clientSignature) {
        const clientCertificate = channel.clientCertificate;
        const securityPolicy = channel.securityPolicy;
        const serverCertificate = this.getCertificate();
        const result = (0, node_opcua_secure_channel_1.verifySignature)(serverCertificate, session.nonce, clientSignature, clientCertificate, securityPolicy);
        return result;
    }
    isValidUserNameIdentityToken(channel, session, userTokenPolicy, userIdentityToken, userTokenSignature, callback) {
        (0, node_opcua_assert_1.assert)(userIdentityToken instanceof node_opcua_service_session_1.UserNameIdentityToken);
        const securityPolicy = adjustSecurityPolicy(channel, userTokenPolicy.securityPolicyUri);
        if (securityPolicy === node_opcua_secure_channel_1.SecurityPolicy.None) {
            return callback(null, node_opcua_status_code_1.StatusCodes.Good);
        }
        const cryptoFactory = (0, node_opcua_secure_channel_1.getCryptoFactory)(securityPolicy);
        /* istanbul ignore next */
        if (!cryptoFactory) {
            return callback(null, node_opcua_status_code_1.StatusCodes.BadSecurityPolicyRejected);
        }
        /* istanbul ignore next */
        if (userIdentityToken.encryptionAlgorithm !== cryptoFactory.asymmetricEncryptionAlgorithm) {
            errorLog("invalid encryptionAlgorithm");
            errorLog("userTokenPolicy", userTokenPolicy.toString());
            errorLog("userTokenPolicy", userIdentityToken.toString());
            return callback(null, node_opcua_status_code_1.StatusCodes.BadIdentityTokenInvalid);
        }
        const userName = userIdentityToken.userName;
        const password = userIdentityToken.password;
        if (!userName || !password) {
            return callback(null, node_opcua_status_code_1.StatusCodes.BadIdentityTokenInvalid);
        }
        return callback(null, node_opcua_status_code_1.StatusCodes.Good);
    }
    isValidX509IdentityToken(channel, session, userTokenPolicy, userIdentityToken, userTokenSignature, callback) {
        (0, node_opcua_assert_1.assert)(userIdentityToken instanceof node_opcua_service_session_1.X509IdentityToken);
        (0, node_opcua_assert_1.assert)(typeof callback === "function");
        const securityPolicy = adjustSecurityPolicy(channel, userTokenPolicy.securityPolicyUri);
        const cryptoFactory = (0, node_opcua_secure_channel_1.getCryptoFactory)(securityPolicy);
        /* istanbul ignore next */
        if (!cryptoFactory) {
            return callback(null, node_opcua_status_code_1.StatusCodes.BadSecurityPolicyRejected);
        }
        if (!userTokenSignature || !userTokenSignature.signature) {
            this.raiseEvent("AuditCreateSessionEventType", {});
            return callback(null, node_opcua_status_code_1.StatusCodes.BadUserSignatureInvalid);
        }
        if (userIdentityToken.policyId !== userTokenPolicy.policyId) {
            errorLog("invalid encryptionAlgorithm");
            errorLog("userTokenPolicy", userTokenPolicy.toString());
            errorLog("userTokenPolicy", userIdentityToken.toString());
            return callback(null, node_opcua_status_code_1.StatusCodes.BadSecurityPolicyRejected);
        }
        const certificate = userIdentityToken.certificateData; /* as Certificate*/
        const nonce = session.nonce;
        const serverCertificate = this.getCertificate();
        (0, node_opcua_assert_1.assert)(serverCertificate instanceof Buffer);
        (0, node_opcua_assert_1.assert)(certificate instanceof Buffer, "expecting certificate to be a Buffer");
        (0, node_opcua_assert_1.assert)(nonce instanceof Buffer, "expecting nonce to be a Buffer");
        (0, node_opcua_assert_1.assert)(userTokenSignature.signature instanceof Buffer, "expecting userTokenSignature to be a Buffer");
        // verify proof of possession by checking certificate signature & server nonce correctness
        if (!(0, node_opcua_secure_channel_1.verifySignature)(serverCertificate, nonce, userTokenSignature, certificate, securityPolicy)) {
            return callback(null, node_opcua_status_code_1.StatusCodes.BadUserSignatureInvalid);
        }
        // verify if certificate is Valid
        this.userCertificateManager.checkCertificate(certificate, (err, certificateStatus) => {
            /* istanbul ignore next */
            if (err) {
                return callback(err);
            }
            if (this.isAuditing) {
                switch (certificateStatus) {
                    case node_opcua_status_code_1.StatusCodes.Good:
                        break;
                    case node_opcua_status_code_1.StatusCodes.BadCertificateUntrusted:
                        this.raiseEvent("AuditCertificateUntrustedEventType", {
                            certificate: { dataType: node_opcua_variant_1.DataType.ByteString, value: certificate },
                            sourceName: { dataType: node_opcua_variant_1.DataType.String, value: "Security/Certificate" }
                        });
                        break;
                    case node_opcua_status_code_1.StatusCodes.BadCertificateTimeInvalid:
                    case node_opcua_status_code_1.StatusCodes.BadCertificateIssuerTimeInvalid:
                        this.raiseEvent("AuditCertificateExpiredEventType", {
                            certificate: { dataType: node_opcua_variant_1.DataType.ByteString, value: certificate },
                            sourceName: { dataType: node_opcua_variant_1.DataType.String, value: "Security/Certificate" }
                        });
                        break;
                    case node_opcua_status_code_1.StatusCodes.BadCertificateRevoked:
                    case node_opcua_status_code_1.StatusCodes.BadCertificateRevocationUnknown:
                    case node_opcua_status_code_1.StatusCodes.BadCertificateIssuerRevocationUnknown:
                        this.raiseEvent("AuditCertificateRevokedEventType", {
                            certificate: { dataType: node_opcua_variant_1.DataType.ByteString, value: certificate },
                            sourceName: { dataType: node_opcua_variant_1.DataType.String, value: "Security/Certificate" }
                        });
                        break;
                    case node_opcua_status_code_1.StatusCodes.BadCertificateIssuerUseNotAllowed:
                    case node_opcua_status_code_1.StatusCodes.BadCertificateUseNotAllowed:
                    case node_opcua_status_code_1.StatusCodes.BadSecurityChecksFailed:
                        this.raiseEvent("AuditCertificateMismatchEventType", {
                            certificate: { dataType: node_opcua_variant_1.DataType.ByteString, value: certificate },
                            sourceName: { dataType: node_opcua_variant_1.DataType.String, value: "Security/Certificate" }
                        });
                        break;
                }
            }
            if (certificateStatus &&
                (node_opcua_status_code_1.StatusCodes.BadCertificateUntrusted.equals(certificateStatus) ||
                    node_opcua_status_code_1.StatusCodes.BadCertificateTimeInvalid.equals(certificateStatus) ||
                    node_opcua_status_code_1.StatusCodes.BadCertificateIssuerTimeInvalid.equals(certificateStatus) ||
                    node_opcua_status_code_1.StatusCodes.BadCertificateIssuerUseNotAllowed.equals(certificateStatus) ||
                    node_opcua_status_code_1.StatusCodes.BadCertificateIssuerRevocationUnknown.equals(certificateStatus) ||
                    node_opcua_status_code_1.StatusCodes.BadCertificateRevocationUnknown.equals(certificateStatus) ||
                    node_opcua_status_code_1.StatusCodes.BadCertificateRevoked.equals(certificateStatus) ||
                    node_opcua_status_code_1.StatusCodes.BadCertificateUseNotAllowed.equals(certificateStatus) ||
                    node_opcua_status_code_1.StatusCodes.BadSecurityChecksFailed.equals(certificateStatus) ||
                    !node_opcua_status_code_1.StatusCodes.Good.equals(certificateStatus))) {
                debugLog("isValidX509IdentityToken => certificateStatus = ", certificateStatus?.toString());
                return callback(null, node_opcua_status_code_1.StatusCodes.BadIdentityTokenRejected);
            }
            if (node_opcua_status_code_1.StatusCodes.Good !== certificateStatus) {
                (0, node_opcua_assert_1.assert)(certificateStatus instanceof node_opcua_status_code_1.StatusCode);
                return callback(null, certificateStatus);
                // return callback(null, StatusCodes.BadIdentityTokenInvalid);
            }
            // verify if certificate is truster or rejected
            // todo: StatusCodes.BadCertificateUntrusted
            // store untrusted certificate to rejected folder
            // todo:
            return callback(null, node_opcua_status_code_1.StatusCodes.Good);
        });
    }
    /**
     * @internal
     */
    userNameIdentityTokenAuthenticateUser(channel, session, userTokenPolicy, userIdentityToken, callback) {
        (0, node_opcua_assert_1.assert)(userIdentityToken instanceof node_opcua_service_session_1.UserNameIdentityToken);
        // assert(this.isValidUserNameIdentityToken(channel, session, userTokenPolicy, userIdentityToken));
        const securityPolicy = adjustSecurityPolicy(channel, userTokenPolicy.securityPolicyUri);
        const userName = userIdentityToken.userName;
        let password = userIdentityToken.password;
        // decrypt password if necessary
        if (securityPolicy === node_opcua_secure_channel_1.SecurityPolicy.None) {
            // not good, password was sent in clear text ...
            password = password.toString();
        }
        else {
            const serverPrivateKey = this.getPrivateKey();
            const serverNonce = session.nonce;
            (0, node_opcua_assert_1.assert)(serverNonce instanceof Buffer);
            const cryptoFactory = (0, node_opcua_secure_channel_1.getCryptoFactory)(securityPolicy);
            /* istanbul ignore next */
            if (!cryptoFactory) {
                return callback(new Error(" Unsupported security Policy"));
            }
            const buff = cryptoFactory.asymmetricDecrypt(password, serverPrivateKey);
            // server certificate may be invalid and asymmetricDecrypt may fail
            if (!buff || buff.length < 4) {
                async_1.default.setImmediate(() => callback(null, false));
                return;
            }
            const length = buff.readUInt32LE(0) - serverNonce.length;
            password = buff.subarray(4, 4 + length).toString("utf-8");
        }
        this.userManager
            .isValidUser(session, userName, password)
            .then((isValid) => callback(null, isValid))
            .catch((err) => callback(err));
    }
    /**
     * @internal
     */
    isValidUserIdentityToken(channel, session, userIdentityToken, userTokenSignature, endpointDescription, callback) {
        (0, node_opcua_assert_1.assert)(typeof callback === "function");
        /* istanbul ignore next */
        if (!userIdentityToken) {
            throw new Error("Invalid token");
        }
        const userTokenType = getTokenType(userIdentityToken);
        const userTokenPolicy = findUserTokenByPolicy(endpointDescription, userTokenType, userIdentityToken.policyId);
        if (!userTokenPolicy) {
            // cannot find token with this policyId
            return callback(null, node_opcua_status_code_1.StatusCodes.BadIdentityTokenInvalid);
        }
        //
        if (userIdentityToken instanceof node_opcua_service_session_1.UserNameIdentityToken) {
            return this.isValidUserNameIdentityToken(channel, session, userTokenPolicy, userIdentityToken, userTokenSignature, callback);
        }
        if (userIdentityToken instanceof node_opcua_service_session_1.X509IdentityToken) {
            return this.isValidX509IdentityToken(channel, session, userTokenPolicy, userIdentityToken, userTokenSignature, callback);
        }
        return callback(null, node_opcua_status_code_1.StatusCodes.Good);
    }
    /**
     *
     * @internal
     * @param channel
     * @param session
     * @param userIdentityToken
     * @param callback
     * @returns {*}
     */
    isUserAuthorized(channel, session, userIdentityToken, callback) {
        (0, node_opcua_assert_1.assert)(userIdentityToken);
        (0, node_opcua_assert_1.assert)(typeof callback === "function");
        const userTokenType = getTokenType(userIdentityToken);
        const userTokenPolicy = findUserTokenByPolicy(session.getEndpointDescription(), userTokenType, userIdentityToken.policyId);
        /** istanbul ignore next */
        if (!userTokenPolicy) {
            return callback(null, false);
        }
        // find if a userToken exists
        if (userIdentityToken instanceof node_opcua_service_session_1.UserNameIdentityToken) {
            return this.userNameIdentityTokenAuthenticateUser(channel, session, userTokenPolicy, userIdentityToken, callback);
        }
        async_1.default.setImmediate(callback.bind(null, null, true));
    }
    makeServerNonce() {
        return (0, crypto_1.randomBytes)(32);
    }
    // session services
    // eslint-disable-next-line max-statements
    async _on_CreateSessionRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_session_1.CreateSessionRequest);
        function rejectConnection(server, statusCode) {
            server.engine.incrementSecurityRejectedSessionCount();
            const response1 = new node_opcua_types_1.ServiceFault({
                responseHeader: { serviceResult: statusCode }
            });
            channel.send_response("MSG", response1, message);
            // and close !
        }
        // From OPCUA V1.03 Part 4 5.6.2 CreateSession
        // A Server application should limit the number of Sessions. To protect against misbehaving Clients and denial
        // of service attacks, the Server shall close the oldest Session that is not activated before reaching the
        // maximum number of supported Sessions
        if (this.currentSessionCount >= this.engine.serverCapabilities.maxSessions) {
            await _attempt_to_close_some_old_unactivated_session(this);
        }
        // check if session count hasn't reach the maximum allowed sessions
        if (this.currentSessionCount >= this.engine.serverCapabilities.maxSessions) {
            return rejectConnection(this, node_opcua_status_code_1.StatusCodes.BadTooManySessions);
        }
        // Release 1.03 OPC Unified Architecture, Part 4 page 24 - CreateSession Parameters
        // client should prove a sessionName
        // Session name is a Human readable string that identifies the Session. The Server makes this name and the
        // sessionId visible in its AddressSpace for diagnostic purposes. The Client should provide a name that is
        // unique for the instance of the Client.
        // If this parameter is not specified the Server shall assign a value.
        if ((0, node_opcua_utils_1.isNullOrUndefined)(request.sessionName)) {
            // see also #198
            // let's the server assign a sessionName for this lazy client.
            debugLog("assigning OPCUAServer.fallbackSessionName because client's sessionName is null ", OPCUAServer.fallbackSessionName);
            request.sessionName = OPCUAServer.fallbackSessionName;
        }
        // Duration Requested maximum number of milliseconds that a Session should remain open without activity.
        // If the Client fails to issue a Service request within this interval, then the Server shall automatically
        // terminate the Client Session.
        const revisedSessionTimeout = _adjust_session_timeout(request.requestedSessionTimeout);
        // Release 1.02 page 27 OPC Unified Architecture, Part 4: CreateSession.clientNonce
        // A random number that should never be used in any other request. This number shall have a minimum length of 32
        // bytes. Profiles may increase the required length. The Server shall use this value to prove possession of
        // its application instance Certificate in the response.
        if (!request.clientNonce || request.clientNonce.length < 32) {
            if (channel.securityMode !== node_opcua_secure_channel_1.MessageSecurityMode.None) {
                errorLog(chalk_1.default.red("SERVER with secure connection: Missing or invalid client Nonce "), request.clientNonce && request.clientNonce.toString("hex"));
                return rejectConnection(this, node_opcua_status_code_1.StatusCodes.BadNonceInvalid);
            }
        }
        if ((0, node_opcua_secure_channel_1.nonceAlreadyBeenUsed)(request.clientNonce)) {
            errorLog(chalk_1.default.red("SERVER with secure connection: None has already been used"), request.clientNonce && request.clientNonce.toString("hex"));
            return rejectConnection(this, node_opcua_status_code_1.StatusCodes.BadNonceInvalid);
        }
        // check application spoofing
        // check if applicationUri in createSessionRequest matches applicationUri in client Certificate
        if (!validate_applicationUri(channel, request)) {
            return rejectConnection(this, node_opcua_status_code_1.StatusCodes.BadCertificateUriInvalid);
        }
        const { errCode, endpoint } = validate_security_endpoint(this, request, channel);
        if (errCode !== node_opcua_status_code_1.StatusCodes.Good) {
            return rejectConnection(this, errCode);
        }
        // see Release 1.02  27  OPC Unified Architecture, Part 4
        const session = this.createSession({
            clientDescription: request.clientDescription,
            sessionTimeout: revisedSessionTimeout,
            server: this
        });
        session.endpoint = endpoint;
        (0, node_opcua_assert_1.assert)(session);
        (0, node_opcua_assert_1.assert)(session.sessionTimeout === revisedSessionTimeout);
        session.clientDescription = request.clientDescription;
        session.sessionName = request.sessionName || `<unknown session name ${unnamed_session_count++}>`;
        // Depending upon on the  SecurityPolicy  and the  SecurityMode  of the  SecureChannel,  the exchange of
        // ApplicationInstanceCertificates   and  Nonces  may be optional and the signatures may be empty. See
        // Part  7  for the definition of  SecurityPolicies  and the handling of these parameters
        // serverNonce:
        // A random number that should never be used in any other request.
        // This number shall have a minimum length of 32 bytes.
        // The Client shall use this value to prove possession of its application instance
        // Certificate in the ActivateSession request.
        // This value may also be used to prove possession of the userIdentityToken it
        // specified in the ActivateSession request.
        //
        // ( this serverNonce will only be used up to the _on_ActivateSessionRequest
        //   where a new nonce will be created)
        session.nonce = this.makeServerNonce();
        session.channelId = channel.channelId;
        session._attach_channel(channel);
        const serverCertificateChain = this.getCertificateChain();
        const hasEncryption = true;
        // If the securityPolicyUri is None and none of the UserTokenPolicies requires encryption
        if (session.channel.securityMode === node_opcua_secure_channel_1.MessageSecurityMode.None) {
            // ToDo: Check that none of our insecure endpoint has a a UserTokenPolicy that require encryption
            // and set hasEncryption = false under this condition
        }
        const response = new node_opcua_service_session_1.CreateSessionResponse({
            // A identifier which uniquely identifies the session.
            sessionId: session.nodeId,
            // A unique identifier assigned by the Server to the Session.
            // The token used to authenticate the client in subsequent requests.
            authenticationToken: session.authenticationToken,
            revisedSessionTimeout,
            serverNonce: session.nonce,
            // serverCertificate: type ApplicationServerCertificate
            // The application instance Certificate issued to the Server.
            // A Server shall prove possession by using the private key to sign the Nonce provided
            // by the Client in the request. The Client shall verify that this Certificate is the same as
            // the one it used to create the SecureChannel.
            // The ApplicationInstanceCertificate type is defined in OpCUA 1.03 part 4 - $7.2 page 108
            // If the securityPolicyUri is None and none of the UserTokenPolicies requires
            // encryption, the Server shall not send an ApplicationInstanceCertificate and the Client
            // shall ignore the ApplicationInstanceCertificate.
            serverCertificate: hasEncryption ? serverCertificateChain : undefined,
            // The endpoints provided by the server.
            // The Server shall return a set of EndpointDescriptions available for the serverUri
            // specified in the request.[...]
            // The Client shall verify this list with the list from a Discovery Endpoint if it used a Discovery
            // Endpoint to fetch the EndpointDescriptions.
            // It is recommended that Servers only include the endpointUrl, securityMode,
            // securityPolicyUri, userIdentityTokens, transportProfileUri and securityLevel with all
            // other parameters set to null. Only the recommended parameters shall be verified by
            // the client.
            serverEndpoints: _serverEndpointsForCreateSessionResponse(this, session.endpoint.endpointUrl, request.serverUri),
            // This parameter is deprecated and the array shall be empty.
            serverSoftwareCertificates: null,
            // This is a signature generated with the private key associated with the
            // serverCertificate. This parameter is calculated by appending the clientNonce to the
            // clientCertificate and signing the resulting sequence of bytes.
            // The SignatureAlgorithm shall be the AsymmetricSignatureAlgorithm specified in the
            // SecurityPolicy for the Endpoint.
            // The SignatureData type is defined in 7.30.
            serverSignature: this.computeServerSignature(channel, request.clientCertificate, request.clientNonce),
            // The maximum message size accepted by the server
            // The Client Communication Stack should return a Bad_RequestTooLarge error to the
            // application if a request message exceeds this limit.
            // The value zero indicates that this parameter is not used.
            maxRequestMessageSize: 0x4000000
        });
        this.emit("create_session", session);
        session.on("session_closed", (session1, deleteSubscriptions, reason) => {
            (0, node_opcua_assert_1.assert)(typeof reason === "string");
            if (this.isAuditing) {
                (0, node_opcua_assert_1.assert)(reason === "Timeout" || reason === "Terminated" || reason === "CloseSession" || reason === "Forcing");
                const sourceName = "Session/" + reason;
                this.raiseEvent("AuditSessionEventType", {
                    /* part 5 -  6.4.3 AuditEventType */
                    actionTimeStamp: { dataType: "DateTime", value: new Date() },
                    status: { dataType: "Boolean", value: true },
                    serverId: { dataType: "String", value: "" },
                    // ClientAuditEntryId contains the human-readable AuditEntryId defined in Part 3.
                    clientAuditEntryId: { dataType: "String", value: "" },
                    // The ClientUserId identifies the user of the client requesting an action. The ClientUserId can be
                    // obtained from the UserIdentityToken passed in the ActivateSession call.
                    clientUserId: { dataType: "String", value: "" },
                    sourceName: { dataType: "String", value: sourceName },
                    /* part 5 - 6.4.7 AuditSessionEventType */
                    sessionId: { dataType: "NodeId", value: session1.nodeId }
                });
            }
            this.emit("session_closed", session1, deleteSubscriptions);
        });
        if (this.isAuditing) {
            // ------------------------------------------------------------------------------------------------------
            this.raiseEvent("AuditCreateSessionEventType", {
                /* part 5 -  6.4.3 AuditEventType */
                actionTimeStamp: { dataType: "DateTime", value: new Date() },
                status: { dataType: "Boolean", value: true },
                serverId: { dataType: "String", value: "" },
                // ClientAuditEntryId contains the human-readable AuditEntryId defined in Part 3.
                clientAuditEntryId: { dataType: "String", value: "" },
                // The ClientUserId identifies the user of the client requesting an action. The ClientUserId can be
                // obtained from the UserIdentityToken passed in the ActivateSession call.
                clientUserId: { dataType: "String", value: "" },
                sourceName: { dataType: "String", value: "Session/CreateSession" },
                /* part 5 - 6.4.7 AuditSessionEventType */
                sessionId: { dataType: "NodeId", value: session.nodeId },
                /* part 5 - 6.4.8 AuditCreateSessionEventType */
                // SecureChannelId shall uniquely identify the SecureChannel. The application shall use the same
                // identifier in all AuditEvents related to the Session Service Set (AuditCreateSessionEventType,
                // AuditActivateSessionEventType and their subtypes) and the SecureChannel Service Set
                // (AuditChannelEventType and its subtypes
                secureChannelId: { dataType: "String", value: session.channel.channelId.toString() },
                // Duration
                revisedSessionTimeout: { dataType: "Duration", value: session.sessionTimeout },
                // clientCertificate
                clientCertificate: { dataType: "ByteString", value: session.channel.clientCertificate },
                // clientCertificateThumbprint
                clientCertificateThumbprint: {
                    dataType: "String",
                    value: thumbprint(session.channel.clientCertificate)
                }
            });
        }
        // -----------------------------------------------------------------------------------------------------------
        (0, node_opcua_assert_1.assert)(response.authenticationToken);
        channel.send_response("MSG", response, message);
    }
    // TODO : implement this:
    //
    // When the ActivateSession Service is called for the first time then the Server shall reject the request
    // if the SecureChannel is not same as the one associated with the CreateSession request.
    // Subsequent calls to ActivateSession may be associated with different SecureChannels. If this is the
    // case then the Server shall verify that the Certificate the Client used to create the new
    // SecureChannel is the same as the Certificate used to create the original SecureChannel. In addition,
    // the Server shall verify that the Client supplied a UserIdentityToken that is identical to the token
    // currently associated with the Session. Once the Server accepts the new SecureChannel it shall
    // reject requests sent via the old SecureChannel.
    /**
     *
     * @private
     *
     *
     */
    _on_ActivateSessionRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_session_1.ActivateSessionRequest);
        // get session from authenticationToken
        const authenticationToken = request.requestHeader.authenticationToken;
        const session = this.getSession(authenticationToken);
        function rejectConnection(server, statusCode) {
            if (statusCode.equals(node_opcua_status_code_1.StatusCodes.BadSessionIdInvalid)) {
                server.engine.incrementRejectedSessionCount();
            }
            else {
                server.engine.incrementRejectedSessionCount();
                server.engine.incrementSecurityRejectedSessionCount();
            }
            const response1 = new node_opcua_service_session_1.ActivateSessionResponse({ responseHeader: { serviceResult: statusCode } });
            channel.send_response("MSG", response1, message);
        }
        let response;
        /* istanbul ignore next */
        if (!session) {
            // this may happen when the server has been restarted and a client tries to reconnect, thinking
            // that the previous session may still be active
            debugLog(chalk_1.default.yellow.bold(" Bad Session in  _on_ActivateSessionRequest"), authenticationToken.toString());
            return rejectConnection(this, node_opcua_status_code_1.StatusCodes.BadSessionIdInvalid);
        }
        // tslint:disable-next-line: no-unused-expression
        session.keepAlive ? session.keepAlive() : void 0;
        // OpcUA 1.02 part 3 $5.6.3.1 ActiveSession Set page 29
        // When the ActivateSession  Service  is called f or the first time then the Server shall reject the request
        // if the  SecureChannel  is not same as the one associated with the  CreateSession  request.
        if (session.status === "new") {
            // xx if (channel.session_nonce !== session.nonce) {
            if (!channel_has_session(channel, session)) {
                // it looks like session activation is being using a channel that is not the
                // one that have been used to create the session
                errorLog(" channel.sessionTokens === " + Object.keys(channel.sessionTokens).join(" "));
                return rejectConnection(this, node_opcua_status_code_1.StatusCodes.BadSessionNotActivated);
            }
        }
        // OpcUA 1.02 part 3 $5.6.3.1 ActiveSession Set page 29
        // ... Subsequent calls to  ActivateSession  may be associated with different  SecureChannels.  If this is the
        // case then  the  Server  shall verify that the  Certificate  the  Client  used to create the new
        // SecureChannel  is the same as the  Certificate  used to create the original  SecureChannel.
        if (session.status === "active") {
            if (session.channel.channelId !== channel.channelId) {
                warningLog(" Session ", session.sessionName, " is being transferred from channel", chalk_1.default.cyan(session.channel.channelId.toString()), " to channel ", chalk_1.default.cyan(channel.channelId.toString()));
                // session is being reassigned to a new Channel,
                // we shall verify that the certificate used to create the Session is the same as the current
                // channel certificate.
                const old_channel_cert_thumbprint = thumbprint(session.channel.clientCertificate);
                const new_channel_cert_thumbprint = thumbprint(channel.clientCertificate);
                if (old_channel_cert_thumbprint !== new_channel_cert_thumbprint) {
                    return rejectConnection(this, node_opcua_status_code_1.StatusCodes.BadNoValidCertificates); // not sure about this code !
                }
                // ... In addition the Server shall verify that the  Client  supplied a  UserIdentityToken  that is
                // identical to the token currently associated with the  Session reassign session to new channel.
                if (!sameIdentityToken(session.userIdentityToken, request.userIdentityToken)) {
                    return rejectConnection(this, node_opcua_status_code_1.StatusCodes.BadIdentityChangeNotSupported); // not sure about this code !
                }
            }
        }
        else if (session.status === "screwed") {
            // session has been used before being activated => this should be detected and session should be dismissed.
            return rejectConnection(this, node_opcua_status_code_1.StatusCodes.BadSessionClosed);
        }
        else if (session.status === "closed") {
            warningLog(chalk_1.default.yellow.bold(" Bad Session Closed in  _on_ActivateSessionRequest"), authenticationToken.toString());
            return rejectConnection(this, node_opcua_status_code_1.StatusCodes.BadSessionClosed);
        }
        // verify clientSignature provided by the client
        if (!this.verifyClientSignature(session, channel, request.clientSignature)) {
            return rejectConnection(this, node_opcua_status_code_1.StatusCodes.BadApplicationSignatureInvalid);
        }
        // userIdentityToken may be missing , assume anonymous access then
        request.userIdentityToken = request.userIdentityToken || createAnonymousIdentityToken(session.endpoint);
        // check request.userIdentityToken is correct ( expected type and correctly formed)
        this.isValidUserIdentityToken(channel, session, request.userIdentityToken, request.userTokenSignature, session.endpoint, (err, statusCode) => {
            if (!statusCode || statusCode.isNotGood()) {
                /* istanbul ignore next */
                if (!(statusCode && statusCode instanceof node_opcua_status_code_1.StatusCode)) {
                    return rejectConnection(this, node_opcua_status_code_1.StatusCodes.BadCertificateInvalid);
                }
                return rejectConnection(this, statusCode);
            }
            // check if user access is granted
            this.isUserAuthorized(channel, session, request.userIdentityToken, (err1, authorized) => {
                /* istanbul ignore next */
                if (err1) {
                    return rejectConnection(this, node_opcua_status_code_1.StatusCodes.BadInternalError);
                }
                if (!authorized) {
                    return rejectConnection(this, node_opcua_status_code_1.StatusCodes.BadUserAccessDenied);
                }
                else {
                    if (session.status === "active") {
                        moveSessionToChannel(session, channel);
                    }
                    session.userIdentityToken = request.userIdentityToken;
                    // extract : OPC UA part 4 - 5.6.3
                    // Once used, a serverNonce cannot be used again. For that reason, the Server returns a new
                    // serverNonce each time the ActivateSession Service is called.
                    session.nonce = this.makeServerNonce();
                    session.status = "active";
                    response = new node_opcua_service_session_1.ActivateSessionResponse({ serverNonce: session.nonce });
                    channel.send_response("MSG", response, message);
                    // send OPCUA Event Notification
                    // see part 5 : 6.4.3 AuditEventType
                    //              6.4.7 AuditSessionEventType
                    //              6.4.10 AuditActivateSessionEventType
                    (0, node_opcua_assert_1.assert)(session.nodeId); // sessionId
                    // xx assert(session.channel.clientCertificate instanceof Buffer);
                    (0, node_opcua_assert_1.assert)(session.sessionTimeout > 0);
                    raiseAuditActivateSessionEventType.call(this, session);
                    this.emit("session_activated", session, userIdentityTokenPasswordRemoved(session.userIdentityToken));
                    session.resendMonitoredItemInitialValues();
                }
            });
        });
    }
    prepare(message, channel) {
        const request = message.request;
        // --- check that session is correct
        const authenticationToken = request.requestHeader.authenticationToken;
        const session = this.getSession(authenticationToken, /*activeOnly*/ true);
        if (!session) {
            message.session_statusCode = node_opcua_status_code_1.StatusCodes.BadSessionIdInvalid;
            return;
        }
        message.session = session;
        // --- check that provided session matches session attached to channel
        if (channel.channelId !== session.channelId) {
            if (!(request instanceof node_opcua_service_session_1.ActivateSessionRequest)) {
                errorLog(chalk_1.default.red.bgWhite("ERROR: channel.channelId !== session.channelId  on processing request " + request.constructor.name), channel.channelId, session.channelId);
            }
            message.session_statusCode = node_opcua_status_code_1.StatusCodes.BadSecureChannelIdInvalid;
        }
        else if (channel_has_session(channel, session)) {
            message.session_statusCode = node_opcua_status_code_1.StatusCodes.Good;
        }
        else {
            // session ma y have been moved to a different channel
            message.session_statusCode = node_opcua_status_code_1.StatusCodes.BadSecureChannelIdInvalid;
        }
    }
    /**
     * ensure that action is performed on a valid session object,
     * @param ResponseClass the constructor of the response Class
     * @param message
     * @param channel
     * @param actionToPerform
     * @param actionToPerform.session {ServerSession}
     * @param actionToPerform.sendResponse
     * @param actionToPerform.sendResponse.response
     * @param actionToPerform.sendError
     * @param actionToPerform.sendError.statusCode
     * @param actionToPerform.sendError.diagnostics
     *
     * @private
     */
    async _apply_on_SessionObject(ResponseClass, message, channel, actionToPerform) {
        (0, node_opcua_assert_1.assert)(typeof actionToPerform === "function");
        function sendResponse(response1) {
            try {
                (0, node_opcua_assert_1.assert)(response1 instanceof ResponseClass || response1 instanceof node_opcua_types_1.ServiceFault);
                if (message.session) {
                    const counterName = ResponseClass.schema.name.replace("Response", "");
                    message.session.incrementRequestTotalCounter(counterName);
                }
                return channel.send_response("MSG", response1, message);
            }
            catch (err) {
                warningLog(err);
                // istanbul ignore next
                if (util_1.types.isNativeError(err)) {
                    // istanbul ignore next
                    errorLog("Internal error in issuing response\nplease contact support@sterfive.com", message.request.toString(), "\n", response1.toString());
                }
                // istanbul ignore next
                throw err;
            }
        }
        function sendError(statusCode) {
            if (message.session) {
                message.session.incrementRequestErrorCounter(ResponseClass.schema.name.replace("Response", ""));
            }
            return g_sendError(channel, message, ResponseClass, statusCode);
        }
        let response;
        /* istanbul ignore next */
        if (!message.session || message.session_statusCode !== node_opcua_status_code_1.StatusCodes.Good) {
            const errMessage = "=>" + message.session_statusCode?.toString();
            response = new node_opcua_types_1.ServiceFault({ responseHeader: { serviceResult: message.session_statusCode } });
            debugLog(chalk_1.default.red.bold(errMessage), chalk_1.default.yellow(message.session_statusCode.toString()), response.constructor.name);
            return sendResponse(response);
        }
        (0, node_opcua_assert_1.assert)(message.session_statusCode.isGood());
        // OPC UA Specification 1.02 part 4 page 26
        // When a  Session  is terminated, all outstanding requests on the  Session  are aborted and
        // Bad_SessionClosed  StatusCodes  are returned to the  Client. In addition,   the  Server  deletes the entry
        // for the  Client  from its  SessionDiagnostics Array  Variable  and notifies any other  Clients  who were
        // subscribed to this entry.
        if (message.session.status === "closed") {
            // note : use StatusCodes.BadSessionClosed , for pending message for this session
            return sendError(node_opcua_status_code_1.StatusCodes.BadSessionIdInvalid);
        }
        if (message.session.status === "new") {
            // mark session as being screwed ! so it cannot be activated anymore
            message.session.status = "screwed";
            return sendError(node_opcua_status_code_1.StatusCodes.BadSessionNotActivated);
        }
        if (message.session.status !== "active") {
            // mark session as being screwed ! so it cannot be activated anymore
            message.session.status = "screwed";
            // note : use StatusCodes.BadSessionClosed , for pending message for this session
            return sendError(node_opcua_status_code_1.StatusCodes.BadSessionIdInvalid);
        }
        // lets also reset the session watchdog so it doesn't
        // (Sessions are terminated by the Server automatically if the Client fails to issue a Service
        // request on the Session within the timeout period negotiated by the Server in the
        // CreateSession Service response. )
        if (message.session.keepAlive) {
            (0, node_opcua_assert_1.assert)(typeof message.session.keepAlive === "function");
            message.session.keepAlive();
        }
        message.session.incrementTotalRequestCount();
        await actionToPerform(message.session, sendResponse, sendError);
    }
    async _apply_on_Subscription(ResponseClass, message, channel, actionToPerform) {
        (0, node_opcua_assert_1.assert)(typeof actionToPerform === "function");
        const request = message.request;
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(request, "subscriptionId"));
        this._apply_on_SessionObject(ResponseClass, message, channel, async (session, sendResponse, sendError) => {
            const subscription = session.getSubscription(request.subscriptionId);
            if (!subscription) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadSubscriptionIdInvalid);
            }
            subscription.resetLifeTimeAndKeepAliveCounters();
            await actionToPerform(session, subscription, sendResponse, sendError);
        });
    }
    _apply_on_SubscriptionIds(ResponseClass, message, channel, actionToPerform) {
        (0, node_opcua_assert_1.assert)(typeof actionToPerform === "function");
        const request = message.request;
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(request, "subscriptionIds"));
        this._apply_on_SessionObject(ResponseClass, message, channel, async (session, sendResponse, sendError) => {
            const subscriptionIds = request.subscriptionIds;
            if (!request.subscriptionIds || request.subscriptionIds.length === 0) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadNothingToDo);
            }
            // check minimal
            if (request.subscriptionIds.length >
                Math.min(this.engine.serverCapabilities.maxSubscriptionsPerSession, this.engine.serverCapabilities.maxSubscriptions)) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadTooManyOperations);
            }
            const promises = subscriptionIds.map((subscriptionId) => actionToPerform(session, subscriptionId));
            const results = await Promise.all(promises);
            const serviceResult = node_opcua_status_code_1.StatusCodes.Good;
            const response = new ResponseClass({
                responseHeader: {
                    serviceResult
                },
                results: results
            });
            sendResponse(response);
        });
    }
    _apply_on_Subscriptions(ResponseClass, message, channel, actionToPerform) {
        this._apply_on_SubscriptionIds(ResponseClass, message, channel, async (session, subscriptionId) => {
            /* istanbul ignore next */
            if (isSubscriptionIdInvalid(subscriptionId)) {
                return node_opcua_status_code_1.StatusCodes.BadSubscriptionIdInvalid;
            }
            const subscription = session.getSubscription(subscriptionId);
            if (!subscription) {
                return node_opcua_status_code_1.StatusCodes.BadSubscriptionIdInvalid;
            }
            return actionToPerform(session, subscription);
        });
    }
    async _closeSession(authenticationToken, deleteSubscriptions, reason) {
        if (deleteSubscriptions && this.options.onDeleteMonitoredItem) {
            const session = this.getSession(authenticationToken);
            if (session) {
                const subscriptions = session.publishEngine.subscriptions;
                for (const subscription of subscriptions) {
                    await subscription.applyOnMonitoredItem(this.options.onDeleteMonitoredItem.bind(null, subscription));
                }
            }
        }
        await this.engine.closeSession(authenticationToken, deleteSubscriptions, reason);
    }
    /**
     * @param message
     * @param channel
     * @private
     */
    _on_CloseSessionRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_session_1.CloseSessionRequest);
        let response;
        message.session_statusCode = node_opcua_status_code_1.StatusCodes.Good;
        function sendError(statusCode) {
            return g_sendError(channel, message, node_opcua_service_session_1.CloseSessionResponse, statusCode);
        }
        function sendResponse(response1) {
            channel.send_response("MSG", response1, message);
        }
        // do not use _apply_on_SessionObject
        // this._apply_on_SessionObject(CloseSessionResponse, message, channel, function (session) {
        // });
        const session = message.session;
        if (!session) {
            return sendError(node_opcua_status_code_1.StatusCodes.BadSessionIdInvalid);
        }
        // session has been created but not activated !
        const wasNotActivated = session.status === "new";
        (async () => {
            try {
                await this._closeSession(request.requestHeader.authenticationToken, request.deleteSubscriptions, "CloseSession");
                // if (false && wasNotActivated) {
                //  return sendError(StatusCodes.BadSessionNotActivated);
                // }
                response = new node_opcua_service_session_1.CloseSessionResponse({});
                sendResponse(response);
            }
            catch (err) {
                sendError(node_opcua_status_code_1.StatusCodes.BadInternalError);
            }
        })();
    }
    // browse services
    /**
     * @param message
     * @param channel
     * @private
     */
    _on_BrowseRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_browse_1.BrowseRequest);
        const diagnostic = {};
        this._apply_on_SessionObject(node_opcua_service_browse_1.BrowseResponse, message, channel, (session, sendResponse, sendError) => {
            let response;
            // test view
            if (request.view && !request.view.viewId.isEmpty()) {
                let theView = this.engine.addressSpace.findNode(request.view.viewId);
                if (theView && theView.nodeClass !== node_opcua_data_model_1.NodeClass.View) {
                    // Error: theView is not a View
                    diagnostic.localizedText = { text: "Expecting a view here" };
                    theView = null;
                }
                if (!theView) {
                    return sendError(node_opcua_status_code_1.StatusCodes.BadViewIdUnknown);
                }
            }
            if (!request.nodesToBrowse || request.nodesToBrowse.length === 0) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadNothingToDo);
            }
            if (this.engine.serverCapabilities.operationLimits.maxNodesPerBrowse > 0) {
                if (request.nodesToBrowse.length > this.engine.serverCapabilities.operationLimits.maxNodesPerBrowse) {
                    return sendError(node_opcua_status_code_1.StatusCodes.BadTooManyOperations);
                }
            }
            // limit results to requestedMaxReferencesPerNode further so it never exceed a too big number
            const requestedMaxReferencesPerNode = Math.min(9876, request.requestedMaxReferencesPerNode);
            (0, node_opcua_assert_1.assert)(request.nodesToBrowse[0].schema.name === "BrowseDescription");
            const context = session.sessionContext;
            const browseAll = (nodesToBrowse, callack) => {
                const f = (0, util_1.callbackify)(this.engine.browseWithAutomaticExpansion).bind(this.engine);
                f(request.nodesToBrowse, context, callack);
            };
            // handle continuation point and requestedMaxReferencesPerNode
            const maxBrowseContinuationPoints = this.engine.serverCapabilities.maxBrowseContinuationPoints;
            (0, node_opcua_address_space_1.innerBrowse)({
                browseAll,
                context,
                continuationPointManager: session.continuationPointManager,
                requestedMaxReferencesPerNode,
                maxBrowseContinuationPoints
            }, request.nodesToBrowse, (err, results) => {
                if (!results) {
                    return sendError(node_opcua_status_code_1.StatusCodes.BadInternalError);
                }
                (0, node_opcua_assert_1.assert)(results[0].schema.name === "BrowseResult");
                response = new node_opcua_service_browse_1.BrowseResponse({
                    diagnosticInfos: undefined,
                    results
                });
                sendResponse(response);
            });
        });
    }
    /**
     */
    _on_BrowseNextRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_browse_1.BrowseNextRequest);
        this._apply_on_SessionObject(node_opcua_service_browse_1.BrowseNextResponse, message, channel, (session, sendResponse, sendError) => {
            if (!request.continuationPoints || request.continuationPoints.length === 0) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadNothingToDo);
            }
            (0, node_opcua_address_space_1.innerBrowseNext)({
                continuationPointManager: session.continuationPointManager
            }, request.continuationPoints, request.releaseContinuationPoints, (err, results) => {
                if (err) {
                    return sendError(node_opcua_status_code_1.StatusCodes.BadInternalError);
                }
                const response = new node_opcua_service_browse_1.BrowseNextResponse({
                    diagnosticInfos: undefined,
                    results
                });
                sendResponse(response);
            });
        });
    }
    // read services
    _on_ReadRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_read_1.ReadRequest);
        this._apply_on_SessionObject(node_opcua_service_read_1.ReadResponse, message, channel, (session, sendResponse, sendError) => {
            const context = session.sessionContext;
            const timestampsToReturn = request.timestampsToReturn;
            if (timestampsToReturn === node_opcua_service_read_1.TimestampsToReturn.Invalid) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadTimestampsToReturnInvalid);
            }
            if (request.maxAge < 0) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadMaxAgeInvalid);
            }
            request.nodesToRead = request.nodesToRead || [];
            if (!request.nodesToRead || request.nodesToRead.length <= 0) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadNothingToDo);
            }
            (0, node_opcua_assert_1.assert)(request.nodesToRead[0].schema.name === "ReadValueId");
            // limit size of nodesToRead array to maxNodesPerRead
            if (this.engine.serverCapabilities.operationLimits.maxNodesPerRead > 0) {
                if (request.nodesToRead.length > this.engine.serverCapabilities.operationLimits.maxNodesPerRead) {
                    return sendError(node_opcua_status_code_1.StatusCodes.BadTooManyOperations);
                }
            }
            // proceed with registered nodes alias resolution
            for (const nodeToRead of request.nodesToRead) {
                nodeToRead.nodeId = session.resolveRegisteredNode(nodeToRead.nodeId);
            }
            // ask for a refresh of asynchronous variables
            this.engine.refreshValues(request.nodesToRead, request.maxAge, (err) => {
                this.engine.read(context, request).then((results) => {
                    (0, node_opcua_assert_1.assert)(results[0].schema.name === "DataValue");
                    (0, node_opcua_assert_1.assert)(results.length === request.nodesToRead.length);
                    const response = new node_opcua_service_read_1.ReadResponse({
                        diagnosticInfos: undefined,
                        results: undefined
                    });
                    // set it here for performance
                    response.results = results;
                    (0, node_opcua_assert_1.assert)(response.diagnosticInfos.length === 0);
                    sendResponse(response);
                });
            });
        });
    }
    // read services
    _on_HistoryReadRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_history_1.HistoryReadRequest);
        this._apply_on_SessionObject(node_opcua_service_history_1.HistoryReadResponse, message, channel, (session, sendResponse, sendError) => {
            let response;
            const timestampsToReturn = request.timestampsToReturn;
            if (timestampsToReturn === node_opcua_service_read_1.TimestampsToReturn.Invalid) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadTimestampsToReturnInvalid);
            }
            request.nodesToRead = request.nodesToRead || [];
            if (!request.nodesToRead || request.nodesToRead.length <= 0) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadNothingToDo);
            }
            (0, node_opcua_assert_1.assert)(request.nodesToRead[0].schema.name === "HistoryReadValueId");
            // limit size of nodesToRead array to maxNodesPerRead
            if (this.engine.serverCapabilities.operationLimits.maxNodesPerRead > 0) {
                if (request.nodesToRead.length > this.engine.serverCapabilities.operationLimits.maxNodesPerRead) {
                    return sendError(node_opcua_status_code_1.StatusCodes.BadTooManyOperations);
                }
            }
            // todo : handle
            if (this.engine.serverCapabilities.operationLimits.maxNodesPerHistoryReadData > 0) {
                if (request.nodesToRead.length > this.engine.serverCapabilities.operationLimits.maxNodesPerHistoryReadData) {
                    return sendError(node_opcua_status_code_1.StatusCodes.BadTooManyOperations);
                }
            }
            if (this.engine.serverCapabilities.operationLimits.maxNodesPerHistoryReadEvents > 0) {
                if (request.nodesToRead.length > this.engine.serverCapabilities.operationLimits.maxNodesPerHistoryReadEvents) {
                    return sendError(node_opcua_status_code_1.StatusCodes.BadTooManyOperations);
                }
            }
            const context = session.sessionContext;
            // ask for a refresh of asynchronous variables
            this.engine.refreshValues(request.nodesToRead, 0, (err) => {
                (0, node_opcua_assert_1.assert)(!err, " error not handled here , fix me"); // TODO
                this.engine
                    .historyRead(context, request)
                    .then((results) => {
                    (0, node_opcua_assert_1.assert)(results[0].schema.name === "HistoryReadResult");
                    (0, node_opcua_assert_1.assert)(results.length === request.nodesToRead.length);
                    response = new node_opcua_service_history_1.HistoryReadResponse({
                        diagnosticInfos: undefined,
                        results
                    });
                    (0, node_opcua_assert_1.assert)(response.diagnosticInfos.length === 0);
                    sendResponse(response);
                })
                    .catch((err) => {
                    return sendError(node_opcua_status_code_1.StatusCodes.BadHistoryOperationInvalid);
                });
            });
        });
    }
    /*
   // write services
   // OPCUA Specification 1.02 Part 3 : 5.10.4 Write
   // This Service is used to write values to one or more Attributes of one or more Nodes. For constructed
   // Attribute values whose elements are indexed, such as an array, this Service allows Clients to write
   // the entire set of indexed values as a composite, to write individual elements or to write ranges of
   // elements of the composite.
   // The values are written to the data source, such as a device, and the Service does not return until it writes
   // the values or determines that the value cannot be written. In certain cases, the Server will successfully
   // to an intermediate system or Server, and will not know if the data source was updated properly. In these cases,
   // the Server should report a success code that indicates that the write was not verified.
   // In the cases where the Server is able to verify that it has successfully written to the data source,
   // it reports an unconditional success.
   */
    _on_WriteRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_write_1.WriteRequest);
        (0, node_opcua_assert_1.assert)(!request.nodesToWrite || Array.isArray(request.nodesToWrite));
        this._apply_on_SessionObject(node_opcua_service_write_1.WriteResponse, message, channel, (session, sendResponse, sendError) => {
            let response;
            if (!request.nodesToWrite || request.nodesToWrite.length === 0) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadNothingToDo);
            }
            if (this.engine.serverCapabilities.operationLimits.maxNodesPerWrite > 0) {
                if (request.nodesToWrite.length > this.engine.serverCapabilities.operationLimits.maxNodesPerWrite) {
                    return sendError(node_opcua_status_code_1.StatusCodes.BadTooManyOperations);
                }
            }
            // proceed with registered nodes alias resolution
            for (const nodeToWrite of request.nodesToWrite) {
                nodeToWrite.nodeId = session.resolveRegisteredNode(nodeToWrite.nodeId);
            }
            const context = session.sessionContext;
            (0, node_opcua_assert_1.assert)(request.nodesToWrite[0].schema.name === "WriteValue");
            this.engine
                .write(context, request.nodesToWrite)
                .then((results) => {
                (0, node_opcua_assert_1.assert)(results.length === request.nodesToWrite.length);
                response = new node_opcua_service_write_1.WriteResponse({
                    diagnosticInfos: undefined,
                    results
                });
                sendResponse(response);
            })
                .catch((err) => {
                errorLog(err);
                sendError(node_opcua_status_code_1.StatusCodes.BadInternalError);
            });
        });
    }
    // subscription services
    _on_CreateSubscriptionRequest(message, channel) {
        const engine = this.engine;
        const addressSpace = engine.addressSpace;
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_subscription_1.CreateSubscriptionRequest);
        this._apply_on_SessionObject(node_opcua_service_subscription_1.CreateSubscriptionResponse, message, channel, (session, sendResponse, sendError) => {
            const context = session.sessionContext;
            if (session.currentSubscriptionCount >= this.engine.serverCapabilities.maxSubscriptionsPerSession) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadTooManySubscriptions);
            }
            if (this.currentSubscriptionCount >= this.engine.serverCapabilities.maxSubscriptions) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadTooManySubscriptions);
            }
            const subscription = session.createSubscription(request);
            subscription.on("monitoredItem", (monitoredItem) => {
                prepareMonitoredItem(context, addressSpace, monitoredItem);
            });
            const response = new node_opcua_service_subscription_1.CreateSubscriptionResponse({
                revisedLifetimeCount: subscription.lifeTimeCount,
                revisedMaxKeepAliveCount: subscription.maxKeepAliveCount,
                revisedPublishingInterval: subscription.publishingInterval,
                subscriptionId: subscription.id
            });
            sendResponse(response);
        });
    }
    _on_DeleteSubscriptionsRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_subscription_1.DeleteSubscriptionsRequest);
        this._apply_on_SubscriptionIds(node_opcua_service_subscription_1.DeleteSubscriptionsResponse, message, channel, async (session, subscriptionId) => {
            let subscription = this.engine.findOrphanSubscription(subscriptionId);
            // istanbul ignore next
            if (subscription) {
                warningLog("Deleting an orphan subscription", subscriptionId);
                await this._beforeDeleteSubscription(subscription);
                return this.engine.deleteOrphanSubscription(subscription);
            }
            subscription = session.getSubscription(subscriptionId);
            if (subscription) {
                await this._beforeDeleteSubscription(subscription);
            }
            return session.deleteSubscription(subscriptionId);
        });
    }
    _on_TransferSubscriptionsRequest(message, channel) {
        //
        // sendInitialValue Boolean
        //    A Boolean parameter with the following values:
        //    TRUE      the first Publish response(s) after the TransferSubscriptions call shall
        //              contain the current values of all Monitored Items in the Subscription where
        //              the Monitoring Mode is set to Reporting.
        //    FALSE     the first Publish response after the TransferSubscriptions call shall contain only the value
        //              changes since the last Publish response was sent.
        //    This parameter only applies to MonitoredItems used for monitoring Attribute changes.
        //
        const engine = this.engine;
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_subscription_1.TransferSubscriptionsRequest);
        this._apply_on_SubscriptionIds(node_opcua_service_subscription_1.TransferSubscriptionsResponse, message, channel, async (session, subscriptionId) => await engine.transferSubscription(session, subscriptionId, request.sendInitialValues));
    }
    _on_CreateMonitoredItemsRequest(message, channel) {
        const engine = this.engine;
        const addressSpace = engine.addressSpace;
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_subscription_1.CreateMonitoredItemsRequest);
        this._apply_on_Subscription(node_opcua_service_subscription_1.CreateMonitoredItemsResponse, message, channel, async (session, subscription, sendResponse, sendError) => {
            const timestampsToReturn = request.timestampsToReturn;
            if (timestampsToReturn === node_opcua_service_read_1.TimestampsToReturn.Invalid) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadTimestampsToReturnInvalid);
            }
            if (!request.itemsToCreate || request.itemsToCreate.length === 0) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadNothingToDo);
            }
            if (this.engine.serverCapabilities.operationLimits.maxMonitoredItemsPerCall > 0) {
                if (request.itemsToCreate.length > this.engine.serverCapabilities.operationLimits.maxMonitoredItemsPerCall) {
                    return sendError(node_opcua_status_code_1.StatusCodes.BadTooManyOperations);
                }
            }
            const options = this.options;
            let results = [];
            if (options.onCreateMonitoredItem) {
                const resultsPromise = request.itemsToCreate.map(async (monitoredItemCreateRequest) => {
                    const { monitoredItem, createResult } = subscription.preCreateMonitoredItem(addressSpace, timestampsToReturn, monitoredItemCreateRequest);
                    if (monitoredItem) {
                        await options.onCreateMonitoredItem(subscription, monitoredItem);
                        subscription.postCreateMonitoredItem(monitoredItem, monitoredItemCreateRequest, createResult);
                    }
                    return createResult;
                });
                results = await Promise.all(resultsPromise);
            }
            else {
                results = request.itemsToCreate.map((monitoredItemCreateRequest) => {
                    const { monitoredItem, createResult } = subscription.preCreateMonitoredItem(addressSpace, timestampsToReturn, monitoredItemCreateRequest);
                    if (monitoredItem) {
                        subscription.postCreateMonitoredItem(monitoredItem, monitoredItemCreateRequest, createResult);
                    }
                    return createResult;
                });
            }
            const response = new node_opcua_service_subscription_1.CreateMonitoredItemsResponse({
                responseHeader: { serviceResult: node_opcua_status_code_1.StatusCodes.Good },
                results
                // ,diagnosticInfos: []
            });
            sendResponse(response);
        });
    }
    _on_ModifySubscriptionRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_subscription_1.ModifySubscriptionRequest);
        this._apply_on_Subscription(node_opcua_service_subscription_1.ModifySubscriptionResponse, message, channel, async (session, subscription, sendResponse, sendError) => {
            subscription.modify(request);
            const response = new node_opcua_service_subscription_1.ModifySubscriptionResponse({
                revisedLifetimeCount: subscription.lifeTimeCount,
                revisedMaxKeepAliveCount: subscription.maxKeepAliveCount,
                revisedPublishingInterval: subscription.publishingInterval
            });
            sendResponse(response);
        });
    }
    _on_ModifyMonitoredItemsRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_subscription_1.ModifyMonitoredItemsRequest);
        this._apply_on_Subscription(node_opcua_service_subscription_1.ModifyMonitoredItemsResponse, message, channel, async (session, subscription, sendResponse, sendError) => {
            const timestampsToReturn = request.timestampsToReturn;
            if (timestampsToReturn === node_opcua_service_read_1.TimestampsToReturn.Invalid) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadTimestampsToReturnInvalid);
            }
            if (!request.itemsToModify || request.itemsToModify.length === 0) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadNothingToDo);
            }
            /* istanbul ignore next */
            if (this.engine.serverCapabilities.operationLimits.maxMonitoredItemsPerCall > 0) {
                if (request.itemsToModify.length > this.engine.serverCapabilities.operationLimits.maxMonitoredItemsPerCall) {
                    return sendError(node_opcua_status_code_1.StatusCodes.BadTooManyOperations);
                }
            }
            const itemsToModify = request.itemsToModify; // MonitoredItemModifyRequest
            function modifyMonitoredItem(item) {
                const monitoredItemId = item.monitoredItemId;
                const monitoredItem = subscription.getMonitoredItem(monitoredItemId);
                if (!monitoredItem) {
                    return new node_opcua_service_subscription_1.MonitoredItemModifyResult({ statusCode: node_opcua_status_code_1.StatusCodes.BadMonitoredItemIdInvalid });
                }
                // adjust samplingInterval if === -1
                if (item.requestedParameters.samplingInterval === -1) {
                    item.requestedParameters.samplingInterval = subscription.publishingInterval;
                }
                return monitoredItem.modify(timestampsToReturn, item.requestedParameters);
            }
            const results = itemsToModify.map(modifyMonitoredItem);
            const response = new node_opcua_service_subscription_1.ModifyMonitoredItemsResponse({
                results
            });
            sendResponse(response);
        });
    }
    _on_PublishRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_subscription_1.PublishRequest);
        this._apply_on_SessionObject(node_opcua_service_subscription_1.PublishResponse, message, channel, (session, sendResponse, sendError) => {
            (0, node_opcua_assert_1.assert)(session);
            (0, node_opcua_assert_1.assert)(session.publishEngine); // server.publishEngine doesn't exists, OPCUAServer has probably shut down already
            session.publishEngine._on_PublishRequest(request, (_request1, response) => {
                sendResponse(response);
            });
        });
    }
    _on_SetPublishingModeRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_subscription_1.SetPublishingModeRequest);
        const publishingEnabled = request.publishingEnabled;
        this._apply_on_Subscriptions(node_opcua_service_subscription_1.SetPublishingModeResponse, message, channel, async (session, subscription) => {
            return subscription.setPublishingMode(publishingEnabled);
        });
    }
    _on_DeleteMonitoredItemsRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_subscription_1.DeleteMonitoredItemsRequest);
        this._apply_on_Subscription(node_opcua_service_subscription_1.DeleteMonitoredItemsResponse, message, channel, async (session, subscription, sendResponse, sendError) => {
            /* istanbul ignore next */
            if (!request.monitoredItemIds || request.monitoredItemIds.length === 0) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadNothingToDo);
            }
            /* istanbul ignore next */
            if (this.engine.serverCapabilities.operationLimits.maxMonitoredItemsPerCall > 0) {
                if (request.monitoredItemIds.length > this.engine.serverCapabilities.operationLimits.maxMonitoredItemsPerCall) {
                    return sendError(node_opcua_status_code_1.StatusCodes.BadTooManyOperations);
                }
            }
            const resultsPromises = request.monitoredItemIds.map(async (monitoredItemId) => {
                if (this.options.onDeleteMonitoredItem) {
                    const monitoredItem = subscription.getMonitoredItem(monitoredItemId);
                    if (monitoredItem) {
                        await this.options.onDeleteMonitoredItem(subscription, monitoredItem);
                    }
                }
                return subscription.removeMonitoredItem(monitoredItemId);
            });
            try {
                const results = await Promise.all(resultsPromises);
                const response = new node_opcua_service_subscription_1.DeleteMonitoredItemsResponse({
                    diagnosticInfos: undefined,
                    results
                });
                sendResponse(response);
            }
            catch (err) {
                warningLog(err);
                return sendError(node_opcua_status_code_1.StatusCodes.BadInternalError);
            }
        });
    }
    _on_SetTriggeringRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_subscription_1.SetTriggeringRequest);
        this._apply_on_Subscription(node_opcua_service_subscription_1.SetTriggeringResponse, message, channel, async (session, subscription, sendResponse, sendError) => {
            /* */
            const { triggeringItemId, linksToAdd, linksToRemove } = request;
            /**
             * The MaxMonitoredItemsPerCall Property indicates
             * [...]
             *  • the maximum size of the sum of the linksToAdd and linksToRemove arrays when a
             *    Client calls the SetTriggering Service.
             *
             */
            const maxElements = (linksToAdd ? linksToAdd.length : 0) + (linksToRemove ? linksToRemove.length : 0);
            if (this.engine.serverCapabilities.operationLimits.maxMonitoredItemsPerCall > 0) {
                if (maxElements > this.engine.serverCapabilities.operationLimits.maxMonitoredItemsPerCall) {
                    return sendError(node_opcua_status_code_1.StatusCodes.BadTooManyOperations);
                }
            }
            const { addResults, removeResults, statusCode } = subscription.setTriggering(triggeringItemId, linksToAdd, linksToRemove);
            if (statusCode.isNotGood()) {
                const response = new node_opcua_types_1.ServiceFault({ responseHeader: { serviceResult: statusCode } });
                sendResponse(response);
            }
            else {
                const response = new node_opcua_service_subscription_1.SetTriggeringResponse({
                    responseHeader: { serviceResult: statusCode },
                    addResults,
                    removeResults,
                    addDiagnosticInfos: null,
                    removeDiagnosticInfos: null
                });
                sendResponse(response);
            }
        });
    }
    async _beforeDeleteSubscription(subscription) {
        if (!this.options.onDeleteMonitoredItem) {
            return;
        }
        await subscription.applyOnMonitoredItem(this.options.onDeleteMonitoredItem.bind(null, subscription));
    }
    _on_RepublishRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_subscription_1.RepublishRequest);
        this._apply_on_Subscription(node_opcua_service_subscription_1.RepublishResponse, message, channel, async (session, subscription, sendResponse, sendError) => {
            // update diagnostic counter
            subscription.subscriptionDiagnostics.republishRequestCount += 1;
            subscription.subscriptionDiagnostics.republishMessageRequestCount += 1;
            const retransmitSequenceNumber = request.retransmitSequenceNumber;
            const notificationMessage = subscription.getMessageForSequenceNumber(retransmitSequenceNumber);
            if (!notificationMessage) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadMessageNotAvailable);
            }
            const response = new node_opcua_service_subscription_1.RepublishResponse({
                notificationMessage,
                responseHeader: {
                    serviceResult: node_opcua_status_code_1.StatusCodes.Good
                }
            });
            // update diagnostic counter
            subscription.subscriptionDiagnostics.republishMessageCount += 1;
            sendResponse(response);
        });
    }
    // Bad_NothingToDo
    // Bad_TooManyOperations
    // Bad_SubscriptionIdInvalid
    // Bad_MonitoringModeInvalid
    _on_SetMonitoringModeRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_subscription_1.SetMonitoringModeRequest);
        this._apply_on_Subscription(node_opcua_service_subscription_1.SetMonitoringModeResponse, message, channel, async (session, subscription, sendResponse, sendError) => {
            /* istanbul ignore next */
            if (!request.monitoredItemIds || request.monitoredItemIds.length === 0) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadNothingToDo);
            }
            /* istanbul ignore next */
            if (this.engine.serverCapabilities.operationLimits.maxMonitoredItemsPerCall > 0) {
                if (request.monitoredItemIds.length > this.engine.serverCapabilities.operationLimits.maxMonitoredItemsPerCall) {
                    return sendError(node_opcua_status_code_1.StatusCodes.BadTooManyOperations);
                }
            }
            const monitoringMode = request.monitoringMode;
            if (!isMonitoringModeValid(monitoringMode)) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadMonitoringModeInvalid);
            }
            const results = request.monitoredItemIds.map((monitoredItemId) => {
                const monitoredItem = subscription.getMonitoredItem(monitoredItemId);
                if (!monitoredItem) {
                    return node_opcua_status_code_1.StatusCodes.BadMonitoredItemIdInvalid;
                }
                monitoredItem.setMonitoringMode(monitoringMode);
                return node_opcua_status_code_1.StatusCodes.Good;
            });
            const response = new node_opcua_service_subscription_1.SetMonitoringModeResponse({
                results
            });
            sendResponse(response);
        });
    }
    // _on_TranslateBrowsePathsToNodeIds service
    _on_TranslateBrowsePathsToNodeIdsRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_translate_browse_path_1.TranslateBrowsePathsToNodeIdsRequest);
        this._apply_on_SessionObject(node_opcua_service_translate_browse_path_1.TranslateBrowsePathsToNodeIdsResponse, message, channel, async (session, sendResponse, sendError) => {
            if (!request.browsePaths || request.browsePaths.length === 0) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadNothingToDo);
            }
            if (this.engine.serverCapabilities.operationLimits.maxNodesPerTranslateBrowsePathsToNodeIds > 0) {
                if (request.browsePaths.length >
                    this.engine.serverCapabilities.operationLimits.maxNodesPerTranslateBrowsePathsToNodeIds) {
                    return sendError(node_opcua_status_code_1.StatusCodes.BadTooManyOperations);
                }
            }
            this.engine
                .translateBrowsePaths(request.browsePaths)
                .then((browsePathsResults) => {
                const response = new node_opcua_service_translate_browse_path_1.TranslateBrowsePathsToNodeIdsResponse({
                    diagnosticInfos: null,
                    results: browsePathsResults
                });
                sendResponse(response);
            })
                .catch((err) => {
                sendError(node_opcua_status_code_1.StatusCodes.BadInternalError);
            });
        });
    }
    // Call Service Result Codes
    // Symbolic Id Description
    // Bad_NothingToDo       See Table 165 for the description of this result code.
    // Bad_TooManyOperations See Table 165 for the description of this result code.
    //
    _on_CallRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_call_1.CallRequest);
        this._apply_on_SessionObject(node_opcua_service_call_1.CallResponse, message, channel, (session, sendResponse, sendError) => {
            let response;
            if (!request.methodsToCall || request.methodsToCall.length === 0) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadNothingToDo);
            }
            // the MaxNodesPerMethodCall Property indicates the maximum size of the methodsToCall array when
            // a Client calls the Call Service.
            let maxNodesPerMethodCall = this.engine.serverCapabilities.operationLimits.maxNodesPerMethodCall;
            maxNodesPerMethodCall = maxNodesPerMethodCall <= 0 ? 1000 : maxNodesPerMethodCall;
            if (request.methodsToCall.length > maxNodesPerMethodCall) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadTooManyOperations);
            }
            const context = session.sessionContext;
            this.engine
                .call(context, request.methodsToCall)
                .then((results) => {
                const response = new node_opcua_service_call_1.CallResponse({ results });
                filterDiagnosticInfo(request.requestHeader.returnDiagnostics, response);
                sendResponse(response);
            })
                .catch((err) => {
                sendError(node_opcua_status_code_1.StatusCodes.BadInternalError);
            });
        });
    }
    _on_RegisterNodesRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_register_node_1.RegisterNodesRequest);
        this._apply_on_SessionObject(node_opcua_service_register_node_1.RegisterNodesResponse, message, channel, (session, sendResponse, sendError) => {
            if (!request.nodesToRegister || request.nodesToRegister.length === 0) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadNothingToDo);
            }
            if (this.engine.serverCapabilities.operationLimits.maxNodesPerRegisterNodes > 0) {
                if (request.nodesToRegister.length > this.engine.serverCapabilities.operationLimits.maxNodesPerRegisterNodes) {
                    return sendError(node_opcua_status_code_1.StatusCodes.BadTooManyOperations);
                }
            }
            // A list of NodeIds which the Client shall use for subsequent access operations. The
            // size and order of this list matches the size and order of the nodesToRegister
            // request parameter.
            // The Server may return the NodeId from the request or a new (an alias) NodeId. It
            // is recommended that the Server return a numeric NodeIds for aliasing.
            // In case no optimization is supported for a Node, the Server shall return the
            // NodeId from the request.
            const registeredNodeIds = request.nodesToRegister.map((nodeId) => session.registerNode(nodeId));
            const response = new node_opcua_service_register_node_1.RegisterNodesResponse({
                registeredNodeIds
            });
            sendResponse(response);
        });
    }
    _on_UnregisterNodesRequest(message, channel) {
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request instanceof node_opcua_service_register_node_1.UnregisterNodesRequest);
        this._apply_on_SessionObject(node_opcua_service_register_node_1.UnregisterNodesResponse, message, channel, (session, sendResponse, sendError) => {
            request.nodesToUnregister = request.nodesToUnregister || [];
            if (!request.nodesToUnregister || request.nodesToUnregister.length === 0) {
                return sendError(node_opcua_status_code_1.StatusCodes.BadNothingToDo);
            }
            if (this.engine.serverCapabilities.operationLimits.maxNodesPerRegisterNodes > 0) {
                if (request.nodesToUnregister.length > this.engine.serverCapabilities.operationLimits.maxNodesPerRegisterNodes) {
                    return sendError(node_opcua_status_code_1.StatusCodes.BadTooManyOperations);
                }
            }
            request.nodesToUnregister.map((nodeId) => session.unRegisterNode(nodeId));
            const response = new node_opcua_service_register_node_1.UnregisterNodesResponse({});
            sendResponse(response);
        });
    }
    /* istanbul ignore next */
    _on_Cancel(message, channel) {
        return g_sendError(channel, message, node_opcua_types_1.CancelResponse, node_opcua_status_code_1.StatusCodes.BadServiceUnsupported);
    }
    // NodeManagement Service Set Overview
    // This Service Set defines Services to add and delete AddressSpace Nodes and References between them. All added
    // Nodes continue to exist in the AddressSpace even if the Client that created them disconnects from the Server.
    //
    /* istanbul ignore next */
    _on_AddNodes(message, channel) {
        return g_sendError(channel, message, node_opcua_service_node_management_1.AddNodesResponse, node_opcua_status_code_1.StatusCodes.BadServiceUnsupported);
    }
    /* istanbul ignore next */
    _on_AddReferences(message, channel) {
        return g_sendError(channel, message, node_opcua_service_node_management_1.AddReferencesResponse, node_opcua_status_code_1.StatusCodes.BadServiceUnsupported);
    }
    /* istanbul ignore next */
    _on_DeleteNodes(message, channel) {
        return g_sendError(channel, message, node_opcua_service_node_management_1.DeleteNodesResponse, node_opcua_status_code_1.StatusCodes.BadServiceUnsupported);
    }
    /* istanbul ignore next */
    _on_DeleteReferences(message, channel) {
        return g_sendError(channel, message, node_opcua_service_node_management_1.DeleteReferencesResponse, node_opcua_status_code_1.StatusCodes.BadServiceUnsupported);
    }
    // Query Service
    /* istanbul ignore next */
    _on_QueryFirst(message, channel) {
        return g_sendError(channel, message, node_opcua_service_query_1.QueryFirstResponse, node_opcua_status_code_1.StatusCodes.BadServiceUnsupported);
    }
    /* istanbul ignore next */
    _on_QueryNext(message, channel) {
        return g_sendError(channel, message, node_opcua_service_query_1.QueryNextResponse, node_opcua_status_code_1.StatusCodes.BadServiceUnsupported);
    }
    /* istanbul ignore next */
    _on_HistoryUpdate(message, channel) {
        return g_sendError(channel, message, node_opcua_service_history_1.HistoryUpdateResponse, node_opcua_status_code_1.StatusCodes.BadServiceUnsupported);
    }
    createEndpoint(port1, serverOptions) {
        // add the tcp/ip endpoint with no security
        const endPoint = new server_end_point_1.OPCUAServerEndPoint({
            port: port1,
            host: serverOptions.host,
            certificateManager: this.serverCertificateManager,
            certificateChain: this.getCertificateChain(),
            privateKey: this.getPrivateKey(),
            defaultSecureTokenLifetime: serverOptions.defaultSecureTokenLifetime || 600000,
            timeout: serverOptions.timeout || 3 * 60 * 1000,
            maxConnections: this.maxConnectionsPerEndpoint,
            objectFactory: this.objectFactory,
            serverInfo: this.serverInfo,
            transportSettings: serverOptions.transportSettings
        });
        return endPoint;
    }
    createEndpointDescriptions(serverOption, endpointOptions) {
        /* istanbul ignore next */
        if (!endpointOptions) {
            throw new Error("internal error");
        }
        const hostname = (0, node_opcua_hostname_1.getFullyQualifiedDomainName)();
        endpointOptions.hostname = endpointOptions.hostname || hostname;
        endpointOptions.port = endpointOptions.port === undefined ? 26543 : endpointOptions.port;
        /* istanbul ignore next */
        if (!Object.prototype.hasOwnProperty.call(endpointOptions, "port") ||
            !isFinite(endpointOptions.port) ||
            typeof endpointOptions.port !== "number") {
            throw new Error("expecting a valid port (number) when specified. alternatively you can specify port:0 and node-opcua will choose the first available port");
        }
        const port = Number(endpointOptions.port || 0);
        const endPoint = this.createEndpoint(port, serverOption);
        endpointOptions.alternateHostname = endpointOptions.alternateHostname || [];
        const alternateHostname = endpointOptions.alternateHostname instanceof Array
            ? endpointOptions.alternateHostname
            : [endpointOptions.alternateHostname];
        const allowAnonymous = endpointOptions.allowAnonymous === undefined ? true : !!endpointOptions.allowAnonymous;
        endPoint.addStandardEndpointDescriptions({
            allowAnonymous,
            securityModes: endpointOptions.securityModes,
            securityPolicies: endpointOptions.securityPolicies,
            hostname: endpointOptions.hostname,
            alternateHostname,
            disableDiscovery: !!endpointOptions.disableDiscovery,
            // xx                hostname,
            resourcePath: serverOption.resourcePath || ""
            // TODO  userTokenTypes: endpointOptions.userTokenTypes || undefined,
            // TODO allowUnsecurePassword: endpointOptions.allowUnsecurePassword || false
        });
        return endPoint;
    }
    async initializeCM() {
        await super.initializeCM();
        await this.userCertificateManager.initialize();
    }
}
exports.OPCUAServer = OPCUAServer;
const userIdentityTokenPasswordRemoved = (userIdentityToken) => {
    if (!userIdentityToken)
        return new node_opcua_service_session_1.AnonymousIdentityToken();
    const a = userIdentityToken.clone();
    // For Username/Password tokens the password shall not be included.
    if (a instanceof node_opcua_service_session_1.UserNameIdentityToken) {
        // remove password
        a.password = Buffer.from("*************", "ascii");
    }
    // if (a instanceof X509IdentityToken) {
    //     a.certificateData = Buffer.alloc(0);
    // }
    return a;
};
function raiseAuditActivateSessionEventType(session) {
    if (this.isAuditing) {
        this.raiseEvent("AuditActivateSessionEventType", {
            /* part 5 -  6.4.3 AuditEventType */
            actionTimeStamp: { dataType: "DateTime", value: new Date() },
            status: { dataType: "Boolean", value: true },
            serverId: { dataType: "String", value: "" },
            // ClientAuditEntryId contains the human-readable AuditEntryId defined in Part 3.
            clientAuditEntryId: { dataType: "String", value: "" },
            // The ClientUserId identifies the user of the client requesting an action.
            // The ClientUserId can be obtained from the UserIdentityToken passed in the
            // ActivateSession call.
            clientUserId: { dataType: "String", value: "cc" },
            sourceName: { dataType: "String", value: "Session/ActivateSession" },
            /* part 5 - 6.4.7 AuditSessionEventType */
            sessionId: { dataType: "NodeId", value: session.nodeId },
            /* part 5 - 6.4.10 AuditActivateSessionEventType */
            clientSoftwareCertificates: {
                arrayType: node_opcua_variant_2.VariantArrayType.Array,
                dataType: "ExtensionObject" /* SignedSoftwareCertificate */,
                value: []
            },
            // UserIdentityToken reflects the userIdentityToken parameter of the ActivateSession
            // Service call.
            // For Username/Password tokens the password should NOT be included.
            userIdentityToken: {
                dataType: "ExtensionObject" /*  UserIdentityToken */,
                value: userIdentityTokenPasswordRemoved(session.userIdentityToken)
            },
            // SecureChannelId shall uniquely identify the SecureChannel. The application shall
            // use the same identifier in all AuditEvents related to the Session Service Set
            // (AuditCreateSessionEventType, AuditActivateSessionEventType and their subtypes) and
            // the SecureChannel Service Set (AuditChannelEventType and its subtypes).
            secureChannelId: { dataType: "String", value: session.channel.channelId.toString() }
        });
    }
}
const opts = { multiArgs: false };
OPCUAServer.prototype.start = (0, thenify_ex_1.withCallback)(OPCUAServer.prototype.start, opts);
OPCUAServer.prototype.initialize = (0, thenify_ex_1.withCallback)(OPCUAServer.prototype.initialize, opts);
OPCUAServer.prototype.shutdown = (0, thenify_ex_1.withCallback)(OPCUAServer.prototype.shutdown, opts);
//# sourceMappingURL=opcua_server.js.map