/**
 * @module node-opcua-server
 */
import { EventEmitter } from "events";
import { ISessionContext, UAMethod, UAObject } from "node-opcua-address-space-base";
import { SessionContext, UAVariable } from "node-opcua-address-space";
import { UInt32 } from "node-opcua-basic-types";
import { NodeClass, QualifiedNameOptions } from "node-opcua-data-model";
import { AttributeIds } from "node-opcua-data-model";
import { DataValue } from "node-opcua-data-value";
import { ExtensionObject } from "node-opcua-extension-object";
import { NodeId } from "node-opcua-nodeid";
import { NumericRange } from "node-opcua-numeric-range";
import { ObjectRegistry } from "node-opcua-object-registry";
import { TimestampsToReturn } from "node-opcua-service-read";
import { MonitoredItemModifyResult, MonitoredItemNotification, MonitoringMode, MonitoringParameters } from "node-opcua-service-subscription";
import { StatusCode } from "node-opcua-status-code";
import { EventFieldList, MonitoringFilter, ReadValueIdOptions, SubscriptionDiagnosticsDataType } from "node-opcua-types";
import { SamplingFunc } from "./sampling_func";
import { MonitoredItemBase } from "./server_subscription";
export type QueueItem = MonitoredItemNotification | EventFieldList;
export interface MonitoredItemOptions extends MonitoringParameters {
    monitoringMode: MonitoringMode;
    /**
     * the monitoredItem Id assigned by the server to this monitoredItem.
     */
    monitoredItemId: number;
    itemToMonitor?: ReadValueIdOptions;
    timestampsToReturn?: TimestampsToReturn;
    filter: ExtensionObject | null;
    /**
     * if discardOldest === true, older items are removed from the queue when queue overflows
     */
    discardOldest: boolean;
    /**
     * the size of the queue.
     */
    queueSize: number;
    /**
     * the monitored item sampling interval ..
     */
    samplingInterval: number;
    /**
     * the client handle
     */
    clientHandle: number;
}
export interface BaseNode2 extends EventEmitter {
    nodeId: NodeId;
    browseName: QualifiedNameOptions;
    nodeClass: NodeClass;
    dataType: NodeId;
    addressSpace: any;
    readAttribute(context: SessionContext | null, attributeId: AttributeIds): DataValue;
}
export interface IServerSession2 {
    sessionContext: ISessionContext;
}
export interface ISubscription {
    $session?: IServerSession2;
    subscriptionDiagnostics: SubscriptionDiagnosticsDataType;
    getMonitoredItem(monitoredItemId: number): MonitoredItem | null;
}
/**
 * a server side monitored item
 *
 * - Once created, the MonitoredItem will raised an "samplingEvent" event every "samplingInterval" millisecond
 *   until {{#crossLink "MonitoredItem/terminate:method"}}{{/crossLink}} is called.
 *
 * - It is up to the  event receiver to call {{#crossLink "MonitoredItem/recordValue:method"}}{{/crossLink}}.
 *
 */
export declare class MonitoredItem extends EventEmitter implements MonitoredItemBase {
    get node(): UAVariable | UAObject | UAMethod | null;
    set node(someNode: UAVariable | UAObject | UAMethod | null);
    static registry: ObjectRegistry;
    static minimumSamplingInterval: number;
    static defaultSamplingInterval: number;
    static maximumSamplingInterval: number;
    samplingInterval: number;
    monitoredItemId: number;
    overflow: boolean;
    oldDataValue: DataValue;
    monitoringMode: MonitoringMode;
    timestampsToReturn: TimestampsToReturn;
    itemToMonitor: any;
    filter: MonitoringFilter | null;
    discardOldest: boolean;
    queueSize: number;
    clientHandle: UInt32;
    $subscription?: ISubscription;
    _samplingId?: NodeJS.Timeout | string;
    samplingFunc: SamplingFunc | null;
    private _node;
    queue: QueueItem[];
    private _semantic_version;
    private _is_sampling;
    private _on_opcua_event_received_callback;
    private _attribute_changed_callback;
    private _value_changed_callback;
    private _semantic_changed_callback;
    private _on_node_disposed_listener;
    private _linkedItems?;
    private _triggeredNotifications?;
    constructor(options: MonitoredItemOptions);
    setNode(node: UAVariable | UAObject | UAMethod): void;
    setMonitoringMode(monitoringMode: MonitoringMode): void;
    /**
     * Terminate the  MonitoredItem.
     * This will stop the internal sampling timer.
     */
    terminate(): void;
    dispose(): void;
    get isSampling(): boolean;
    toString(): string;
    /**
     * @param dataValue       the whole dataValue
     * @param skipChangeTest  indicates whether recordValue should  not check that dataValue is really
     *                                  different from previous one, ( by checking timestamps but also variant value)
     * @private
     *
     * Notes:
     *  - recordValue can only be called within timer event
     *  - for performance reason, dataValue may be a shared value with the underlying node,
     *    therefore recordValue must clone the dataValue to make sure it retains a snapshot
     *    of the contain at the time recordValue was called.
     *
     * return true if the value has been recorded, false if not.
     *
     * Value will not be recorded :
     *   * if the range do not overlap
     *   * is !skipChangeTest and value is equal to previous value
     *
     */
    recordValue(dataValue: DataValue, skipChangeTest?: boolean, indexRange?: NumericRange): boolean;
    hasLinkItem(linkedMonitoredItemId: number): boolean;
    addLinkItem(linkedMonitoredItemId: number): StatusCode;
    removeLinkItem(linkedMonitoredItemId: number): StatusCode;
    /**
     * @private
     */
    private triggerLinkedItems;
    get hasMonitoredItemNotifications(): boolean;
    /**
     * @private
     */
    private trigger;
    extractMonitoredItemNotifications(bForce?: boolean): QueueItem[];
    modify(timestampsToReturn: TimestampsToReturn | null, monitoringParameters: MonitoringParameters | null): MonitoredItemModifyResult;
    resendInitialValue(): Promise<void>;
    private getSessionContext;
    /**
     * @private
     */
    private _on_sampling_timer;
    private _stop_sampling;
    private _on_value_changed;
    private _on_semantic_changed;
    private _on_opcua_event;
    private _getSession;
    private _start_sampling;
    private __acquireInitialValue;
    private __start_sampling;
    private _set_parameters;
    private _setOverflowBit;
    private _enqueue_notification;
    private _makeDataChangeNotification;
    /**
     * @param dataValue {DataValue} the dataValue to enqueue
     * @private
     */
    _enqueue_value(dataValue: DataValue): void;
    private _makeEventFieldList;
    private _enqueue_event;
    private _empty_queue;
    private _clear_timer;
    private _set_timer;
    private _adjust_queue_to_match_new_queue_size;
    private _adjustSampling;
    private _on_node_disposed;
}
