"use strict";
/**
 * @module node-opcua-server-discovery
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.OPCUADiscoveryServer = void 0;
const os_1 = __importDefault(require("os"));
const path_1 = __importDefault(require("path"));
const url_1 = require("url");
const chalk_1 = __importDefault(require("chalk"));
const env_paths_1 = __importDefault(require("env-paths"));
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_common_1 = require("node-opcua-common");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_secure_channel_1 = require("node-opcua-secure-channel");
const node_opcua_server_1 = require("node-opcua-server");
const node_opcua_service_discovery_1 = require("node-opcua-service-discovery");
const node_opcua_certificate_manager_1 = require("node-opcua-certificate-manager");
const node_opcua_service_endpoints_1 = require("node-opcua-service-endpoints");
const node_opcua_service_endpoints_2 = require("node-opcua-service-endpoints");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const mdns_responder_1 = require("./mdns_responder");
const debugLog = (0, node_opcua_debug_1.make_debugLog)("LDSSERVER");
const doDebug = (0, node_opcua_debug_1.checkDebugFlag)("LDSSERVER");
const errorLog = (0, node_opcua_debug_1.make_errorLog)("LDSSERVER");
function hasCapabilities(serverCapabilities, serverCapabilityFilter) {
    if (serverCapabilities == null) {
        return true; // filter is empty => no filtering should take place
    }
    if (serverCapabilityFilter.length === 0) {
        return true; // filter is empty => no filtering should take place
    }
    return !!serverCapabilities.join(" ").match(serverCapabilityFilter);
}
const defaultProductUri = "NodeOPCUA-LocalDiscoveryServer";
const defaultApplicationUri = (0, node_opcua_common_1.makeApplicationUrn)(os_1.default.hostname(), defaultProductUri);
function getDefaultCertificateManager() {
    const config = (0, env_paths_1.default)(defaultProductUri).config;
    return new node_opcua_certificate_manager_1.OPCUACertificateManager({
        name: "PKI",
        rootFolder: path_1.default.join(config, "PKI"),
        automaticallyAcceptUnknownCertificate: true
    });
}
// const weakMap = new WeakMap<MdnsDiscoveryConfiguration, BonjourHolder>;
class OPCUADiscoveryServer extends node_opcua_server_1.OPCUABaseServer {
    mDnsLDSAnnouncer;
    mDnsResponder;
    registeredServers;
    _delayInit;
    constructor(options) {
        options.serverInfo = options.serverInfo || {};
        const serverInfo = options.serverInfo;
        serverInfo.applicationType = node_opcua_service_endpoints_2.ApplicationType.DiscoveryServer;
        serverInfo.applicationUri = serverInfo.applicationUri || defaultApplicationUri;
        serverInfo.productUri = serverInfo.productUri || defaultProductUri;
        serverInfo.applicationName = serverInfo.applicationName || {
            text: defaultProductUri,
            locale: null
        };
        serverInfo.gatewayServerUri = serverInfo.gatewayServerUri || "";
        serverInfo.discoveryProfileUri = serverInfo.discoveryProfileUri || "";
        serverInfo.discoveryUrls = serverInfo.discoveryUrls || [];
        options.serverCertificateManager = options.serverCertificateManager || getDefaultCertificateManager();
        super(options);
        // see OPC UA Spec 1.2 part 6 : 7.4 Well Known Addresses
        // opc.tcp://localhost:4840/UADiscovery
        const port = options.port || 4840;
        this.capabilitiesForMDNS = ["LDS"];
        this.registeredServers = new Map();
        this.mDnsResponder = undefined;
        this._delayInit = async () => {
            const endPoint = new node_opcua_server_1.OPCUAServerEndPoint({
                port,
                certificateChain: this.getCertificateChain(),
                certificateManager: this.serverCertificateManager,
                privateKey: this.getPrivateKey(),
                serverInfo: this.serverInfo
            });
            const options1 = {
                allowAnonymous: true,
                securityModes: options.securityModes,
                securityPolicies: options.securityPolicies,
                userTokenTypes: [], // << intentionally empty (discovery server without session)
                hostname: options.hostname,
                alternateHostname: options.alternateHostname,
                disableDiscovery: true
            };
            endPoint.addStandardEndpointDescriptions(options1);
            this.endpoints.push(endPoint);
            endPoint.on("message", (message, channel) => {
                this.on_request(message, channel);
            });
        };
    }
    async start() {
        (0, node_opcua_assert_1.assert)(!this.mDnsResponder);
        (0, node_opcua_assert_1.assert)(Array.isArray(this.capabilitiesForMDNS));
        this._preInitTask.push(async () => {
            await this._delayInit();
        });
        await new Promise((resolve, reject) => super.start((err) => err ? reject(err) : resolve()));
        const endpointUri = this.getEndpointUrl();
        const { hostname } = new url_1.URL(endpointUri);
        this.mDnsResponder = new mdns_responder_1.MDNSResponder();
        this.mDnsLDSAnnouncer = new node_opcua_service_discovery_1.BonjourHolder();
        // declare the discovery server itself in bonjour
        await this.mDnsLDSAnnouncer.announcedOnMulticastSubnet({
            capabilities: this.capabilitiesForMDNS,
            name: this.serverInfo.applicationUri,
            path: "/DiscoveryServer",
            host: hostname || "",
            port: this.endpoints[0].port
        });
    }
    #shutting_down = false;
    async shutdown() {
        if (this.#shutting_down)
            return;
        this.#shutting_down = true;
        debugLog("stopping announcement of LDS on mDNS");
        // 
        for (const registeredServer of this.registeredServers.values()) {
            debugLog("LDS is shutting down and is forcefuly unregistering server", registeredServer.serverUri);
            await this.#internalRegisterServerOffline(registeredServer, true);
        }
        if (this.mDnsResponder) {
            debugLog("disposing mDnsResponder");
            await this.mDnsResponder.dispose();
            this.mDnsResponder = undefined;
            debugLog(" mDnsResponder disposed");
        }
        if (this.mDnsLDSAnnouncer) {
            debugLog("disposing mDnsLDSAnnouncer of this LDS to the mDNS");
            await this.mDnsLDSAnnouncer.stopAnnouncedOnMulticastSubnet();
            this.mDnsLDSAnnouncer = undefined;
        }
        debugLog("Shutting down Discovery Server");
        await new Promise((resolve, reject) => super.shutdown((err) => err ? reject(err) : resolve()));
        debugLog("stopping announcement of LDS on mDNS - DONE");
        // add a extra delay to ensure that the port is really closed
        // and registered server propagated the fact that LDS is not here anymore
        await new Promise((resolve) => setTimeout(resolve, 1000));
    }
    /**
     * returns the number of registered servers
     */
    get registeredServerCount() {
        return this.registeredServers.size;
    }
    getServers(channel) {
        this.serverInfo.discoveryUrls = this.getDiscoveryUrls();
        const servers = [this.serverInfo];
        for (const registered_server of this.registeredServers.values()) {
            const serverInfo = new node_opcua_service_endpoints_1.ApplicationDescription(registered_server.serverInfo);
            servers.push(serverInfo);
        }
        return servers;
    }
    _on_RegisterServer2Request(message, channel) {
        (0, node_opcua_assert_1.assert)(message.request instanceof node_opcua_service_discovery_1.RegisterServer2Request);
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request.schema.name === "RegisterServer2Request");
        request.discoveryConfiguration = request.discoveryConfiguration || [];
        this.#internalRegisterServer(node_opcua_service_discovery_1.RegisterServer2Response, request.server, request.discoveryConfiguration).then((response) => {
            channel.send_response("MSG", response, message);
        }).catch((err) => {
            errorLog("What shall I do ?", err.message);
            errorLog(err);
            let additional_messages = [];
            additional_messages.push("EXCEPTION CAUGHT WHILE PROCESSING REQUEST !!! " + request.schema.name);
            additional_messages.push(err.message);
            if (err.stack) {
                additional_messages = additional_messages.concat(err.stack.split("\n"));
            }
            const response = OPCUADiscoveryServer.makeServiceFault(node_opcua_status_code_1.StatusCodes.BadInternalError, additional_messages);
            channel.send_response("MSG", response, message);
        });
        // istanbul ignore next
    }
    _on_RegisterServerRequest(message, channel) {
        (0, node_opcua_assert_1.assert)(message.request instanceof node_opcua_service_discovery_1.RegisterServerRequest);
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request.schema.name === "RegisterServerRequest");
        this.#internalRegisterServer(node_opcua_service_discovery_1.RegisterServerResponse, request.server, undefined).then((response) => {
            channel.send_response("MSG", response, message);
        }).catch((err) => {
            let additional_messages = [];
            additional_messages.push("EXCEPTION CAUGHT WHILE PROCESSING REQUEST !!! " + request.schema.name);
            additional_messages.push(err.message);
            if (err.stack) {
                additional_messages = additional_messages.concat(err.stack.split("\n"));
            }
            const response = OPCUADiscoveryServer.makeServiceFault(node_opcua_status_code_1.StatusCodes.BadInternalError, additional_messages);
            channel.send_response("MSG", response, message);
        });
    }
    _on_FindServersOnNetworkRequest(message, channel) {
        // from OPCUA 1.04 part 4
        // This Service returns the Servers known to a Discovery Server. Unlike FindServer, this Service is
        // only implemented by Discovery Servers.
        // The Client may reduce the number of results returned by specifying filter criteria. An empty list is
        // returned if no Server matches the criteria specified by the Client.
        // This Service shall not require message security but it may require transport layer security.
        // Each time the Discovery Server creates or updates a record in its cache it shall assign a
        // monotonically increasing identifier to the record. This allows Clients to request records in batches
        // by specifying the identifier for the last record received in the last call to FindServersOnNetwork.
        // To support this the Discovery Server shall return records in numerical order starting from the
        // lowest record identifier. The Discovery Server shall also return the last time the counter was reset
        // for example due to a restart of the Discovery Server. If a Client detects that this time is more
        // recent than the last time the Client called the Service it shall call the Service again with a
        // startingRecordId of 0.
        // This Service can be used without security and it is therefore vulnerable to Denial Of Service
        // (DOS) attacks. A Server should minimize the amount of processing required to send the response
        // for this Service. This can be achieved by preparing the result in advance
        (0, node_opcua_assert_1.assert)(message.request instanceof node_opcua_service_discovery_1.FindServersOnNetworkRequest);
        const request = message.request;
        (0, node_opcua_assert_1.assert)(request.schema.name === "FindServersOnNetworkRequest");
        function sendError(statusCode) {
            const response1 = new node_opcua_service_discovery_1.FindServersOnNetworkResponse({ responseHeader: { serviceResult: statusCode } });
            return channel.send_response("MSG", response1, message);
        }
        sendError;
        if (this.#shutting_down) {
            return sendError(node_opcua_status_code_1.StatusCodes.BadShutdown);
        }
        //     startingRecordId         Counter Only records with an identifier greater than this number will be
        //                              returned.
        //                              Specify 0 to start with the first record in the cache.
        //     maxRecordsToReturn       UInt32 The maximum number of records to return in the response.
        //                              0 indicates that there is no limit.
        //     serverCapabilityFilter[] String List of Server capability filters. The set of allowed server capabilities
        //                              are defined in Part 12.
        //                              Only records with all of the specified server capabilities are
        //                              returned.
        //                              The comparison is case insensitive.
        //                              If this list is empty then no filtering is performed
        // ------------------------
        // The last time the counters were reset.
        const lastCounterResetTime = new Date();
        //  servers[] ServerOnNetwork List of DNS service records that meet criteria specified in the
        // request. This list is empty if no Servers meet the criteria
        const servers = [];
        request.serverCapabilityFilter = request.serverCapabilityFilter || [];
        const serverCapabilityFilter = request.serverCapabilityFilter
            .map((x) => x.toUpperCase())
            .sort()
            .join(" ");
        debugLog(" startingRecordId = ", request.startingRecordId);
        if (this.mDnsResponder) {
            for (const serverOnNetwork of this.mDnsResponder.registeredServers) {
                debugLog("Exploring server ", serverOnNetwork.serverName);
                if (serverOnNetwork.recordId <= request.startingRecordId) {
                    continue;
                }
                if (!hasCapabilities(serverOnNetwork.serverCapabilities, serverCapabilityFilter)) {
                    // istanbul ignore next
                    if (doDebug) {
                        debugLog("   server ", serverOnNetwork.serverName, serverOnNetwork.serverCapabilities ? serverOnNetwork.serverCapabilities.join(",") : [], " does not match serverCapabilities ", serverCapabilityFilter);
                    }
                    continue;
                }
                debugLog("   server ", serverOnNetwork.serverName, " found");
                servers.push(serverOnNetwork);
                if (servers.length === request.maxRecordsToReturn) {
                    debugLog("max records to return reached", request.maxRecordsToReturn);
                    break;
                }
            }
        }
        const response = new node_opcua_service_discovery_1.FindServersOnNetworkResponse({
            lastCounterResetTime, //  UtcTime The last time the counters were reset
            servers
        });
        channel.send_response("MSG", response, message);
    }
    async #stopAnnouncedOnMulticastSubnet(conf) {
        const b = conf.bonjourHolder;
        await b.stopAnnouncedOnMulticastSubnet();
        conf.bonjourHolder = undefined;
    }
    async #announcedOnMulticastSubnet(conf, announcement) {
        const serviceConfig = (0, node_opcua_service_discovery_1.announcementToServiceConfig)(announcement);
        let b = conf.bonjourHolder;
        if (b && b.serviceConfig) {
            if ((0, node_opcua_service_discovery_1.isSameService)(b.serviceConfig, serviceConfig)) {
                debugLog("Configuration ", conf.mdnsServerName, " has not changed !");
                // nothing to do
                return;
            }
            else {
                // istanbul ignore next
                if (doDebug) {
                    debugLog("Configuration ", conf.mdnsServerName, " HAS changed !");
                    debugLog(" Was ", (0, node_opcua_service_discovery_1.serviceToString)(b.serviceConfig));
                    debugLog(" is  ", announcement);
                }
            }
            await this.#stopAnnouncedOnMulticastSubnet(conf);
        }
        b = new node_opcua_service_discovery_1.BonjourHolder();
        conf.bonjourHolder = b;
        await b.announcedOnMulticastSubnet(announcement);
    }
    async #dealWithDiscoveryConfiguration(previousConfMap, server1, serverInfo, discoveryConfiguration) {
        // mdnsServerName     String     The name of the Server when it is announced via mDNS.
        //                               See Part 12 for the details about mDNS. This string shall be 
        //                               less than 64 bytes.
        //                               If not specified the first element of the serverNames array 
        //                               is used (truncated to 63 bytes if necessary).
        // serverCapabilities [] String  The set of Server capabilities supported by the Server.
        //                               A Server capability is a short identifier for a feature
        //                               The set of allowed Server capabilities are defined in Part 12.
        discoveryConfiguration.mdnsServerName ??= server1.serverNames[0].text;
        serverInfo.discoveryUrls ??= [];
        const endpointUrl = serverInfo.discoveryUrls[0];
        const parsedUrl = new url_1.URL(endpointUrl);
        discoveryConfiguration.serverCapabilities = discoveryConfiguration.serverCapabilities || [];
        const announcement = {
            capabilities: discoveryConfiguration.serverCapabilities.map((x) => x) || ["DA"],
            name: discoveryConfiguration.mdnsServerName,
            host: parsedUrl.hostname || "",
            path: parsedUrl.pathname || "/",
            port: parseInt(parsedUrl.port, 10)
        };
        if (previousConfMap.has(discoveryConfiguration.mdnsServerName)) {
            // configuration already exists
            debugLog("Configuration ", discoveryConfiguration.mdnsServerName, " already exists !");
            const prevConf = previousConfMap.get(discoveryConfiguration.mdnsServerName);
            previousConfMap.delete(discoveryConfiguration.mdnsServerName);
            discoveryConfiguration.bonjourHolder = prevConf.bonjourHolder;
        }
        // let's announce the server on the  multicast DNS
        await this.#announcedOnMulticastSubnet(discoveryConfiguration, announcement);
        return node_opcua_status_code_1.StatusCodes.Good;
    }
    /**
     *
     * @param server
     * @param forced  true :indicated if the LDS is forcing the Server to be seen as unregistered, false
     * when the offline comes from the server it self.
     * @returns
     */
    async #internalRegisterServerOffline(server, forced) {
        const key = server.serverUri;
        let configurationResults = null;
        // server is announced offline
        if (this.registeredServers.has(key)) {
            const serverToUnregister = this.registeredServers.get(key);
            debugLog(chalk_1.default.cyan("unregistering server : "), chalk_1.default.yellow(serverToUnregister.serverUri));
            configurationResults = [];
            const discoveryConfigurations = serverToUnregister.discoveryConfiguration || [];
            for (const conf of discoveryConfigurations) {
                await this.#stopAnnouncedOnMulticastSubnet(conf);
                configurationResults.push(node_opcua_status_code_1.StatusCodes.Good);
            }
            this.registeredServers.delete(key);
            serverToUnregister.isOnline = false;
            this.emit("onUnregisterServer", serverToUnregister, forced);
        }
        return configurationResults;
    }
    async #internalRegisterServerOnline(server, discoveryConfigurations) {
        (0, node_opcua_assert_1.assert)(discoveryConfigurations);
        const key = server.serverUri;
        let configurationResults = null;
        debugLog(chalk_1.default.cyan(" registering server : "), chalk_1.default.yellow(server.serverUri));
        // prepare serverInfo which will be used by FindServers
        const serverInfo = {
            applicationName: server.serverNames[0], // which one shall we use ?
            applicationType: server.serverType,
            applicationUri: server.serverUri,
            discoveryUrls: server.discoveryUrls,
            gatewayServerUri: server.gatewayServerUri,
            productUri: server.productUri
            // XXX ?????? serverInfo.discoveryProfileUri = serverInfo.discoveryProfileUri;
        };
        const previousConfMap = new Map();
        // let check in the server has already been registed on this LDS 
        let firstTimeRegistration = true;
        if (this.registeredServers.has(key)) {
            // server already exists and must only be updated
            const previousServer = this.registeredServers.get(key);
            for (const conf of previousServer.discoveryConfiguration) {
                previousConfMap.set(conf.mdnsServerName, conf);
            }
            firstTimeRegistration = false;
        }
        this.registeredServers.set(key, server);
        this.emit("onRegisterServer", server, firstTimeRegistration);
        // xx server.semaphoreFilePath = server.semaphoreFilePath;
        // xx server.serverNames = server.serverNames;
        server.serverInfo = serverInfo;
        server.discoveryConfiguration = discoveryConfigurations;
        configurationResults = [];
        for (const conf of discoveryConfigurations) {
            const statusCode = await this.#dealWithDiscoveryConfiguration(previousConfMap, server, serverInfo, conf);
            configurationResults.push(statusCode);
        }
        // now also unregister unprocessed
        if (previousConfMap.size !== 0) {
            debugLog(" Warning some conf need to be removed !");
        }
        return configurationResults;
    }
    // eslint-disable-next-line max-statements
    async #internalRegisterServer(RegisterServerXResponse, rawServer, discoveryConfigurations) {
        // #region check parameter validity
        function sendError(statusCode) {
            debugLog(chalk_1.default.red("_on_RegisterServer(2)Request error"), statusCode.toString());
            const response1 = new node_opcua_secure_channel_1.ServiceFault({
                responseHeader: { serviceResult: statusCode }
            });
            return response1;
        }
        if (this.#shutting_down) {
            return sendError(node_opcua_status_code_1.StatusCodes.BadShutdown);
        }
        const server = rawServer;
        // check serverType is valid
        if (!_isValidServerType(server.serverType)) {
            debugLog("Invalid server Type", node_opcua_service_endpoints_2.ApplicationType[server.serverType]);
            return sendError(node_opcua_status_code_1.StatusCodes.BadInvalidArgument);
        }
        if (!server.serverUri) {
            debugLog("Missing serverURI");
            return sendError(node_opcua_status_code_1.StatusCodes.BadInvalidArgument);
        }
        server.serverNames = server.serverNames || [];
        // BadServerNameMissing
        if (server.serverNames.length === 0 || !server.serverNames[0].text) {
            return sendError(node_opcua_status_code_1.StatusCodes.BadServerNameMissing);
        }
        // BadDiscoveryUrlMissing
        server.discoveryUrls = server.discoveryUrls || [];
        if (server.discoveryUrls.length === 0 || !server.discoveryUrls[0]) {
            return sendError(node_opcua_status_code_1.StatusCodes.BadDiscoveryUrlMissing);
        }
        // BadServerUriInvalid
        // TODO
        // #endregion
        if (!discoveryConfigurations) {
            discoveryConfigurations = [
                new node_opcua_service_discovery_1.MdnsDiscoveryConfiguration({
                    mdnsServerName: undefined,
                    serverCapabilities: ["NA"]
                })
            ];
        }
        const configurationResults = server?.isOnline ?
            await this.#internalRegisterServerOnline(server, discoveryConfigurations) :
            await this.#internalRegisterServerOffline(server, false);
        const response = new RegisterServerXResponse({});
        if (response instanceof node_opcua_service_discovery_1.RegisterServer2Response) {
            response.configurationResults = configurationResults;
        }
        return response;
    }
}
exports.OPCUADiscoveryServer = OPCUADiscoveryServer;
/*== private
 * returns true if the serverType can be added to a discovery server.
 * @param serverType
 * @return {boolean}
 * @private
 */
function _isValidServerType(serverType) {
    switch (serverType) {
        case node_opcua_service_endpoints_2.ApplicationType.Client:
            return false;
        case node_opcua_service_endpoints_2.ApplicationType.Server:
        case node_opcua_service_endpoints_2.ApplicationType.ClientAndServer:
        case node_opcua_service_endpoints_2.ApplicationType.DiscoveryServer:
            return true;
    }
    return false;
}
//# sourceMappingURL=opcua_discovery_server.js.map