"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClientSecureChannelLayer = exports.requestHandleNotSetValue = void 0;
exports.coerceConnectionStrategy = coerceConnectionStrategy;
/* eslint-disable @typescript-eslint/no-unsafe-declaration-merging */
/**
 * @module node-opcua-secure-channel
 */
const crypto_1 = require("crypto");
const events_1 = require("events");
const util_1 = require("util");
const chalk_1 = __importDefault(require("chalk"));
const async_1 = __importDefault(require("async"));
const web_1 = require("node-opcua-crypto/web");
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_binary_stream_1 = require("node-opcua-binary-stream");
const node_opcua_utils_1 = require("node-opcua-utils");
const node_opcua_chunkmanager_1 = require("node-opcua-chunkmanager");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_service_secure_channel_1 = require("node-opcua-service-secure-channel");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_transport_1 = require("node-opcua-transport");
const node_opcua_transport_2 = require("node-opcua-transport");
const node_opcua_common_1 = require("node-opcua-common");
const message_builder_1 = require("../message_builder");
const message_chunker_1 = require("../message_chunker");
const message_header_to_string_1 = require("../message_header_to_string");
const security_policy_1 = require("../security_policy");
const services_1 = require("../services");
const debugLog = (0, node_opcua_debug_1.make_debugLog)("SecureChannel");
const errorLog = (0, node_opcua_debug_1.make_errorLog)("SecureChannel");
const doDebug = (0, node_opcua_debug_1.checkDebugFlag)("SecureChannel");
const warningLog = (0, node_opcua_debug_1.make_warningLog)("SecureChannel");
const checkChunks = doDebug && false;
const doDebug1 = false;
// set checkTimeout to true to enable timeout trace checking
const checkTimeout = !!process.env.NODEOPCUACHECKTIMEOUT || false;
const common_1 = require("../common");
const utils_1 = require("../utils");
const duration_to_string_1 = require("./duration_to_string");
const token_stack_1 = require("../token_stack");
// tslint:disable-next-line: no-var-requires
const backoff = require("backoff");
exports.requestHandleNotSetValue = 0xdeadbeef;
function process_request_callback(requestData, err, response) {
    (0, node_opcua_assert_1.assert)(typeof requestData.callback === "function");
    const request = requestData.request;
    if (!response && !err && requestData.msgType !== "CLO") {
        // this case happens when CLO is called and when some pending transactions
        // remains in the queue...
        err = new Error(" Connection has been closed by client , but this transaction cannot be honored");
    }
    if (response && response instanceof services_1.ServiceFault) {
        response.responseHeader.stringTable = [...(response.responseHeader.stringTable || [])];
        err = new Error(" serviceResult = " + response.responseHeader.serviceResult.toString());
        //  "  returned by server \n response:" + response.toString() + "\n  request: " + request.toString());
        err.response = response;
        err.request = request;
        response = undefined;
    }
    const theCallbackFunction = requestData.callback;
    /* istanbul ignore next */
    if (!theCallbackFunction) {
        throw new Error("Internal error");
    }
    (0, node_opcua_assert_1.assert)(requestData.msgType === "CLO" || (err && !response) || (!err && response));
    // let set callback to undefined to prevent callback to be called again
    requestData.callback = undefined;
    theCallbackFunction(err || null, !err && response !== null ? response : undefined);
}
function coerceConnectionStrategy(options) {
    options = options || {};
    const maxRetry = options.maxRetry === undefined ? 10 : options.maxRetry;
    const initialDelay = options.initialDelay || 10;
    const maxDelay = options.maxDelay || 10000;
    const randomisationFactor = options.randomisationFactor === undefined ? 0 : options.randomisationFactor;
    return {
        initialDelay,
        maxDelay,
        maxRetry,
        randomisationFactor
    };
}
/**
 * a ClientSecureChannelLayer represents the client side of the OPCUA secure channel.
 */
class ClientSecureChannelLayer extends events_1.EventEmitter {
    static g_counter = 0;
    #_counter = _a.g_counter++;
    #_bytesRead = 0;
    #_bytesWritten = 0;
    #_timeDrift = 0;
    static minTransactionTimeout = 5 * 1000; // 5 sec
    static defaultTransactionTimeout = 15 * 1000; // 15 seconds
    static defaultTransportTimeout = 60 * 1000; // 60 seconds
    /**
     *
     * maxClockSkew: The amount of clock skew that can be tolerated between server and client clocks
     *
     * from https://reference.opcfoundation.org/Core/Part6/v104/docs/6.3
     *
     *  The amount of clock skew that can be tolerated depends on the system security requirements
     *  and applications shall allow administrators to configure the acceptable clock skew when
     *  verifying times. A suitable default value is 5 minutes.
     */
    static maxClockSkew = 5 * 60 * 1000; // 5 minutes
    defaultTransactionTimeout;
    /**
     * true if the secure channel is trying to establish the connection with the server. In this case, the client
     * may be in the middle of the backoff connection process.
     *
     */
    get isConnecting() {
        return !!this.__call;
    }
    get bytesRead() {
        return this.#_bytesRead + (this.#_transport ? this.#_transport.bytesRead : 0);
    }
    get bytesWritten() {
        return this.#_bytesWritten + (this.#_transport ? this.#_transport.bytesWritten : 0);
    }
    get timeDrift() {
        return this.#_timeDrift;
    }
    get transactionsPerformed() {
        return this.#_lastRequestId;
    }
    get timedOutRequestCount() {
        return this.#_timeout_request_count;
    }
    #requestedTransportSettings;
    protocolVersion;
    securityMode;
    securityPolicy;
    endpointUrl;
    channelId;
    activeSecurityToken;
    #_lastRequestId;
    #_transport;
    #_pending_transport;
    #parent;
    #messageChunker;
    #requestedLifetime;
    #tokenRenewalInterval;
    #messageBuilder;
    #_requests;
    #__in_normal_close_operation;
    #_timeout_request_count;
    #_securityTokenTimeoutId;
    #transportTimeout;
    #connectionStrategy;
    last_transaction_stats;
    #serverCertificate; // the receiverCertificate => Receiver is Server
    #receiverPublicKey;
    __call;
    #_isOpened;
    #lastError;
    #startReceivingTick = 0;
    #_isDisconnecting = false;
    #tokenStack;
    constructor(options) {
        super();
        this.defaultTransactionTimeout = options.defaultTransactionTimeout || _a.defaultTransactionTimeout;
        this.activeSecurityToken = null;
        this.#receiverPublicKey = null;
        this.endpointUrl = "";
        if (global.hasResourceLeakDetector && !global.ResourceLeakDetectorStarted) {
            throw new Error("ClientSecureChannelLayer not in ResourceLeakDetectorStarted");
        }
        (0, node_opcua_assert_1.assert)(this instanceof _a);
        this.#_isOpened = false;
        this.#_transport = undefined;
        this.#_lastRequestId = 0;
        this.#parent = options.parent;
        this.protocolVersion = 0;
        this.#tokenStack = new token_stack_1.TokenStack(1);
        this.#requestedLifetime = options.defaultSecureTokenLifetime || 30000;
        this.#tokenRenewalInterval = options.tokenRenewalInterval || 0;
        this.securityMode = (0, node_opcua_service_secure_channel_1.coerceMessageSecurityMode)(options.securityMode);
        this.securityPolicy = (0, security_policy_1.coerceSecurityPolicy)(options.securityPolicy);
        this.#serverCertificate = (0, common_1.extractFirstCertificateInChain)(options.serverCertificate);
        (0, node_opcua_assert_1.assert)(this.securityMode == node_opcua_service_secure_channel_1.MessageSecurityMode.None || this.#serverCertificate);
        // use to send Request => we use client keys
        this.#messageChunker = new message_chunker_1.MessageChunker({
            securityMode: this.securityMode
            // note maxMessageSize cannot be set at this stage, transport is not known
        });
        this.#_requests = {};
        this.#__in_normal_close_operation = false;
        this.#_timeout_request_count = 0;
        this.#_securityTokenTimeoutId = null;
        this.#transportTimeout = options.transportTimeout || _a.defaultTransportTimeout;
        this.#requestedTransportSettings = options.transportSettings || {};
        this.#connectionStrategy = coerceConnectionStrategy(options.connectionStrategy);
        this.channelId = 0;
    }
    getTransportSettings() {
        const { maxMessageSize } = this.#_transport ? this.#_transport.getTransportSettings() : { maxMessageSize: 2048 };
        return { maxMessageSize: maxMessageSize || 0 };
    }
    get transportTimeout() {
        return this.#transportTimeout;
    }
    getPrivateKey() {
        return this.#parent ? this.#parent.getPrivateKey() : null;
    }
    getCertificateChain() {
        return this.#parent ? this.#parent.getCertificateChain() : null;
    }
    getCertificate() {
        return this.#parent ? this.#parent.getCertificate() : null;
    }
    toString() {
        let str = "";
        str += "\n securityMode ............. : " + node_opcua_service_secure_channel_1.MessageSecurityMode[this.securityMode];
        str += "\n securityPolicy............ : " + this.securityPolicy;
        str += "\n securityToken ............ : " + (this.activeSecurityToken ? this.activeSecurityToken.toString() : "null");
        str += "\n timedOutRequestCount.....  : " + this.timedOutRequestCount;
        str += "\n transportTimeout ......... : " + this.#transportTimeout;
        str += "\n is transaction in progress : " + this.isTransactionInProgress();
        str += "\n is connecting ............ : " + this.isConnecting;
        str += "\n is disconnecting ......... : " + this.#_isDisconnecting;
        str += "\n is opened ................ : " + this.isOpened();
        str += "\n is valid ................. : " + this.isValid();
        str += "\n channelId ................ : " + this.channelId;
        str += "\n transportParameters: ..... : ";
        str += "\n   maxMessageSize (to send) : " + (this.#_transport?.parameters?.maxMessageSize || "<not set>");
        str += "\n   maxChunkCount  (to send) : " + (this.#_transport?.parameters?.maxChunkCount || "<not set>");
        str += "\n   receiveBufferSize(server): " + (this.#_transport?.parameters?.receiveBufferSize || "<not set>");
        str += "\n   sendBufferSize (to send) : " + (this.#_transport?.parameters?.sendBufferSize || "<not set>");
        str += "\ntime drift with server      : " + (0, duration_to_string_1.durationToString)(this.#_timeDrift);
        str += "\n";
        return str;
    }
    isTransactionInProgress() {
        return Object.keys(this.#_requests).length > 0;
    }
    /**
     * establish a secure channel with the provided server end point.
     *
     * @example
     *
     *    ```javascript
     *
     *    const secureChannel  = new ClientSecureChannelLayer({});
     *
     *    secureChannel.on("end", function(err) {
     *         console.log("secure channel has ended",err);
     *         if(err) {
     *            console.log(" the connection was closed by an external cause such as server shutdown");
     *        }
     *    });
     *    secureChannel.create("opc.tcp://localhost:1234/UA/Sample", (err) => {
     *         if(err) {
     *              console.log(" cannot establish secure channel" , err);
     *         } else {
     *              console.log("secure channel has been established");
     *         }
     *    });
     *
     *    ```
     */
    create(endpointUrl, callback) {
        (0, node_opcua_assert_1.assert)(typeof callback === "function");
        if (this.securityMode !== node_opcua_service_secure_channel_1.MessageSecurityMode.None) {
            // istanbul ignore next
            if (!this.#serverCertificate) {
                return callback(new Error("ClientSecureChannelLayer#create : expecting a server certificate when securityMode is not None"));
            }
            // take the opportunity of this async method to perform some async pre-processing
            if (!this.#receiverPublicKey) {
                (0, web_1.extractPublicKeyFromCertificate)(this.#serverCertificate, (err, publicKey) => {
                    /* istanbul ignore next */
                    if (err) {
                        return callback(err);
                    }
                    /* istanbul ignore next */
                    if (!publicKey) {
                        throw new Error("Internal Error");
                    }
                    this.#receiverPublicKey = (0, crypto_1.createPublicKey)(publicKey);
                    this.create(endpointUrl, callback);
                });
                return;
            }
        }
        this.endpointUrl = endpointUrl;
        const transport = new node_opcua_transport_1.ClientTCP_transport(this.#requestedTransportSettings);
        transport.timeout = this.#transportTimeout;
        doDebug &&
            debugLog("ClientSecureChannelLayer#create creating ClientTCP_transport with  transport.timeout = ", transport.timeout);
        (0, node_opcua_assert_1.assert)(!this.#_pending_transport);
        this.#_pending_transport = transport;
        this.#_establish_connection(transport, endpointUrl, (err) => {
            if (err) {
                doDebug && debugLog(chalk_1.default.red("cannot connect to server"));
                this.#_pending_transport = undefined;
                transport.dispose();
                return callback(err);
            }
            this.#_on_connection(transport, callback);
        });
    }
    dispose() {
        this.#_dispose_transports();
        this.abortConnection(() => {
            /* empty */
        });
        this.#_cancel_security_token_watchdog();
    }
    sabotageConnection() {
        this.#_closeWithError(new Error("Sabotage"), node_opcua_status_code_1.StatusCodes.Bad);
    }
    /**
     * forceConnectionBreak is a private api method that
     * can be used to simulate a connection break or
     * terminate the channel in case of a socket timeout that
     * do not produce a socket close event
     * @private
     */
    forceConnectionBreak() {
        const transport = this.#_transport;
        if (!transport) {
            return;
        }
        transport.forceConnectionBreak();
    }
    abortConnection(callback) {
        if (this.#_isDisconnecting) {
            doDebug && debugLog("abortConnection already aborting!");
            return callback();
        }
        this.#_isDisconnecting = true;
        doDebug && debugLog("abortConnection ", !!this.__call);
        async_1.default.series([
            (inner_callback) => {
                if (this.__call) {
                    this.__call.once("abort", () => inner_callback());
                    this.__call._cancelBackoff = true;
                    this.__call.abort();
                    this.__call = null;
                }
                else {
                    inner_callback();
                }
            },
            (inner_callback) => {
                if (!this.#_pending_transport) {
                    return inner_callback();
                }
                this.#_pending_transport.disconnect(() => {
                    inner_callback();
                });
            },
            (inner_callback) => {
                if (!this.#_transport) {
                    return inner_callback();
                }
                this.#_transport.disconnect(() => {
                    inner_callback();
                });
            }
        ], () => {
            callback();
        });
    }
    /**
     * perform a OPC-UA message transaction, asynchronously.
     * During a transaction, the client sends a request to the server. The provided callback will be invoked
     * at a later stage with the reply from the server, or the error.
     *
     * preconditions:
     *   - the channel must be opened
     *
     * @example
     *
     *    ```javascript
     *    let secure_channel ; // get a  ClientSecureChannelLayer somehow
     *
     *    const request = new BrowseRequest({...});
     *    secure_channel.performMessageTransaction(request, (err,response) => {
     *       if (err) {
     *         // an error has occurred
     *       } else {
     *          assert(response instanceof BrowseResponse);
     *         // do something with response.
     *       }
     *    });
     *    ```
     *
     */
    performMessageTransaction(request, callback) {
        this.#_performMessageTransaction("MSG", request, callback);
    }
    isValid() {
        if (!this.#_transport) {
            return false;
        }
        return this.#_transport.isValid();
    }
    isOpened() {
        return this.isValid() && this.#_isOpened;
    }
    getDisplayName() {
        if (!this.#parent) {
            return "";
        }
        return "" + (this.#parent.applicationName ? this.#parent.applicationName + " " : "") + this.#parent.clientName;
    }
    cancelPendingTransactions(callback) {
        (0, node_opcua_assert_1.assert)(typeof callback === "function", "expecting a callback function, but got " + callback);
        // istanbul ignore next
        if (doDebug) {
            debugLog("cancelPendingTransactions ", this.getDisplayName(), " = ", Object.keys(this.#_requests)
                .map((k) => this.#_requests[k].request.constructor.name)
                .join(" "));
        }
        for (const key of Object.keys(this.#_requests)) {
            // kill timer id
            const transaction = this.#_requests[key];
            if (transaction.callback) {
                transaction.callback(new Error("Transaction has been canceled because client channel  is being closed"));
            }
        }
        callback();
    }
    /**
     * Close a client SecureChannel ,by sending a CloseSecureChannelRequest to the server.
     *
     * After this call, the connection is closed and no further transaction can be made.
     */
    close(callback) {
        (0, node_opcua_assert_1.assert)(typeof callback === "function", "expecting a callback function, but got " + callback);
        // cancel any pending transaction
        this.cancelPendingTransactions(( /* err?: Error */) => {
            // what the specs says:
            // --------------------
            //   The client closes the connection by sending a CloseSecureChannelRequest and closing the
            //   socket gracefully. When the server receives this message it shall release all resources
            //   allocated for the channel. The server does not send a CloseSecureChannel response
            //
            // ( Note : some servers do  send a CloseSecureChannel though !)
            // there is no need for the security token expiration event to trigger anymore
            this.#_cancel_security_token_watchdog();
            doDebug && debugLog("Sending CloseSecureChannelRequest to server");
            const request = new services_1.CloseSecureChannelRequest();
            this.#__in_normal_close_operation = true;
            if (!this.#_transport || this.#_transport.isDisconnecting()) {
                this.dispose();
                return callback(new Error("Transport disconnected"));
            }
            this.#_performMessageTransaction("CLO", request, (err) => {
                // istanbul ignore next
                if (err) {
                    warningLog("CLO transaction terminated with error: ", err.message, " has transport ?: ", !!this.#_transport);
                }
                if (this.#_transport) {
                    this.#_transport.disconnect(() => {
                        this.dispose();
                        callback();
                    });
                }
                else {
                    this.dispose();
                    callback();
                }
            });
        });
    }
    /**
     * @private internal use only : (used for test)
     */
    getTransport() {
        return this.#_transport;
    }
    /**
     * @private internal use only : (use for testing purpose)
     */
    _getMessageBuilder() {
        return this.#messageBuilder;
    }
    // #region private
    #_dispose_transports() {
        if (this.#_transport) {
            this.#_bytesRead += this.#_transport.bytesRead || 0;
            this.#_bytesWritten += this.#_transport.bytesWritten || 0;
            this.#_transport.dispose();
            this.#_transport = undefined;
        }
        if (this.#_pending_transport) {
            this.#_bytesRead += this.#_pending_transport.bytesRead || 0;
            this.#_bytesWritten += this.#_pending_transport.bytesWritten || 0;
            this.#_pending_transport.dispose();
            this.#_pending_transport = undefined;
        }
    }
    #_install_message_builder() {
        // istanbul ignore next
        if (!this.#_transport || !this.#_transport.parameters) {
            throw new Error("internal error");
        }
        // use to receive Server response
        this.#messageBuilder = new message_builder_1.MessageBuilder(this.#tokenStack.serverKeyProvider(), {
            name: "client",
            privateKey: this.getPrivateKey() || undefined,
            securityMode: this.securityMode,
            maxChunkSize: this.#_transport.receiveBufferSize || 0,
            maxChunkCount: this.#_transport.maxChunkCount || 0,
            maxMessageSize: this.#_transport.maxMessageSize || 0
        });
        /* istanbul ignore next */
        if (node_opcua_transport_2.doTraceChunk) {
            console.log(chalk_1.default.cyan((0, node_opcua_utils_1.timestamp)()), "   MESSAGE BUILDER LIMITS", "maxMessageSize = ", this.#messageBuilder.maxMessageSize, "maxChunkCount = ", this.#messageBuilder.maxChunkCount, "maxChunkSize = ", this.#messageBuilder.maxChunkSize, "(", this.#messageBuilder.maxChunkSize * this.#messageBuilder.maxChunkCount, ")");
        }
        this.#messageBuilder
            .on("message", (response, msgType, securityHeader, requestId, channelId) => {
            this.emit("message", response, msgType, securityHeader, requestId, channelId);
            this.#_on_message_received(response, msgType, securityHeader, requestId);
        })
            .on("startChunk", () => {
            //
            if (utils_1.doPerfMonitoring) {
                this.#startReceivingTick = (0, node_opcua_utils_1.get_clock_tick)();
            }
        })
            .on("abandon", (requestId) => {
            const requestData = this.#_requests[requestId];
            // istanbul ignore next
            if (doDebug) {
                debugLog("request id = ", requestId, "message was ", requestData);
            }
            const err = new services_1.ServiceFault({
                responseHeader: {
                    requestHandle: requestId,
                    serviceResult: node_opcua_status_code_1.StatusCodes.BadOperationAbandoned
                }
            });
            const callback = requestData.callback;
            delete this.#_requests[requestId];
            callback && callback(null, err);
        })
            .on("error", (err, statusCode, requestId) => {
            // istanbul ignore next
            if (!requestId) {
                return;
            }
            const requestData = this.#_requests[requestId];
            // istanbul ignore next
            doDebug && debugLog("request id = ", requestId, err, "message was ", requestData);
            if (utils_1.doTraceClientRequestContent) {
                errorLog(" message was 2:", requestData?.request?.toString() || "<null>");
            }
            if (!requestData) {
                warningLog("requestData not found for requestId = ", requestId);
                // istanbul ignore next
                doDebug && warningLog("err = ", err);
                return;
            }
            const callback = requestData.callback;
            delete this.#_requests[requestId];
            callback && callback(err, undefined);
            this.#_closeWithError(err, statusCode);
            return;
        });
    }
    #_closeWithError(err, statusCode) {
        if (this.#_transport) {
            this.#_transport.prematureTerminate(err, statusCode);
        }
        this.dispose();
    }
    #_on_transaction_completed(transactionStatistics) {
        /* istanbul ignore next */
        if (utils_1.doTraceStatistics) {
            // dump some statistics about transaction ( time and sizes )
            (0, utils_1._dump_client_transaction_statistics)(transactionStatistics);
        }
        this.emit("end_transaction", transactionStatistics);
    }
    #_on_message_received(response, msgType, securityHeader, requestId) {
        /* istanbul ignore next */
        if (response.responseHeader.requestHandle !== requestId) {
            warningLog(msgType, response.toString());
            errorLog(chalk_1.default.red.bgWhite.bold("xxxxx  <<<<<< _on_message_received  ERROR!   requestHandle !== requestId"), "requestId=", requestId, this.#_requests[requestId]?.constructor.name, "response.responseHeader.requestHandle=", response.responseHeader.requestHandle, response.schema.name.padStart(30));
        }
        if (response instanceof services_1.OpenSecureChannelResponse) {
            if (this.channelId === 0) {
                this.channelId = response.securityToken?.channelId || 0;
            }
            else {
                if (this.channelId !== response.securityToken?.channelId) {
                    warningLog("channelId is supposed to be  ", this.channelId, " but is ", response.securityToken?.channelId);
                }
            }
        }
        else {
        }
        /* istanbul ignore next */
        if (utils_1.doTraceClientMessage) {
            (0, utils_1.traceClientResponseMessage)(response, this.channelId, this.#_counter);
        }
        const requestData = this.#_requests[requestId];
        /* istanbul ignore next */
        if (!requestData) {
            if (this.#__in_normal_close_operation) {
                // may be some responses that are received from the server
                // after the communication is closed. We can just ignore them
                // ( this happens with Dotnet C# stack for instance)
                return;
            }
            errorLog(chalk_1.default.cyan.bold("xxxxx  <<<<<< _on_message_received for unknown or timeout request "), requestId, response.schema.name.padStart(30), response.responseHeader.serviceResult.toString(), this.channelId);
            throw new Error(" =>  invalid requestId =" + requestId);
        }
        const request = requestData.request;
        /* istanbul ignore next */
        if (utils_1.doPerfMonitoring) {
            requestData.startReceivingTick = this.#startReceivingTick;
        }
        delete this.#_requests[requestId];
        /* istanbul ignore next */
        if (response.responseHeader.requestHandle !== request.requestHeader.requestHandle) {
            const expected = request.requestHeader.requestHandle;
            const actual = response.responseHeader.requestHandle;
            if (actual !== 0x0) {
                // note some old OPCUA Server, like siemens with OPCUA 1.2 may send 0x00 as a
                // requestHandle, this is not harmful. THis happened with OpenSecureChannelRequest
                // so we only display the warning message if we have a real random discrepancy between the two requestHandle.
                const moreInfo = "Request= " + request.schema.name + " Response = " + response.schema.name;
                const message = " WARNING SERVER responseHeader.requestHandle is invalid" +
                    ": expecting 0x" +
                    expected.toString(16) +
                    "(" +
                    expected +
                    ")" +
                    "  but got 0x" +
                    actual.toString(16) +
                    "(" +
                    actual +
                    ")" +
                    " ";
                debugLog(chalk_1.default.red.bold(message), chalk_1.default.yellow(moreInfo));
                warningLog(chalk_1.default.red.bold(message), chalk_1.default.yellow(moreInfo));
                warningLog(request.toString());
            }
        }
        requestData.response = response;
        /* istanbul ignore next */
        if (utils_1.doPerfMonitoring) {
            // record tick2 : after response message has been received, before message processing
            requestData.startReceivingTick = this.#messageBuilder._tick1;
        }
        requestData.bytesRead = this.#messageBuilder.totalMessageSize;
        /* istanbul ignore next */
        if (utils_1.doPerfMonitoring) {
            // record tick3 : after response message has been received, before message processing
            requestData.endReceivingTick = (0, node_opcua_utils_1.get_clock_tick)();
        }
        process_request_callback(requestData, null, response);
        if (utils_1.doPerfMonitoring) {
            // record tick4 after callback
            requestData.afterProcessingTick = (0, node_opcua_utils_1.get_clock_tick)();
        } // store some statistics
        this.#_record_transaction_statistics(requestData);
        // notify that transaction is completed
        this.#_on_transaction_completed(this.last_transaction_stats);
    }
    #_record_transaction_statistics(requestData) {
        const request = requestData.request;
        const response = requestData.response;
        // ---------------------------------------------------------------------------------------------------------|-
        //      _tick0                _tick1                         _tick2                       _tick3          _tick4
        //          sending request
        //        |---------------------|  waiting response
        //                              |------------------------------|      receiving response
        //                                                             |---------------------------| process.resp
        //                                                                                  |---------------|
        this.last_transaction_stats = {
            bytesRead: requestData.bytesRead,
            bytesWritten: requestData.bytesWritten_after - requestData.bytesWritten_before,
            lap_processing_response: requestData.afterProcessingTick - requestData.endReceivingTick,
            lap_receiving_response: requestData.endReceivingTick - requestData.startReceivingTick,
            lap_sending_request: requestData.afterSendTick - requestData.beforeSendTick,
            lap_transaction: requestData.afterProcessingTick - requestData.beforeSendTick,
            lap_waiting_response: requestData.startReceivingTick - requestData.afterSendTick,
            request,
            response
        };
        if (utils_1.doTraceStatistics) {
            (0, utils_1._dump_client_transaction_statistics)(this.last_transaction_stats);
        }
    }
    #_cancel_pending_transactions(err) {
        if (doDebug && this.#_requests) {
            debugLog("_cancel_pending_transactions  ", Object.keys(this.#_requests), this.#_transport ? this.#_transport.name : "no transport");
        }
        if (this.#_requests) {
            for (const requestData of Object.values(this.#_requests)) {
                if (requestData) {
                    const request = requestData.request;
                    doDebug &&
                        debugLog("Cancelling pending transaction ", requestData.key, requestData.msgType, request?.schema.name);
                    process_request_callback(requestData, err);
                }
            }
        }
        this.#_requests = {};
    }
    #_on_transport_closed(err) {
        doDebug && debugLog(" =>ClientSecureChannelLayer#_on_transport_closed  err=", err ? err.message : "null");
        if (this.#__in_normal_close_operation) {
            err = undefined;
        }
        if (utils_1.doTraceClientMessage) {
            (0, utils_1.traceClientConnectionClosed)(err, this.channelId, this.#_counter);
        }
        this.emit("close", err);
        this.#_dispose_transports();
        this.#_cancel_pending_transactions(err);
        this.#_cancel_security_token_watchdog();
        this.dispose();
    }
    #_on_security_token_about_to_expire(securityToken) {
        /* istanbul ignore next */
        doDebug &&
            debugLog(" client: Security Token ", securityToken.tokenId, " is about to expired, let's raise lifetime_75 event ");
        this.emit("lifetime_75", securityToken);
        this.#_renew_security_token();
    }
    #_cancel_security_token_watchdog() {
        if (this.#_securityTokenTimeoutId) {
            clearTimeout(this.#_securityTokenTimeoutId);
            this.#_securityTokenTimeoutId = null;
        }
    }
    #_install_security_token_watchdog(securityToken) {
        // install timer event to raise a 'lifetime_75' when security token is about to expired
        // so that client can request for a new security token
        // note that, for speedup in test,
        // it is possible to tweak this interval for test by specifying a tokenRenewalInterval value
        //
        let lifeTime = securityToken.revisedLifetime;
        lifeTime = Math.max(lifeTime, 20);
        const percent = 75 / 100.0;
        let timeout = this.#tokenRenewalInterval || lifeTime * percent;
        timeout = Math.min(timeout, (lifeTime * 75) / 100);
        timeout = Math.max(timeout, 50);
        // istanbul ignore next
        if (doDebug) {
            debugLog(chalk_1.default.red.bold(" time until next security token renewal = "), timeout, "( lifetime = ", lifeTime + " -  tokenRenewalInterval =" + this.#tokenRenewalInterval);
        }
        (0, node_opcua_assert_1.assert)(this.#_securityTokenTimeoutId === null);
        // security token renewal should happen without overlapping
        this.#_securityTokenTimeoutId = setTimeout(() => {
            this.#_securityTokenTimeoutId = null;
            this.#_on_security_token_about_to_expire(securityToken);
        }, timeout);
    }
    #_build_client_nonce() {
        if (this.securityMode === node_opcua_service_secure_channel_1.MessageSecurityMode.None) {
            return undefined;
        }
        // create a client Nonce if secure mode is requested
        // Release 1.02 page 23 OPC Unified Architecture, Part 4 Table 7 – OpenSecureChannel Service Parameters
        // clientNonce
        // "This parameter shall have a length equal to key size used for the symmetric
        //  encryption algorithm that is identified by the securityPolicyUri"
        const cryptoFactory = (0, security_policy_1.getCryptoFactory)(this.securityPolicy);
        /* istanbul ignore next */
        if (!cryptoFactory) {
            // this securityPolicy may not be support yet ... let's return null
            return undefined;
        }
        return (0, crypto_1.randomBytes)(cryptoFactory.symmetricKeyLength);
    }
    #_send_open_secure_channel_request(isInitial, callback) {
        // Verify that we have a valid and known Security policy
        if (this.securityPolicy !== security_policy_1.SecurityPolicy.None) {
            const cryptoFactory = (0, security_policy_1.getCryptoFactory)(this.securityPolicy);
            if (!cryptoFactory) {
                return callback(new Error(`_open_secure_channel_request :  invalid securityPolicy : ${this.securityPolicy} `));
            }
        }
        (0, node_opcua_assert_1.assert)(this.securityMode !== node_opcua_service_secure_channel_1.MessageSecurityMode.Invalid, "invalid security mode");
        // from the specs:
        // The OpenSecureChannel Messages are not signed or encrypted if the SecurityMode is None. The
        // nonces  are ignored and should be set to null. The SecureChannelId and the TokenId are still
        // assigned but no security is applied to Messages exchanged via the channel.
        const msgType = "OPN";
        const requestType = isInitial ? services_1.SecurityTokenRequestType.Issue : services_1.SecurityTokenRequestType.Renew;
        const clientNonce = this.#_build_client_nonce();
        this.#_isOpened = !isInitial;
        // OpenSecureChannel
        const msg = new services_1.OpenSecureChannelRequest({
            clientNonce,
            clientProtocolVersion: this.protocolVersion,
            requestHeader: {
                auditEntryId: null
            },
            requestType: requestType,
            requestedLifetime: this.#requestedLifetime,
            securityMode: this.securityMode
        });
        const startDate = new Date();
        this.#_performMessageTransaction(msgType, msg, (err, response) => {
            // istanbul ignore next
            if (response && response.responseHeader && response.responseHeader.serviceResult !== node_opcua_status_code_1.StatusCodes.Good) {
                warningLog("OpenSecureChannelRequest Error: response.responseHeader.serviceResult ", response.constructor.name, response.responseHeader.serviceResult.toString());
                err = new Error(response.responseHeader.serviceResult.toString());
            }
            if (!err && response) {
                const openSecureChannelResponse = response;
                // record channelId for future transactions
                this.channelId = openSecureChannelResponse.securityToken.channelId;
                // istanbul ignore next
                if (openSecureChannelResponse.securityToken.tokenId <= 0 && msgType !== "OPN") {
                    return callback(new Error(`_open_secure_channel_request : response has an  invalid token ${openSecureChannelResponse.securityToken.tokenId} Id or msgType ${msgType} `));
                }
                const securityToken = openSecureChannelResponse.securityToken;
                // Check time
                const endDate = new Date();
                const midDate = new Date((startDate.getTime() + endDate.getTime()) / 2);
                if (securityToken.createdAt) {
                    const delta = securityToken.createdAt.getTime() - midDate.getTime();
                    this.#_timeDrift = delta;
                    if (Math.abs(delta) > _a.maxClockSkew) {
                        warningLog(`[NODE-OPCUA-W33]  client : server token creation date exposes a time discrepancy ${(0, duration_to_string_1.durationToString)(delta)}\n` +
                            "remote server clock doesn't match this computer date !\n" +
                            " please check both server and client clocks are properly set !\n" +
                            ` server time :${chalk_1.default.cyan(securityToken.createdAt?.toISOString())}\n` +
                            ` client time :${chalk_1.default.cyan(midDate.toISOString())}\n` +
                            ` transaction duration = ${(0, duration_to_string_1.absoluteDurationToString)(endDate.getTime() - startDate.getTime())}\n` +
                            ` server URL = ${this.endpointUrl} \n` +
                            ` token.createdAt  has been updated to reflect client time`);
                    }
                }
                securityToken.createdAt = midDate;
                const serverNonce = openSecureChannelResponse.serverNonce;
                let derivedKeys = null;
                if (this.securityMode !== node_opcua_service_secure_channel_1.MessageSecurityMode.None) {
                    // verify that server nonce if provided is at least 32 bytes long
                    /* istanbul ignore next */
                    if (!openSecureChannelResponse.serverNonce) {
                        warningLog(" client : server nonce is missing !");
                        return callback(new Error(" Invalid server nonce"));
                    }
                    // This parameter shall have a length equal to key size used for the symmetric
                    // encryption algorithm that is identified by the securityPolicyUri.
                    /* istanbul ignore next */
                    if (openSecureChannelResponse.serverNonce.length !== clientNonce?.length) {
                        warningLog(" client : server nonce is invalid  (invalid length)!");
                        return callback(new Error(" Invalid server nonce length"));
                    }
                    const cryptoFactory = (0, security_policy_1.getCryptoFactory)(this.#messageBuilder.securityPolicy);
                    derivedKeys = (0, security_policy_1.computeDerivedKeys)(cryptoFactory, serverNonce, clientNonce);
                    // istanbul ignore next
                    if (doDebug) {
                        debugLog("Server has send a new security Token");
                    }
                }
                this.#tokenStack.pushNewToken(securityToken, derivedKeys);
                this.emit("security_token_created", securityToken);
                this.#_install_security_token_watchdog(securityToken);
                this.activeSecurityToken = securityToken;
                this.#_isOpened = true;
            }
            callback(err || undefined);
        });
    }
    /**
     * install message builder and send first OpenSecureChannelRequest
     */
    #_on_connection(transport, callback) {
        (0, node_opcua_assert_1.assert)(this.#_pending_transport === transport);
        this.#_pending_transport = undefined;
        this.#_transport = transport;
        // install message chunker limits:
        this.#messageChunker.maxMessageSize = this.#_transport?.maxMessageSize || 0;
        this.#messageChunker.maxChunkCount = this.#_transport?.maxChunkCount || 0;
        this.#_install_message_builder();
        this.#_transport.on("chunk", (messageChunk) => {
            this.emit("receive_chunk", messageChunk);
            this.#_on_receive_message_chunk(messageChunk);
        });
        this.#_transport.on("close", (err) => this.#_on_transport_closed(err));
        this.#_transport.on("connection_break", () => {
            doDebug && debugLog(chalk_1.default.red("Client => CONNECTION BREAK  <="));
            this.#_on_transport_closed(new Error("Connection Break"));
        });
        setImmediate(() => {
            doDebug && debugLog(chalk_1.default.red("Client now sending OpenSecureChannel"));
            const isInitial = true;
            this.#_send_open_secure_channel_request(isInitial, callback);
        });
    }
    #_backoff_completion(err, lastError, transport, callback) {
        // Node 20.11.1 on windows now reports a AggregateError when a connection is refused
        // this is a workaround to fix the error message, that is empty when the error is
        // an AggregateError
        const fixError = (err) => {
            if (!err)
                return err;
            if (err.errors) {
                const _err = err;
                err.message = _err.errors.map((e) => e.message).join("\n");
            }
            return err;
        };
        if (this.__call) {
            transport.numberOfRetry = transport.numberOfRetry || 0;
            transport.numberOfRetry += this.__call.getNumRetries();
            this.__call.removeAllListeners();
            this.__call = null;
            if (err) {
                const err_ = fixError(lastError || err);
                callback(err_);
            }
            else {
                callback();
            }
        }
    }
    #_connect(transport, endpointUrl, _i_callback) {
        const on_connect = (err) => {
            doDebug && debugLog("Connection => err", err ? err.message : "null");
            // force Backoff to fail if err is not ECONNRESET or ECONNREFUSED
            // this mean that the connection to the server has succeeded but for some reason
            // the server has denied the connection
            // the cause could be:
            //   - invalid protocol version specified by client
            //   - server going to shutdown
            //   - server too busy -
            //   - server shielding itself from a DDOS attack
            if (err) {
                let should_abort = this.#_isDisconnecting;
                if (err.message.match(/ECONNRESET/)) {
                    // this situation could arise when the socket try to connect and timeouts...  should_abort = false;
                }
                if (err.message.match(/BadProtocolVersionUnsupported/)) {
                    should_abort = true;
                }
                if (err.message.match(/BadTcpInternalError/)) {
                    should_abort = true;
                }
                if (err.message.match(/BadTcpMessageTooLarge/)) {
                    should_abort = true;
                }
                if (err.message.match(/BadTcpEndpointUriInvalid/)) {
                    should_abort = true;
                }
                if (err.message.match(/BadTcpMessageTypeInvalid/)) {
                    should_abort = true;
                }
                this.#lastError = err;
                if (this.__call) {
                    // connection cannot be establish ? if not, abort the backoff process
                    if (should_abort) {
                        doDebug && debugLog(" Aborting backoff process prematurely - err = ", err.message);
                        this.__call.abort();
                    }
                    else {
                        doDebug && debugLog(" backoff - keep trying - err = ", err.message);
                    }
                }
            }
            _i_callback(err);
        };
        transport.connect(endpointUrl, on_connect);
    }
    #_establish_connection(transport, endpointUrl, callback) {
        transport.protocolVersion = this.protocolVersion;
        this.#lastError = undefined;
        if (this.#connectionStrategy.maxRetry === 0) {
            doDebug && debugLog(chalk_1.default.cyan("max Retry === 0 =>  No backoff required -> call the _connect function directly"));
            this.__call = 0;
            return this.#_connect(transport, endpointUrl, callback);
        }
        const connectFunc = (callback2) => {
            return this.#_connect(transport, endpointUrl, callback2);
        };
        const completionFunc = (err) => {
            return this.#_backoff_completion(err, this.#lastError, transport, callback);
        };
        this.__call = backoff.call(connectFunc, completionFunc);
        if (this.#connectionStrategy.maxRetry >= 0) {
            const maxRetry = Math.max(this.#connectionStrategy.maxRetry, 1);
            doDebug && debugLog(chalk_1.default.cyan("backoff will failed after "), maxRetry);
            this.__call.failAfter(maxRetry);
        }
        else {
            // retry will be infinite
            doDebug && debugLog(chalk_1.default.cyan("backoff => starting a infinite retry"));
        }
        const onBackoffFunc = (retryCount, delay) => {
            doDebug &&
                debugLog(chalk_1.default.bgWhite.cyan(" Backoff #"), retryCount, "delay = ", delay, " ms", " maxRetry ", this.#connectionStrategy.maxRetry);
            // Do something when backoff starts, e.g. show to the
            // user the delay before next reconnection attempt.
            this.emit("backoff", retryCount, delay);
        };
        this.__call.on("backoff", onBackoffFunc);
        this.__call.on("abort", () => {
            doDebug && debugLog(chalk_1.default.bgWhite.cyan(` abort #   after ${this.__call.getNumRetries()} retries.`));
            // Do something when backoff starts, e.g. show to the
            // user the delay before next reconnection attempt.
            this.emit("abort");
            setImmediate(() => {
                this.#_backoff_completion(undefined, new Error("Connection abandoned"), transport, callback);
            });
        });
        this.__call.setStrategy(new backoff.ExponentialStrategy(this.#connectionStrategy));
        this.__call.start();
    }
    /**
     * @private internal function
     */
    beforeSecurityRenew = async () => { };
    #_renew_security_token() {
        this.beforeSecurityRenew()
            .then(() => {
            // istanbul ignore next
            doDebug && debugLog("ClientSecureChannelLayer#_renew_security_token");
            // istanbul ignore next
            if (!this.isValid()) {
                // this may happen if the communication has been closed by the client or the sever
                warningLog("Invalid socket => Communication has been lost, cannot renew token");
                return;
            }
            const isInitial = false;
            this.#_send_open_secure_channel_request(isInitial, (err) => {
                /* istanbul ignore else */
                if (!err) {
                    doDebug && debugLog(" token renewed");
                    this.emit("security_token_renewed", this.activeSecurityToken);
                }
                else {
                    errorLog("ClientSecureChannelLayer: Warning: securityToken hasn't been renewed -> err ", err.message);
                }
            });
        })
            .catch((err) => {
            errorLog("ClientSecureChannelLayer#beforeSecurityRenew error", err);
        });
    }
    #_on_receive_message_chunk(messageChunk) {
        /* istanbul ignore next */
        if (doDebug1) {
            const _stream = new node_opcua_binary_stream_1.BinaryStream(messageChunk);
            const messageHeader = (0, node_opcua_chunkmanager_1.readMessageHeader)(_stream);
            debugLog("CLIENT RECEIVED " + chalk_1.default.yellow(JSON.stringify(messageHeader) + ""));
            debugLog("\n" + (0, node_opcua_debug_1.hexDump)(messageChunk));
            debugLog((0, message_header_to_string_1.messageHeaderToString)(messageChunk));
        }
        this.#messageBuilder.feed(messageChunk);
    }
    /**
     * @return  newly generated request id
     */
    #makeRequestId() {
        this.#_lastRequestId += 1;
        return this.#_lastRequestId;
    }
    #undoRequestId() {
        this.#_lastRequestId -= 1;
        return this.#_lastRequestId;
    }
    /**
     * internal version of _performMessageTransaction.
     *
     * - this method takes a extra parameter : msgType
     * TODO:
     * - this method can be re-entrant, meaning that a new transaction can be started before any pending transaction
     *   is fully completed.
     * - Any error on transport will cause all pending transactions to be cancelled
     *
     * - the method returns a timeout Error if the server fails to return a response within the timeoutHint interval.
     *
     *
     */
    #_make_timeout_callback(request, callback, timeout) {
        let localCallback = callback;
        const optionalTrace = !checkTimeout || new Error().stack;
        let hasTimedOut = false;
        let timerId = setTimeout(() => {
            timerId = null;
            hasTimedOut = true;
            if (checkTimeout) {
                warningLog(" Timeout .... waiting for response for ", request.constructor.name, request.requestHeader.toString());
                warningLog(" Timeout was ", timeout, "ms");
                warningLog(request.toString());
                warningLog(optionalTrace);
            }
            modified_callback(new Error("Transaction has timed out ( timeout = " + timeout + " ms , request = " + request.constructor.name + ")"));
            this.#_timeout_request_count += 1;
            this.emit("timed_out_request", request);
        }, timeout);
        const modified_callback = (err, response) => {
            /* istanbul ignore next */
            if (doDebug) {
                debugLog(chalk_1.default.cyan("------------------------------------- Client receiving response "), request.constructor.name, request.requestHeader.requestHandle, response ? response.constructor.name : "null", "err=", err ? err.message : "null", "securityTokenId=", this.activeSecurityToken?.tokenId);
            }
            if (response && utils_1.doTraceClientResponseContent) {
                (0, utils_1.traceClientResponseContent)(response, this.channelId);
            }
            if (!localCallback) {
                return; // already processed by time  out
            }
            // when response === null we are processing the timeout , therefore there is no need to clearTimeout
            if (!hasTimedOut && timerId) {
                clearTimeout(timerId);
            }
            timerId = null;
            if (!err && response) {
                this.emit("receive_response", response);
            }
            (0, node_opcua_assert_1.assert)(!err || util_1.types.isNativeError(err));
            delete this.#_requests[request.requestHeader.requestHandle];
            // invoke user callback if it has not been intercepted first ( by a abrupt disconnection for instance )
            try {
                localCallback.call(this, err || null, response);
            }
            catch (err1) {
                errorLog("ERROR !!! callback has thrown en error ", err1);
            }
            finally {
                localCallback = null;
            }
        };
        return modified_callback;
    }
    #_adjustRequestTimeout(request) {
        let timeout = request.requestHeader.timeoutHint ||
            this.defaultTransactionTimeout ||
            _a.defaultTransactionTimeout;
        timeout = Math.max(_a.minTransactionTimeout, timeout);
        /* istanbul ignore next */
        if (request.requestHeader.timeoutHint != timeout) {
            debugLog("Adjusted timeout = ", request.requestHeader.timeoutHint);
        }
        request.requestHeader.timeoutHint = timeout;
        return timeout;
    }
    #_performMessageTransaction(msgType, request, callback) {
        this.emit("beforePerformTransaction", msgType, request);
        /* istanbul ignore next */
        if (!this.isValid()) {
            return callback(new Error("ClientSecureChannelLayer => Socket is closed ! while processing " + request.constructor.name));
        }
        const timeout = this.#_adjustRequestTimeout(request);
        const modifiedCallback = this.#_make_timeout_callback(request, callback, timeout);
        const transactionData = {
            callback: modifiedCallback,
            msgType: msgType,
            request: request
        };
        this.#_internal_perform_transaction(transactionData);
    }
    /**
     *
     * @param transactionData
     * @param transactionData.msgType
     * @param transactionData.request
     * @param transactionData.callback
     * @private
     */
    #_internal_perform_transaction(transactionData) {
        if (!this.#_transport) {
            setTimeout(() => {
                transactionData.callback(new Error("Client not connected"));
            }, 100);
            return;
        }
        const msgType = transactionData.msgType;
        const request = transactionData.request;
        /* istanbul ignore next */
        if (request.requestHeader.requestHandle !== exports.requestHandleNotSetValue) {
            errorLog(chalk_1.default.bgRed.white("xxxxx   >>>>>> request has already been set with a requestHandle"), request.requestHeader.requestHandle, request.constructor.name);
            errorLog(Object.keys(this.#_requests).join(" "));
            errorLog(new Error("Investigate me"));
        }
        // get a new requestId
        request.requestHeader.requestHandle = this.#makeRequestId();
        /* istanbul ignore next */
        if (utils_1.doTraceClientMessage) {
            (0, utils_1.traceClientRequestMessage)(request, this.channelId, this.#_counter);
        }
        const requestData = {
            callback: transactionData.callback,
            msgType: msgType,
            request: request,
            bytesRead: 0,
            bytesWritten_after: 0,
            bytesWritten_before: this.bytesWritten,
            beforeSendTick: 0,
            afterSendTick: 0,
            startReceivingTick: 0,
            endReceivingTick: 0,
            afterProcessingTick: 0,
            key: "",
            chunk_count: 0
        };
        this.#_requests[request.requestHeader.requestHandle] = requestData;
        /* istanbul ignore next */
        if (utils_1.doPerfMonitoring) {
            const stats = requestData;
            // record tick0 : before request is being sent to server
            stats.beforeSendTick = (0, node_opcua_utils_1.get_clock_tick)();
        }
        // check that limits are OK
        this.#_sendSecureOpcUARequest(msgType, request);
    }
    #_send_chunk(requestId, chunk) {
        const requestData = this.#_requests[requestId];
        if (chunk) {
            this.emit("send_chunk", chunk);
            /* istanbul ignore next */
            if (checkChunks) {
                (0, node_opcua_chunkmanager_1.verify_message_chunk)(chunk);
                debugLog(chalk_1.default.yellow("CLIENT SEND chunk "));
                debugLog(chalk_1.default.yellow((0, message_header_to_string_1.messageHeaderToString)(chunk)));
                debugLog(chalk_1.default.red((0, node_opcua_debug_1.hexDump)(chunk)));
            }
            (0, node_opcua_assert_1.assert)(this.#_transport);
            this.#_transport?.write(chunk);
            requestData.chunk_count += 1;
        }
        else {
            // last chunk ....
            /* istanbul ignore next */
            if (checkChunks) {
                debugLog(chalk_1.default.yellow("CLIENT SEND done."));
            }
            if (requestData) {
                if (utils_1.doPerfMonitoring) {
                    requestData.afterSendTick = (0, node_opcua_utils_1.get_clock_tick)();
                }
                requestData.bytesWritten_after = this.bytesWritten;
                if (requestData.msgType === "CLO") {
                    setTimeout(() => {
                        // We sdo not expect any response from the server for a CLO message
                        if (requestData.callback) {
                            // if server do not initiates the disconnection, we may need to call the callback
                            // from here
                            requestData.callback(null, undefined);
                            requestData.callback = undefined;
                        }
                    }, 100);
                }
            }
        }
    }
    #_construct_asymmetric_security_header() {
        const calculateMaxSenderCertificateSize = () => {
            /**
             * The SenderCertificate, including any chains, shall be small enough to fit
             * into a single MessageChunk and leave room for at least one byte of body
             * information.The maximum size for the SenderCertificate can be calculated
             * with this formula:
             */
            const cryptoFactory = (0, security_policy_1.getCryptoFactory)(this.securityPolicy);
            if (!cryptoFactory) {
                // we have a unknown security policy
                // let's assume that maxSenderCertificateSize is not an issue
                return 1 * 8192;
            }
            const { signatureLength, blockPaddingSize } = cryptoFactory;
            const securityPolicyUriLength = this.securityPolicy.length;
            const messageChunkSize = this.#_transport?.parameters?.sendBufferSize || 0;
            const padding = blockPaddingSize;
            const extraPadding = 0; // ???
            const asymmetricSignatureSize = signatureLength;
            const maxSenderCertificateSize = messageChunkSize -
                12 - // Header size
                4 - // SecurityPolicyUriLength
                securityPolicyUriLength - // UTF-8 encoded string
                4 - // SenderCertificateLength
                4 - // ReceiverCertificateThumbprintLength
                20 - // ReceiverCertificateThumbprint
                8 - // SequenceHeader size
                1 - // Minimum body size
                1 - // PaddingSize if present
                padding - // Padding if present
                extraPadding - // ExtraPadding if present
                asymmetricSignatureSize; // If present
            return maxSenderCertificateSize;
        };
        switch (this.securityMode) {
            case node_opcua_service_secure_channel_1.MessageSecurityMode.None:
                {
                    return new services_1.AsymmetricAlgorithmSecurityHeader({
                        securityPolicyUri: (0, security_policy_1.toURI)(this.securityPolicy),
                        receiverCertificateThumbprint: null,
                        senderCertificate: null
                    });
                }
                break;
            case node_opcua_service_secure_channel_1.MessageSecurityMode.Sign:
            case node_opcua_service_secure_channel_1.MessageSecurityMode.SignAndEncrypt: {
                // get a partial portion of the client certificate chain that matches the maxSenderCertificateSize
                const maxSenderCertificateSize = calculateMaxSenderCertificateSize();
                const partialCertificateChain = (0, node_opcua_common_1.getPartialCertificateChain)(this.getCertificateChain(), maxSenderCertificateSize);
                // get the thumbprint of the  receiverCertificate certificate
                const evaluateReceiverThumbprint = () => {
                    if (this.securityMode === node_opcua_service_secure_channel_1.MessageSecurityMode.None) {
                        return null;
                    }
                    const chain = (0, web_1.split_der)(this.#serverCertificate);
                    (0, node_opcua_assert_1.assert)(chain.length === 1);
                    const receiverCertificateThumbprint = (0, common_1.getThumbprint)(this.#serverCertificate);
                    doDebug && debugLog("XXXXXXserver certificate thumbprint = ", receiverCertificateThumbprint.toString("hex"));
                    return receiverCertificateThumbprint;
                };
                const receiverCertificateThumbprint = evaluateReceiverThumbprint();
                if (this.securityPolicy === security_policy_1.SecurityPolicy.Invalid) {
                    warningLog("SecurityPolicy is invalid", this.securityPolicy.toString());
                }
                const securityHeader = new services_1.AsymmetricAlgorithmSecurityHeader({
                    securityPolicyUri: (0, security_policy_1.toURI)(this.securityPolicy),
                    /**
                     * The thumbprint of the X.509 v3 Certificate assigned to the receiving application Instance.
                     * The thumbprint is the CertificateDigest of the DER encoded form of the Certificate.
                     * This indicates what public key was used to encrypt the MessageChunk.
                     * This field shall be null if the Message is not encrypted.
                     */
                    receiverCertificateThumbprint,
                    /**
                     * The X.509 v3 Certificate assigned to the sending application Instance.
                     * This is a DER encoded blob.
                     * The structure of an X.509 v3 Certificate is defined in X.509 v3.
                     * The DER format for a Certificate is defined in X690
                     * This indicates what Private Key was used to sign the MessageChunk.
                     * The Stack shall close the channel and report an error to the application
                     * if the SenderCertificate is too large for the buffer size supported by the transport layer.
                     * This field shall be null if the Message is not signed.
                     * If the Certificate is signed by a CA, the DER encoded CA Certificate may be
                     * appended after the Certificate in the byte array. If the CA Certificate is also
                     * signed by another CA this process is repeated until the entire Certificate chain
                     * is in the buffer or if MaxSenderCertificateSize limit is reached (the process
                     * stops after the last whole Certificate that can be added without exceeding
                     * the MaxSenderCertificateSize limit).
                     * Receivers can extract the Certificates from the byte array by using the Certificate
                     * size contained in DER header (see X.509 v3).
                     */
                    senderCertificate: partialCertificateChain // certificate of the private key used to sign the message
                });
                /* istanbul ignore next */
                if (utils_1.dumpSecurityHeader) {
                    warningLog("HEADER !!!! ", securityHeader.toString());
                }
                return securityHeader;
                break;
            }
            default:
                /* istanbul ignore next */
                throw new Error("invalid security mode");
        }
    }
    #_get_security_options_for_OPN() {
        // The OpenSecureChannel Messages are signed and encrypted if the SecurityMode is
        // not None(even  if the SecurityMode is Sign).
        if (this.securityMode === node_opcua_service_secure_channel_1.MessageSecurityMode.None) {
            return null;
        }
        const senderPrivateKey = this.getPrivateKey();
        /* istanbul ignore next */
        if (!senderPrivateKey) {
            throw new Error("invalid or missing senderPrivateKey : necessary to sign");
        }
        const cryptoFactory = (0, security_policy_1.getCryptoFactory)(this.securityPolicy);
        /* istanbul ignore next */
        if (!cryptoFactory) {
            throw new Error("Internal Error: ServerSecureChannelLayer must have a crypto strategy");
        }
        /* istanbul ignore next */
        if (!this.#receiverPublicKey) {
            throw new Error("Internal error: invalid receiverPublicKey");
        }
        const receiverPublicKey = this.#receiverPublicKey;
        const keyLength = (0, web_1.rsaLengthPublicKey)(receiverPublicKey);
        const signatureLength = (0, web_1.rsaLengthPrivateKey)(senderPrivateKey);
        const options = {
            // for signing
            signatureLength,
            signBufferFunc: (chunk) => cryptoFactory.asymmetricSign(chunk, senderPrivateKey),
            // for encrypting
            cipherBlockSize: keyLength,
            plainBlockSize: keyLength - cryptoFactory.blockPaddingSize,
            encryptBufferFunc: (chunk) => cryptoFactory.asymmetricEncrypt(chunk, receiverPublicKey)
        };
        return options;
    }
    #_get_security_options_for_MSG(tokenId) {
        if (this.securityMode === node_opcua_service_secure_channel_1.MessageSecurityMode.None) {
            return null;
        }
        const derivedClientKeys = this.#tokenStack.clientKeyProvider().getDerivedKey(tokenId);
        // istanbul ignore next
        if (!derivedClientKeys) {
            errorLog("derivedKeys not set but security mode = ", node_opcua_service_secure_channel_1.MessageSecurityMode[this.securityMode]);
            return null;
        }
        const options = (0, security_policy_1.getOptionsForSymmetricSignAndEncrypt)(this.securityMode, derivedClientKeys);
        return options;
    }
    #_get_security_options(msgType) {
        if (msgType == "OPN") {
            const securityHeader = this.#_construct_asymmetric_security_header();
            const securityOptions = this.#_get_security_options_for_OPN();
            return {
                securityHeader,
                securityOptions
            };
        }
        else {
            const securityToken = this.activeSecurityToken;
            const tokenId = securityToken ? securityToken.tokenId : 0;
            const securityHeader = new node_opcua_service_secure_channel_1.SymmetricAlgorithmSecurityHeader({ tokenId });
            const securityOptions = this.#_get_security_options_for_MSG(tokenId);
            return {
                securityHeader,
                securityOptions
            };
        }
    }
    #_sendSecureOpcUARequest(msgType, request) {
        const evaluateChunkSize = () => {
            // use chunk size that has been negotiated by the transport layer
            if (this.#_transport?.parameters && this.#_transport?.parameters.sendBufferSize) {
                return this.#_transport.parameters.sendBufferSize;
            }
            return 0;
        };
        const { securityOptions, securityHeader } = this.#_get_security_options(msgType);
        const requestId = request.requestHeader.requestHandle;
        const chunkSize = evaluateChunkSize();
        let options = {
            channelId: this.channelId,
            securityOptions: {
                channelId: this.channelId,
                requestId,
                chunkSize,
                cipherBlockSize: 0,
                plainBlockSize: 0,
                sequenceHeaderSize: 0,
                signatureLength: 0,
                ...securityOptions
            },
            securityHeader: securityHeader
        };
        /* istanbul ignore next */
        if (utils_1.doTraceClientRequestContent) {
            (0, utils_1.traceClientRequestContent)(request, this.channelId, this.activeSecurityToken);
        }
        this.emit("send_request", request, msgType, securityHeader);
        const statusCode = this.#messageChunker.chunkSecureMessage(msgType, options, request, (chunk) => this.#_send_chunk(requestId, chunk));
        if (statusCode.isNotGood()) {
            // chunkSecureMessage has refused to send the message
            const response = new services_1.ServiceFault({
                responseHeader: {
                    requestHandle: requestId,
                    serviceResult: statusCode,
                    stringTable: [statusCode.toString()]
                }
            });
            this.#_on_message_received(response, "ERR", securityHeader, request.requestHeader.requestHandle);
        }
    }
}
exports.ClientSecureChannelLayer = ClientSecureChannelLayer;
_a = ClientSecureChannelLayer;
//# sourceMappingURL=client_secure_channel_layer.js.map