"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findInTypeOrSuperType = findInTypeOrSuperType;
exports.findMethodId = findMethodId;
const node_opcua_service_translate_browse_path_1 = require("node-opcua-service-translate-browse-path");
const node_opcua_data_model_1 = require("node-opcua-data-model");
async function findInTypeOrSuperType(session, browsePath) {
    const nodeId = browsePath.startingNode;
    const result = await session.translateBrowsePath(browsePath);
    if (result.statusCode.isGood()) {
        return { nodeId: result.targets[0].targetId };
    }
    // cannot be found here, go one step up
    const br = await session.browse({
        nodeId,
        referenceTypeId: "HasSubtype",
        browseDirection: node_opcua_data_model_1.BrowseDirection.Inverse,
        includeSubtypes: true,
        nodeClassMask: 0,
        resultMask: 0
    });
    if (br.statusCode.isNotGood()) {
        // cannot find typeDefinition
        return { nodeId: null, err: new Error("cannot find typeDefinition") };
    }
    const typeDefinition = br.references[0].nodeId;
    browsePath = new node_opcua_service_translate_browse_path_1.BrowsePath({
        startingNode: typeDefinition,
        relativePath: browsePath.relativePath
    });
    return await findInTypeOrSuperType(session, browsePath);
}
/**
 *
 * find a MethodId in a object or in its super type
 *
 * note:
 *   - methodName is a browse name and may therefore be prefixed with a namespace index.
 *   - if method is not found on the object specified by nodeId, then the findMethodId will
 *     recursively browse up the hierarchy of object typeDefinition Node
 *     until it reaches the root type. and try to find the first method that matches the
 *     provided name.
 *
 * @param session
 * @param nodeId     the nodeId of the object to find
 * @param methodName the method name to find prefixed with a namespace index (unless ns=0)
 *                   ( e.g "Add" or "Add" or "1:BumpCounter" )
 */
async function findMethodId(session, nodeId, methodName) {
    const browsePath = (0, node_opcua_service_translate_browse_path_1.makeBrowsePath)(nodeId, "/" + (0, node_opcua_data_model_1.coerceQualifiedName)(methodName).toString());
    const result = await session.translateBrowsePath(browsePath);
    if (result.statusCode.isNotGood()) {
        const br = await session.browse({
            nodeId,
            referenceTypeId: "HasTypeDefinition",
            browseDirection: node_opcua_data_model_1.BrowseDirection.Forward,
            includeSubtypes: true,
            nodeClassMask: 0,
            resultMask: 0
        });
        if (br.statusCode.isNotGood()) {
            // cannot find typeDefinition
            return { methodId: null, err: new Error("cannot find typeDefinition") };
        }
        const typeDefinition = br.references[0].nodeId;
        // need to find method on objectType
        const browsePath = (0, node_opcua_service_translate_browse_path_1.makeBrowsePath)(typeDefinition, "/" + methodName);
        const result = await findInTypeOrSuperType(session, browsePath);
        if (!result.nodeId) {
            return { err: result.err, methodId: null };
        }
        return { methodId: result.nodeId };
    }
    result.targets = result.targets || [];
    // istanbul ignore else
    if (result.targets.length > 0) {
        const methodId = result.targets[0].targetId;
        return { methodId };
    }
    else {
        // cannot find objectWithMethodNodeId
        const err = new Error(" cannot find " + methodName + " Method");
        return { methodId: null, err };
    }
}
//# sourceMappingURL=find_method_id.js.map