"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractFields = extractFields;
exports.simpleBrowsePathToString = simpleBrowsePathToString;
exports.simpleBrowsePathsToString = simpleBrowsePathsToString;
exports.stringPathToSimpleBrowsePath = stringPathToSimpleBrowsePath;
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_types_1 = require("node-opcua-types");
const node_opcua_debug_1 = require("node-opcua-debug");
const doDebug = false;
const debugLog = (0, node_opcua_debug_1.make_debugLog)(__filename);
/**
 *
 * recursively work down an node definition and find
 * the components and property ...
 * also navigate the sub
 * @param session the session
 * @param nodeId the object to investigate , could be the nodeId of a Object/Variable/ObjectType or VariableType.
 * @returns a array of {path: QualifiedName[], nodeId: NodeId}}
 *
 * @private
 */
async function extractFields(session, nodeId) {
    const _duplicateMap = {};
    const fields1 = [];
    function addField(parent, browseName, nodeId) {
        const e = [...parent, browseName];
        const key = simpleBrowsePathToString(e);
        // istanbul ignore next
        doDebug && debugLog("adding field ", key);
        if (!_duplicateMap[key]) {
            fields1.push({ path: e, nodeId });
            _duplicateMap[key] = e;
        }
    }
    const stack = [];
    function _pushInvestigation(parent, objectId) {
        stack.push({
            parent,
            nodeId: objectId
        });
    }
    async function _flushPendingInvestigations() {
        if (stack.length === 0) {
            return;
        }
        const extracted = stack.splice(0);
        const nodesToBrowse = extracted.map((e) => {
            const { parent, nodeId } = e;
            const b = {
                browseDirection: node_opcua_data_model_1.BrowseDirection.Forward,
                includeSubtypes: true,
                nodeClassMask: node_opcua_data_model_1.NodeClassMask.Object | node_opcua_data_model_1.NodeClassMask.Variable,
                nodeId,
                referenceTypeId: "HasChild",
                resultMask: 63
            };
            return b;
        });
        const results = await session.browse(nodesToBrowse);
        for (let index = 0; index < results.length; index++) {
            const result = results[index];
            const parent = extracted[index].parent;
            if (!result.references || result.references.length === 0)
                continue;
            // istanbul ignore next
            doDebug &&
                debugLog("exploring", simpleBrowsePathToString(parent), result.references.map((a) => a.browseName.toString()));
            for (const ref of result.references) {
                if (ref.nodeClass === node_opcua_types_1.NodeClass.Variable) {
                    addField(parent, ref.browseName, ref.nodeId);
                }
                _pushInvestigation([...parent, ref.browseName], ref.nodeId);
            }
        }
        await _flushPendingInvestigations();
    }
    async function _investigateTopLevel(parent, eventNodeId) {
        const browseDescriptionForInverseSubType = {
            browseDirection: node_opcua_data_model_1.BrowseDirection.Inverse,
            includeSubtypes: true,
            nodeClassMask: node_opcua_data_model_1.NodeClassMask.ObjectType,
            nodeId: eventNodeId,
            referenceTypeId: (0, node_opcua_nodeid_1.resolveNodeId)("HasSubtype"),
            resultMask: 63
        };
        const nodeToBrowse2 = {
            browseDirection: node_opcua_data_model_1.BrowseDirection.Forward,
            includeSubtypes: true,
            nodeClassMask: node_opcua_data_model_1.NodeClassMask.Object | node_opcua_data_model_1.NodeClassMask.Variable,
            nodeId: eventNodeId,
            referenceTypeId: (0, node_opcua_nodeid_1.resolveNodeId)("HasChild"),
            resultMask: 63
        };
        const nodesToBrowse = [browseDescriptionForInverseSubType, nodeToBrowse2];
        const browseResults = await session.browse(nodesToBrowse);
        const [browseResultForInverseSubType, browseResultForChildren] = browseResults;
        if (browseResultForChildren && browseResultForChildren.references) {
            for (const ref of browseResultForChildren.references) {
                if (ref.nodeClass === node_opcua_types_1.NodeClass.Variable) {
                    addField(parent, ref.browseName, ref.nodeId);
                }
                _pushInvestigation([...parent, ref.browseName], ref.nodeId);
            }
        }
        await _flushPendingInvestigations();
        if (browseResultForInverseSubType && browseResultForInverseSubType.references) {
            const promises = [];
            for (const reference of browseResultForInverseSubType.references) {
                // istanbul ignore next
                doDebug && debugLog(" investigating super-type", reference.browseName.toString());
                promises.push(_investigateTopLevel([], reference.nodeId));
            }
            await Promise.all(promises);
        }
    }
    // istanbul ignore next
    doDebug &&
        debugLog("investigating ", nodeId.toString(), (await session.read({ nodeId, attributeId: node_opcua_data_model_1.AttributeIds.BrowseName })).value.value.toString());
    await _investigateTopLevel([], nodeId);
    return fields1;
}
function simpleBrowsePathToString(bp) {
    return bp.map((qn) => qn.toString()).join(".");
}
function simpleBrowsePathsToString(simpleBrowsePathArray) {
    return simpleBrowsePathArray.map(simpleBrowsePathToString);
}
function stringPathToSimpleBrowsePath(bp) {
    return bp.split(".").map((s) => (0, node_opcua_data_model_1.stringToQualifiedName)(s));
}
//# sourceMappingURL=extract_fields.js.map