"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getArgumentDefinitionHelper = getArgumentDefinitionHelper;
exports.readNamespaceArray = readNamespaceArray;
exports.clearSessionCache = clearSessionCache;
/**
 * @module node-opcua-pseudo-session
 */
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_service_browse_1 = require("node-opcua-service-browse");
const node_opcua_variant_1 = require("node-opcua-variant");
const node_opcua_constants_1 = require("node-opcua-constants");
function isValid(result) {
    (0, node_opcua_assert_1.assert)(result.statusCode.isGood());
    if (result.value.dataType !== node_opcua_variant_1.DataType.Null) {
        (0, node_opcua_assert_1.assert)(result.value.dataType === node_opcua_variant_1.DataType.ExtensionObject);
        (0, node_opcua_assert_1.assert)(result.value.arrayType === node_opcua_variant_1.VariantArrayType.Array);
    }
    return true;
}
async function getArgumentDefinitionHelper(session, methodId) {
    const browseDescription = new node_opcua_service_browse_1.BrowseDescription({
        browseDirection: node_opcua_data_model_1.BrowseDirection.Forward,
        includeSubtypes: true,
        nodeClassMask: 0, // makeNodeClassMask("Variable"),
        nodeId: methodId,
        referenceTypeId: (0, node_opcua_nodeid_1.resolveNodeId)("HasProperty"),
        resultMask: (0, node_opcua_data_model_1.makeResultMask)("BrowseName")
    });
    const browseResult = await session.browse(browseDescription);
    browseResult.references = browseResult.references || [];
    const inputArgumentRefArray = browseResult.references.filter((r) => r.browseName.name === "InputArguments");
    // note : InputArguments property is optional thus may be missing
    const inputArgumentRef = inputArgumentRefArray.length === 1 ? inputArgumentRefArray[0] : null;
    const outputArgumentRefArray = browseResult.references.filter((r) => r.browseName.name === "OutputArguments");
    // note : OutputArguments property is optional thus may be missing
    const outputArgumentRef = outputArgumentRefArray.length === 1 ? outputArgumentRefArray[0] : null;
    let inputArguments = [];
    let outputArguments = [];
    const nodesToRead = [];
    const actions = [];
    if (inputArgumentRef) {
        nodesToRead.push({
            attributeId: node_opcua_data_model_1.AttributeIds.Value,
            nodeId: inputArgumentRef.nodeId
        });
        actions.push((result) => {
            if (isValid(result)) {
                inputArguments = result.value.value;
            }
        });
    }
    if (outputArgumentRef) {
        nodesToRead.push({
            attributeId: node_opcua_data_model_1.AttributeIds.Value,
            nodeId: outputArgumentRef.nodeId
        });
        actions.push((result) => {
            (0, node_opcua_assert_1.assert)(result.statusCode.isGood());
            if (isValid(result)) {
                outputArguments = result.value.value;
            }
        });
    }
    if (nodesToRead.length === 0) {
        return { inputArguments, outputArguments };
    }
    // now read the variable
    const dataValues = await session.read(nodesToRead);
    dataValues.forEach((dataValue, index) => {
        actions[index].call(null, dataValue);
    });
    return { inputArguments, outputArguments };
}
function followSession(session) {
    if (session.session) {
        return followSession(session.session);
    }
    return session;
}
async function readNamespaceArray(session) {
    const sessionHoldingCache = followSession(session);
    if (sessionHoldingCache.$$namespaceArray) {
        return sessionHoldingCache.$$namespaceArray;
    }
    const nodeId = (0, node_opcua_nodeid_1.resolveNodeId)(node_opcua_constants_1.VariableIds.Server_NamespaceArray);
    const dataValue = await session.read({
        nodeId,
        attributeId: node_opcua_data_model_1.AttributeIds.Value
    });
    if (dataValue.statusCode.isNotGood()) {
        // errorLog("namespaceArray is not populated ! Your server must expose a list of namespaces in node ", nodeId.toString());
        return [];
    }
    sessionHoldingCache.$$namespaceArray = dataValue.value.value; // keep a cache
    return sessionHoldingCache.$$namespaceArray;
}
async function clearSessionCache(session) {
    const sessionHoldingCache = followSession(session);
    sessionHoldingCache.$$namespaceArray = undefined;
}
//# sourceMappingURL=basic_session_interface.js.map