"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.analyzePacket = analyzePacket;
exports.analyseExtensionObject = analyseExtensionObject;
exports.analyze_object_binary_encoding = analyze_object_binary_encoding;
const util_1 = require("util");
const chalk_1 = __importDefault(require("chalk"));
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_basic_types_1 = require("node-opcua-basic-types");
const node_opcua_binary_stream_1 = require("node-opcua-binary-stream");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_factory_1 = require("node-opcua-factory");
const node_opcua_utils_1 = require("node-opcua-utils");
const spaces = "                                                                                                                                                                             ";
function f(n, width) {
    const s = n.toString();
    return (s + "      ").substring(0, Math.max(s.length, width));
}
function display_encoding_mask(padding, encodingMask, encodingInfo) {
    for (const v in encodingInfo) {
        if (!Object.prototype.hasOwnProperty.call(encodingInfo, v)) {
            continue;
        }
        const enumKey = encodingInfo[v];
        if (typeof enumKey === "number") {
            continue;
        }
        const mask = encodingInfo[enumKey];
        const bit = Math.log(mask) / Math.log(2);
        const bits = [".", ".", ".", ".", ".", ".", ".", ".", "."];
        bits[bit] = (encodingMask & mask) === mask ? "Y" : "n";
        console.log(padding + " ", bits.join(""), " <- has " + enumKey + " 0x" + mask.toString(16));
    }
    // DataValueEncodingByte
}
function hex_block(start, end, buffer) {
    const n = end - start;
    const strBuf = (0, node_opcua_utils_1.buffer_ellipsis)(buffer);
    return (chalk_1.default.cyan("s:") + f(start, 4) + chalk_1.default.cyan(" e:") + f(end, 4) + chalk_1.default.cyan(" n:") + f(n, 4) + " " + chalk_1.default.yellow(strBuf));
}
function make_tracer(buffer, padding, offset) {
    padding = !padding ? 0 : padding;
    offset = offset || 0;
    const pad = () => "                                                       ".substring(0, padding);
    function _display(str, hexInfo) {
        hexInfo = hexInfo || "";
        // account for ESC codes for colors
        const nbColorAttributes = [...str.split("")].filter((c) => c === "\u001b").length;
        const extra = nbColorAttributes * 5;
        console.log((pad() + str + spaces).substring(0, 132 + extra) + "|" + hexInfo);
    }
    function display(str, hexInfo) {
        const lines = str.split("\n");
        for (const line of lines) {
            _display(line, hexInfo);
        }
    }
    function display_encodeable(value, buffer1, start, end) {
        const bufferExtract = buffer1.subarray(start, end);
        const stream = new node_opcua_binary_stream_1.BinaryStream(bufferExtract);
        const nodeId = (0, node_opcua_basic_types_1.decodeNodeId)(stream);
        const encodingMask = (0, node_opcua_basic_types_1.decodeByte)(stream); // 1 bin 2: xml
        const length = (0, node_opcua_basic_types_1.decodeUInt32)(stream);
        display(chalk_1.default.green("     ExpandedNodId =") + " " + nodeId);
        display(chalk_1.default.green("     encoding mask =") + " " + encodingMask);
        display(chalk_1.default.green("            length =") + " " + length);
        analyzePacket(bufferExtract.subarray(stream.length), value.encodingDefaultBinary, padding + 2, start + stream.length);
    }
    return {
        tracer: {
            dump: (title, value) => display(title + "  " + chalk_1.default.green(value.toString())),
            encoding_byte: (encodingMask, valueEnum, start, end) => {
                (0, node_opcua_assert_1.assert)(valueEnum);
                const b = buffer.subarray(start, end);
                display("  012345678", hex_block(start, end, b));
                display_encoding_mask(pad(), encodingMask, valueEnum);
            },
            trace: (operation, name, value, start, end, fieldType) => {
                const b = buffer.subarray(start, end);
                let _hexDump = "";
                switch (operation) {
                    case "start":
                        padding += 2;
                        display(name.toString());
                        break;
                    case "end":
                        padding -= 2;
                        break;
                    case "start_array":
                        display("." + name + " (length = " + value + ") " + "[", hex_block(start, end, b));
                        padding += 2;
                        break;
                    case "end_array":
                        padding -= 2;
                        display("] // " + name);
                        break;
                    case "start_element":
                        display(" #" + value + " {");
                        padding += 2;
                        break;
                    case "end_element":
                        padding -= 2;
                        display(" } // # " + value);
                        break;
                    case "member":
                        display("." + name + " : " + fieldType);
                        _hexDump = "";
                        if (value instanceof Buffer) {
                            _hexDump = (0, node_opcua_debug_1.hexDump)(value);
                            console.log(_hexDump);
                            value = "<BUFFER>";
                        }
                        if (value && value.encode) {
                            if (fieldType === "ExtensionObject") {
                                display_encodeable(value, buffer, start, end);
                            }
                            else {
                                const str = value.toString() || "<empty>";
                                display(str);
                            }
                        }
                        else {
                            display(" " + value, hex_block(start, end, b));
                        }
                        break;
                }
            }
        }
    };
}
function analyzePacket(buffer, objMessage, padding, offset, customOptions) {
    const stream = new node_opcua_binary_stream_1.BinaryStream(buffer);
    _internalAnalyzePacket(buffer, stream, objMessage, padding, customOptions, offset);
}
function analyseExtensionObject(buffer, padding, offset, customOptions) {
    const stream = new node_opcua_binary_stream_1.BinaryStream(buffer);
    let id;
    let objMessage;
    try {
        id = (0, node_opcua_basic_types_1.decodeExpandedNodeId)(stream);
        objMessage = (0, node_opcua_factory_1.getStandardDataTypeFactory)().constructObject(id);
    }
    catch (err) {
        console.log(id);
        console.log(err);
        console.log("Cannot read decodeExpandedNodeId  on stream " + stream.buffer.toString("hex"));
    }
    _internalAnalyzePacket(buffer, stream, objMessage, padding, customOptions, offset);
}
function _internalAnalyzePacket(buffer, stream, objMessage, padding, customOptions, offset) {
    let options = make_tracer(buffer, padding, offset);
    options.name = "message";
    if (customOptions)
        options = { ...options, ...customOptions };
    try {
        if (objMessage) {
            objMessage.decodeDebug(stream, options);
        }
        else {
            console.log(" Invalid object", objMessage);
        }
    }
    catch (err) {
        console.log(" Error in ", err);
        if (util_1.types.isNativeError(err)) {
            console.log(" Error in ", err.stack);
        }
        console.log(" objMessage ", (0, util_1.inspect)(objMessage, { colors: true }));
    }
}
function analyze_object_binary_encoding(obj) {
    (0, node_opcua_assert_1.assert)(obj);
    const size = obj.binaryStoreSize();
    console.log("-------------------------------------------------");
    console.log(" size = ", size);
    const stream = new node_opcua_binary_stream_1.BinaryStream(size);
    obj.encode(stream);
    stream.rewind();
    console.log("-------------------------------------------------");
    if (stream.buffer.length < 256) {
        console.log((0, node_opcua_debug_1.hexDump)(stream.buffer));
        console.log("-------------------------------------------------");
    }
    const reloadedObject = new obj.constructor();
    analyzePacket(stream.buffer, reloadedObject, 0);
}
//# sourceMappingURL=packet_analyzer.js.map