"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.StructuredTypeSchema = void 0;
exports.extractAllPossibleFields = extractAllPossibleFields;
exports.check_options_correctness_against_schema = check_options_correctness_against_schema;
exports.buildStructuredType = buildStructuredType;
/**
 * @module node-opcua-factory
 */
const chalk_1 = __importDefault(require("chalk"));
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_utils_1 = require("node-opcua-utils");
const builtin_types_1 = require("./builtin_types");
const parameters_1 = require("./parameters");
const get_standard_data_type_factory_1 = require("./get_standard_data_type_factory");
const types_1 = require("./types");
const warningLog = (0, node_opcua_debug_1.make_warningLog)(__filename);
const errorLog = (0, node_opcua_debug_1.make_errorLog)(__filename);
function figureOutFieldCategory(field, dataTypeFactory) {
    const fieldType = field.fieldType;
    if (field.category) {
        return field.category;
    }
    if (dataTypeFactory.hasEnumeration(fieldType)) {
        return types_1.FieldCategory.enumeration;
    }
    else if (dataTypeFactory.hasBuiltInType(fieldType)) {
        return types_1.FieldCategory.basic;
    }
    else if (dataTypeFactory.hasStructureByTypeName(fieldType)) {
        (0, node_opcua_assert_1.assert)(fieldType !== "LocalizedText"); // LocalizedText should be treated as BasicType!!!
        return types_1.FieldCategory.complex;
    }
    warningLog("Cannot figure out field category for ", field);
    return types_1.FieldCategory.basic;
}
const regExp = /((ns[0-9]+:)?)(.*)/;
function figureOutSchema(underConstructSchema, dataTypeFactory, field, category) {
    if (field.schema) {
        return field.schema;
    }
    if (underConstructSchema.name === field.fieldType) {
        return underConstructSchema;
    }
    let returnValue = null;
    // may be the field.type  contains a ns<X>: prefix !! like the one found in Beckhoff PLC !
    const m = field.fieldType.match(regExp);
    /* istanbul ignore next */
    if (!m) {
        throw new Error("malformed fieldType ? : " + field.fieldType);
    }
    const fieldTypeWithoutNS = m[3];
    switch (category) {
        case types_1.FieldCategory.complex:
            if (dataTypeFactory.hasStructureByTypeName(field.fieldType)) {
                returnValue = dataTypeFactory.getStructuredTypeSchema(fieldTypeWithoutNS);
            }
            else {
                // LocalizedText etc ...
                returnValue = dataTypeFactory.getBuiltInType(fieldTypeWithoutNS);
            }
            break;
        case types_1.FieldCategory.basic:
            returnValue = dataTypeFactory.getBuiltInType(fieldTypeWithoutNS);
            if (!returnValue) {
                if (dataTypeFactory.hasEnumeration(fieldTypeWithoutNS)) {
                    warningLog("expecting a enumeration!");
                }
                returnValue = dataTypeFactory.getStructuredTypeSchema(fieldTypeWithoutNS);
                // istanbul ignore next
                if (returnValue) {
                    warningLog("Why can't we find a basic type here ?");
                }
            }
            break;
        case types_1.FieldCategory.enumeration:
            returnValue = dataTypeFactory.getEnumeration(fieldTypeWithoutNS);
            break;
    }
    if (null === returnValue || undefined === returnValue) {
        try {
            returnValue = dataTypeFactory.getEnumeration(fieldTypeWithoutNS);
        }
        catch (err) {
            warningLog("dataTypeFactory.getEnumeration has failed", err);
        }
        throw new Error("Cannot find Schema for field with name " +
            field.name +
            " fieldTypeWithoutNS= " +
            fieldTypeWithoutNS +
            " with type " +
            field.fieldType +
            " category = " +
            category +
            JSON.stringify(field, null, "\t"));
    }
    return returnValue;
}
function buildField(underConstructSchema, dataTypeFactory, fieldLight, _index) {
    const category = (fieldLight.fieldType == underConstructSchema.name)
        ? underConstructSchema.category
        : figureOutFieldCategory(fieldLight, dataTypeFactory);
    const schema = figureOutSchema(underConstructSchema, dataTypeFactory, fieldLight, category);
    /* istanbul ignore next */
    if (!schema) {
        throw new Error("expecting a valid schema for field with name " +
            fieldLight.name +
            " with type " +
            fieldLight.fieldType +
            " category" +
            category +
            " at index" +
            _index);
    }
    const { defaultValue, isArray, documentation, fieldType, switchBit, switchValue, allowSubType, dataType, basicDataType } = fieldLight;
    return {
        name: (0, node_opcua_utils_1.lowerFirstLetter)(fieldLight.name),
        originalName: fieldLight.name,
        category,
        defaultValue,
        isArray,
        documentation,
        fieldType,
        switchBit,
        switchValue,
        allowSubType,
        dataType,
        basicDataType,
        schema
    };
}
class StructuredTypeSchema extends builtin_types_1.TypeSchemaBase {
    fields;
    dataTypeNodeId;
    baseType;
    _baseSchema;
    documentation;
    isValid;
    decodeDebug;
    constructHook;
    encodingDefaultBinary;
    encodingDefaultXml;
    encodingDefaultJson;
    bitFields;
    _dataTypeFactory;
    constructor(options) {
        super(options);
        this.bitFields = options.bitFields;
        this.baseType = options.baseType;
        this.category = options.category || types_1.FieldCategory.complex;
        this._dataTypeFactory = options.dataTypeFactory;
        if (this._dataTypeFactory.hasBuiltInType(options.name)) {
            this.category = types_1.FieldCategory.basic;
        }
        this.dataTypeNodeId = new node_opcua_nodeid_1.NodeId();
        this._baseSchema = undefined;
        this.fields = options.fields.map(buildField.bind(null, this, this._dataTypeFactory));
    }
    getDataTypeFactory() {
        return this._dataTypeFactory || (0, get_standard_data_type_factory_1.getStandardDataTypeFactory)();
    }
    getBaseSchema() {
        if (this._baseSchema !== undefined && this._baseSchema === null && this.baseType === "ExtensionObject") {
            return this._baseSchema;
        }
        const _schemaBase = _get_base_schema(this);
        this._baseSchema = _schemaBase;
        return _schemaBase || null;
    }
    getPossibleFieldsLocal() {
        return this.fields.map((field) => field.name);
    }
    toString() {
        const str = [];
        str.push("name           = " + this.name);
        str.push("baseType       = " + this.baseType);
        str.push("bitFields      = " + (this.bitFields ? this.bitFields.map((b) => b.name).join(" ") : undefined));
        str.push("dataTypeNodeId = " + (this.dataTypeNodeId ? this.dataTypeNodeId.toString() : undefined));
        str.push("documentation  = " + this.documentation);
        str.push("encodingDefaultBinary  = " + this.encodingDefaultBinary?.toString());
        str.push("encodingDefaultXml     = " + this.encodingDefaultXml?.toString());
        str.push("encodingDefaultJson    = " + this.encodingDefaultJson?.toString());
        for (const f of this.fields) {
            str.push("  field   =  " +
                f.name.padEnd(30) +
                " isArray= " +
                (f.isArray ? true : false) +
                " " +
                f.fieldType.toString().padEnd(30) +
                (f.switchBit !== undefined ? " switchBit " + f.switchBit : "") +
                (f.switchValue !== undefined ? " switchValue    " + f.switchValue : ""));
        }
        return str.join("\n");
    }
}
exports.StructuredTypeSchema = StructuredTypeSchema;
function _get_base_schema(schema) {
    const dataTypeFactory = schema.getDataTypeFactory();
    if (schema.baseType === "ExtensionObject" || schema.baseType === "DataTypeDefinition") {
        return null;
    }
    if (schema.baseType === "Union") {
        return null;
    }
    if (schema.baseType &&
        schema.baseType !== "BaseUAObject" &&
        schema.baseType !== "Structure" &&
        schema.baseType !== "DataTypeDefinition") {
        if (!dataTypeFactory.hasStructureByTypeName(schema.baseType)) {
            //    warningLog(`Cannot find schema for ${schema.baseType} in dataTypeFactory for ${schema.name} and schema is not abstract ! fix me !`);
            return undefined;
        }
        const structureInfo = dataTypeFactory.getStructureInfoByTypeName(schema.baseType);
        // istanbul ignore next
        if (!structureInfo) {
            throw new Error(" cannot find factory for " + schema.baseType);
        }
        if (structureInfo.schema) {
            return structureInfo.schema;
        }
    }
    // put in  cache for speedup
    return null;
}
/**
 * extract a list of all possible fields for a schema
 * (by walking up the inheritance chain)
 *
 */
function extractAllPossibleFields(schema) {
    // returns cached result if any
    // istanbul ignore next
    // extract the possible fields from the schema.
    let possibleFields = schema.fields.map((field) => field.name);
    const baseSchema = schema.getBaseSchema();
    // istanbul ignore next
    if (baseSchema) {
        const fields = extractAllPossibleFields(baseSchema);
        possibleFields = fields.concat(possibleFields);
    }
    return possibleFields;
}
/**
 * check correctness of option fields against scheme
 *

 *
 */
function check_options_correctness_against_schema(obj, schema, options) {
    if (!parameters_1.parameters.debugSchemaHelper) {
        return true; // ignoring set
    }
    options = options || {};
    // istanbul ignore next
    if (!(options !== null && typeof options === "object") && !(typeof options === "object")) {
        let message = chalk_1.default.red(" Invalid options specified while trying to construct a ") + " " + chalk_1.default.yellow(schema.name);
        message += "\n";
        message += chalk_1.default.red(" expecting a ") + chalk_1.default.yellow(" Object ");
        message += "\n";
        message += chalk_1.default.red(" and got a ") + chalk_1.default.yellow(typeof options) + chalk_1.default.red(" instead ");
        throw new Error(message);
    }
    // istanbul ignore next
    if (options instanceof obj.constructor) {
        return true;
    }
    // extract the possible fields from the schema.
    const possibleFields = obj.constructor.possibleFields || extractAllPossibleFields(schema);
    // extracts the fields exposed by the option object
    const currentFields = Object.keys(options);
    // get a list of field that are in the 'options' object but not in schema
    // https://github.com/you-dont-need/You-Dont-Need-Lodash-Underscore
    function difference(a1, a2) {
        return [a1, a2].reduce((a, b) => a.filter((value) => !b.includes(value)));
    }
    const invalidOptionsFields = difference(currentFields, possibleFields);
    /* istanbul ignore next */
    if (invalidOptionsFields.length > 0) {
        errorLog("expected schema", schema.name);
        errorLog(chalk_1.default.yellow("possible fields= "), possibleFields.sort().join(" "));
        errorLog(chalk_1.default.red("current fields= "), currentFields.sort().join(" "));
        errorLog(chalk_1.default.cyan("invalid_options_fields= "), invalidOptionsFields.sort().join(" "));
        errorLog("options = ", options);
    }
    /* istanbul ignore next */
    if (invalidOptionsFields.length !== 0) {
        errorLog(chalk_1.default.yellow("possible fields= "), possibleFields.sort().join(" "));
        errorLog(chalk_1.default.red("current fields= "), currentFields.sort().join(" "));
        throw new Error(" invalid field found in option :" + JSON.stringify(invalidOptionsFields));
    }
    return true;
}
function buildStructuredType(schemaLight) {
    return new StructuredTypeSchema({
        ...schemaLight,
        dataTypeFactory: (0, get_standard_data_type_factory_1.getStandardDataTypeFactory)()
    });
}
//# sourceMappingURL=structured_type_schema.js.map