"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UAProxyManager = void 0;
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_data_value_1 = require("node-opcua-data-value");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_debug_1 = require("node-opcua-debug");
const object_explorer_1 = require("./object_explorer");
const proxy_1 = require("./proxy");
const proxy_object_1 = require("./proxy_object");
const state_machine_proxy_1 = require("./state_machine_proxy");
const debugLog = (0, node_opcua_debug_1.make_debugLog)(__filename);
async function internalGetObject(proxyManager, nodeId, options) {
    const session = proxyManager.session;
    nodeId = (0, node_opcua_nodeid_1.coerceNodeId)(nodeId);
    if (nodeId.isEmpty()) {
        throw new Error(" Invalid empty node in getObject");
    }
    const nodesToRead = [
        {
            attributeId: node_opcua_data_model_1.AttributeIds.BrowseName,
            nodeId
        },
        {
            attributeId: node_opcua_data_model_1.AttributeIds.Description,
            nodeId
        },
        {
            attributeId: node_opcua_data_model_1.AttributeIds.NodeClass,
            nodeId
        }
    ];
    async function read_accessLevels(clientObject) {
        const nodesToRead = [
            {
                attributeId: node_opcua_data_model_1.AttributeIds.Value,
                nodeId
            },
            {
                attributeId: node_opcua_data_model_1.AttributeIds.UserAccessLevel,
                nodeId
            },
            {
                attributeId: node_opcua_data_model_1.AttributeIds.AccessLevel,
                nodeId
            }
        ];
        const dataValues = await session.read(nodesToRead);
        if (dataValues[0].statusCode.isGood()) {
            clientObject.dataValue = dataValues[0].value;
        }
        if (dataValues[1].statusCode.isGood()) {
            clientObject.userAccessLevel = (0, node_opcua_data_model_1.coerceAccessLevelFlag)(dataValues[1].value.value);
        }
        if (dataValues[2].statusCode.isGood()) {
            clientObject.accessLevel = (0, node_opcua_data_model_1.coerceAccessLevelFlag)(dataValues[2].value.value);
        }
    }
    let clientObject;
    const dataValues = await session.read(nodesToRead);
    if (dataValues[0].statusCode.equals(node_opcua_status_code_1.StatusCodes.BadNodeIdUnknown)) {
        throw new Error("Invalid Node " + nodeId.toString());
    }
    clientObject = new proxy_object_1.ProxyObject(proxyManager, nodeId);
    clientObject.browseName = dataValues[0].value.value;
    clientObject.description = dataValues[1].value ? dataValues[1].value.value : "";
    clientObject.nodeClass = dataValues[2].value.value;
    if (clientObject.nodeClass === node_opcua_data_model_1.NodeClass.Variable) {
        await read_accessLevels(clientObject);
    }
    // install monitored item
    if (clientObject.nodeClass === node_opcua_data_model_1.NodeClass.Variable) {
        await proxyManager._monitor_value(clientObject);
    }
    return await (0, object_explorer_1.readUAStructure)(proxyManager, clientObject);
}
// tslint:disable-next-line: max-classes-per-file
class UAProxyManager {
    session;
    subscription;
    #_map;
    constructor(session) {
        this.session = session;
        this.#_map = {};
    }
    async start() {
        const createSubscriptionRequest = {
            maxNotificationsPerPublish: 1000,
            priority: 10,
            publishingEnabled: true,
            requestedLifetimeCount: 6000,
            requestedMaxKeepAliveCount: 100,
            requestedPublishingInterval: 100
        };
        const subscription = await this.session.createSubscription2(createSubscriptionRequest);
        this.subscription = subscription;
        this.subscription.on("terminated", () => {
            this.subscription = undefined;
        });
    }
    async stop() {
        if (this.subscription) {
            await this.subscription.terminate();
            this.subscription = undefined;
        }
        else {
            // throw new Error("UAProxyManager already stopped ?");
        }
    }
    // todo: rename getObject as getNode
    async getObject(nodeId) {
        let options = {};
        options = options || {};
        options.depth = options.depth || 1;
        const key = nodeId.toString();
        // the object already exist in the map ?
        if (Object.prototype.hasOwnProperty.call(this.#_map, key)) {
            return this.#_map[key];
        }
        const obj = await internalGetObject(this, nodeId, options);
        this.#_map[key] = obj;
        return obj;
    }
    async _monitor_value(proxyObject) {
        if (!this.subscription) {
            // debugLog("cannot monitor _monitor_value: no subscription");
            // some server do not provide subscription support, do not treat this as an error.
            return; // new Error("No subscription"));
        }
        const itemToMonitor = {
            // ReadValueId
            attributeId: node_opcua_data_model_1.AttributeIds.Value,
            nodeId: proxyObject.nodeId
        };
        const monitoringParameters = {
            // MonitoringParameters
            discardOldest: true,
            queueSize: 10,
            samplingInterval: 0 /* event-based */
        };
        const requestedParameters = node_opcua_data_value_1.TimestampsToReturn.Both;
        const monitoredItem = await this.subscription.monitor(itemToMonitor, monitoringParameters, requestedParameters);
        Object.defineProperty(proxyObject, "__monitoredItem", { value: monitoredItem, enumerable: false });
        proxyObject.__monitoredItem.on("changed", (dataValue) => {
            proxyObject.dataValue = dataValue;
            proxyObject.emit("value_changed", dataValue);
        });
        proxyObject.__monitoredItem.on("err", (err) => {
            debugLog("Proxy: cannot monitor variable ", itemToMonitor.nodeId?.toString(), err.message);
        });
    }
    async _monitor_execution_flag(proxyObject) {
        // note : proxyObject must wrap a method
        (0, node_opcua_assert_1.assert)(proxyObject.nodeId instanceof node_opcua_nodeid_1.NodeId);
        if (!this.subscription) {
            // some server do not provide subscription support, do not treat this as an error.
            return; // new Error("No subscription"));
        }
        const itemToMonitor = {
            // ReadValueId
            attributeId: node_opcua_data_model_1.AttributeIds.Executable,
            nodeId: proxyObject.nodeId
        };
        const monitoringParameters = {
            // MonitoringParameters
            discardOldest: true,
            queueSize: 10,
            samplingInterval: 0 /* event-based */
        };
        const requestedParameters = node_opcua_data_value_1.TimestampsToReturn.Neither;
        const monitoredItem = await this.subscription.monitor(itemToMonitor, monitoringParameters, requestedParameters);
        Object.defineProperty(proxyObject, "__monitoredItem_execution_flag", {
            value: monitoredItem,
            enumerable: false
        });
        proxyObject.__monitoredItem_execution_flag.on("changed", (dataValue) => {
            proxyObject.executableFlag = dataValue.value.value;
        });
    }
    async getStateMachineType(nodeId) {
        if (typeof nodeId === "string") {
            nodeId = (0, proxy_1.makeRefId)(nodeId);
        }
        const obj = await this.getObject(nodeId);
        return new state_machine_proxy_1.ProxyStateMachineType(obj);
    }
}
exports.UAProxyManager = UAProxyManager;
//# sourceMappingURL=proxy_manager.js.map