"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProxyBaseNode = void 0;
/**
 * @module node-opcua-client-proxy
 */
const events_1 = require("events");
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_data_model_1 = require("node-opcua-data-model");
class ProxyBaseNode extends events_1.EventEmitter {
    /**
     * the object nodeId
     * @property nodeId
     * @type {NodeId}
     */
    nodeId;
    /**
     * the object's components
     * @property $components
     * @type {Array<ProxyBaseNode>}
     */
    $components;
    /**
     * the object's properties
     * @property $properties
     * @type {Array<ProxyBaseNode>}
     */
    $properties;
    /**
     * the object's properties
     * @property $methods
     * @type {Array<ProxyBaseNode>}
     */
    $methods;
    /**
     * the Folder's elements
     * @property $organizes
     * @type {Array<ProxyBaseNode>}
     */
    $organizes;
    /**
     * the object's description
     * @property description
     * @type {String}
     */
    description;
    /**
     * the object's browseName
     * @property browseName
     * @type {String}
     */
    browseName;
    /**
     * the object's NodeClass
     * @property nodeClass
     * @type {NodeClass}
     */
    nodeClass;
    proxyManager;
    constructor(proxyManager, nodeId, nodeClass) {
        super();
        this.nodeId = nodeId;
        this.proxyManager = proxyManager;
        (0, node_opcua_assert_1.assert)(this.proxyManager.session, "expecting valid session");
        Object.defineProperty(this, "proxyManager", {
            enumerable: false,
            writable: true
        });
        this.$components = [];
        this.$properties = [];
        this.$methods = [];
        this.$organizes = [];
        this.description = "";
        this.browseName = "";
        this.nodeClass = nodeClass;
    }
    /**
     * get a updated Value of the Variable , by using a ReadRequest
     */
    async readValue() {
        (0, node_opcua_assert_1.assert)(this.proxyManager);
        const session = this.proxyManager.session;
        (0, node_opcua_assert_1.assert)(session);
        const nodeToRead = {
            attributeId: node_opcua_data_model_1.AttributeIds.Value,
            nodeId: this.nodeId
        };
        const dataValue = await this.proxyManager.session.read(nodeToRead);
        const data = dataValue.value;
        return data;
    }
    /**
     * set the Value of the Variable, by using a WriteRequest
     */
    async writeValue(dataValue) {
        (0, node_opcua_assert_1.assert)(this.proxyManager);
        const session = this.proxyManager.session;
        (0, node_opcua_assert_1.assert)(session);
        const nodeToWrite = {
            attributeId: node_opcua_data_model_1.AttributeIds.Value,
            nodeId: this.nodeId,
            value: dataValue
        };
        const statusCode = await this.proxyManager.session.write(nodeToWrite);
        return statusCode;
    }
    toString() {
        const str = [];
        str.push(" ProxyObject ");
        str.push("   browseName     : " + this.browseName.toString());
        // str.push("   typeDefinition : " + this.typeDefinition.toString());
        str.push("   $components#   : " + this.$components.length.toString());
        str.push("   $properties#   : " + this.$properties.length.toString());
        return str.join("\n");
    }
}
exports.ProxyBaseNode = ProxyBaseNode;
//# sourceMappingURL=proxy_base_node.js.map