"use strict";
/**
 * @module node-opcua-client-proxy
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ObjectExplorer = void 0;
exports.readUAStructure = readUAStructure;
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_service_call_1 = require("node-opcua-service-call");
const node_opcua_utils_1 = require("node-opcua-utils");
const node_opcua_variant_1 = require("node-opcua-variant");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_constants_1 = require("node-opcua-constants");
const proxy_1 = require("./proxy");
const proxy_variable_1 = require("./proxy_variable");
const doDebug = false;
const debugLog = (0, node_opcua_debug_1.make_debugLog)("Proxy");
const resultMask = (0, node_opcua_data_model_1.makeResultMask)("ReferenceType | IsForward | BrowseName | NodeClass | TypeDefinition");
/**

 *
 * @param session
 * @param dataTypeId
 * @param callback
 * @param callback.err
 * @param callback.dataType
 *
 *  @example
 *
 *      const dataTypeId  ="ns=0;i=11"; // Double
 *      convertNodeIdToDataTypeAsync(session,dataTypeId,function(err,dataType) {
 *          assert(!err && dataType === DataType.Double);
 *      });
 *
 *      const dataTypeId  ="ns=0;i=290"; // Duration => SubTypeOf Double
 *      convertNodeIdToDataTypeAsync(session,dataTypeId,function(err,dataType) {
 *          assert(!err && dataType === DataType.Double);
 *      });
 *
 * see also AddressSpace#findCorrespondingBasicDataType
 *
 * for an enumeration dataType will be DataType.Int32
 */
async function convertNodeIdToDataTypeAsync(session, dataTypeId) {
    const nodeToRead = {
        attributeId: node_opcua_data_model_1.AttributeIds.BrowseName,
        nodeId: dataTypeId
    };
    const dataValue = await session.read(nodeToRead);
    let dataType;
    // istanbul ignore next
    if (dataValue.statusCode.isNotGood()) {
        dataType = node_opcua_variant_1.DataType.Null;
        return dataType;
    }
    const dataTypeName = dataValue.value.value;
    if (dataTypeId.namespace === 0 && dataTypeId.value === node_opcua_constants_1.DataTypeIds.Enumeration) {
        dataType = node_opcua_variant_1.DataType.Int32;
        return dataType;
    }
    if (dataTypeId.namespace === 0 && node_opcua_variant_1.DataType[dataTypeId.value]) {
        dataType = node_opcua_variant_1.DataType[dataTypeId.value];
        return dataType;
    }
    /// example => Duration (i=290) => Double (i=11)
    // read subTypeOf
    const nodeToBrowse = {
        browseDirection: node_opcua_data_model_1.BrowseDirection.Inverse,
        includeSubtypes: false,
        nodeId: dataTypeId,
        // BrowseDescription
        referenceTypeId: (0, proxy_1.makeRefId)("HasSubtype"),
        // xx nodeClassMask: makeNodeClassMask("ObjectType"),
        resultMask
    };
    // tslint:disable:no-shadowed-variable
    const browseResult = await session.browse(nodeToBrowse);
    const references = browseResult.references;
    if (!references || references.length !== 1) {
        throw new Error("cannot find SuperType of " + dataTypeName.toString());
    }
    const nodeId = references[0].nodeId;
    return convertNodeIdToDataTypeAsync(session, nodeId);
}
function convertToVariant(value, arg, propName) {
    const dataType = arg._basicDataType || node_opcua_variant_1.DataType.Null;
    const arrayType = arg.valueRank === 1 ? node_opcua_variant_1.VariantArrayType.Array : arg.valueRank === -1 ? node_opcua_variant_1.VariantArrayType.Scalar : node_opcua_variant_1.VariantArrayType.Matrix;
    if (value === undefined) {
        throw new Error("expecting input argument ");
    }
    if (arrayType === node_opcua_variant_1.VariantArrayType.Array) {
        if (!Array.isArray(value)) {
            throw new Error("expecting value to be an Array or a TypedArray");
        }
    }
    return new node_opcua_variant_1.Variant({ arrayType, dataType, value });
}
function convertToVariantArray(inputArgsDef, inputArgs) {
    const inputArguments = inputArgsDef.map((arg) => {
        const propName = (0, node_opcua_utils_1.lowerFirstLetter)(arg.name || "");
        const value = inputArgs[propName];
        return convertToVariant(value, arg, propName);
    });
    return inputArguments;
}
function makeFunction(obj, methodName) {
    return async function functionCaller(inputArgs) {
        const session = this.proxyManager.session;
        const methodDef = this.$methods[methodName];
        // convert input arguments into Variants
        const inputArgsDef = methodDef.inputArguments || [];
        const inputArguments = convertToVariantArray(inputArgsDef, inputArgs);
        const methodToCall = new node_opcua_service_call_1.CallMethodRequest({
            inputArguments,
            methodId: methodDef.nodeId,
            objectId: obj.nodeId
        });
        const callResult = await session.call(methodToCall);
        if (callResult.statusCode.isNotGood()) {
            return { statusCode: callResult.statusCode };
        }
        callResult.outputArguments = callResult.outputArguments || [];
        if (callResult.outputArguments.length !== methodDef.outputArguments.length) {
            throw new Error("Internal error callResult.outputArguments.length " +
                callResult.outputArguments.length +
                " " +
                obj[methodName].outputArguments.length);
        }
        const output = {};
        methodDef.outputArguments.map((arg, index) => {
            const variant = callResult.outputArguments[index];
            const propName = (0, node_opcua_utils_1.lowerFirstLetter)(arg.name);
            output[propName] = variant.value;
        });
        return { statusCode: callResult.statusCode, output };
    };
}
async function extractDataType(session, arg) {
    if (arg.dataType && arg._basicDataType) {
        return;
    }
    const dataType = await convertNodeIdToDataTypeAsync(session, arg.dataType);
    arg._basicDataType = dataType;
}
/**
 
 * @private
 */
async function add_method(proxyManager, obj, reference) {
    const session = proxyManager.session;
    const methodName = (0, node_opcua_utils_1.lowerFirstLetter)(reference.browseName.name);
    let inputArguments = [];
    let outputArguments = [];
    // tslint:disable:no-shadowed-variable
    const argumentDefinition = await session.getArgumentDefinition(reference.nodeId);
    inputArguments = argumentDefinition.inputArguments || [];
    outputArguments = argumentDefinition.outputArguments || [];
    const promises = [];
    for (const arg of inputArguments || []) {
        promises.push(extractDataType(session, arg));
    }
    for (const arg of outputArguments || []) {
        promises.push(extractDataType(session, arg));
    }
    await Promise.all(promises);
    const methodObj = {
        browseName: methodName,
        executableFlag: false,
        func: makeFunction(obj, methodName),
        nodeId: reference.nodeId,
        inputArguments,
        outputArguments
    };
    obj.$methods[methodName] = methodObj;
    obj[methodName] = methodObj.func;
    obj[methodName].inputArguments = inputArguments;
    obj[methodName].outputArguments = outputArguments;
    doDebug && debugLog("installing method name", methodName);
    await proxyManager._monitor_execution_flag(methodObj);
}
async function add_component(proxyManager, obj, reference) {
    const name = (0, node_opcua_utils_1.lowerFirstLetter)(reference.browseName.name || "");
    await proxyManager.getObject(reference.nodeId);
    const childObj = new ObjectExplorer({
        name,
        nodeId: reference.nodeId,
        parent: obj,
        proxyManager
    });
    obj[name] = childObj;
    obj.$components.push(childObj);
    await childObj.$resolve();
}
async function addFolderElement(proxyManager, obj, reference) {
    const name = (0, node_opcua_utils_1.lowerFirstLetter)(reference.browseName.name || "");
    const childObj = new ObjectExplorer({
        name,
        nodeId: reference.nodeId,
        parent: obj,
        proxyManager
    });
    obj[name] = childObj;
    obj.$organizes.push(childObj);
    await childObj.$resolve();
}
async function add_property(proxyManager, obj, reference) {
    const name = (0, node_opcua_utils_1.lowerFirstLetter)(reference.browseName.name || "");
    obj[name] = new proxy_variable_1.ProxyVariable(proxyManager, reference.nodeId, reference);
    obj.$properties[name] = obj[name];
}
async function add_typeDefinition(proxyManager, obj, references) {
    references = references || [];
    if (references.length !== 1) {
        return;
    }
    const reference = references[0];
    (0, node_opcua_assert_1.assert)(!obj.typeDefinition, "type definition can only be set once");
    obj.typeDefinition = reference.browseName.name || "";
}
async function addFromState(proxyManager, obj, reference) {
    const childObj = await proxyManager.getObject(reference.nodeId);
    obj.$fromState = childObj;
}
async function addToState(proxyManager, obj, reference) {
    const childObj = await proxyManager.getObject(reference.nodeId);
    obj.$toState = childObj;
}
class ObjectExplorer {
    proxyManager;
    name;
    nodeId;
    parent;
    constructor(options) {
        this.proxyManager = options.proxyManager;
        this.name = options.name;
        this.nodeId = options.nodeId;
        this.parent = options.parent;
    }
    async $resolve() {
        const childObj = await this.proxyManager.getObject(this.nodeId);
        this.parent[this.name] = childObj;
        this.parent.$components.push(childObj);
    }
}
exports.ObjectExplorer = ObjectExplorer;
function t(references) {
    if (!references)
        return "";
    return references.map((r) => r.browseName.name + " " + r.nodeId.toString());
}
async function readUAStructure(proxyManager, obj) {
    const session = proxyManager.session;
    //   0   Object
    //   1   Variable
    //   2   Method
    const nodeId = obj.nodeId;
    const nodesToBrowse = [
        // 0. Components (except Methods)
        {
            // BrowseDescription
            browseDirection: node_opcua_data_model_1.BrowseDirection.Forward,
            includeSubtypes: true,
            nodeClassMask: (0, node_opcua_data_model_1.makeNodeClassMask)("Object | Variable"), // we don't want Method here
            nodeId,
            referenceTypeId: (0, proxy_1.makeRefId)("HasComponent"),
            resultMask
        },
        // 1. Properties
        {
            // BrowseDescription
            browseDirection: node_opcua_data_model_1.BrowseDirection.Forward,
            includeSubtypes: true,
            // nodeClassMask: makeNodeClassMask("Variable"),
            nodeId,
            referenceTypeId: (0, proxy_1.makeRefId)("HasProperty"),
            resultMask
        },
        // 2.  Methods
        {
            // BrowseDescription
            browseDirection: node_opcua_data_model_1.BrowseDirection.Forward,
            includeSubtypes: true,
            nodeClassMask: (0, node_opcua_data_model_1.makeNodeClassMask)("Method"),
            nodeId,
            referenceTypeId: (0, proxy_1.makeRefId)("HasComponent"),
            resultMask
        },
        // TypeDefinition
        {
            // BrowseDescription
            browseDirection: node_opcua_data_model_1.BrowseDirection.Both,
            includeSubtypes: true,
            nodeId,
            referenceTypeId: (0, proxy_1.makeRefId)("HasTypeDefinition"),
            resultMask
        },
        // FromState
        {
            // BrowseDescription
            browseDirection: node_opcua_data_model_1.BrowseDirection.Forward,
            includeSubtypes: true,
            nodeId,
            referenceTypeId: (0, proxy_1.makeRefId)("FromState"),
            resultMask
        },
        // ToState
        {
            // BrowseDescription
            browseDirection: node_opcua_data_model_1.BrowseDirection.Forward,
            includeSubtypes: true,
            nodeId,
            referenceTypeId: (0, proxy_1.makeRefId)("ToState"),
            resultMask
        },
        // (for folders ) Organizes
        {
            // BrowseDescription
            browseDirection: node_opcua_data_model_1.BrowseDirection.Forward,
            includeSubtypes: true,
            nodeId,
            referenceTypeId: (0, proxy_1.makeRefId)("Organizes"),
            resultMask
        }
    ];
    const browseResults = await session.browse(nodesToBrowse);
    // istanbul ignore next
    if (doDebug) {
        debugLog("Components", t(browseResults[0].references));
        debugLog("Properties", t(browseResults[1].references));
        debugLog("Methods", t(browseResults[2].references));
    }
    const promises = [];
    for (const reference of browseResults[0].references || []) {
        promises.push(add_component(proxyManager, obj, reference));
    }
    for (const reference of browseResults[1].references || []) {
        promises.push(add_property(proxyManager, obj, reference));
    }
    for (const reference of browseResults[2].references || []) {
        promises.push(add_method(proxyManager, obj, reference));
    }
    browseResults[3].references &&
        browseResults[3].references.length &&
        promises.push(add_typeDefinition(proxyManager, obj, browseResults[3].references || []));
    browseResults[4].references &&
        browseResults[4].references.length &&
        promises.push(addFromState(proxyManager, obj, browseResults[4].references[0]));
    browseResults[5].references &&
        browseResults[5].references.length &&
        promises.push(addToState(proxyManager, obj, browseResults[5].references[0]));
    for (const reference of browseResults[6].references || []) {
        promises.push(addFolderElement(proxyManager, obj, reference));
    }
    await Promise.all(promises);
    return obj;
}
//# sourceMappingURL=object_explorer.js.map