"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.callMethodCondition = callMethodCondition;
exports.acknowledgeCondition = acknowledgeCondition;
exports.confirmCondition = confirmCondition;
const node_opcua_assert_1 = __importDefault(require("node-opcua-assert"));
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_types_1 = require("node-opcua-types");
const node_opcua_variant_1 = require("node-opcua-variant");
const node_opcua_pseudo_session_1 = require("node-opcua-pseudo-session");
const node_opcua_constants_1 = require("node-opcua-constants");
async function callMethodCondition(session, methodName, conditionId, eventId, comment) {
    conditionId = (0, node_opcua_nodeid_1.coerceNodeId)(conditionId);
    (0, node_opcua_assert_1.default)(conditionId instanceof node_opcua_nodeid_1.NodeId);
    (0, node_opcua_assert_1.default)(eventId instanceof Buffer);
    (0, node_opcua_assert_1.default)(typeof comment === "string" || comment instanceof node_opcua_data_model_1.LocalizedText);
    comment = node_opcua_data_model_1.LocalizedText.coerce(comment) || new node_opcua_data_model_1.LocalizedText();
    const r = await (0, node_opcua_pseudo_session_1.findMethodId)(session, conditionId, methodName);
    let methodId = r.methodId;
    if (!methodId) {
        // https://reference.opcfoundation.org/Core/Part9/v104/docs/5.7.3#_Ref224987672
        //  The Acknowledge Method is used to acknowledge an Event Notification for a Condition instance
        //  state where AckedState is False. Normally, the NodeId of the object instance is passed as the
        //  ObjectId to the Call Service. However, some Servers do not expose Condition instances in the AddressSpace.
        //  Therefore, Servers shall allow Clients to call the Acknowledge Method by specifying ConditionId as the ObjectId.
        //  The Method cannot be called with an ObjectId of the AcknowledgeableConditionType Node.
        //
        // The Confirm Method is used to confirm an Event Notifications for a Condition instance state where ConfirmedState is False.
        switch (methodName) {
            case "Acknowledge":
                methodId = (0, node_opcua_nodeid_1.resolveNodeId)(node_opcua_constants_1.MethodIds.AcknowledgeableConditionType_Acknowledge);
                break;
            case "Confirm":
                methodId = (0, node_opcua_nodeid_1.resolveNodeId)(node_opcua_constants_1.MethodIds.AcknowledgeableConditionType_Confirm);
                break;
            default:
                return node_opcua_status_code_1.StatusCodes.BadNodeIdUnknown;
        }
    }
    const methodToCalls = [];
    methodToCalls.push(new node_opcua_types_1.CallMethodRequest({
        inputArguments: [
            /* eventId */ new node_opcua_variant_1.Variant({ dataType: "ByteString", value: eventId }),
            /* comment */ new node_opcua_variant_1.Variant({ dataType: "LocalizedText", value: comment })
        ],
        methodId,
        objectId: conditionId
    }));
    const results = await session.call(methodToCalls);
    const statusCode = results[0].statusCode;
    return statusCode;
}
async function acknowledgeCondition(session, conditionId, eventId, comment) {
    return await callMethodCondition(session, "Acknowledge", conditionId, eventId, comment);
}
async function confirmCondition(session, conditionId, eventId, comment) {
    return await callMethodCondition(session, "Confirm", conditionId, eventId, comment);
}
//# sourceMappingURL=call_method_condition.js.map