"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createHistoryServerCapabilities = createHistoryServerCapabilities;
exports.addAggregateFunctionSupport = addAggregateFunctionSupport;
exports.addAggregateStandardFunctionSupport = addAggregateStandardFunctionSupport;
exports.addAggregateSupport = addAggregateSupport;
exports.getAggregateFunctions = getAggregateFunctions;
exports.installAggregateConfigurationOptions = installAggregateConfigurationOptions;
exports.getAggregateConfiguration = getAggregateConfiguration;
/**
 * @module node-opcua-aggregates
 */
const node_opcua_constants_1 = require("node-opcua-constants");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_variant_1 = require("node-opcua-variant");
const node_opcua_utils_1 = require("node-opcua-utils");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const read_processed_details_1 = require("./read_processed_details");
// import { HistoryServerCapabilities } from "node-opcua-server";
/*
HasProperty Variable AccessHistoryDataCapability Boolean PropertyType Mandatory
HasProperty Variable AccessHistoryEventsCapability Boolean PropertyType Mandatory
HasProperty Variable MaxReturnDataValues UInt32 PropertyType Mandatory
HasProperty Variable MaxReturnEventValues UInt32 PropertyType Mandatory
HasProperty Variable InsertDataCapability Boolean PropertyType Mandatory
HasProperty Variable ReplaceDataCapability Boolean PropertyType Mandatory
HasProperty Variable UpdateDataCapability Boolean PropertyType Mandatory
HasProperty Variable DeleteRawCapability Boolean PropertyType Mandatory
HasProperty Variable DeleteAtTimeCapability Boolean PropertyType Mandatory
HasProperty Variable InsertEventCapability Boolean PropertyType Mandatory
HasProperty Variable ReplaceEventCapability Boolean PropertyType Mandatory
HasProperty Variable UpdateEventCapability Boolean PropertyType Mandatory
HasProperty Variable DeleteEventCapability Boolean PropertyType Mandatory
HasProperty Variable InsertAnnotationsCapability Boolean PropertyType Mandatory
 */
const historicalCapabilitiesDefaultProperties /*: HistoryServerCapabilities */ = {
    accessHistoryDataCapability: true, // Boolean PropertyType Mandatory
    accessHistoryEventsCapability: true, // Boolean PropertyType Mandatory
    deleteAtTimeCapability: false, // Boolean PropertyType Mandatory
    deleteEventCapability: false, // Boolean PropertyType Mandatory
    deleteRawCapability: false, // Boolean PropertyType Mandatory
    insertAnnotationCapability: false, // Boolean PropertyType Mandatory
    insertDataCapability: false, // Boolean PropertyType Mandatory
    insertEventCapability: false, // Boolean PropertyType Mandatory
    maxReturnDataValues: 0,
    maxReturnEventValues: 0, // UInt32 PropertyType Mandatory
    replaceDataCapability: false, // Boolean PropertyType Mandatory
    replaceEventCapability: false, // Boolean PropertyType Mandatory
    updateDataCapability: false, // Boolean PropertyType Mandatory
    updateEventCapability: false // Boolean PropertyType Mandatory
};
function createHistoryServerCapabilities(addressSpace, serverCapabilities) {
    /* istanbul ignore next */
    if (serverCapabilities.browseName.toString() !== "ServerCapabilities") {
        throw new Error("Expecting server Capabilities");
    }
    const historyServerCapabilitiesType = addressSpace.getNamespace(0).findObjectType("HistoryServerCapabilitiesType");
    /* istanbul ignore next */
    if (!historyServerCapabilitiesType) {
        throw new Error("Cannot find HistoryServerCapabilitiesType");
    }
    return historyServerCapabilitiesType.instantiate({
        browseName: "HistoryServerCapabilities",
        componentOf: serverCapabilities
    });
}
function setHistoricalServerCapabilities(historyServerCapabilities, defaultProperties) {
    function setBoolean(propName) {
        const lowerCase = (0, node_opcua_utils_1.lowerFirstLetter)(propName);
        /* istanbul ignore next */
        if (!Object.prototype.hasOwnProperty.call(defaultProperties, lowerCase)) {
            throw new Error("cannot find " + lowerCase);
        }
        const value = defaultProperties[lowerCase];
        const prop = historyServerCapabilities.getChildByName(propName);
        /* istanbul ignore next */
        if (!prop) {
            throw new Error(" Cannot find property " + propName);
        }
        prop.setValueFromSource({ dataType: node_opcua_variant_1.DataType.Boolean, value });
    }
    function setUInt32(propName) {
        const lowerCase = (0, node_opcua_utils_1.lowerFirstLetter)(propName);
        /* istanbul ignore next */
        if (!Object.prototype.hasOwnProperty.call(historyServerCapabilities, lowerCase)) {
            throw new Error("cannot find " + lowerCase);
        }
        const value = defaultProperties[lowerCase];
        const prop = historyServerCapabilities.getChildByName(propName);
        prop.setValueFromSource({ dataType: node_opcua_variant_1.DataType.UInt32, value });
    }
    setBoolean("AccessHistoryDataCapability");
    setBoolean("AccessHistoryEventsCapability");
    setUInt32("MaxReturnDataValues");
    setUInt32("MaxReturnEventValues");
    setBoolean("InsertDataCapability");
    setBoolean("ReplaceDataCapability");
    setBoolean("UpdateDataCapability");
    setBoolean("DeleteRawCapability");
    setBoolean("DeleteAtTimeCapability");
    setBoolean("InsertEventCapability");
    setBoolean("ReplaceEventCapability");
    setBoolean("UpdateEventCapability");
    setBoolean("DeleteEventCapability");
    /// FOUND A BUG HERE spec says InsertAnnotationsCapability
    /// Standard nodeset2 says InsertAnnotationCapability ( without s )
    // xx setBoolean("InsertAnnotationsCapability");
}
function addAggregateFunctionSupport(addressSpace, aggregateFunctionNodeId) {
    const serverCapabilities = addressSpace.rootFolder.objects.server.serverCapabilities;
    /* istanbul ignore next */
    if (!serverCapabilities.historyServerCapabilities) {
        throw new Error("missing serverCapabilities.historyServerCapabilities");
    }
    const aggregateFunctions = serverCapabilities.aggregateFunctions;
    const aggregateFunctionsInHist = serverCapabilities.historyServerCapabilities.aggregateFunctions;
    const functionNode = addressSpace.findNode(aggregateFunctionNodeId);
    /* istanbul ignore next */
    if (!functionNode) {
        throw new Error("Cannot find node " + aggregateFunctionNodeId.toString() + " in addressSpace");
    }
    /* istanbul ignore next */
    if (functionNode.nodeClass !== node_opcua_data_model_1.NodeClass.Object) {
        throw new Error("Expecting an object Node");
    }
    /* istanbul ignore next */
    if (!(0, node_opcua_nodeid_1.sameNodeId)(functionNode.typeDefinition, (0, node_opcua_nodeid_1.coerceNodeId)(node_opcua_constants_1.ObjectTypeIds.AggregateFunctionType))) {
        throw new Error("Expecting an object with TypeDefinition AggregateFunctionType");
    }
    aggregateFunctions.addReference({
        nodeId: functionNode.nodeId,
        referenceType: "Organizes"
    });
    aggregateFunctionsInHist.addReference({
        nodeId: functionNode.nodeId,
        referenceType: "Organizes"
    });
}
function addAggregateStandardFunctionSupport(addressSpace, functionName) {
    /* istanbul ignore next */
    if (!functionName) {
        throw new Error("Invalid function name");
    }
    const functionNodeId = (0, node_opcua_nodeid_1.makeNodeId)(functionName);
    addAggregateFunctionSupport(addressSpace, functionNodeId);
}
function addAggregateSupport(addressSpace, aggregatedFunctions) {
    aggregatedFunctions = aggregatedFunctions || [
        node_opcua_constants_1.AggregateFunction.Interpolative,
        node_opcua_constants_1.AggregateFunction.Minimum,
        node_opcua_constants_1.AggregateFunction.Maximum,
        node_opcua_constants_1.AggregateFunction.Average
    ];
    const aggregateConfigurationType = addressSpace.getNamespace(0).findObjectType("AggregateConfigurationType");
    /* istanbul ignore next */
    if (!aggregateConfigurationType) {
        throw new Error("addressSpace do not expose AggregateConfigurationType");
    }
    const aggregateFunctionType = addressSpace.getNamespace(0).findObjectType("AggregateFunctionType");
    /* istanbul ignore next */
    if (!aggregateFunctionType) {
        throw new Error("addressSpace do not expose AggregateFunctionType");
    }
    const serverObject = addressSpace.rootFolder.objects.getFolderElementByName("Server");
    /* istanbul ignore next */
    if (!serverObject) {
        throw new Error("addressSpace do not expose a ServerObject");
    }
    // xx serverObject.
    const serverCapabilities = serverObject.getChildByName("ServerCapabilities");
    // Let see if HistoryServer Capabilities object exists
    let historyServerCapabilities = serverCapabilities.getChildByName("HistoryServerCapabilities");
    /* istanbul ignore next */
    if (!historyServerCapabilities) {
        historyServerCapabilities = createHistoryServerCapabilities(addressSpace, serverCapabilities);
    }
    setHistoricalServerCapabilities(historyServerCapabilities, historicalCapabilitiesDefaultProperties);
    for (const f of aggregatedFunctions) {
        addAggregateStandardFunctionSupport(addressSpace, f);
    }
    const addressSpaceInternal = addressSpace;
    addressSpaceInternal._readProcessedDetails = read_processed_details_1.readProcessedDetails;
}
function getAggregateFunctions(addressSpace) {
    const aggregateFunctionTypeNodeId = (0, node_opcua_nodeid_1.resolveNodeId)(node_opcua_constants_1.ObjectTypeIds.AggregateFunctionType);
    const aggregateFunctions = addressSpace.findNode(node_opcua_constants_1.ObjectIds.Server_ServerCapabilities_AggregateFunctions);
    if (!aggregateFunctions) {
        return [];
    }
    const referenceDescripitions = aggregateFunctions.browseNode({
        referenceTypeId: node_opcua_constants_1.ReferenceTypeIds.HierarchicalReferences,
        resultMask: 63,
        nodeClassMask: node_opcua_data_model_1.NodeClassMask.Object,
        browseDirection: node_opcua_data_model_1.BrowseDirection.Forward,
        includeSubtypes: true
    });
    const aggregateFunctionsNodeIds = referenceDescripitions
        .filter((a) => (0, node_opcua_nodeid_1.sameNodeId)(a.typeDefinition, aggregateFunctionTypeNodeId))
        .map((a) => a.nodeId);
    return aggregateFunctionsNodeIds;
}
/**
 * Install aggregateConfiguration on an historizing variable
 *
 * @param node the variable on which to add the aggregateConfiguration.
 * @param options the default AggregateConfigurationOptions.
 * @param aggregateFunctions the aggregatedFunctions, if not specified the aggregatedFunction of ServerCapabilities.AggregatedFunction will be used.

 */
function installAggregateConfigurationOptions(node, options, aggregateFunctions) {
    const nodePriv = node;
    // istanbul ignore next
    if (!nodePriv.historizing) {
        throw new Error("variable.historizing is not set\n make sure addressSpace.installHistoricalDataNode(variable) has been called");
    }
    const aggregateConfiguration = nodePriv.$historicalDataConfiguration.aggregateConfiguration;
    const f = (a, defaultValue) => a === undefined ? defaultValue : a;
    aggregateConfiguration.percentDataBad.setValueFromSource({ dataType: "Byte", value: f(options.percentDataBad, 100) });
    aggregateConfiguration.percentDataGood.setValueFromSource({ dataType: "Byte", value: f(options.percentDataGood, 100) });
    aggregateConfiguration.treatUncertainAsBad.setValueFromSource({
        dataType: "Boolean",
        value: f(options.treatUncertainAsBad, false)
    });
    aggregateConfiguration.useSlopedExtrapolation.setValueFromSource({
        dataType: "Boolean",
        value: f(options.useSlopedExtrapolation, false)
    });
    nodePriv.$historicalDataConfiguration.stepped.setValueFromSource({
        dataType: "Boolean",
        value: f(options.stepped, false)
    });
    // https://reference.opcfoundation.org/v104/Core/docs/Part13/4.4/
    // Exposing Supported Functions and Capabilities
    if (!aggregateFunctions) {
        aggregateFunctions = getAggregateFunctions(node.addressSpace);
    }
    let uaAggregateFunctions = nodePriv.$historicalDataConfiguration.aggregateFunctions;
    if (!uaAggregateFunctions) {
        const namespace = nodePriv.namespace;
        uaAggregateFunctions = namespace.addObject({
            browseName: (0, node_opcua_data_model_1.coerceQualifiedName)({ name: "AggregateFunctions", namespaceIndex: 0 }),
            componentOf: nodePriv.$historicalDataConfiguration
        });
        uaAggregateFunctions = nodePriv.$historicalDataConfiguration.aggregateFunctions;
    }
    // verify that all aggregateFunctions are of type AggregateFunctionType
    // ... to do
    const referenceType = (0, node_opcua_nodeid_1.resolveNodeId)(node_opcua_constants_1.ReferenceTypeIds.Organizes);
    for (const nodeId of aggregateFunctions) {
        uaAggregateFunctions.addReference({
            nodeId,
            referenceType,
            isForward: true
        });
    }
}
function getAggregateConfiguration(node) {
    const nodePriv = node;
    /* istanbul ignore next */
    if (!nodePriv.$historicalDataConfiguration) {
        throw new Error("internal error");
    }
    const aggregateConfiguration = nodePriv.$historicalDataConfiguration.aggregateConfiguration;
    // Beware ! Stepped value comes from Historical Configuration !
    const stepped = nodePriv.$historicalDataConfiguration.stepped.readValue().value.value;
    return {
        percentDataBad: aggregateConfiguration.percentDataBad.readValue().value.value,
        percentDataGood: aggregateConfiguration.percentDataGood.readValue().value.value,
        stepped,
        treatUncertainAsBad: aggregateConfiguration.treatUncertainAsBad.readValue().value.value,
        // xx stepped:                aggregateConfiguration.stepped.readValue().value,
        useSlopedExtrapolation: aggregateConfiguration.useSlopedExtrapolation.readValue().value.value
    };
}
//# sourceMappingURL=aggregates.js.map