"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.construct_demo_alarm_in_address_space = construct_demo_alarm_in_address_space;
/**
 * @module node-opcua-address-space
 */
const node_opcua_assert_1 = require("node-opcua-assert");
function construct_demo_alarm_in_address_space(test, addressSpace) {
    const a = addressSpace;
    if (a.construct_demo_alarm_in_address_space_called) {
        return;
    }
    a.construct_demo_alarm_in_address_space_called = true;
    addressSpace.installAlarmsAndConditionsService();
    const namespace = addressSpace.getOwnNamespace();
    const tank = namespace.addObject({
        browseName: "Tank",
        description: "The Object representing the Tank",
        eventNotifier: 0x01,
        notifierOf: addressSpace.rootFolder.objects.server,
        organizedBy: addressSpace.rootFolder.objects
    });
    (0, node_opcua_assert_1.assert)(tank.getNotifiers().length === 0, "expecting a notifier now");
    const tankLevel = namespace.addVariable({
        browseName: "TankLevel",
        componentOf: tank,
        dataType: "Double",
        description: "Fill level in percentage (0% to 100%) of the water tank",
        eventSourceOf: tank,
        value: { dataType: "Double", value: 0.5 }
    });
    //    assert(tank.getNotifiers().length === 1, "expecting a notifier now");
    // --------------------------------------------------------------------------------
    // Let's create a exclusive Limit Alarm that automatically raise itself
    // when the tank level is out of limit
    // --------------------------------------------------------------------------------
    const exclusiveLimitAlarmType = addressSpace.findEventType("ExclusiveLimitAlarmType");
    if (!exclusiveLimitAlarmType) {
        throw new Error("cannot find ExclusiveLimitAlarmType in namespace 0");
    }
    const tankLevelCondition = namespace.instantiateExclusiveLimitAlarm(exclusiveLimitAlarmType, {
        browseName: "TankLevelCondition",
        componentOf: tank,
        conditionName: "TankLevelCondition",
        conditionSource: tankLevel,
        highHighLimit: 0.9,
        highLimit: 0.8,
        inputNode: tankLevel, // the variable that will be monitored for change
        lowLimit: 0.2,
        optionals: [
            "ConfirmedState",
            "Confirm" // confirm state and confirm Method
        ],
        setpointNode: null,
    });
    (0, node_opcua_assert_1.assert)(tankLevelCondition.browseName.toString() === "1:TankLevelCondition");
    (0, node_opcua_assert_1.assert)(tankLevel.findReferences("HasCondition").length === 1);
    (0, node_opcua_assert_1.assert)(tankLevel.findReferencesAsObject("HasCondition", true).length === 1);
    const conditionName = tankLevel.findReferencesAsObject("HasCondition", true)[0].browseName.toString();
    const conditionTypeDefinition = tankLevel.findReferencesAsObject("HasCondition", true)[0].typeDefinitionObj.browseName.toString();
    const conditionJavascriptClass = tankLevel.findReferencesAsObject("HasCondition", true)[0].constructor.name.toString();
    if (false) {
        console.log(conditionName, conditionTypeDefinition, conditionJavascriptClass);
    }
    (0, node_opcua_assert_1.assert)("1:TankLevelCondition" === conditionName);
    (0, node_opcua_assert_1.assert)("ExclusiveLimitAlarmType" === conditionTypeDefinition);
    (0, node_opcua_assert_1.assert)("UAExclusiveLimitAlarmImpl" === conditionJavascriptClass);
    // ----------------------------------------------------------------
    // tripAlarm that signals that the "Tank lid" is opened
    const tripAlarmType = addressSpace.findEventType("TripAlarmType");
    const tankTripCondition = null;
    // to
    // ---------------------------
    // create a retain condition
    // xx tankLevelCondition.currentBranch().setRetain(true);
    // xx tankLevelCondition.raiseNewCondition({message: "Tank is almost 70% full",
    //                                          severity: 100, quality: StatusCodes.Good});
    // -------------------------------------------------------------
    // Let's create a second variable with no Exclusive alarm
    // -------------------------------------------------------------
    const tankLevel2 = namespace.addVariable({
        browseName: "tankLevel2",
        componentOf: tank,
        dataType: "Double",
        description: "Fill level in percentage (0% to 100%) of the water tank",
        eventSourceOf: tank,
        value: { dataType: "Double", value: 0.5 }
    });
    const nonExclusiveLimitAlarmType = addressSpace.findEventType("NonExclusiveLimitAlarmType");
    if (!nonExclusiveLimitAlarmType) {
        throw new Error("!!");
    }
    const tankLevelCondition2 = namespace.instantiateNonExclusiveLimitAlarm(nonExclusiveLimitAlarmType, {
        browseName: "TankLevelCondition2",
        componentOf: tank,
        conditionName: "TankLevel2",
        conditionSource: tankLevel2,
        highHighLimit: 0.9,
        highLimit: 0.8,
        inputNode: tankLevel2, // the variable that will be monitored for change
        lowLimit: 0.2,
        optionals: [
            "ConfirmedState",
            "Confirm" // confirm state and confirm Method
        ]
    });
    (0, node_opcua_assert_1.assert)(tankLevel2.findReferences("HasCondition").length === 1);
    (0, node_opcua_assert_1.assert)(tankLevel2.findReferencesAsObject("HasCondition", true).length === 1);
    test.tankLevel = tankLevel;
    test.tankLevelCondition = tankLevelCondition;
    test.tankLevel2 = tankLevel2;
    test.tankLevelCondition2 = tankLevelCondition2;
    test.tankTripCondition = tankTripCondition;
}
//# sourceMappingURL=alarms_and_conditions_demo.js.map