"use strict";
/**
 * @module node-opcua-address-space
 */
// tslint:disable:max-classes-per-file
// tslint:disable:no-console
Object.defineProperty(exports, "__esModule", { value: true });
exports.UAVariableTypeImpl = void 0;
exports.topMostParentIsObjectTypeOrVariableType = topMostParentIsObjectTypeOrVariableType;
exports.assertUnusedChildBrowseName = assertUnusedChildBrowseName;
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_data_value_1 = require("node-opcua-data-value");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_utils_1 = require("node-opcua-utils");
const node_opcua_variant_1 = require("node-opcua-variant");
const session_context_1 = require("../source/session_context");
const _instantiate_helpers_1 = require("./_instantiate_helpers");
const base_node_impl_1 = require("./base_node_impl");
const base_node_private_1 = require("./base_node_private");
const tool_isSubtypeOf_1 = require("./tool_isSubtypeOf");
const tool_isSubtypeOf_2 = require("./tool_isSubtypeOf");
const tool_isSubtypeOf_3 = require("./tool_isSubtypeOf");
const check_value_rank_compatibility_1 = require("./check_value_rank_compatibility");
const get_basic_datatype_1 = require("./get_basic_datatype");
const debugLog = (0, node_opcua_debug_1.make_debugLog)(__filename);
const doDebug = (0, node_opcua_debug_1.checkDebugFlag)(__filename);
const warningLog = (0, node_opcua_debug_1.make_warningLog)(__filename);
const errorLog = (0, node_opcua_debug_1.make_errorLog)(__filename);
// eslint-disable-next-line prefer-const
let doTrace = (0, node_opcua_debug_1.checkDebugFlag)("INSTANTIATE");
const traceLog = errorLog;
function topMostParentIsObjectTypeOrVariableType(addressSpace, options) {
    if (options.copyAlsoModellingRules) {
        return true;
    }
    if (options.modellingRule) {
        return true;
    }
    const parent = options.propertyOf || options.componentOf;
    if (!parent) {
        return false;
    }
    const parentNode = addressSpace._coerceNode(parent);
    if (!parentNode) {
        return false;
    }
    let currentNode = parentNode;
    while (currentNode) {
        const nodeClass = parentNode.nodeClass;
        if (nodeClass === node_opcua_data_model_1.NodeClass.ObjectType || nodeClass === node_opcua_data_model_1.NodeClass.VariableType) {
            return true;
        }
        if (nodeClass === node_opcua_data_model_1.NodeClass.Object || nodeClass === node_opcua_data_model_1.NodeClass.Variable || nodeClass === node_opcua_data_model_1.NodeClass.Method) {
            /** */
        }
        currentNode = currentNode.findReferencesEx("HasChild", node_opcua_data_model_1.BrowseDirection.Inverse)[0]?.node;
    }
    return false;
}
function deprecate(func) {
    return func;
}
class UAVariableTypeImpl extends base_node_impl_1.BaseNodeImpl {
    nodeClass = node_opcua_data_model_1.NodeClass.VariableType;
    get subtypeOf() {
        return tool_isSubtypeOf_3.get_subtypeOf.call(this);
    }
    get subtypeOfObj() {
        return tool_isSubtypeOf_2.get_subtypeOfObj.call(this);
    }
    isSubtypeOf = (0, tool_isSubtypeOf_1.construct_isSubtypeOf)(UAVariableTypeImpl);
    /** @deprecated - use  isSubtypeOf instead */
    isSupertypeOf = deprecate((0, tool_isSubtypeOf_1.construct_isSubtypeOf)(UAVariableTypeImpl));
    isAbstract;
    dataType;
    valueRank;
    arrayDimensions;
    minimumSamplingInterval;
    value;
    historizing;
    constructor(options) {
        super(options);
        (0, node_opcua_variant_1.verifyRankAndDimensions)(options);
        this.valueRank = options.valueRank || -1;
        this.arrayDimensions = options.arrayDimensions || null;
        this.minimumSamplingInterval = 0;
        this.historizing = (0, node_opcua_utils_1.isNullOrUndefined)(options.historizing) ? false : options.historizing;
        this.isAbstract = (0, node_opcua_utils_1.isNullOrUndefined)(options.isAbstract) ? false : options.isAbstract;
        this.value = options.value; // optional default value for instances of this UAVariableType
        this.dataType = (0, node_opcua_nodeid_1.coerceNodeId)(options.dataType); // DataType (NodeId)
        if (options.value) {
            this.value = new node_opcua_variant_1.Variant(options.value);
        }
    }
    readAttribute(context, attributeId) {
        (0, node_opcua_assert_1.assert)(!context || context instanceof session_context_1.SessionContext);
        const options = {};
        switch (attributeId) {
            case node_opcua_data_model_1.AttributeIds.IsAbstract:
                options.value = { dataType: node_opcua_variant_1.DataType.Boolean, value: this.isAbstract ? true : false };
                options.statusCode = node_opcua_status_code_1.StatusCodes.Good;
                break;
            case node_opcua_data_model_1.AttributeIds.Value:
                if (Object.prototype.hasOwnProperty.call(this, "value") && this.value !== undefined) {
                    (0, node_opcua_assert_1.assert)(this.value.schema.name === "Variant");
                    options.value = this.value;
                    options.statusCode = node_opcua_status_code_1.StatusCodes.Good;
                }
                else {
                    debugLog(" warning Value not implemented");
                    options.value = { dataType: node_opcua_variant_1.DataType.Null };
                    options.statusCode = node_opcua_status_code_1.StatusCodes.BadAttributeIdInvalid;
                }
                break;
            case node_opcua_data_model_1.AttributeIds.DataType:
                (0, node_opcua_assert_1.assert)(this.dataType instanceof node_opcua_nodeid_1.NodeId);
                options.value = { dataType: node_opcua_variant_1.DataType.NodeId, value: this.dataType };
                options.statusCode = node_opcua_status_code_1.StatusCodes.Good;
                break;
            case node_opcua_data_model_1.AttributeIds.ValueRank:
                options.value = { dataType: node_opcua_variant_1.DataType.Int32, value: this.valueRank };
                options.statusCode = node_opcua_status_code_1.StatusCodes.Good;
                break;
            case node_opcua_data_model_1.AttributeIds.ArrayDimensions:
                (0, node_opcua_assert_1.assert)(Array.isArray(this.arrayDimensions) || this.arrayDimensions === null);
                options.value = {
                    arrayType: node_opcua_variant_1.VariantArrayType.Array,
                    dataType: node_opcua_variant_1.DataType.UInt32,
                    value: this.arrayDimensions
                };
                options.statusCode = node_opcua_status_code_1.StatusCodes.Good;
                break;
            default:
                return super.readAttribute(context, attributeId);
        }
        return new node_opcua_data_value_1.DataValue(options);
    }
    toString() {
        const options = new base_node_private_1.ToStringBuilder();
        base_node_private_1.UAVariableType_toString.call(this, options);
        return options.toString();
    }
    /**
     * instantiate an object of this UAVariableType
     * The instantiation takes care of object type inheritance when constructing inner properties
   
     * Note : HasComponent usage scope
     *
     * ```text
     *    Source          |     Destination
     * -------------------+---------------------------
     *  Object            | Object, Variable,Method
     *  ObjectType        |
     * -------------------+---------------------------
     *  DataVariable      | Variable
     *  DataVariableType  |
     * ```
     *
     *  see : OPCUA 1.03 page 44 $6.4 Instances of ObjectTypes and VariableTypes
     */
    instantiate(options) {
        const addressSpace = this.addressSpace;
        // xx assert(!this.isAbstract, "cannot instantiate abstract UAVariableType");
        (0, node_opcua_assert_1.assert)(options, "missing option object");
        (0, node_opcua_assert_1.assert)(typeof options.browseName === "string" || (options.browseName !== null && typeof options.browseName === "object"), "expecting a browse name");
        (0, node_opcua_assert_1.assert)(!Object.prototype.hasOwnProperty.call(options, "propertyOf"), "Use addressSpace#addVariable({ propertyOf: xxx}); to add a property");
        assertUnusedChildBrowseName(addressSpace, options);
        const baseVariableType = addressSpace.findVariableType("BaseVariableType");
        (0, node_opcua_assert_1.assert)(baseVariableType, "BaseVariableType must be defined in the address space");
        let dataType = options.dataType !== undefined ? options.dataType : this.dataType;
        // may be required (i.e YArrayItemType )
        dataType = this.resolveNodeId(dataType); // DataType (NodeId)
        (0, node_opcua_assert_1.assert)(dataType instanceof node_opcua_nodeid_1.NodeId);
        const valueRank = options.valueRank !== undefined ? options.valueRank : this.valueRank;
        const { result, errorMessage } = (0, check_value_rank_compatibility_1.checkValueRankCompatibility)(valueRank, this.valueRank);
        if (!result) {
            errorLog(errorMessage);
            throw new Error(errorMessage);
        }
        const arrayDimensions = options.arrayDimensions !== undefined ? options.arrayDimensions : this.arrayDimensions;
        // istanbul ignore next
        if (!dataType || dataType.isEmpty()) {
            warningLog(" options.dataType", options.dataType ? options.dataType.toString() : "<null>");
            warningLog(" this.dataType", this.dataType ? this.dataType.toString() : "<null>");
            throw new Error(" A valid dataType must be specified");
        }
        const copyAlsoModellingRules = topMostParentIsObjectTypeOrVariableType(addressSpace, options);
        const defaultDataType = this.dataType;
        // BadAttributeIdInvalid
        const defaultDataValue = this.readAttribute(null, node_opcua_data_model_1.AttributeIds.Value);
        const defaultValue = (defaultDataType.namespace === 0 && defaultDataType.value == 0) || defaultDataValue.statusCode.isNotGood()
            ? undefined
            : defaultDataValue.value;
        const opts = {
            arrayDimensions,
            browseName: options.browseName,
            componentOf: options.componentOf,
            dataType,
            description: options.description === undefined ? this.description?.clone() : options.description,
            displayName: options.displayName || "",
            eventSourceOf: options.eventSourceOf,
            minimumSamplingInterval: options.minimumSamplingInterval,
            modellingRule: options.modellingRule,
            nodeId: options.nodeId,
            notifierOf: options.notifierOf,
            organizedBy: options.organizedBy,
            typeDefinition: this.nodeId,
            value: options.value || defaultValue,
            valueRank
        };
        const namespace = options.namespace || addressSpace.getOwnNamespace();
        const instance = namespace.addVariable(opts);
        // xx assert(instance.minimumSamplingInterval === options.minimumSamplingInterval);
        const copyAlsoAllOptionals = options.copyAlsoAllOptionals || false;
        (0, _instantiate_helpers_1.initialize_properties_and_components)(instance, baseVariableType, this, copyAlsoModellingRules, copyAlsoAllOptionals, options.optionals);
        // if VariableType is a type of Structure DataType
        // we need to instantiate a dataValue
        // and create a bidirectional binding with the individual properties of this type
        instance.bindExtensionObject(options.extensionObject, { createMissingProp: false });
        (0, node_opcua_assert_1.assert)(instance.typeDefinition.toString() === this.nodeId.toString());
        instance.install_extra_properties();
        if (this._postInstantiateFunc) {
            this._postInstantiateFunc(instance, this);
        }
        return instance;
    }
    getBasicDataType() {
        return (0, get_basic_datatype_1._getBasicDataType)(this);
    }
}
exports.UAVariableTypeImpl = UAVariableTypeImpl;
/**

 * returns true if the parent object has a child  with the provided browseName
 * @param parent
 * @param childBrowseName
 */
function hasChildWithBrowseName(parent, childBrowseName) {
    if (!parent) {
        throw Error("Internal error");
    }
    // extract children
    const children = parent.findReferencesAsObject("HasChild", true);
    return (children.filter((child) => {
        return child.browseName.name?.toString() === childBrowseName.name?.toString();
    }).length > 0);
}
function getParent(addressSpace, options) {
    const parent = options.componentOf || options.organizedBy;
    if (parent instanceof node_opcua_nodeid_1.NodeId) {
        return addressSpace.findNode(parent);
    }
    return parent;
}
function assertUnusedChildBrowseName(addressSpace, options) {
    const resolveOptionalObject = (node) => node ? addressSpace._coerceNode(node) || undefined : undefined;
    options.componentOf = resolveOptionalObject(options.componentOf);
    options.organizedBy = resolveOptionalObject(options.organizedBy);
    (0, node_opcua_assert_1.assert)(!(options.componentOf && options.organizedBy));
    const parent = getParent(addressSpace, options);
    if (!parent) {
        return;
    }
    (0, node_opcua_assert_1.assert)(parent !== null && typeof parent === "object");
    if (!(parent instanceof base_node_impl_1.BaseNodeImpl)) {
        throw new Error("Invalid parent  parent is " + parent.constructor.name);
    }
    // istanbul ignore next
    // verify that no components already exists in parent
    if (parent && hasChildWithBrowseName(parent, (0, node_opcua_data_model_1.coerceQualifiedName)(options.browseName))) {
        throw new Error("object " +
            parent.browseName.name.toString() +
            " have already a child with browseName " +
            options.browseName.toString());
    }
}
//# sourceMappingURL=ua_variable_type_impl.js.map