"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getProxyTarget = getProxyTarget;
exports._touchValue = _touchValue;
exports.propagateTouchValueUpward = propagateTouchValueUpward;
exports.propagateTouchValueDownward = propagateTouchValueDownward;
exports.setExtensionObjectPartialValue = setExtensionObjectPartialValue;
exports._installExtensionObjectBindingOnProperties = _installExtensionObjectBindingOnProperties;
exports._bindExtensionObject = _bindExtensionObject;
exports._bindExtensionObjectArrayOrMatrix = _bindExtensionObjectArrayOrMatrix;
exports.getElement = getElement;
exports.setElement = setElement;
exports.incrementElement = incrementElement;
exports.extractPartialData = extractPartialData;
exports.propagateTouchValueDownwardArray = propagateTouchValueDownwardArray;
const node_opcua_assert_1 = __importDefault(require("node-opcua-assert"));
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_date_time_1 = require("node-opcua-date-time");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_extension_object_1 = require("node-opcua-extension-object");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_utils_1 = require("node-opcua-utils");
const node_opcua_variant_1 = require("node-opcua-variant");
const ua_variable_impl_1 = require("./ua_variable_impl");
const idx_iterator_1 = require("./idx_iterator");
const doDebug = (0, node_opcua_debug_1.checkDebugFlag)(__filename);
const debugLog = (0, node_opcua_debug_1.make_debugLog)(__filename);
const warningLog = (0, node_opcua_debug_1.make_warningLog)(__filename);
const errorLog = (0, node_opcua_debug_1.make_errorLog)(__filename);
function w(str, n) {
    return str.padEnd(n).substring(n);
}
function isProxy(ext) {
    return ext.$isProxy ? true : false;
}
function getProxyVariable(ext) {
    (0, node_opcua_assert_1.default)(isProxy(ext));
    return ext.$variable;
}
function getProxyVariableForProp(ext, prop) {
    const uaVariable = getProxyVariable(ext);
    if (!uaVariable)
        return undefined;
    return uaVariable[prop];
}
function getProxyTarget(ext) {
    (0, node_opcua_assert_1.default)(isProxy(ext));
    const target = ext.$proxyTarget;
    if (target && isProxy(target)) {
        return getProxyTarget(target);
    }
    return target;
}
function unProxy(ext) {
    return isProxy(ext) ? getProxyTarget(ext) : ext;
}
function _extensionObjectFieldGetter(getVariable, target, key /*, receiver*/) {
    if (key === "$isProxy") {
        return true;
    }
    if (key === "$proxyTarget") {
        return target;
    }
    if (key === "$variable") {
        return getVariable();
    }
    if (target[key] === undefined) {
        return undefined;
    }
    return target[key];
}
function _extensionObjectFieldSetter(getVariable, target, key, value /*, receiver*/) {
    target[key] = value;
    if (isProxy(target)) {
        return true;
    }
    const uaVariable = getVariable();
    if (!uaVariable)
        return true;
    const child = uaVariable[key];
    if (child && child.touchValue) {
        child.touchValue();
    }
    return true; // true means the set operation has succeeded
}
function makeHandler(getVariable) {
    const handler = {
        get: _extensionObjectFieldGetter.bind(null, getVariable),
        set: _extensionObjectFieldSetter.bind(null, getVariable)
    };
    return handler;
}
/**
 * inconditionnaly change the time stamp of the variable
 * if the variable is being listened to, and if the minimumSamplingInterval is exactly zero,
 * then the change will be reported to the observer
 *
 */
function _touchValue(property, now) {
    property.$dataValue.sourceTimestamp = now.timestamp;
    property.$dataValue.sourcePicoseconds = now.picoseconds;
    property.$dataValue.serverTimestamp = now.timestamp;
    property.$dataValue.serverPicoseconds = now.picoseconds;
    // don't change statusCode ! property.$dataValue.statusCode = StatusCodes.Good;
    if (property.listenerCount("value_changed") > 0) {
        property.emit("value_changed", property.$dataValue.clone());
    }
}
function propagateTouchValueUpward(self, now, cache) {
    _touchValue(self, now);
    if (self.parent && self.parent.nodeClass === node_opcua_data_model_1.NodeClass.Variable) {
        const parentVar = self.parent;
        if (!parentVar.isExtensionObject())
            return;
        if (cache) {
            if (cache.has(parentVar))
                return;
            cache.add(parentVar);
        }
        propagateTouchValueUpward(parentVar, now, cache);
    }
}
function propagateTouchValueDownward(self, now, cache) {
    if (!self.isExtensionObject())
        return;
    // also propagate changes to embeded variables
    const dataTypeNode = self.getDataTypeNode();
    const definition = dataTypeNode.getStructureDefinition();
    for (const field of definition.fields || []) {
        const property = self.getChildByName(field.name);
        if (property) {
            if (cache) {
                if (cache.has(property)) {
                    continue;
                }
                cache.add(property);
            }
            _touchValue(property, now);
            // to do cascade recursivelly ?
        }
    }
}
function setExtensionObjectPartialValue(node, partialObject, sourceTimestamp) {
    const variablesToUpdate = new Set();
    const extensionObject = node.$extensionObject;
    if (!extensionObject) {
        throw new Error("setExtensionObjectValue node has no extension object " + node.browseName.toString());
    }
    function _update_extension_object(extObject, partialObject1) {
        const keys = Object.keys(partialObject1);
        for (const prop of keys) {
            if (extObject[prop] instanceof Object) {
                _update_extension_object(extObject[prop], partialObject1[prop]);
            }
            else {
                if (isProxy(extObject)) {
                    // collect element we have to update
                    const target = getProxyTarget(extObject);
                    (0, node_opcua_assert_1.default)(!isProxy(target), "something wierd!");
                    target[prop] = partialObject1[prop];
                    const variable = getProxyVariableForProp(extObject, prop);
                    variable && variablesToUpdate.add(variable);
                }
                else {
                    extObject[prop] = partialObject1[prop];
                }
            }
        }
    }
    _update_extension_object(extensionObject, partialObject);
    const now = sourceTimestamp || (0, node_opcua_date_time_1.getCurrentClock)();
    const cache = new Set();
    for (const c of variablesToUpdate) {
        if (cache.has(c))
            continue;
        propagateTouchValueUpward(c, now, cache);
        propagateTouchValueDownward(c, now, cache);
        cache.add(c);
    }
}
function getOrCreateProperty(variableNode, field, options) {
    const dt = variableNode.getDataTypeNode();
    // the namespace for the structure browse name elements
    const structureNamespace = dt.nodeId.namespace;
    const components = variableNode.getComponents();
    let property;
    const selectedComponents = components.filter((f) => f instanceof ua_variable_impl_1.UAVariableImpl && f.browseName.name.toString() === field.name);
    // istanbul ignore next
    if (field.dataType.value === node_opcua_variant_1.DataType.Variant) {
        // this means that any type of extensions being used here
        debugLog("Warning : variant is not supported in ExtensionObject");
    }
    if (selectedComponents.length === 1) {
        property = selectedComponents[0];
        /* istanbul ignore next */
    }
    else {
        if (!options.createMissingProp) {
            return null;
        }
        debugLog("adding missing array variable", field.name, variableNode.browseName.toString(), variableNode.nodeId.toString());
        // todo: Handle array appropriately...
        (0, node_opcua_assert_1.default)(selectedComponents.length === 0);
        // create a variable (Note we may use ns=1;s=parentName/0:PropertyName)
        property = variableNode.namespace.addVariable({
            browseName: { namespaceIndex: structureNamespace, name: field.name.toString() },
            componentOf: variableNode,
            dataType: field.dataType,
            minimumSamplingInterval: variableNode.minimumSamplingInterval,
            accessLevel: variableNode.accessLevel,
            accessRestrictions: variableNode.accessRestrictions,
            rolePermissions: variableNode.rolePermissions
        });
        (0, node_opcua_assert_1.default)(property.minimumSamplingInterval === variableNode.minimumSamplingInterval);
    }
    return property;
}
function prepareVariantValue(dataType, value) {
    if ((dataType === node_opcua_variant_1.DataType.Int32 || dataType === node_opcua_variant_1.DataType.UInt32) && value && value.key) {
        value = value.value;
    }
    return value;
}
function installExt(uaVariable, ext) {
    ext = unProxy(ext);
    uaVariable.$extensionObject = new Proxy(ext, makeHandler(() => uaVariable));
    const addressSpace = uaVariable.addressSpace;
    const definition = uaVariable.dataTypeObj.getStructureDefinition();
    const structure = addressSpace.findDataType("Structure");
    for (const field of definition.fields || []) {
        if (field.dataType) {
            const dataTypeNode = addressSpace.findDataType(field.dataType);
            // istanbul ignore next
            if (dataTypeNode && dataTypeNode.isSubtypeOf(structure)) {
                // sub structure .. let make an handler too
                const camelCaseName = (0, node_opcua_utils_1.lowerFirstLetter)(field.name);
                const subExtObj = uaVariable.$extensionObject[camelCaseName];
                if (subExtObj) {
                    uaVariable.$extensionObject[camelCaseName] = new Proxy(subExtObj, makeHandler(() => {
                        return uaVariable.getComponentByName(field.name);
                    }));
                }
                else {
                    doDebug && warningLog("extension object is null");
                }
            }
        }
    }
}
function _installExtensionObjectBindingOnProperties(uaVariable, options) {
    // may be extension object mechanism has alreday been install
    // in this case we just need to rebind the properties...
    if (uaVariable.$extensionObject) {
        const extObj = uaVariable.$extensionObject;
        uaVariable.bindExtensionObject(extObj, { createMissingProp: true, force: true });
        return;
    }
    if (uaVariable.$$extensionObjectArray) {
        const extObj = uaVariable.$$extensionObjectArray;
        _bindExtensionObjectArrayOrMatrix(uaVariable, extObj, { createMissingProp: true, force: true });
        return;
    }
    const dataValue = uaVariable.readValue();
    const extObj = dataValue.value.value;
    if (extObj instanceof node_opcua_extension_object_1.ExtensionObject) {
        uaVariable.bindExtensionObject(extObj, { createMissingProp: true, force: true });
    }
    else if (extObj instanceof Array) {
        // istanbul ignore else
        if (dataValue.value.arrayType === node_opcua_variant_1.VariantArrayType.Array || dataValue.value.arrayType === node_opcua_variant_1.VariantArrayType.Matrix) {
            _bindExtensionObjectArrayOrMatrix(uaVariable, extObj, { createMissingProp: true, force: true });
        }
        else {
            throw new Error("Internal Error, unexpected case");
        }
    }
}
function _installFields2(uaVariable, { get, set }, options) {
    options = options || { createMissingProp: false };
    const dt = uaVariable.getDataTypeNode();
    const definition = dt.getStructureDefinition();
    for (const field of definition.fields || []) {
        if (node_opcua_nodeid_1.NodeId.sameNodeId(node_opcua_nodeid_1.NodeId.nullNodeId, field.dataType)) {
            warningLog("field.dataType is null ! ", field.name, node_opcua_nodeid_1.NodeId.nullNodeId.toString());
            warningLog(field.toString());
            warningLog(" dataType replaced with BaseDataType ");
            warningLog(definition.toString());
            field.dataType = uaVariable.resolveNodeId("BaseDataType");
        }
        const propertyNode = getOrCreateProperty(uaVariable, field, options);
        if (!propertyNode) {
            continue;
        }
        propertyNode.$dataValue.statusCode = node_opcua_status_code_1.StatusCodes.Good;
        propertyNode.$dataValue.sourceTimestamp = uaVariable.$dataValue.sourceTimestamp;
        propertyNode.$dataValue.sourcePicoseconds = uaVariable.$dataValue.sourcePicoseconds;
        propertyNode.$dataValue.serverTimestamp = uaVariable.$dataValue.serverTimestamp;
        propertyNode.$dataValue.serverPicoseconds = uaVariable.$dataValue.serverPicoseconds;
        propertyNode.$dataValue.value.dataType = propertyNode.dataTypeObj.basicDataType;
        propertyNode.$dataValue.value.arrayType =
            propertyNode.valueRank === -1
                ? node_opcua_variant_1.VariantArrayType.Scalar
                : propertyNode.valueRank === 1
                    ? node_opcua_variant_1.VariantArrayType.Array
                    : node_opcua_variant_1.VariantArrayType.Matrix;
        propertyNode.$dataValue.value.dimensions = propertyNode.valueRank > 1 ? propertyNode.arrayDimensions : null;
        const fieldName = field.name;
        installDataValueGetter(propertyNode, () => get(fieldName));
        (0, node_opcua_assert_1.default)(propertyNode._inner_replace_dataValue);
        propertyNode._inner_replace_dataValue = (dataValue, indexRange) => {
            /** */
            const sourceTime = (0, node_opcua_date_time_1.coerceClock)(dataValue.sourceTimestamp, dataValue.sourcePicoseconds);
            const value = dataValue.value.value;
            set(field.name, value, sourceTime);
            propertyNode.touchValue(sourceTime);
        };
        if (propertyNode.dataTypeObj.basicDataType === node_opcua_variant_1.DataType.ExtensionObject) {
            _installFields2(propertyNode, {
                get: (fieldName) => {
                    const mainFieldName = field.name;
                    return get(mainFieldName)[(0, node_opcua_utils_1.lowerFirstLetter)(fieldName)];
                },
                set: (fieldName, value, sourceTime) => {
                    const mainFieldName = field.name;
                    get(mainFieldName)[(0, node_opcua_utils_1.lowerFirstLetter)(fieldName)] = value;
                }
            }, options);
        }
    }
}
function installDataValueGetter(propertyNode, get) {
    Object.defineProperty(propertyNode.$dataValue.value, "value", { get });
    const $ = propertyNode.$dataValue;
    Object.defineProperty(propertyNode, "$dataValue", {
        get() {
            return $;
        },
        set: (value) => {
            throw new Error("$dataValue is now frozen and should not be modified this way !\n contact sterfive.com");
        }
    });
}
function isVariableContainingExtensionObject(uaVariable) {
    const addressSpace = uaVariable.addressSpace;
    const structure = addressSpace.findDataType("Structure");
    if (!structure) {
        // the addressSpace is limited and doesn't provide extension object
        // bindExtensionObject cannot be performed and shall finish here.
        return false;
    }
    (0, node_opcua_assert_1.default)(structure.browseName.toString() === "Structure", "expecting DataType Structure to be in IAddressSpace");
    const dt = uaVariable.getDataTypeNode();
    if (!dt.isSubtypeOf(structure)) {
        return false;
    }
    return true;
}
function _innerBindExtensionObjectScalar(uaVariable, { get, set, setField }, options) {
    uaVariable.$dataValue.statusCode = node_opcua_status_code_1.StatusCodes.Good;
    uaVariable.$dataValue.value.dataType = node_opcua_variant_1.DataType.ExtensionObject;
    uaVariable.$dataValue.value.arrayType = node_opcua_variant_1.VariantArrayType.Scalar;
    uaVariable.setValueFromSource = function (variant) {
        setExtensionObjectPartialValue(this, variant.value);
    };
    installDataValueGetter(uaVariable, get);
    uaVariable.$set_ExtensionObject = set;
    _installFields2(uaVariable, {
        get: (fieldName) => {
            const extObj = get();
            return extObj[(0, node_opcua_utils_1.lowerFirstLetter)(fieldName)];
        },
        set: (fieldName, value, sourceTime) => {
            setField(fieldName, value, sourceTime);
        }
    }, options);
}
// eslint-disable-next-line complexity
function _bindExtensionObject(uaVariable, optionalExtensionObject, options) {
    options = options || { createMissingProp: false };
    // istanbul ignore next
    if (!isVariableContainingExtensionObject(uaVariable)) {
        return null;
    }
    // istanbul ignore next
    if (optionalExtensionObject && uaVariable.valueRank === 0) {
        warningLog(uaVariable.browseName.toString() +
            ": valueRank was zero but needed to be adjusted to -1 (Scalar) in bindExtensionObject");
        uaVariable.valueRank = -1;
    }
    const addressSpace = uaVariable.addressSpace;
    let extensionObject_;
    // istanbul ignore next
    if (uaVariable.valueRank !== -1 && uaVariable.valueRank !== 1) {
        throw new Error("Cannot bind an extension object here, valueRank must be scalar (-1) or one-dimensional (1)");
    }
    // istanbul ignore next
    doDebug && debugLog(" ------------------------------ binding ", uaVariable.browseName.toString(), uaVariable.nodeId.toString());
    // ignore bindExtensionObject on sub extension object, bindExtensionObject has to be called from the top most object
    if (!options.force &&
        uaVariable.parent &&
        (uaVariable.parent.nodeClass === node_opcua_data_model_1.NodeClass.Variable || uaVariable.parent.nodeClass === node_opcua_data_model_1.NodeClass.VariableType)) {
        const parentDataType = uaVariable.parent.dataType;
        const dataTypeNode = addressSpace.findNode(parentDataType);
        const structure = addressSpace.findDataType("Structure");
        // istanbul ignore next
        if (dataTypeNode && dataTypeNode.isSubtypeOf(structure)) {
            // warningLog(
            //     "Ignoring bindExtensionObject on sub extension object",
            //     "child=",
            //     self.browseName.toString(),
            //     "parent=",
            //     self.parent.browseName.toString()
            // );
            return null;
        }
    }
    // -------------------- make sure we do not bind a variable twice ....
    if (uaVariable.$extensionObject && !optionalExtensionObject) {
        // istanbul ignore next
        if (!uaVariable.checkExtensionObjectIsCorrect(uaVariable.$extensionObject)) {
            warningLog("on node : ", uaVariable.browseName.toString(), uaVariable.nodeId.toString(), "dataType=", uaVariable.dataType.toString({ addressSpace: uaVariable.addressSpace }));
            warningLog(uaVariable.$extensionObject?.toString());
            throw new Error("bindExtensionObject: $extensionObject is incorrect: we are expecting a " +
                uaVariable.dataType.toString({ addressSpace: uaVariable.addressSpace }) +
                " but we got a " +
                uaVariable.$extensionObject?.schema.name);
        }
        return uaVariable.$extensionObject;
    }
    if (uaVariable.dataTypeObj.isAbstract) {
        // istanbul ignore next
        if (!optionalExtensionObject) {
            warningLog("Warning the DataType associated with this Variable is abstract ", uaVariable.dataTypeObj.browseName.toString());
            warningLog("You need to provide a extension object yourself ");
            throw new Error("bindExtensionObject requires a extensionObject as associated dataType is only abstract");
        }
    }
    const s = uaVariable.readValue();
    if (s.value && s.value.dataType === node_opcua_variant_1.DataType.ExtensionObject && s.value.value && optionalExtensionObject) {
        // we want to replace the extension object
        s.value.value = null;
    }
    innerBindExtensionObject();
    (0, node_opcua_assert_1.default)(uaVariable.$extensionObject instanceof Object);
    return uaVariable.$extensionObject;
    function innerBindExtensionObject() {
        if (s.value && (s.value.dataType === node_opcua_variant_1.DataType.Null || (s.value.dataType === node_opcua_variant_1.DataType.ExtensionObject && !s.value.value))) {
            if (uaVariable.valueRank === -1 /** Scalar */) {
                extensionObject_ = optionalExtensionObject || addressSpace.constructExtensionObject(uaVariable.dataType, {});
                installExt(uaVariable, extensionObject_);
                _innerBindExtensionObjectScalar(uaVariable, {
                    get: () => uaVariable.$extensionObject,
                    set: (value) => installExt(uaVariable, value),
                    setField: (fieldName, value) => {
                        const extObj = uaVariable.$extensionObject;
                        getProxyTarget(extObj)[(0, node_opcua_utils_1.lowerFirstLetter)(fieldName)] = value;
                    }
                }, options);
                return;
            }
            else if (uaVariable.valueRank === 1 /** Array */) {
                throw new Error("Should not get there ! Please fix me");
            }
            else {
                errorLog(uaVariable.toString());
                errorLog("Unsupported case ! valueRank= ", uaVariable.valueRank);
            }
        }
        else {
            // verify that variant has the correct type
            (0, node_opcua_assert_1.default)(s.value.dataType === node_opcua_variant_1.DataType.ExtensionObject);
            installExt(uaVariable, s.value.value);
            _innerBindExtensionObjectScalar(uaVariable, {
                get: () => uaVariable.$extensionObject,
                set: (value) => installExt(uaVariable, value),
                setField: (fieldName, value) => {
                    const extObj = uaVariable.$extensionObject;
                    getProxyTarget(extObj)[(0, node_opcua_utils_1.lowerFirstLetter)(fieldName)] = value;
                }
            }, options);
        }
    }
}
const getIndexAsText = (index) => {
    if (typeof index === "number")
        return `${index}`;
    return `${index.map((a) => a.toString()).join(",")}`;
};
const composeBrowseNameAndNodeId = (uaVariable, indexes) => {
    const iAsText = getIndexAsText(indexes);
    const browseName = (0, node_opcua_data_model_1.coerceQualifiedName)(iAsText);
    let nodeId;
    if (uaVariable.nodeId.identifierType === node_opcua_nodeid_1.NodeIdType.STRING) {
        nodeId = new node_opcua_nodeid_1.NodeId(node_opcua_nodeid_1.NodeIdType.STRING, uaVariable.nodeId.value + `[${iAsText}]`, uaVariable.nodeId.namespace);
    }
    return { browseName, nodeId };
};
// eslint-disable-next-line max-statements, complexity
function _bindExtensionObjectArrayOrMatrix(uaVariable, optionalExtensionObjectArray, options) {
    options = options || { createMissingProp: false };
    options.createMissingProp = options.createMissingProp || false;
    // istanbul ignore next
    if (uaVariable.valueRank < 1) {
        throw new Error("Variable must be a MultiDimensional array");
    }
    const arrayDimensions = uaVariable.arrayDimensions || [];
    // istanbul ignore next
    if (!isVariableContainingExtensionObject(uaVariable)) {
        return [];
    }
    if (!optionalExtensionObjectArray && uaVariable.$dataValue.value.value) {
        (0, node_opcua_assert_1.default)(Array.isArray(uaVariable.$dataValue.value.value));
        optionalExtensionObjectArray = uaVariable.$dataValue.value.value;
    }
    if ((arrayDimensions.length === 0 || (arrayDimensions.length === 1 && arrayDimensions[0] === 0)) &&
        optionalExtensionObjectArray) {
        arrayDimensions[0] = optionalExtensionObjectArray.length;
    }
    const totalLength = arrayDimensions.reduce((p, c) => p * c, 1);
    /** */
    const addressSpace = uaVariable.addressSpace;
    if (optionalExtensionObjectArray && optionalExtensionObjectArray.length != 0) {
        if (optionalExtensionObjectArray.length !== totalLength) {
            throw new Error(`optionalExtensionObjectArray must have the expected number of element matching ${arrayDimensions} but was ${optionalExtensionObjectArray.length}`);
        }
    }
    if (!optionalExtensionObjectArray || optionalExtensionObjectArray.length == 0) {
        optionalExtensionObjectArray = [];
        for (let i = 0; i < totalLength; i++) {
            optionalExtensionObjectArray[i] = addressSpace.constructExtensionObject(uaVariable.dataType, {});
        }
    }
    uaVariable.$$extensionObjectArray = optionalExtensionObjectArray;
    uaVariable.$dataValue.value.arrayType = uaVariable.valueRank === 1 ? node_opcua_variant_1.VariantArrayType.Array : node_opcua_variant_1.VariantArrayType.Matrix;
    uaVariable.$dataValue.value.dimensions = uaVariable.valueRank === 1 ? null : uaVariable.arrayDimensions || [];
    uaVariable.$dataValue.value.dataType = node_opcua_variant_1.DataType.ExtensionObject;
    uaVariable.$dataValue.value.value = uaVariable.$$extensionObjectArray;
    // make sure uaVariable.$dataValue cannot be inadvertantly changed from this point onward
    const $dataValue = uaVariable.$dataValue;
    Object.defineProperty(uaVariable, "$dataValue", {
        get() {
            return $dataValue;
        },
        set() {
            throw new Error("$dataValue is now sealed , you should not change internal $dataValue!");
        },
        //    writable: true,
        enumerable: true,
        configurable: true
    });
    uaVariable.bindVariable({
        get: () => uaVariable.$dataValue.value
    }, true);
    const namespace = uaVariable.namespace;
    const indexIterator = new idx_iterator_1.IndexIterator(arrayDimensions);
    for (let i = 0; i < totalLength; i++) {
        const index = indexIterator.next();
        const { browseName, nodeId } = composeBrowseNameAndNodeId(uaVariable, index);
        let uaElement = uaVariable.getComponentByName(browseName);
        if (!uaElement) {
            if (!options.createMissingProp) {
                continue;
            }
            uaElement = namespace.addVariable({
                browseName,
                nodeId,
                componentOf: uaVariable,
                dataType: uaVariable.dataType,
                valueRank: -1,
                accessLevel: uaVariable.accessLevel
            });
        }
        uaElement.$dataValue.statusCode = node_opcua_status_code_1.StatusCodes.Good;
        uaElement.$dataValue.sourceTimestamp = uaVariable.$dataValue.sourceTimestamp;
        uaElement.$dataValue.sourcePicoseconds = uaVariable.$dataValue.sourcePicoseconds;
        uaElement.$dataValue.serverTimestamp = uaVariable.$dataValue.serverTimestamp;
        uaElement.$dataValue.serverPicoseconds = uaVariable.$dataValue.serverPicoseconds;
        uaElement.$dataValue.value.dataType = node_opcua_variant_1.DataType.ExtensionObject;
        uaElement.$dataValue.value.arrayType = node_opcua_variant_1.VariantArrayType.Scalar;
        {
            const capturedIndex = i;
            const capturedUaElement = uaElement;
            _innerBindExtensionObjectScalar(uaElement, {
                get: () => uaVariable.$$extensionObjectArray[capturedIndex],
                set: (newValue, sourceTimestamp, cache) => {
                    (0, node_opcua_assert_1.default)(!isProxy(uaVariable.$$extensionObjectArray[capturedIndex]));
                    uaVariable.$$extensionObjectArray[capturedIndex] = newValue;
                    // istanbul ignore next
                    if (uaVariable.$$extensionObjectArray !== uaVariable.$dataValue.value.value) {
                        warningLog("uaVariable", uaVariable.nodeId.toString());
                        warningLog("Houston! We have a problem ");
                    }
                    propagateTouchValueDownward(capturedUaElement, sourceTimestamp, cache);
                    propagateTouchValueUpward(capturedUaElement, sourceTimestamp, cache);
                },
                setField: (fieldName, newValue, sourceTimestamp, cache) => {
                    // istanbul ignore next doDebug && debugLog("setField", fieldName, newValue, sourceTimestamp, cache);
                    const extObj = uaVariable.$$extensionObjectArray[capturedIndex];
                    (isProxy(extObj) ? getProxyTarget(extObj) : extObj)[(0, node_opcua_utils_1.lowerFirstLetter)(fieldName)] = newValue;
                    propagateTouchValueUpward(capturedUaElement, sourceTimestamp, cache);
                }
            }, { ...options, force: true });
        }
    }
    return uaVariable.$$extensionObjectArray;
}
function getElement(path, data) {
    if (typeof path === "string") {
        path = path.split(".");
    }
    let a = data;
    for (const e of path) {
        a = a[e];
    }
    return a;
}
function setElement(path, data, value) {
    if (typeof path === "string") {
        path = path.split(".");
    }
    const last = path.pop();
    let a = data;
    for (const e of path) {
        a = a[e];
    }
    a[last] = value;
}
function incrementElement(path, data) {
    const value = getElement(path, data);
    setElement(path, data, value + 1);
}
function extractPartialData(path, extensionObject) {
    let name;
    if (typeof path === "string") {
        path = path.split(".");
    }
    (0, node_opcua_assert_1.default)(path instanceof Array);
    let i;
    // read partial value
    const partialData = {};
    let p = partialData;
    for (i = 0; i < path.length - 1; i++) {
        name = path[i];
        p[name] = {};
        p = p[name];
    }
    name = path[path.length - 1];
    p[name] = 0;
    let c1 = partialData;
    let c2 = extensionObject;
    for (i = 0; i < path.length - 1; i++) {
        name = path[i];
        c1 = partialData[name];
        c2 = extensionObject[name];
    }
    name = path[path.length - 1];
    c1[name] = c2[name];
    return partialData;
}
function propagateTouchValueDownwardArray(uaVariable, now, cache) {
    if (!uaVariable.$$extensionObjectArray)
        return;
    const arrayDimensions = uaVariable.arrayDimensions || [];
    const totalLength = uaVariable.$$extensionObjectArray.length;
    const indexIterator = new idx_iterator_1.IndexIterator(arrayDimensions);
    for (let i = 0; i < totalLength; i++) {
        const index = indexIterator.next();
        const { browseName, nodeId } = composeBrowseNameAndNodeId(uaVariable, index);
        const uaElement = uaVariable.getComponentByName(browseName);
        if (uaElement?.nodeClass === node_opcua_data_model_1.NodeClass.Variable) {
            uaElement.touchValue(now);
            propagateTouchValueDownward(uaElement, now, cache);
        }
    }
}
//# sourceMappingURL=ua_variable_impl_ext_obj.js.map