"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.UAMethodImpl = void 0;
/**
 * @module node-opcua-address-space
 */
const util_1 = require("util");
const chalk_1 = __importDefault(require("chalk"));
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_data_model_2 = require("node-opcua-data-model");
const node_opcua_data_value_1 = require("node-opcua-data-value");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_service_call_1 = require("node-opcua-service-call");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_types_1 = require("node-opcua-types");
const node_opcua_variant_1 = require("node-opcua-variant");
const node_opcua_variant_2 = require("node-opcua-variant");
const node_opcua_address_space_base_1 = require("node-opcua-address-space-base");
const base_node_private_1 = require("./base_node_private");
const namespace_impl_1 = require("./namespace_impl");
const base_node_impl_1 = require("./base_node_impl");
const warningLog = (0, node_opcua_debug_1.make_warningLog)(__filename);
const debugLog = (0, node_opcua_debug_1.make_debugLog)(__filename);
const errorLog = (0, node_opcua_debug_1.make_errorLog)(__filename);
function default_check_valid_argument(arg) {
    return arg instanceof node_opcua_service_call_1.Argument;
}
class UAMethodImpl extends base_node_impl_1.BaseNodeImpl {
    static checkValidArgument(args) {
        return default_check_valid_argument(args);
    }
    nodeClass = node_opcua_data_model_2.NodeClass.Method;
    get typeDefinitionObj() {
        return super.typeDefinitionObj;
    }
    get parent() {
        return super.parent;
    }
    value;
    methodDeclarationId;
    _getExecutableFlag;
    _asyncExecutionFunction;
    constructor(options) {
        super(options);
        this.value = options.value;
        this.methodDeclarationId = options.methodDeclarationId;
    }
    /**
     *
     *
     */
    getExecutableFlag(context) {
        if (!this.isBound()) {
            return false;
        }
        if (this._getExecutableFlag) {
            return this._getExecutableFlag(context);
        }
        return true;
    }
    getUserExecutableFlag(context) {
        if (context && !context.checkPermission(this, node_opcua_types_1.PermissionType.Call)) {
            return false;
        }
        if (!this.getExecutableFlag(context))
            return false;
        return true;
    }
    /**
     *
     * @returns  true if the method is bound
     */
    isBound() {
        return typeof this._asyncExecutionFunction === "function";
    }
    readAttribute(context, attributeId, indexRange, dataEncoding) {
        const options = {};
        switch (attributeId) {
            case node_opcua_data_model_1.AttributeIds.Executable:
                options.value = { dataType: node_opcua_variant_2.DataType.Boolean, value: this.getExecutableFlag(context) };
                options.statusCode = node_opcua_status_code_1.StatusCodes.Good;
                break;
            case node_opcua_data_model_1.AttributeIds.UserExecutable:
                options.value = { dataType: node_opcua_variant_2.DataType.Boolean, value: this.getUserExecutableFlag(context) };
                options.statusCode = node_opcua_status_code_1.StatusCodes.Good;
                break;
            default:
                return base_node_impl_1.BaseNodeImpl.prototype.readAttribute.call(this, context, attributeId, indexRange, dataEncoding);
        }
        return new node_opcua_data_value_1.DataValue(options);
    }
    getInputArguments() {
        return this._getArguments("InputArguments");
    }
    getOutputArguments() {
        return this._getArguments("OutputArguments");
    }
    bindMethod(async_func) {
        (0, node_opcua_assert_1.assert)(typeof async_func === "function");
        if (async_func.length === 2) {
            async_func = (0, util_1.callbackify)(async_func);
        }
        (0, node_opcua_assert_1.assert)(async_func.length === 3, " a method with callback should have 3 arguments : got " + async_func.length);
        this._asyncExecutionFunction = async_func;
    }
    execute(object, inputArguments, context, callback) {
        // istanbul ignore next
        if (!callback) {
            throw new Error("execute need to be promisified");
        }
        (0, node_opcua_assert_1.assert)(inputArguments === null || Array.isArray(inputArguments));
        inputArguments = inputArguments || [];
        inputArguments = inputArguments.map(node_opcua_variant_1.Variant.coerce);
        (0, node_opcua_assert_1.assert)(inputArguments.length === 0 || inputArguments[0] instanceof node_opcua_variant_1.Variant);
        (0, node_opcua_assert_1.assert)(context !== null && typeof context === "object");
        (0, node_opcua_assert_1.assert)(typeof callback === "function");
        object = object || this.parent;
        // istanbul ignore next
        if (!object) {
            errorLog("UAMethod#execute expects a valid object");
            return callback(null, { statusCode: node_opcua_status_code_1.StatusCodes.BadInternalError });
        }
        if (object.nodeClass !== node_opcua_data_model_2.NodeClass.Object && object.nodeClass !== node_opcua_data_model_2.NodeClass.ObjectType) {
            warningLog("Method " +
                this.nodeId.toString() +
                " " +
                this.browseName.toString() +
                " called for a node that is not a Object/ObjectType but " +
                node_opcua_data_model_2.NodeClass[context.object.nodeClass]);
            return callback(null, { statusCode: node_opcua_status_code_1.StatusCodes.BadNodeIdInvalid });
        }
        if (!this._asyncExecutionFunction) {
            warningLog("Method " + this.nodeId.toString() + " " + this.browseName.toString() + " has not been bound");
            return callback(null, { statusCode: node_opcua_status_code_1.StatusCodes.BadInternalError });
        }
        if (!this.getExecutableFlag(context)) {
            warningLog("Method " + this.nodeId.toString() + " " + this.browseName.toString() + " is not executable");
            return callback(null, { statusCode: node_opcua_status_code_1.StatusCodes.BadNotExecutable });
        }
        if (context.isAccessRestricted(this)) {
            return callback(null, { statusCode: node_opcua_status_code_1.StatusCodes.BadSecurityModeInsufficient });
        }
        if (!context.checkPermission(this, node_opcua_types_1.PermissionType.Call)) {
            return callback(null, { statusCode: node_opcua_status_code_1.StatusCodes.BadUserAccessDenied });
        }
        // verify that input arguments are correct
        // todo :
        const inputArgumentDiagnosticInfos = [];
        context.object = object;
        try {
            this._asyncExecutionFunction.call(this, inputArguments, context, (err, callMethodResult) => {
                if (err) {
                    debugLog(err.message);
                    debugLog(err);
                }
                callMethodResult = callMethodResult || {};
                callMethodResult.statusCode = callMethodResult.statusCode || node_opcua_status_code_1.StatusCodes.Good;
                callMethodResult.outputArguments = callMethodResult.outputArguments || [];
                callMethodResult.inputArgumentResults =
                    callMethodResult.inputArgumentResults?.length === inputArguments?.length
                        ? callMethodResult.inputArgumentResults
                        : inputArguments?.map(() => node_opcua_status_code_1.StatusCodes.Good);
                callMethodResult.inputArgumentDiagnosticInfos =
                    callMethodResult.inputArgumentDiagnosticInfos || inputArgumentDiagnosticInfos;
                // verify that output arguments are correct according to schema
                // Todo : ...
                // const outputArgsDef = this.getOutputArguments();
                // xx assert(outputArgsDef.length === callMethodResponse.outputArguments.length,
                // xx     "_asyncExecutionFunction did not provide the expected number of output arguments");
                // to be continued ...
                callback(err, callMethodResult);
            });
        }
        catch (err) {
            if (util_1.types.isNativeError(err)) {
                warningLog(chalk_1.default.red("ERR in method  handler"), err.message);
                warningLog(err.stack);
            }
            const callMethodResponse = { statusCode: node_opcua_status_code_1.StatusCodes.BadInternalError };
            callback(err, callMethodResponse);
        }
    }
    clone(options, optionalFilter, extraInfo) {
        (0, node_opcua_assert_1.assert)(!options.componentOf || options.componentOf, "trying to create an orphan method ?");
        const addressSpace = this.addressSpace;
        options = {
            ...options,
            methodDeclarationId: this.nodeId
        };
        options.references = options.references || [];
        (0, namespace_impl_1._handle_hierarchy_parent)(addressSpace, options.references, options);
        if (!extraInfo) {
            extraInfo = (0, node_opcua_address_space_base_1.makeDefaultCloneExtraInfo)(this);
        }
        const clonedMethod = (0, base_node_private_1._clone)(this, UAMethodImpl, options, optionalFilter || node_opcua_address_space_base_1.defaultCloneFilter, extraInfo);
        clonedMethod._asyncExecutionFunction = this._asyncExecutionFunction;
        clonedMethod._getExecutableFlag = this._getExecutableFlag;
        if (options.componentOf) {
            const m = options.componentOf.getMethodByName(clonedMethod.browseName.name);
            (0, node_opcua_assert_1.assert)(m);
        }
        return clonedMethod;
    }
    _getArguments(name) {
        (0, node_opcua_assert_1.assert)(name === "InputArguments" || name === "OutputArguments");
        const argsVariable = this.getPropertyByName(name);
        if (!argsVariable) {
            return [];
        }
        (0, node_opcua_assert_1.assert)(argsVariable.nodeClass === node_opcua_data_model_2.NodeClass.Variable);
        const args = argsVariable.readValue().value.value;
        if (!args) {
            return [];
        }
        // a list of extension object
        (0, node_opcua_assert_1.assert)(Array.isArray(args));
        (0, node_opcua_assert_1.assert)(args.length === 0 || UAMethodImpl.checkValidArgument(args[0]));
        return args;
    }
}
exports.UAMethodImpl = UAMethodImpl;
// tslint:disable:no-var-requires
// tslint:disable:max-line-length
const thenify_ex_1 = require("thenify-ex");
UAMethodImpl.prototype.execute = (0, thenify_ex_1.withCallback)(UAMethodImpl.prototype.execute);
//# sourceMappingURL=ua_method_impl.js.map