"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodeIdManager = exports.NamespaceOptions = void 0;
/* eslint-disable max-depth */
/* eslint-disable max-statements */
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_address_space_base_1 = require("node-opcua-address-space-base");
const base_node_impl_1 = require("./base_node_impl");
const reference_impl_1 = require("./reference_impl");
const debugLog = (0, node_opcua_debug_1.make_debugLog)(__filename);
const warningLog = (0, node_opcua_debug_1.make_warningLog)(__filename);
exports.NamespaceOptions = {
    nodeIdNameSeparator: "-"
};
function isValidNodeClass(nodeClass) {
    return typeof node_opcua_data_model_1.NodeClass[nodeClass] === "string";
}
const regExp1 = /^(s|i|b|g)=/;
const regExp2 = /^ns=[0-9]+;(s|i|b|g)=/;
const hasEncoding = (0, node_opcua_nodeid_1.resolveNodeId)("HasEncoding");
function _filterAggregates(addressSpace, references) {
    const aggregatesRefType = addressSpace.findNode((0, node_opcua_nodeid_1.resolveNodeId)("Aggregates"));
    const hasEncodinRefType = addressSpace.findNode((0, node_opcua_nodeid_1.resolveNodeId)("HasEncoding"));
    const checkAggregate = (reference) => {
        if (reference.isForward)
            return false;
        const r = (0, reference_impl_1.resolveReferenceType)(addressSpace, reference);
        if (!r) {
            return false;
        }
        return r.isSubtypeOf(aggregatesRefType) || r.isSubtypeOf(hasEncodinRefType);
    };
    const candidates = references.filter(checkAggregate);
    (0, node_opcua_assert_1.assert)(candidates.length <= 1, "a node shall not have more than one parent (link to a parent with a reference derived from 'Aggregates')");
    if (candidates.length === 0) {
        return null;
    }
    const ref = candidates[0];
    if ((0, node_opcua_nodeid_1.sameNodeId)(ref.referenceType, hasEncoding)) {
        return [ref.nodeId, "_Encoding"];
    }
    return [ref.nodeId, ""];
}
function _findParentNodeId(addressSpace, options) {
    if (!options.references) {
        return null;
    }
    for (const ref of options.references) {
        ref._referenceType = addressSpace.findReferenceType(ref.referenceType);
        /* istanbul ignore next */
        if (!(0, base_node_impl_1.getReferenceType)(ref)) {
            throw new Error("Cannot find referenceType " + JSON.stringify(ref));
        }
        ref.referenceType = ref._referenceType.nodeId;
    }
    // find HasComponent, or has Property reverse
    return _filterAggregates(addressSpace, options.references);
}
function prepareName(browseName) {
    const m = browseName.name.toString().replace(/[ ]/g, "").replace(/(<|>)/g, "");
    return m;
}
class NodeIdManager {
    _cacheSymbolicName = {};
    _cacheSymbolicNameRev = new Set();
    _internal_id_counter;
    namespaceIndex;
    addressSpace;
    constructor(namespaceIndex, addressSpace) {
        this._internal_id_counter = 1000;
        this.namespaceIndex = namespaceIndex;
        this.addressSpace = addressSpace;
    }
    setSymbols(symbols) {
        function convertNodeClass(nodeClass) {
            return node_opcua_data_model_1.NodeClass[nodeClass];
        }
        const symbols2 = symbols.map((e) => [e[0], e[1], convertNodeClass(e[2])]);
        for (const [name, value, nodeClass] of symbols2) {
            this._cacheSymbolicName[name] = [value, nodeClass];
            this._cacheSymbolicNameRev.add(value);
        }
    }
    getSymbols() {
        const line = [];
        for (const [key, [value, nodeClass1]] of Object.entries(this._cacheSymbolicName)) {
            const node = this.addressSpace.findNode((0, node_opcua_nodeid_1.makeNodeId)(value, this.namespaceIndex));
            const nodeClass = node_opcua_data_model_1.NodeClass[nodeClass1 || node_opcua_data_model_1.NodeClass.Unspecified];
            line.push([key, value, nodeClass]);
        }
        return line;
    }
    getSymbolCSV() {
        const line = [];
        for (const [name, value, nodeClass] of this.getSymbols()) {
            line.push([name, value, nodeClass].join(";"));
        }
        return line.join("\n");
    }
    buildNewNodeId() {
        let nodeId;
        do {
            nodeId = (0, node_opcua_nodeid_1.makeNodeId)(this._internal_id_counter, this.namespaceIndex);
            this._internal_id_counter += 1;
        } while (this.addressSpace.findNode(nodeId) || this._isInCache(nodeId));
        return nodeId;
    }
    constructNodeId(options) {
        const compose = (left, right) => { return right ? (left ? left + '_' + right : right) : left; };
        const buildUpName2 = (nodeId, suffix) => {
            const namespaceIndex = nodeId.namespace;
            let name = "";
            let n = this.addressSpace.findNode(nodeId);
            while (n && n.nodeId.namespace === namespaceIndex) {
                const e = prepareName(n.browseName) + suffix;
                name = compose(e, name);
                n = n.parentNodeId ? this.addressSpace.findNode(n.parentNodeId) : null;
            }
            return name;
        };
        if (!options.nodeId && options.registerSymbolicNames) {
            const parentInfo = this.findParentNodeId(options);
            let fullParentName = "";
            if (parentInfo) {
                const [parentNodeId, suffix] = parentInfo;
                fullParentName = buildUpName2(parentNodeId, suffix);
            }
            const fullName = compose(fullParentName, prepareName(options.browseName));
            if (this._cacheSymbolicName[fullName]) {
                return (0, node_opcua_nodeid_1.makeNodeId)(this._cacheSymbolicName[fullName][0], this.namespaceIndex);
            }
            const nodeId = this._constructNodeId(options);
            if (nodeId.identifierType === node_opcua_nodeid_1.NodeIdType.NUMERIC) {
                this._cacheSymbolicName[fullName] = [nodeId.value, options.nodeClass];
                this._cacheSymbolicNameRev.add(nodeId.value);
            }
            return nodeId;
        }
        return this._constructNodeId(options);
    }
    _constructNodeId(options) {
        const resolveNodeIdEx = (nodeId) => (nodeId && typeof nodeId == "object" && nodeId instanceof node_opcua_address_space_base_1.BaseNode) ? nodeId.nodeId : (0, node_opcua_nodeid_1.resolveNodeId)(nodeId);
        let nodeId = options.nodeId;
        if (!nodeId) {
            const parentInfo = this.findParentNodeId(options);
            if (parentInfo) {
                const [parentNodeId, linkName] = parentInfo;
                const name = prepareName(options.browseName);
                nodeId = null;
                if (parentNodeId.identifierType === node_opcua_nodeid_1.NodeId.NodeIdType.STRING) {
                    // combining string nodeId => not stored in chache
                    const childName = parentNodeId.value + exports.NamespaceOptions.nodeIdNameSeparator + name;
                    nodeId = new node_opcua_nodeid_1.NodeId(node_opcua_nodeid_1.NodeId.NodeIdType.STRING, childName, parentNodeId.namespace);
                    return nodeId;
                }
            }
        }
        else if (typeof nodeId === "string") {
            if (this.namespaceIndex !== 0) {
                if (nodeId.match(regExp2)) {
                    // nothing
                }
                else if (nodeId.match(regExp1)) {
                    nodeId = "ns=" + this.namespaceIndex + ";" + nodeId;
                    // } else {
                    //     nodeId = this._getOrCreateFromName(nodeId, nodeClass);
                }
            }
        }
        nodeId = nodeId || this.buildNewNodeId();
        if (nodeId instanceof node_opcua_nodeid_1.NodeId) {
            (0, node_opcua_assert_1.assert)(nodeId.namespace === this.namespaceIndex);
            return nodeId;
        }
        nodeId = resolveNodeIdEx(nodeId);
        (0, node_opcua_assert_1.assert)(nodeId.namespace === this.namespaceIndex);
        return nodeId;
    }
    findParentNodeId(options) {
        return _findParentNodeId(this.addressSpace, options);
    }
    _isInCache(nodeId) {
        if (nodeId.namespace !== this.namespaceIndex || nodeId.identifierType !== node_opcua_nodeid_1.NodeIdType.NUMERIC)
            return false;
        return this._cacheSymbolicNameRev.has(nodeId.value) ? true : false;
    }
}
exports.NodeIdManager = NodeIdManager;
//# sourceMappingURL=nodeid_manager.js.map