"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createExtObjArrayNode = createExtObjArrayNode;
exports.bindExtObjArrayNode = bindExtObjArrayNode;
exports.addElement = addElement;
exports.removeElement = removeElement;
/**
 * @module node-opcua-address-space.Private
 */
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_variant_1 = require("node-opcua-variant");
const node_opcua_variant_2 = require("node-opcua-variant");
const node_opcua_variant_3 = require("node-opcua-variant");
const node_opcua_extension_object_1 = require("node-opcua-extension-object");
const ua_variable_impl_1 = require("./ua_variable_impl");
const doDebug = (0, node_opcua_debug_1.checkDebugFlag)(__filename);
const debugLog = (0, node_opcua_debug_1.make_debugLog)(__filename);
const errorLog = (0, node_opcua_debug_1.make_errorLog)(__filename);
const warningLog = (0, node_opcua_debug_1.make_warningLog)(__filename);
/*
 * define a complex Variable containing a array of extension objects
 * each element of the array is also accessible as a component variable.
 *
 */
function getExtObjArrayNodeValue() {
    return new node_opcua_variant_1.Variant({
        arrayType: node_opcua_variant_3.VariantArrayType.Array,
        dataType: node_opcua_variant_2.DataType.ExtensionObject,
        value: this.$$extensionObjectArray
    });
}
function removeElementByIndex(uaArrayVariableNode, elementIndex) {
    const _array = uaArrayVariableNode.$$extensionObjectArray;
    (0, node_opcua_assert_1.assert)(typeof elementIndex === "number");
    const addressSpace = uaArrayVariableNode.addressSpace;
    const extObj = _array[elementIndex];
    const browseName = uaArrayVariableNode.$$getElementBrowseName(extObj, elementIndex);
    // remove element from global array (inefficient)
    uaArrayVariableNode.$$extensionObjectArray.splice(elementIndex, 1);
    if (uaArrayVariableNode.$$extensionObjectArray !== uaArrayVariableNode.$dataValue.value.value) {
        //    throw new Error("internal error");
    }
    uaArrayVariableNode.touchValue();
    // remove matching component
    const node = uaArrayVariableNode.getComponentByName(browseName);
    if (!node) {
        throw new Error(" cannot find component ");
    }
    const hasComponent = uaArrayVariableNode.addressSpace.findReferenceType("HasComponent");
    // remove the hasComponent reference toward node
    uaArrayVariableNode.removeReference({
        isForward: true,
        nodeId: node.nodeId,
        referenceType: hasComponent.nodeId
    });
    // now check if node has still some parent
    const parents = node.findReferencesEx("HasChild", node_opcua_data_model_1.BrowseDirection.Inverse);
    if (parents.length === 0) {
        addressSpace.deleteNode(node.nodeId);
    }
}
/**
 *
 * create a node Variable that contains a array of ExtensionObject of a given type
 */
function createExtObjArrayNode(parentFolder, options) {
    (0, node_opcua_assert_1.assert)(typeof options.variableType === "string");
    (0, node_opcua_assert_1.assert)(typeof options.indexPropertyName === "string");
    const addressSpace = parentFolder.addressSpace;
    const namespace = parentFolder.namespace;
    const complexVariableType = addressSpace.findVariableType(options.complexVariableType);
    // istanbul ignore next
    if (!complexVariableType) {
        throw new Error("cannot find complex variable type");
    }
    (0, node_opcua_assert_1.assert)(!complexVariableType.nodeId.isEmpty());
    const variableType = addressSpace.findVariableType(options.variableType);
    if (!variableType) {
        throw new Error("cannot find variable Type");
    }
    (0, node_opcua_assert_1.assert)(!variableType.nodeId.isEmpty());
    const structure = addressSpace.findDataType("Structure");
    (0, node_opcua_assert_1.assert)(structure, "Structure Type not found: please check your nodeset file");
    const dataType = addressSpace.findDataType(variableType.dataType);
    // istanbul ignore next
    if (!dataType) {
        errorLog(variableType.toString());
        throw new Error("cannot find Data Type");
    }
    (0, node_opcua_assert_1.assert)(dataType.isSubtypeOf(structure), "expecting a structure (= ExtensionObject) here ");
    const inner_options = {
        componentOf: parentFolder,
        browseName: options.browseName,
        dataType: dataType.nodeId,
        typeDefinition: complexVariableType.nodeId,
        value: { dataType: node_opcua_variant_2.DataType.ExtensionObject, value: [], arrayType: node_opcua_variant_3.VariantArrayType.Array },
        valueRank: 1
    };
    const uaArrayVariableNode = namespace.addVariable(inner_options);
    bindExtObjArrayNode(uaArrayVariableNode, options.variableType, options.indexPropertyName);
    return uaArrayVariableNode;
}
function _getElementBrowseName(extObj, index) {
    const indexPropertyName1 = this.$$indexPropertyName;
    if (!Object.prototype.hasOwnProperty.call(extObj, indexPropertyName1)) {
        warningLog(" extension object does not have ", indexPropertyName1, extObj);
    }
    // assert(extObj.constructor === addressSpace.constructExtensionObject(dataType));
    (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(extObj, indexPropertyName1));
    const browseName = extObj[indexPropertyName1].toString();
    return browseName;
}
;
function bindExtObjArrayNode(uaArrayVariableNode, variableTypeNodeId, indexPropertyName) {
    (0, node_opcua_assert_1.assert)(uaArrayVariableNode.valueRank === 1, "expecting a one dimension array");
    const addressSpace = uaArrayVariableNode.addressSpace;
    const variableType = addressSpace.findVariableType(variableTypeNodeId);
    // istanbul ignore next
    if (!variableType || variableType.nodeId.isEmpty()) {
        throw new Error("Cannot find VariableType " + variableTypeNodeId.toString());
    }
    const structure = addressSpace.findDataType("Structure");
    // istanbul ignore next
    if (!structure) {
        throw new Error("Structure Type not found: please check your nodeset file");
    }
    let dataType = addressSpace.findDataType(variableType.dataType);
    // istanbul ignore next
    if (!dataType) {
        throw new Error("Cannot find DataType " + variableType.dataType.toString());
    }
    (0, node_opcua_assert_1.assert)(dataType.isSubtypeOf(structure), "expecting a structure (= ExtensionObject) here ");
    (0, node_opcua_assert_1.assert)(!uaArrayVariableNode.$$variableType, "uaArrayVariableNode has already been bound !");
    uaArrayVariableNode.$$variableType = variableType;
    // verify that an object with same doesn't already exist
    dataType = addressSpace.findDataType(variableType.dataType);
    (0, node_opcua_assert_1.assert)(dataType.isSubtypeOf(structure), "expecting a structure (= ExtensionObject) here ");
    (0, node_opcua_assert_1.assert)(!uaArrayVariableNode.$$extensionObjectArray, "UAVariable ExtensionObject array already bounded");
    uaArrayVariableNode.$$dataType = dataType;
    uaArrayVariableNode.$$extensionObjectArray = [];
    uaArrayVariableNode.$$indexPropertyName = indexPropertyName;
    uaArrayVariableNode.$$getElementBrowseName = _getElementBrowseName;
    uaArrayVariableNode.$dataValue.value.value = uaArrayVariableNode.$$extensionObjectArray;
    uaArrayVariableNode.$dataValue.value.arrayType = node_opcua_variant_3.VariantArrayType.Array;
    const bindOptions = {
        get: getExtObjArrayNodeValue,
        set: undefined // readonly
    };
    // bind the readonly
    uaArrayVariableNode.bindVariable(bindOptions, true);
    return uaArrayVariableNode;
}
/**

 * add a new element in a ExtensionObject Array variable
 * @param options {Object}   data used to construct the underlying ExtensionObject
 * @param uaArrayVariableNode {UAVariable}
 * @return {UAVariable}
 *
 */
function addElement(options, uaArrayVariableNode) {
    (0, node_opcua_assert_1.assert)(uaArrayVariableNode, " must provide an UAVariable containing the array");
    // verify that arr has been created correctly
    (0, node_opcua_assert_1.assert)(!!uaArrayVariableNode.$$variableType && !!uaArrayVariableNode.$$dataType, "did you create the array Node with createExtObjArrayNode ?");
    (0, node_opcua_assert_1.assert)(uaArrayVariableNode.$$dataType.nodeClass === node_opcua_data_model_1.NodeClass.DataType);
    const addressSpace = uaArrayVariableNode.addressSpace;
    const Constructor = addressSpace.getExtensionObjectConstructor(uaArrayVariableNode.$$dataType);
    (0, node_opcua_assert_1.assert)(Constructor instanceof Function);
    let extensionObject;
    let elVar = null;
    let browseName;
    if (options instanceof ua_variable_impl_1.UAVariableImpl) {
        elVar = options;
        extensionObject = elVar.$extensionObject; // get shared extension object
        (0, node_opcua_assert_1.assert)(extensionObject instanceof Constructor, "the provided variable must expose a Extension Object of the expected type ");
        // add a reference
        uaArrayVariableNode.addReference({
            isForward: true,
            nodeId: elVar.nodeId,
            referenceType: "HasComponent"
        });
        // xx elVar.bindExtensionObject();
    }
    else {
        if (options instanceof node_opcua_extension_object_1.ExtensionObject) {
            // extension object has already been created
            extensionObject = options;
        }
        else {
            extensionObject = addressSpace.constructExtensionObject(uaArrayVariableNode.$$dataType, options);
        }
        const index = uaArrayVariableNode.$$extensionObjectArray?.length || 0;
        browseName = uaArrayVariableNode.$$getElementBrowseName(extensionObject, index);
        elVar = uaArrayVariableNode.$$variableType.instantiate({
            browseName,
            componentOf: uaArrayVariableNode.nodeId,
            value: { dataType: node_opcua_variant_2.DataType.ExtensionObject, value: extensionObject }
        });
        elVar.bindExtensionObject(extensionObject, { force: true });
    }
    if (uaArrayVariableNode.$$extensionObjectArray !== uaArrayVariableNode.$dataValue.value.value) {
        //    throw new Error("internal error");
    }
    // also add the value inside
    uaArrayVariableNode.$$extensionObjectArray.push(elVar.$extensionObject);
    uaArrayVariableNode.touchValue();
    return elVar;
}
/**
 *
 */
function removeElement(uaArrayVariableNode, element) {
    (0, node_opcua_assert_1.assert)(element, "removeElement: element must exist");
    const _array = uaArrayVariableNode.$$extensionObjectArray;
    // istanbul ignore next
    if (_array.length === 0) {
        throw new Error(" cannot remove an element from an empty array ");
    }
    let elementIndex = -1;
    if (typeof element === "number") {
        // find element by index
        elementIndex = element;
        (0, node_opcua_assert_1.assert)(elementIndex >= 0 && elementIndex < _array.length);
    }
    else if (typeof element === "function") {
        // find element by functor
        elementIndex = _array.findIndex(element);
    }
    else if (element && element.nodeClass) {
        // find element by name
        const browseNameToFind = element.browseName.name.toString();
        elementIndex = _array.findIndex((obj, i) => {
            const browseName = uaArrayVariableNode.$$getElementBrowseName(obj, elementIndex).toString();
            return browseName === browseNameToFind;
        });
    }
    else {
        throw new Error("Unsupported anymore!!! please use a functor instead");
    }
    // istanbul ignore next
    if (elementIndex < 0) {
        throw new Error("removeElement: cannot find element matching " + element.toString());
    }
    return removeElementByIndex(uaArrayVariableNode, elementIndex);
}
//# sourceMappingURL=extension_object_array_node.js.map