"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateDataType = exports.UAMultiStateValueDiscreteImpl = void 0;
exports.promoteToMultiStateValueDiscrete = promoteToMultiStateValueDiscrete;
exports._addMultiStateValueDiscrete = _addMultiStateValueDiscrete;
exports.validateIsNumericDataType = validateIsNumericDataType;
/**
 * @module node-opcua-address-space.DataAccess
 */
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_variant_1 = require("node-opcua-variant");
const node_opcua_basic_types_1 = require("node-opcua-basic-types");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_constants_1 = require("node-opcua-constants");
const get_basic_datatype_1 = require("../get_basic_datatype");
const register_node_promoter_1 = require("../../source/loader/register_node_promoter");
const coerce_enum_value_1 = require("../../source/helpers/coerce_enum_value");
const ua_variable_impl_1 = require("../ua_variable_impl");
const add_dataItem_stuff_1 = require("./add_dataItem_stuff");
function convertToArray(array) {
    if (Array.isArray(array))
        return array;
    const result = [];
    for (let i = 0; i < array.length; i++) {
        result[i] = array[i];
    }
    return result;
}
const getCoerceToInt32 = (dataType) => {
    switch (dataType) {
        case node_opcua_variant_1.DataType.UInt64:
            return node_opcua_basic_types_1.coerceUInt64toInt32;
        case node_opcua_variant_1.DataType.Int64:
            return node_opcua_basic_types_1.coerceInt64toInt32;
        default:
            return node_opcua_basic_types_1.coerceInt32;
    }
};
function install_synchronization(variable) {
    const _variable = variable;
    _variable.on("value_changed", (dataValue) => {
        const valueAsTextNode = variable.valueAsText || _variable.getComponentByName("ValueAsText");
        if (!valueAsTextNode) {
            return;
        }
        if (dataValue.value.arrayType === node_opcua_variant_1.VariantArrayType.Array || dataValue.value.arrayType === node_opcua_variant_1.VariantArrayType.Matrix) {
            //
            const coerce = getCoerceToInt32(_variable.getBasicDataType());
            const values = convertToArray(dataValue.value.value).map((a) => (0, node_opcua_basic_types_1.coerceInt32)(a));
            const variantArray = values.map((a) => _variable.findValueAsText(a));
            const localizedText = variantArray.map((a) => a.value);
            const valueAsText1 = new node_opcua_variant_1.Variant({
                arrayType: dataValue.value.arrayType,
                dataType: node_opcua_variant_1.DataType.LocalizedText,
                value: localizedText,
                dimensions: dataValue.value.dimensions
            });
            valueAsTextNode.setValueFromSource(valueAsText1);
        }
        else {
            const valueAsText1 = _variable.findValueAsText(dataValue.value.value);
            valueAsTextNode.setValueFromSource(valueAsText1);
        }
    });
    const dataValue = _variable.readValue();
    // detect value changes to update valueAsText  (initial state)
    _variable.emit("value_changed", dataValue);
}
class UAMultiStateValueDiscreteImpl extends ua_variable_impl_1.UAVariableImpl {
    setValue(value, options) {
        if (typeof value === "string") {
            const enumValues = this.enumValues.readValue().value.value;
            const selected = enumValues.filter((a) => a.displayName.text === value)[0];
            if (selected) {
                this._setValue(selected.value);
            }
            else {
                throw new Error("cannot find enum string " + value + " in " + enumValues.toString());
            }
        }
        else {
            this._setValue((0, node_opcua_basic_types_1.coerceUInt64)(value), options);
        }
    }
    getValueAsString() {
        const v = this.valueAsText.readValue().value.value;
        if (Array.isArray(v)) {
            return v.map((a) => a.text);
        }
        return v.text || "";
    }
    getValueAsNumber() {
        return this.readValue().value.value;
    }
    checkVariantCompatibility(value) {
        if (this.enumValues) {
            if (!this._isValueInRange((0, node_opcua_basic_types_1.coerceInt32)(value.value))) {
                return node_opcua_status_code_1.StatusCodes.BadOutOfRange;
            }
        }
        return node_opcua_status_code_1.StatusCodes.Good;
    }
    clone(options1, optionalFilter, extraInfo) {
        const variable1 = ua_variable_impl_1.UAVariableImpl.prototype.clone.call(this, options1, optionalFilter, extraInfo);
        return promoteToMultiStateValueDiscrete(variable1);
    }
    /**
     * @private
     */
    _isValueInRange(value) {
        // MultiStateValueDiscreteType
        const enumValues = this.enumValues.readValue().value.value;
        const e = enumValues.findIndex((x) => (0, node_opcua_basic_types_1.coerceInt64toInt32)(x.value) === value);
        return !(e === -1);
    }
    /**
     *
     * @private
     */
    _enumValueIndex() {
        // construct an index to quickly find a EnumValue from a value
        const enumValues = this.enumValues.readValue().value.value;
        const enumValueIndex = Object.create(null);
        if (!enumValues || !enumValues.forEach) {
            return enumValueIndex;
        }
        enumValues.forEach((e) => {
            const index = (0, node_opcua_basic_types_1.coerceInt64toInt32)(e.value);
            enumValueIndex[index] = e;
        });
        return enumValueIndex;
    }
    /**
     *
     * @private
     */
    _setValue(value, options) {
        const int32Value = (0, node_opcua_basic_types_1.coerceInt64toInt32)(value);
        // check that value is in bound
        if (!this._isValueInRange(int32Value)) {
            throw new Error("UAMultiStateValueDiscrete#_setValue out of range " + value);
        }
        const dataType = this._getDataType();
        if (dataType === node_opcua_variant_1.DataType.Int64 || dataType === node_opcua_variant_1.DataType.UInt64) {
            this.setValueFromSource({ dataType, arrayType: node_opcua_variant_1.VariantArrayType.Scalar, value });
        }
        else {
            this.setValueFromSource({ dataType, arrayType: node_opcua_variant_1.VariantArrayType.Scalar, value: int32Value });
        }
    }
    /**
     *
     * @private
     */
    findValueAsText(value) {
        const enumValueIndex = this._enumValueIndex();
        if (value === undefined) {
            throw new Error("Unexpected undefined value");
        }
        const valueAsInt32 = (0, node_opcua_basic_types_1.coerceInt64toInt32)(value);
        let valueAsText1 = "Invalid";
        if (enumValueIndex[valueAsInt32] !== undefined) {
            valueAsText1 = enumValueIndex[valueAsInt32].displayName.text || `Invalid:${value}`;
        }
        const result = new node_opcua_variant_1.Variant({
            dataType: node_opcua_variant_1.DataType.LocalizedText,
            value: (0, node_opcua_data_model_1.coerceLocalizedText)(valueAsText1)
        });
        return result;
    }
    _getDataType() {
        if (this.dataType.value === 26 /* Number */) {
            return node_opcua_variant_1.DataType.UInt32;
        }
        const dataTypeStr = node_opcua_variant_1.DataType[this.dataType.value];
        return node_opcua_variant_1.DataType[dataTypeStr];
    }
    /**
     *
     * @private
     */
    _post_initialize() {
        // MultiStateValueDiscrete Variables can have any numeric Data Type;
        // this includes signed and unsigned integers from 8 to 64 Bit length.
        // istanbul ignore next
        validateIsNumericDataType(this.dataType.value);
        // find the enum value type
        install_synchronization(this);
    }
}
exports.UAMultiStateValueDiscreteImpl = UAMultiStateValueDiscreteImpl;
function promoteToMultiStateValueDiscrete(node) {
    if (node instanceof UAMultiStateValueDiscreteImpl) {
        return node; // already promoted
    }
    Object.setPrototypeOf(node, UAMultiStateValueDiscreteImpl.prototype);
    (0, node_opcua_assert_1.assert)(node instanceof UAMultiStateValueDiscreteImpl, "should now  be a State Machine");
    node._post_initialize();
    return node;
}
(0, register_node_promoter_1.registerNodePromoter)(node_opcua_constants_1.VariableTypeIds.MultiStateValueDiscreteType, promoteToMultiStateValueDiscrete);
function _addMultiStateValueDiscrete(namespace, options) {
    (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(options, "enumValues"));
    (0, node_opcua_assert_1.assert)(!Object.prototype.hasOwnProperty.call(options, "ValuePrecision"));
    const addressSpace = namespace.addressSpace;
    const multiStateValueDiscreteType = addressSpace.findVariableType("MultiStateValueDiscreteType");
    if (!multiStateValueDiscreteType) {
        throw new Error("expecting MultiStateValueDiscreteType to be defined , check nodeset xml file");
    }
    // todo : if options.typeDefinition is specified, check that type is SubTypeOf MultiStateDiscreteType
    // EnumValueType
    //   value: Int64, displayName: LocalizedText, Description: LocalizedText
    const enumValues = (0, coerce_enum_value_1.coerceEnumValues)(options.enumValues);
    if (options.value === undefined && enumValues[0]) {
        options.value = enumValues[0].value; // Int64
    }
    // Only DataTypes that can be represented with EnumValues are allowed for Variables of MultiStateValueDiscreteType.
    // These are Integers up to 64 Bits (signed and unsigned).:
    const dataType = (0, get_basic_datatype_1._getBasicDataTypeFromDataTypeNodeId)(addressSpace, options.dataType || node_opcua_variant_1.DataType.UInt32);
    let value;
    if (typeof options.value === "number" ||
        (0, node_opcua_basic_types_1.isValidUInt64)(options.value) ||
        (0, node_opcua_basic_types_1.isValidInt64)(options.value)) {
        value = new node_opcua_variant_1.Variant({
            dataType,
            value: options.value
        });
    }
    else {
        value = options.value;
    }
    const cloned_options = {
        ...options,
        dataType,
        typeDefinition: multiStateValueDiscreteType.nodeId,
        // valueRank:
        // note : OPCUA Spec 1.03 specifies -1:Scalar (part 8 page 8) but nodeset file specifies -2:Any
        value,
        // limitation:  although the Specs specify -2:any, we only support -1(Scalar)
        valueRank: -1 // -1 : Scalar
    };
    const variable = namespace.addVariable(cloned_options);
    (0, add_dataItem_stuff_1.add_dataItem_stuff)(variable, options);
    namespace.addVariable({
        accessLevel: "CurrentRead",
        browseName: { name: "EnumValues", namespaceIndex: 0 },
        dataType: "EnumValueType",
        valueRank: 1,
        minimumSamplingInterval: 0,
        modellingRule: options.modellingRule ? "Mandatory" : undefined,
        propertyOf: variable,
        typeDefinition: "PropertyType",
        userAccessLevel: "CurrentRead",
        value: new node_opcua_variant_1.Variant({
            arrayType: node_opcua_variant_1.VariantArrayType.Array,
            dataType: node_opcua_variant_1.DataType.ExtensionObject,
            value: enumValues
        })
    });
    namespace.addVariable({
        accessLevel: "CurrentRead",
        browseName: { name: "ValueAsText", namespaceIndex: 0 },
        dataType: node_opcua_variant_1.DataType.LocalizedText,
        minimumSamplingInterval: 0,
        modellingRule: options.modellingRule ? "Mandatory" : undefined,
        propertyOf: variable,
        arrayDimensions: options.arrayDimensions,
        valueRank: options.valueRank,
        typeDefinition: "PropertyType",
        userAccessLevel: "CurrentRead"
        // value: valueAsText
    });
    // install additional helpers methods
    variable.install_extra_properties();
    promoteToMultiStateValueDiscrete(variable);
    (0, node_opcua_assert_1.assert)(variable.enumValues.browseName.toString() === "EnumValues");
    (0, node_opcua_assert_1.assert)(variable.valueAsText.browseName.toString() === "ValueAsText");
    return variable;
}
const validBasicNumericDataTypes = [
    node_opcua_variant_1.DataType.UInt64,
    node_opcua_variant_1.DataType.Int64,
    node_opcua_variant_1.DataType.UInt32,
    node_opcua_variant_1.DataType.Int32,
    node_opcua_variant_1.DataType.UInt16,
    node_opcua_variant_1.DataType.Int16,
    node_opcua_variant_1.DataType.Byte,
    node_opcua_variant_1.DataType.Byte,
    node_opcua_variant_1.DataType.SByte,
    26, /* Number  (abstract)*/
    27, /* Integer  (abstract)*/
    28 /* UInteger (abstract)*/
];
function validateIsNumericDataType(dataTypeValue) {
    if (typeof dataTypeValue !== "number" || validBasicNumericDataTypes.indexOf(dataTypeValue) < 0) {
        throw new Error(`Invalid DataType in UAMultiStateValueDiscrete => ${dataTypeValue.toString()}`);
    }
}
/** @deprecated: use validateIsNumericDataType instead */
exports.validateDataType = validateIsNumericDataType;
//# sourceMappingURL=ua_multistate_value_discrete_impl.js.map