/**
 * @module node-opcua-address-space
 */
import { EventEmitter } from "events";
import { AddReferenceOpts, ModellingRuleType, ISessionContext, UAMethod, UAObject, UAObjectType, UAReference, UAVariable, UAVariableType, BaseNode, UAReferenceType, IAddressSpace, INamespace, BrowseDescriptionOptions2, UAProperty } from "node-opcua-address-space-base";
import { AttributeIds, BrowseDirection, LocalizedText, LocalizedTextLike, NodeClass, QualifiedName, QualifiedNameLike, QualifiedNameOptions, AccessRestrictionsFlag } from "node-opcua-data-model";
import { DataValue } from "node-opcua-data-value";
import { NodeId, NodeIdLike } from "node-opcua-nodeid";
import { NumericRange } from "node-opcua-numeric-range";
import { ReferenceDescription } from "node-opcua-service-browse";
import { StatusCode } from "node-opcua-status-code";
import { RelativePathElement, RolePermissionType, RolePermissionTypeOptions, WriteValueOptions } from "node-opcua-types";
import { DataType } from "node-opcua-variant";
import { UAStateVariable } from "node-opcua-nodeset-ua";
import { XmlWriter } from "../source/xml_writer";
import { AddressSpacePrivate } from "./address_space_private";
import { UAString } from "node-opcua-basic-types";
export interface InternalBaseNodeOptions {
    /**
     * the parent address space
     */
    addressSpace: AddressSpacePrivate;
    browseName: QualifiedName;
    nodeId: NodeId;
    references?: UAReference[];
    displayName?: LocalizedTextLike | LocalizedTextLike[];
    description?: LocalizedTextLike | null;
    browseFilter?: (this: BaseNode, context?: ISessionContext) => boolean;
    /**
     * https://reference.opcfoundation.org/v104/Core/docs/Part3/8.56/
     */
    accessRestrictions?: AccessRestrictionsFlag;
    rolePermissions?: RolePermissionTypeOptions[];
}
export declare function makeAttributeEventName(attributeId: AttributeIds): string;
/**
 * Base class for all Node classes
 *
 * BaseNode is the base class for all the OPCUA objects in the address space
 * It provides attributes and a set of references to other nodes.
 * see:
 * {{#crossLink "UAObject"}}{{/crossLink}},
 * {{#crossLink "UAVariable"}}{{/crossLink}},
 * {{#crossLink "Reference"}}{{/crossLink}},
 * {{#crossLink "UAMethod"}}{{/crossLink}},
 * {{#crossLink "UAView"}}{{/crossLink}},
 * {{#crossLink "UAObjectType"}}{{/crossLink}},
 * {{#crossLink "UADataType"}}{{/crossLink}},
 * {{#crossLink "UAVariableType"}}{{/crossLink}},
 *
 *
 */
export declare class BaseNodeImpl extends EventEmitter implements BaseNode {
    static makeAttributeEventName(attributeId: AttributeIds): string;
    private _accessRestrictions?;
    private _rolePermissions?;
    onFirstBrowseAction?: (this: BaseNode) => Promise<void>;
    get addressSpace(): IAddressSpace;
    protected get addressSpacePrivate(): AddressSpacePrivate;
    get displayName(): LocalizedText[];
    setDisplayName(value: LocalizedTextLike[] | LocalizedTextLike): void;
    get description(): LocalizedText;
    setDescription(value: LocalizedTextLike | null): void;
    /**
     * returns the nodeId of this node's Type Definition
     */
    get typeDefinition(): NodeId;
    /**
     * returns the nodeId of this node's Type Definition
     */
    get typeDefinitionObj(): UAObjectType | UAVariableType;
    get parentNodeId(): NodeId | undefined;
    /**
     * namespace index
     */
    get namespaceIndex(): number;
    /**
     * namespace uri
     */
    get namespaceUri(): string;
    /**
     * the parent node
     */
    get parent(): BaseNode | null;
    /**
     * @property modellingRule
     * @type {String|undefined}
     */
    get modellingRule(): ModellingRuleType | undefined;
    readonly nodeClass: NodeClass;
    readonly nodeId: NodeId;
    readonly browseName: QualifiedName;
    protected _postInstantiateFunc?: (instance: BaseNode, instanceType: BaseNode, options?: any) => void;
    /**
     * @internal
     * @param options
     */
    constructor(options: InternalBaseNodeOptions);
    getDisplayName(locale?: string): string;
    get namespace(): INamespace;
    findReferencesEx(referenceType: string | NodeId | UAReferenceType, browseDirection?: BrowseDirection): UAReference[];
    findReferences_no_cache(referenceTypeNode: UAReferenceType, isForward?: boolean): UAReference[];
    findReferences(referenceType: string | NodeId | UAReferenceType, isForward?: boolean): UAReference[];
    findReference(strReference: string | NodeId | UAReferenceType, isForward?: boolean): UAReference | null;
    findReferencesExAsObject(referenceType: string | NodeId | UAReferenceType, browseDirection?: BrowseDirection): BaseNode[];
    findReferencesAsObject(referenceType: string | NodeId | UAReferenceType, isForward?: boolean): BaseNode[];
    /**
     * return an array with the Aggregates of this object.
     */
    getAggregates(): BaseNode[];
    /**
     * return an array with the components of this object.
     */
    getComponents(): BaseNode[];
    /**
     *  return a array with the properties of this object.
     */
    getProperties(): BaseNode[];
    private static _hasChild;
    getChildren(): BaseNode[];
    /**
     * return a array with the notifiers of this object.
     * only reference of exact type HasNotifier are returned.
     */
    getNotifiers(): BaseNode[];
    /**
     * return a array with the event source of this object.
     *  only reference of exact type HasEventSource are returned.
     */
    getEventSources(): BaseNode[];
    /**
     * return a array of the objects for which this node is an EventSource
     */
    getEventSourceOfs(): BaseNode[];
    /**
     * retrieve a component by name
     */
    getComponentByName(browseName: QualifiedNameOptions): UAVariable | UAObject | null;
    getComponentByName(browseName: string, namespaceIndex?: number): UAVariable | UAObject | null;
    /**
     * retrieve a property by name
     */
    getPropertyByName(browseName: QualifiedNameOptions): UAVariable | null;
    getPropertyByName(browseName: string, namespaceIndex?: number): UAVariable | null;
    /**
     * retrieve a folder element by name
     */
    getFolderElementByName(browseName: QualifiedNameOptions): BaseNode | null;
    getFolderElementByName(browseName: string, namespaceIndex?: number): BaseNode | null;
    /**
     * returns the list of nodes that this folder object organizes
     */
    getFolderElements(): BaseNodeImpl[];
    /**
     * returns the list of methods that this object provides
     * @return an array with Method objects.
     *
     *
     * Note: internally, methods are special types of components
     */
    getMethods(): UAMethod[];
    /**
     * returns the method exposed by this object and with the given nodeId
     */
    getMethodById(nodeId: NodeId): UAMethod | null;
    getMethodByName(methodName: QualifiedNameOptions): UAMethod | null;
    getMethodByName(methodName: string, namespaceIndex?: number): UAMethod | null;
    getWriteMask(): number;
    getUserWriteMask(): number;
    readAttribute(context: ISessionContext | null, attributeId: AttributeIds, indexRange?: NumericRange, dataEncoding?: QualifiedNameLike | null): DataValue;
    writeAttribute(context: ISessionContext | null, writeValue: WriteValueOptions, callback: (err: Error | null, statusCode?: StatusCode) => void): void;
    fullName(): string;
    ownReferences(): UAReference[];
    /**
     *
     * @param relativePathElement
     * @param isLast
     * @return {NodeId[]}
     */
    browseNodeByTargetName(relativePathElement: RelativePathElement, isLast: boolean): NodeId[];
    /**
     * browse the node to extract information requested in browseDescription
     * and returns an array with reference descriptions
     *
     *
     *
     */
    browseNode(browseDescription: BrowseDescriptionOptions2, context?: ISessionContext): ReferenceDescription[];
    allReferences(): UAReference[];
    /**
     * @param reference
     * @param reference.referenceType {String}
     * @param [reference.isForward = true] {Boolean}
     * @param reference.nodeId {Node|NodeId|String}
     *
     * @example
     *
     *     view.addReference({ referenceType: "Organizes", nodeId: myDevice });
     *
     * or
     *
     *     myDevice1.addReference({ referenceType: "OrganizedBy", nodeId: view });
     */
    addReference(reference: AddReferenceOpts): void;
    removeReference(referenceOpts: AddReferenceOpts): void;
    /**
     *
     */
    resolveNodeId(nodeId: NodeIdLike): NodeId;
    install_extra_properties(): void;
    uninstall_extra_properties(reference: UAReference): void;
    toString(): string;
    /**
     * @property isFalseSubStateOf
     * @type {BaseNode|null}
     */
    get isFalseSubStateOf(): BaseNode | null;
    /**
     * @property isTrueSubStateOf
     * @type {BaseNode|null}
     */
    get isTrueSubStateOf(): BaseNode | null;
    /**
     * @return {UAStateVariable[]} return an array with the SubStates of this object.
     */
    getFalseSubStates(): UAStateVariable<LocalizedText>[];
    /**

     * @return {UAStateVariable[]} return an array with the SubStates of this object.
     */
    getTrueSubStates(): UAStateVariable<LocalizedText>[];
    findHierarchicalReferences(): UAReference[];
    getChildByName(browseName: QualifiedNameOptions): BaseNode | null;
    getChildByName(browseName: string, namespaceIndex?: number): BaseNode | null;
    getNodeVersion(): UAProperty<UAString, DataType.String> | null;
    get nodeVersion(): UAProperty<UAString, DataType.String> | undefined;
    set nodeVersion(n: any);
    get toStateNode(): BaseNode | null;
    get fromStateNode(): BaseNode | null;
    /**
     * this methods propagates the forward references to the pointed node
     * by inserting backward references to the counter part node
     * @private
     */
    propagate_back_references(): void;
    /**
     * the dispose method should be called when the node is no longer used, to release
     * back pointer to the address space and clear caches.
     *
     * @private
     */
    dispose(): void;
    isDisposed(): boolean;
    dumpXML(xmlWriter: XmlWriter): void;
    /**
     * Undo the effect of propagate_back_references
     */
    unpropagate_back_references(): void;
    installPostInstallFunc(f: (instance: BaseNode, tpyeNode: BaseNode, opts?: any) => void): void;
    _on_child_added(childNode: BaseNode): void;
    _on_child_removed(obj: BaseNode): void;
    /**
     * @private
     * @param reference
     */
    _add_backward_reference(reference: UAReference): void;
    protected _coerceReferenceType(referenceType: string | NodeId | UAReferenceType): UAReferenceType | null;
    private __addReference;
    private _setDisplayName;
    private _setDescription;
    private _notifyAttributeChange;
    private _clear_caches;
    canUserWriteAttribute(context: ISessionContext | null, attributeId: AttributeIds): boolean;
    private _readAccessRestrictions;
    private _readRolePermissions;
    private _readUserRolePermissions;
    /**
     *
     * @param rolePermissions
     */
    setRolePermissions(rolePermissions: RolePermissionTypeOptions[]): void;
    getRolePermissions(inherited: boolean): RolePermissionType[] | null;
    get rolePermissions(): RolePermissionType[] | undefined;
    setAccessRestrictions(accessRestrictions: AccessRestrictionsFlag): void;
    get accessRestrictions(): AccessRestrictionsFlag | undefined;
    getAccessRestrictions(inherited: boolean): AccessRestrictionsFlag;
}
export declare function getReferenceType(reference: UAReference): UAReferenceType;
