"use strict";
/**
 * @module node-opcua-address-space
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AddressSpace = void 0;
const crypto_1 = require("crypto");
const chalk_1 = __importDefault(require("chalk"));
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_constants_1 = require("node-opcua-constants");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_object_registry_1 = require("node-opcua-object-registry");
const node_opcua_service_browse_1 = require("node-opcua-service-browse");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_types_1 = require("node-opcua-types");
const node_opcua_utils_1 = require("node-opcua-utils");
const node_opcua_utils_2 = require("node-opcua-utils");
const node_opcua_variant_1 = require("node-opcua-variant");
const node_opcua_debug_1 = require("node-opcua-debug");
const adjust_browse_direction_1 = require("../source/helpers/adjust_browse_direction");
const alarms_and_conditions_1 = require("./alarms_and_conditions");
const event_data_1 = require("./event_data");
const address_space_historical_data_node_1 = require("./historical_access/address_space_historical_data_node");
const namespace_impl_1 = require("./namespace_impl");
const namespace_impl_2 = require("./namespace_impl");
const ua_data_type_impl_1 = require("./ua_data_type_impl");
const ua_object_type_impl_1 = require("./ua_object_type_impl");
const ua_object_impl_1 = require("./ua_object_impl");
const reference_impl_1 = require("./reference_impl");
const ua_reference_type_impl_1 = require("./ua_reference_type_impl");
const base_node_impl_1 = require("./base_node_impl");
const doDebug = false;
const errorLog = (0, node_opcua_debug_1.make_errorLog)(__filename);
const debugLog = (0, node_opcua_debug_1.make_debugLog)(__filename);
const warningLog = (0, node_opcua_debug_1.make_warningLog)(__filename);
// tslint:disable-next-line:no-var-requires
const Dequeue = require("dequeue");
const regexNumberColumnString = /^([0-9]+):(.*)/;
const enumerationTypeNodeId = (0, node_opcua_nodeid_1.coerceNodeId)(node_opcua_constants_1.DataTypeIds.Enumeration);
/**
 * Extracts the namespace and browse name as a string from the given input.
 *
 * @param addressSpace The address space instance.
 * @param browseNameOrNodeId A NodeIdLike, QualifiedName, or string representing the browse name or node id.
 * @param namespaceIndex Optional namespace index.
 * @returns A tuple containing the NamespacePrivate and the browse name as a string.
 */
function _extract_namespace_and_browse_name_as_string(addressSpace, browseNameOrNodeId, namespaceIndex) {
    (0, node_opcua_assert_1.assert)(!namespaceIndex || namespaceIndex >= 0);
    let result;
    if (namespaceIndex !== undefined && namespaceIndex > 0) {
        (0, node_opcua_assert_1.assert)(typeof browseNameOrNodeId === "string", "expecting a string when namespaceIndex is specified");
        result = [addressSpace.getNamespace(namespaceIndex), browseNameOrNodeId];
    }
    else if (typeof browseNameOrNodeId === "string") {
        // we may have a string like "ns=1;i=1234" or "ns=1:MyBrowseName"
        if (isNodeIdString(browseNameOrNodeId)) {
            return _extract_namespace_and_browse_name_as_string(addressSpace, addressSpace.resolveNodeId(browseNameOrNodeId), namespaceIndex);
        }
        // we must have a BrowseName string
        if (browseNameOrNodeId.indexOf(":") >= 0) {
            const a = browseNameOrNodeId.split(":");
            namespaceIndex = a.length === 2 ? parseInt(a[0], 10) : namespaceIndex;
            browseNameOrNodeId = a.length === 2 ? a[1] : browseNameOrNodeId;
        }
        result = [addressSpace.getNamespace(namespaceIndex || 0), browseNameOrNodeId];
    }
    else if (browseNameOrNodeId instanceof node_opcua_data_model_1.QualifiedName) {
        namespaceIndex = browseNameOrNodeId.namespaceIndex;
        result = [addressSpace.getNamespace(namespaceIndex), browseNameOrNodeId.name];
    }
    else if (typeof browseNameOrNodeId === "number") {
        result = [addressSpace.getDefaultNamespace(), node_opcua_variant_1.DataType[browseNameOrNodeId]];
    }
    else if (browseNameOrNodeId instanceof node_opcua_nodeid_1.NodeId) {
        // we have a nodeId
        const nodeId = browseNameOrNodeId;
        namespaceIndex = nodeId.namespace;
        const node = addressSpace.findNode(nodeId);
        const browseName = node ? node.browseName.toString() : "UnknownNode";
        result = [addressSpace.getNamespace(namespaceIndex), browseName];
    }
    /* istanbul ignore next */
    if (!result || !result[0]) {
        throw new Error(` Cannot find namespace associated with ${browseNameOrNodeId} ${namespaceIndex}`);
    }
    return result;
}
/**
 * returns true if str matches a nodeID, e.g i=123 or ns=...
 */
function isNodeIdString(str) {
    if (typeof str !== "string") {
        return false;
    }
    return str.substring(0, 2) === "i=" || str.substring(0, 3) === "ns=";
}
/**
 * `AddressSpace` is a collection of UA nodes.
 *
 *     const addressSpace = AddressSpace.create();
 */
class AddressSpace {
    get rootFolder() {
        const rootFolder = this.findNode(this.resolveNodeId("RootFolder"));
        if (!rootFolder) {
            // throw new Error("AddressSpace doesn't contain rootFolder object");
            return null;
        }
        return rootFolder;
    }
    static isNonEmptyQualifiedName = namespace_impl_2.isNonEmptyQualifiedName;
    static historizerFactory;
    static create() {
        return new AddressSpace();
    }
    static registry = new node_opcua_object_registry_1.ObjectRegistry();
    /***
     * @internal
     * @private
     */
    suspendBackReference = false;
    isFrugal = false;
    historizingNodes = new Set();
    _condition_refresh_in_progress = false;
    isNodeIdString = isNodeIdString;
    _private_namespaceIndex;
    _namespaceArray;
    _shutdownTask;
    _modelChangeTransactionCounter = 0;
    _modelChanges = [];
    constructor() {
        this._private_namespaceIndex = 1;
        this._namespaceArray = [];
        // special namespace 0 is reserved for the UA namespace
        this.registerNamespace("http://opcfoundation.org/UA/");
        AddressSpace.registry.register(this);
    }
    /**
     * @internal
     */
    getDataTypeManager() {
        const addressSpacePriv = this;
        (0, node_opcua_assert_1.assert)(addressSpacePriv.$$extraDataTypeManager, "expecting a $$extraDataTypeManager please make sure ensureDatatypeExtracted is called");
        return addressSpacePriv.$$extraDataTypeManager;
    }
    getNamespaceUri(namespaceIndex) {
        (0, node_opcua_assert_1.assert)(namespaceIndex >= 0 && namespaceIndex < this._namespaceArray.length);
        return this._namespaceArray[namespaceIndex].namespaceUri;
    }
    /***
     */
    getNamespace(namespaceIndexOrName) {
        if (typeof namespaceIndexOrName === "number") {
            const namespaceIndex = namespaceIndexOrName;
            (0, node_opcua_assert_1.assert)(namespaceIndex >= 0 && namespaceIndex < this._namespaceArray.length, "invalid namespace index ( out of bound)");
            return this._namespaceArray[namespaceIndex];
        }
        else {
            const namespaceUri = namespaceIndexOrName;
            (0, node_opcua_assert_1.assert)(typeof namespaceUri === "string");
            const index = this.getNamespaceIndex(namespaceUri);
            return this._namespaceArray[index];
        }
    }
    /***
     * @return  the  default namespace (standard OPCUA namespace)
     */
    getDefaultNamespace() {
        return this.getNamespace(0);
    }
    /***
     *
     * objects instances managed by the server will be created in this namespace.
     *
     * @return  address space own namespace
     */
    getOwnNamespace() {
        /* istanbul ignore next */
        if (this._private_namespaceIndex >= this._namespaceArray.length) {
            throw new Error("please create the private namespace");
        }
        return this.getNamespace(this._private_namespaceIndex);
    }
    /**
     * @return the namespace index of a namespace given by its namespace uri
     *
     */
    getNamespaceIndex(namespaceUri) {
        (0, node_opcua_assert_1.assert)(typeof namespaceUri === "string");
        return this._namespaceArray.findIndex((ns) => ns.namespaceUri === namespaceUri);
    }
    /**
     *
     * register a new namespace,
     * it is OK to call registerNamespace even if namespace has already been registered;
     * in this case the registerNamespace has no effect and returns the existing namespace.
     *
     * @param namespaceUri {string}
     * @returns {Namespace}
     */
    registerNamespace(namespaceUri) {
        let index = this._namespaceArray.findIndex((ns) => ns.namespaceUri === namespaceUri);
        if (index !== -1) {
            (0, node_opcua_assert_1.assert)(this._namespaceArray[index].addressSpace === this);
            return this._namespaceArray[index];
        }
        index = this._namespaceArray.length;
        this._namespaceArray.push(new namespace_impl_1.NamespaceImpl({
            addressSpace: this,
            index,
            namespaceUri,
            publicationDate: new Date(),
            version: "undefined"
        }));
        return this._namespaceArray[index];
    }
    /***
     * @return {Namespace[]} the namespace array
     */
    getNamespaceArray() {
        return this._namespaceArray;
    }
    /**
     *
     * @param alias {String} the alias name
     * @param nodeId {NodeId}
     * @internal
     */
    addAlias(alias, nodeId) {
        (0, node_opcua_assert_1.assert)(typeof alias === "string");
        (0, node_opcua_assert_1.assert)(nodeId instanceof node_opcua_nodeid_1.NodeId);
        this.getNamespace(nodeId.namespace).addAlias(alias, nodeId);
    }
    /**
     * find an node by node Id
     * @param nodeId   a nodeId or a string coerce-able to nodeID, representing the object to find.
     * @return {BaseNode|null}
     */
    findNode(nodeId) {
        if (!(nodeId instanceof node_opcua_nodeid_1.NodeId)) {
            nodeId = this.resolveNodeId(nodeId);
        }
        if (nodeId.namespace < 0 || nodeId.namespace >= this._namespaceArray.length) {
            // namespace index is out of bound
            return null;
        }
        const namespace = this._namespaceArray[nodeId.namespace];
        return namespace.findNode2(nodeId);
    }
    findMethod(nodeId) {
        const node = this.findNode(nodeId);
        if (!node || node.nodeClass !== node_opcua_data_model_1.NodeClass.Method) {
            return null;
        }
        return node;
    }
    /**
     * resolved a string or a nodeId to a nodeID
     */
    resolveNodeId(nodeId) {
        if (typeof nodeId === "string") {
            const m = nodeId.match(regexNumberColumnString);
            if (m && m.length === 3) {
                const namespaceIndex = parseInt(m[1], 10);
                const aliasName = m[2];
                const namespace = this.getNamespace(namespaceIndex);
                // check if the string is a known alias
                const aliasNodeId = namespace.resolveAlias(aliasName);
                if (aliasNodeId !== null) {
                    return aliasNodeId;
                }
            }
        }
        const namespaceArray = this.getNamespaceArray().map((a) => a.namespaceUri);
        return (0, node_opcua_nodeid_1.resolveNodeId)(nodeId, { namespaceArray });
    }
    /**
     *
     * @param objectType  {String|NodeId|QualifiedName}
     * @param [namespaceIndex=0 {Number}] an optional namespace index
     * @return {UAObjectType|null}
     *
     * @example
     *
     * ```javascript
     *     const objectType = addressSpace.findObjectType("ns=0;i=58");
     *     objectType.browseName.toString().should.eql("BaseObjectType");
     *
     *     const objectType = addressSpace.findObjectType("BaseObjectType");
     *     objectType.browseName.toString().should.eql("BaseObjectType");
     *
     *     const objectType = addressSpace.findObjectType(resolveNodeId("ns=0;i=58"));
     *     objectType.browseName.toString().should.eql("BaseObjectType");
     *
     *     const objectType = addressSpace.findObjectType("CustomObjectType",36);
     *     objectType.nodeId.namespace.should.eql(36);
     *     objectType.browseName.toString().should.eql("BaseObjectType");
     *
     *     const objectType = addressSpace.findObjectType("36:CustomObjectType");
     *     objectType.nodeId.namespace.should.eql(36);
     *     objectType.browseName.toString().should.eql("BaseObjectType");
     * ```
     */
    findObjectType(objectType, namespaceIndex) {
        if (objectType instanceof node_opcua_nodeid_1.NodeId) {
            return _find_by_node_id(this, objectType, namespaceIndex);
        }
        const [namespace, browseName] = _extract_namespace_and_browse_name_as_string(this, objectType, namespaceIndex);
        return namespace.findObjectType(browseName);
    }
    /**
     * @param variableType  {String|NodeId}
     * @param [namespaceIndex=0 {Number}] an optional namespace index
     * @return {UAObjectType|null}
     *
     * @example
     *
     * ```javascript
     *     const objectType = addressSpace.findVariableType("ns=0;i=62");
     *     objectType.browseName.toString().should.eql("BaseVariableType");
     *
     *     const objectType = addressSpace.findVariableType("BaseVariableType");
     *     objectType.browseName.toString().should.eql("BaseVariableType");
     *
     *     const objectType = addressSpace.findVariableType(resolveNodeId("ns=0;i=62"));
     *     objectType.browseName.toString().should.eql("BaseVariableType");
     * ```
     */
    findVariableType(variableType, namespaceIndex) {
        if (variableType instanceof node_opcua_nodeid_1.NodeId) {
            return _find_by_node_id(this, variableType, namespaceIndex);
        }
        const [namespace, browseName] = _extract_namespace_and_browse_name_as_string(this, variableType, namespaceIndex);
        return namespace.findVariableType(browseName);
    }
    /**
     * Find the DataType node from a NodeId or a browseName
     * @param dataType {String|NodeId}
     * @param [namespaceIndex=0 {Number}] an optional namespace index
     * @return {DataType|null}
     *
     *
     * @example
     *
     * ```javascript
     *      const dataDouble = addressSpace.findDataType("Double");
     *      const dataDouble = addressSpace.findDataType(resolveNodeId("ns=0;i=3"));
     * ```
     */
    findDataType(dataType, namespaceIndex) {
        // startingNode i=24  :
        // BaseDataType
        // +-> Boolean (i=1) {BooleanDataType (ns=2:9898)
        // +-> String (i=12)
        //     +->NumericRange
        //     +->Time
        // +-> DateTime
        // +-> Structure
        //       +-> Node
        //            +-> ObjectNode
        if (dataType instanceof ua_data_type_impl_1.UADataTypeImpl) {
            return this.findDataType(dataType.nodeId);
        }
        else if (dataType instanceof node_opcua_nodeid_1.NodeId) {
            return _find_by_node_id(this, dataType, namespaceIndex);
        }
        if (typeof dataType === "number") {
            if (node_opcua_variant_1.DataType[dataType] !== undefined) {
                return this.findDataType((0, node_opcua_nodeid_1.makeNodeId)(dataType, 0));
            }
            else {
                return this.findDataType((0, node_opcua_nodeid_1.resolveNodeId)(dataType));
            }
        }
        const res = _extract_namespace_and_browse_name_as_string(this, dataType, namespaceIndex);
        const namespace = res[0];
        const browseName = res[1];
        return namespace.findDataType(browseName);
    }
    /**
     *
     * @example
     *
     *     const dataType = addressSpace.findDataType("ns=0;i=12");
     *     addressSpace.findCorrespondingBasicDataType(dataType).should.eql(DataType.String);
     *
     *     const dataType = addressSpace.findDataType("ServerStatusDataType"); // ServerStatus
     *     addressSpace.findCorrespondingBasicDataType(dataType).should.eql(DataType.ExtensionObject);
     *
     */
    findCorrespondingBasicDataType(dataTypeNode) {
        const _orig_dataTypeNode = dataTypeNode;
        if (typeof dataTypeNode === "string") {
            const resolvedDataType = this.resolveNodeId(dataTypeNode);
            /* istanbul ignore next */
            if (!resolvedDataType) {
                throw new Error("Cannot resolve " + _orig_dataTypeNode.toString());
            }
            dataTypeNode = resolvedDataType;
        }
        if (dataTypeNode instanceof node_opcua_nodeid_1.NodeId) {
            dataTypeNode = this.findDataType(dataTypeNode);
            /* istanbul ignore next */
            if (!dataTypeNode) {
                throw Error("cannot find dataTypeNode " + _orig_dataTypeNode.toString());
            }
        }
        /* istanbul ignore next */
        if (!(dataTypeNode instanceof ua_data_type_impl_1.UADataTypeImpl)) {
            throw new Error("we are expecting an UADataType here :  " +
                _orig_dataTypeNode.toString() +
                " should not refer to a  " +
                dataTypeNode.browseName.name);
        }
        if ((0, node_opcua_nodeid_1.sameNodeId)(enumerationTypeNodeId, dataTypeNode.nodeId)) {
            return node_opcua_variant_1.DataType.Int32;
        }
        const n = dataTypeNode.nodeId;
        if (n.namespace === 0 && n.value === 29) {
            // Number
            return node_opcua_variant_1.DataType.Null; //which one ?
        }
        if (n.namespace === 0 && n.value === 0) {
            return node_opcua_variant_1.DataType.Null;
        }
        if (n.identifierType === node_opcua_nodeid_1.NodeIdType.NUMERIC && n.namespace === 0 && n.value <= 25) {
            return n.value;
        }
        const result = this.findCorrespondingBasicDataType(dataTypeNode.subtypeOfObj);
        return result;
    }
    /**
     * find a ReferenceType by its inverse name.
     * @param inverseName  the inverse name of the ReferenceType to find
     * @deprecated
     */
    findReferenceTypeFromInverseName(inverseName) {
        return this.getDefaultNamespace().findReferenceTypeFromInverseName(inverseName);
    }
    /**
     * @param refType {String|NodeId}
     * @param [namespaceIndex=0 {Number}] an optional namespace index
     * @return {ReferenceType|null}
     *
     * refType could be
     *   - a string representing a nodeid       : e.g.    `i=9004` or ns=1;i=6030
     *   - a string representing a browse name  : e.g     `"HasTypeDefinition"`.
     *     In this case it should be in the alias list.
     *
     */
    findReferenceType(refType, namespaceIndex) {
        // startingNode ns=0;i=31 : References
        //  +-> References i=31
        //           +->(hasSubtype) NonHierarchicalReferences
        //                  +->(hasSubtype) HasTypeDefinition
        //           +->(hasSubtype) HierarchicalReferences
        //                  +->(hasSubtype) HasChild/ChildOf
        //                                  +->(hasSubtype) Aggregates/AggregatedBy
        //                                                  +-> HasProperty/PropertyOf
        //                                                  +-> HasComponent/ComponentOf
        //                                                  +-> HasHistoricalConfiguration/HistoricalConfigurationOf
        //                                 +->(hasSubtype) HasSubtype/HasSupertype
        //                  +->(hasSubtype) Organizes/OrganizedBy
        //                  +->(hasSubtype) HasEventSource/EventSourceOf
        let node;
        if (!(refType instanceof node_opcua_nodeid_1.NodeId) && isNodeIdString(refType) || typeof refType === "number") {
            refType = (0, node_opcua_nodeid_1.resolveNodeId)(refType);
        }
        if (refType instanceof node_opcua_nodeid_1.NodeId) {
            node = this.findNode(refType);
            /* istanbul ignore next */
            if (!(node && node.nodeClass === node_opcua_data_model_1.NodeClass.ReferenceType)) {
                // throw new Error("cannot resolve referenceId "+ refType.toString());
                return null;
            }
        }
        else {
            (0, node_opcua_assert_1.assert)(typeof refType === "string");
            node = this._findReferenceType(refType, namespaceIndex);
        }
        return node;
    }
    /**
     * returns the inverse name of the referenceType.
     *
     * @param referenceType {String} : the reference type name
     * @return {String} the name of the inverse reference type.
     *
     * @example
     *
     *    ```javascript
     *    addressSpace.inverseReferenceType("OrganizedBy").should.eql("Organizes");
     *    addressSpace.inverseReferenceType("Organizes").should.eql("OrganizedBy");
     *    ```
     *
     */
    inverseReferenceType(referenceType) {
        (0, node_opcua_assert_1.assert)(typeof referenceType === "string");
        const n1 = this.findReferenceType(referenceType);
        const n2 = this.findReferenceTypeFromInverseName(referenceType);
        if (n1) {
            (0, node_opcua_assert_1.assert)(!n2);
            return n1.inverseName.text;
        }
        else {
            (0, node_opcua_assert_1.assert)(n2);
            return n2.browseName.toString();
        }
    }
    /**
     * find an EventType node in the address space
     * @param eventTypeId {String|NodeId|UAObjectType} the eventType to find
     * @param namespaceIndex the namespace index of the event to find
     * @return {UAObjectType|null} the EventType found or null.
     *
     * note:
     *    - the method with throw an exception if a node is found
     *      that is not a BaseEventType or a subtype of it.
     *
     * @example
     *
     *     var evtType = addressSpace.findEventType("AuditEventType");
     *
     */
    findEventType(eventTypeId, namespaceIndex) {
        let eventType;
        if (eventTypeId instanceof base_node_impl_1.BaseNodeImpl) {
            eventType = eventTypeId;
        }
        else {
            eventType = this.findObjectType(eventTypeId, namespaceIndex);
        }
        if (!eventType) {
            return null;
        }
        const baseEventType = this.findObjectType("BaseEventType");
        /* istanbul ignore next */
        if (!baseEventType) {
            throw new Error("expecting BaseEventType - please check you nodeset xml file!");
        }
        if ((0, node_opcua_nodeid_1.sameNodeId)(eventType.nodeId, baseEventType.nodeId)) {
            return eventType;
        }
        /* eventTypeNode should be a sub type of "BaseEventType" */
        /* istanbul ignore next */
        if (!eventType.isSubtypeOf(baseEventType)) {
            throw new Error("findEventType: event found is not subType of BaseEventType");
        }
        return eventType;
    }
    /**
     * EventId is generated by the Server to uniquely identify a particular Event Notification.
     * @return {Variant}  dataType: "ByteString"
     */
    generateEventId() {
        /*
         * OpcUA 1.02 part 5 : 6.4.2 BaseEventType
         * The Server is responsible to ensure that each Event has its unique EventId.
         * It may do this, for example, by putting GUIDs into the ByteString.
         * Clients can use the EventId to assist in minimizing or eliminating gaps and overlaps that may occur during
         * a redundancy fail-over. The EventId shall always be returned as value and the Server is not allowed to
         * return a StatusCode for the EventId indicating an error.
         *
         */
        const offset = 16;
        const self = this;
        if (!self._eventIdCounter) {
            self._eventIdCounter = (0, crypto_1.randomBytes)(20);
            self._eventIdCounter.writeInt32BE(0, offset);
        }
        self._eventIdCounter.writeInt32BE(self._eventIdCounter.readInt32BE(offset) + 1, offset);
        return new node_opcua_variant_1.Variant({
            dataType: node_opcua_variant_1.DataType.ByteString,
            value: Buffer.from(self._eventIdCounter)
        });
    }
    /*=
     * construct a simple javascript object with all the default properties of the event
     *
     * @return result.$eventDataSource {BaseNode} the event type node
     * @return result.eventId {NodeId} the
     * ...
     *
     *
     * eventTypeId can be a UAEventType
     *
     * @private
     */
    constructEventData(eventTypeId, data) {
        data = data || {};
        // construct the reference dataStructure to store event Data
        let eventTypeNode = eventTypeId;
        // make sure that eventType is really a object that derived from EventType
        if (eventTypeId instanceof ua_object_type_impl_1.UAObjectTypeImpl) {
            eventTypeNode = this.findEventType(eventTypeId);
        }
        /* istanbul ignore next */
        if (!eventTypeNode) {
            throw new Error(" cannot find EvenType for " + eventTypeId);
        }
        (0, node_opcua_assert_1.assert)(eventTypeNode instanceof ua_object_type_impl_1.UAObjectTypeImpl, "eventTypeId must represent a UAObjectType");
        // eventId
        (0, node_opcua_assert_1.assert)(!Object.prototype.hasOwnProperty.call(data, "eventId"), "eventId constructEventData : options object should not have eventId property");
        data.eventId = data.eventId || this.generateEventId();
        // eventType
        data.eventType = { dataType: node_opcua_variant_1.DataType.NodeId, value: eventTypeNode.nodeId };
        // sourceNode
        (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(data, "sourceNode"), "expecting a source node to be defined");
        data.sourceNode = new node_opcua_variant_1.Variant(data.sourceNode);
        (0, node_opcua_assert_1.assert)(data.sourceNode.dataType === node_opcua_variant_1.DataType.NodeId);
        // sourceName
        const sourceNode = this.findNode(data.sourceNode.value);
        data.sourceName = data.sourceName || {
            dataType: node_opcua_variant_1.DataType.String,
            value: sourceNode.getDisplayName("en")
        };
        const nowUTC = new Date();
        // time (UtcTime)
        // TODO
        data.time = data.time || { dataType: node_opcua_variant_1.DataType.DateTime, value: nowUTC };
        // receivedTime  (UtcTime)
        // TODO
        data.receiveTime = data.receiveTime || { dataType: node_opcua_variant_1.DataType.DateTime, value: nowUTC };
        // localTime  (UtcTime)
        // TODO
        data.localTime = data.localTime || { dataType: node_opcua_variant_1.DataType.DateTime, value: nowUTC };
        // message  (LocalizedText)
        data.message = data.message || { dataType: node_opcua_variant_1.DataType.LocalizedText, value: { text: "" } };
        // severity  (UInt16)
        data.severity = data.severity || { dataType: node_opcua_variant_1.DataType.UInt16, value: 0 };
        // xx // reminder : event type cannot be instantiated directly !
        // xx assert(eventTypeNode.isAbstract);
        const baseObjectType = this.findObjectType("BaseObjectType"); // i=58
        /* istanbul ignore next */
        if (!baseObjectType) {
            throw new Error("BaseObjectType must be defined in the address space");
        }
        const hasProperty = (data, propertyName) => Object.prototype.hasOwnProperty.call(data, propertyName);
        const visitedProperties = {};
        const alreadyVisited = (key) => Object.prototype.hasOwnProperty.call(visitedProperties, key);
        const markAsVisited = (key) => (visitedProperties[key] = 1);
        function _process_var(self, prefixLower, prefixStandard, node) {
            const lowerName = prefixLower + (0, node_opcua_utils_2.lowerFirstLetter)(node.browseName.name);
            const fullBrowsePath = prefixStandard + node.browseName.toString();
            if (alreadyVisited(lowerName)) {
                return;
            }
            markAsVisited(lowerName);
            if (hasProperty(data, lowerName)) {
                eventData._createValue(fullBrowsePath, node, data[lowerName]);
            }
            else {
                // add a property , but with a null variant
                eventData._createValue(fullBrowsePath, node, { dataType: node_opcua_variant_1.DataType.Null });
                // istanbul ignore next
                if (doDebug) {
                    if (node.modellingRule === "Mandatory") {
                        // tslint:disable:no-console
                        errorLog(chalk_1.default.red("ERROR : AddressSpace#constructEventData(eventType,options) " + "cannot find property ") +
                            self.browseName.toString() +
                            " => " +
                            chalk_1.default.cyan(lowerName));
                    }
                    else {
                        errorLog(chalk_1.default.yellow("Warning : AddressSpace#constructEventData(eventType,options)" + " cannot find property ") +
                            self.browseName.toString() +
                            " => " +
                            chalk_1.default.cyan(lowerName));
                    }
                }
            }
        }
        // verify that all elements of data are valid
        function verify_data_is_valid(data1) {
            Object.keys(data1).map((k) => {
                if (k === "$eventDataSource") {
                    return;
                }
                /* istanbul ignore next */
                if (!alreadyVisited(k)) {
                    warningLog("constructEventData:  cannot find property '" +
                        k +
                        "' in [ " +
                        Object.keys(visitedProperties).join(", ") +
                        "] when filling " +
                        eventTypeNode.browseName.toString());
                }
            });
        }
        const populate_data = (self, eventData1) => {
            if ((0, node_opcua_nodeid_1.sameNodeId)(baseObjectType.nodeId, self.nodeId)) {
                return; // nothing to do
            }
            const baseTypeNodeId = self.subtypeOf;
            /* istanbul ignore next */
            if (!baseTypeNodeId) {
                throw new Error("Object " + self.browseName.toString() + " with nodeId " + self.nodeId + " has no Type");
            }
            const baseType = this.findNode(baseTypeNodeId);
            /* istanbul ignore next */
            if (!baseType) {
                throw new Error(chalk_1.default.red("Cannot find object with nodeId ") + baseTypeNodeId);
            }
            populate_data(baseType, eventData1);
            // get properties and components from base class
            const properties = self.getProperties();
            const components = self.getComponents();
            const children = [].concat(properties, components);
            // istanbul ignore next
            if (doDebug) {
                debugLog(" " + chalk_1.default.bgWhite.cyan(self.browseName.toString()));
            }
            for (const node of children) {
                // only keep those that have a "HasModellingRule"
                if (!node.modellingRule) {
                    continue;
                }
                // ignore also methods
                if (node.nodeClass === node_opcua_data_model_1.NodeClass.Method) {
                    continue;
                }
                _process_var(self, "", "", node);
                // also store value in index
                // xx eventData.__nodes[node.nodeId.toString()] = eventData[lowerName];
                const children2 = node.getAggregates();
                if (children2.length > 0) {
                    const lowerName = (0, node_opcua_utils_2.lowerFirstLetter)(node.browseName.name);
                    const standardName = node.browseName.toString();
                    for (const child2 of children2) {
                        _process_var(self, lowerName + ".", standardName + ".", child2);
                    }
                }
            }
        };
        const eventData = new event_data_1.EventData(eventTypeNode);
        // verify standard properties...
        populate_data(eventTypeNode, eventData);
        verify_data_is_valid(data);
        return eventData;
    }
    // - Browse --------------------------------------------------------------------------------------------------------
    /**
     * browse some path.
     *
     * @param  {BrowsePath} browsePath
     * @return {BrowsePathResult}
     *
     * This service can be used translates one or more browse paths into NodeIds.
     * A browse path is constructed of a starting Node and a RelativePath. The specified starting Node
     * identifies the Node from which the RelativePath is based. The RelativePath contains a sequence of
     * ReferenceTypes and BrowseNames.
     *
     *   |StatusCode                    |                                                            |
     *   |------------------------------|:-----------------------------------------------------------|
     *   |BadNodeIdUnknown              |                                                            |
     *   |BadNodeIdInvalid              |                                                            |
     *   |BadNothingToDo                | - the relative path contains an empty list )               |
     *   |BadBrowseNameInvalid          | - target name is missing in relative path                  |
     *   |UncertainReferenceOutOfServer | - The path element has targets which are in another server.|
     *   |BadTooManyMatches             |                                                            |
     *   |BadQueryTooComplex            |                                                            |
     *   |BadNoMatch                    |                                                            |
     *
     *
     *
     */
    browsePath(browsePath) {
        (0, node_opcua_assert_1.assert)(browsePath instanceof node_opcua_types_1.BrowsePath);
        const startingNode = this.findNode(browsePath.startingNode);
        if (!startingNode) {
            return new node_opcua_types_1.BrowsePathResult({ statusCode: node_opcua_status_code_1.StatusCodes.BadNodeIdUnknown });
        }
        if (!browsePath.relativePath.elements || browsePath.relativePath.elements.length === 0) {
            return new node_opcua_types_1.BrowsePathResult({
                statusCode: node_opcua_status_code_1.StatusCodes.BadNothingToDo,
                targets: []
            });
        }
        const elements_length = browsePath.relativePath.elements.length;
        // -------------------------------------------------------------------------------------------------------
        // verify standard RelativePath construction
        //   from OPCUA 1.03 - PArt 3 - 7.6 RelativePath:
        //   TargetName  The BrowseName of the target node.
        //               The final element may have an empty targetName. In this situation all targets of the
        //               references identified by the referenceTypeId are the targets of the RelativePath.
        //               The targetName shall be specified for all other elements.
        //               The current path cannot be followed any further if no targets with the specified
        //               BrowseName exist.
        //   Let's detect null targetName which are not in last position and return BadBrowseNameInvalid if not
        //
        const empty_targetName_not_in_lastPos = browsePath.relativePath.elements.reduce((prev, e, index) => {
            const is_last = index + 1 === elements_length;
            const isBad = !is_last && (!e.targetName || e.targetName.isEmpty());
            return prev + (!is_last && (!e.targetName || e.targetName.isEmpty()) ? 1 : 0);
        }, 0);
        if (empty_targetName_not_in_lastPos) {
            return new node_opcua_types_1.BrowsePathResult({ statusCode: node_opcua_status_code_1.StatusCodes.BadBrowseNameInvalid });
        }
        // from OPCUA 1.03 - PArt 3 - 5.8.4 TranslateBrowsePathToNodeIds
        // TranslateBrowsePathToNodeIds further restrict RelativePath targetName rules:
        // The last element in the relativePath shall always have a targetName specified.
        const last_el = browsePath.relativePath.elements[elements_length - 1];
        if (!last_el.targetName || !last_el.targetName.name || last_el.targetName.name.length === 0) {
            return new node_opcua_types_1.BrowsePathResult({ statusCode: node_opcua_status_code_1.StatusCodes.BadBrowseNameInvalid });
        }
        const res = [];
        const explore_element = (curNodeObject, elements, index) => {
            const element = elements[index];
            (0, node_opcua_assert_1.assert)(element instanceof node_opcua_types_1.RelativePathElement);
            const is_last = index + 1 === elements.length;
            const nodeIds = curNodeObject.browseNodeByTargetName(element, is_last);
            const targets = nodeIds.map((nodeId) => {
                return {
                    remainingPathIndex: elements.length - index,
                    targetId: nodeId
                };
            });
            if (!is_last) {
                // explorer
                for (const target of targets) {
                    const node = this.findNode(target.targetId);
                    if (!node) {
                        continue;
                    }
                    explore_element(node, elements, index + 1);
                }
            }
            else {
                for (const target of targets) {
                    res.push({
                        remainingPathIndex: 0xffffffff,
                        targetId: (0, node_opcua_nodeid_1.coerceExpandedNodeId)(target.targetId)
                    });
                }
            }
        };
        explore_element(startingNode, browsePath.relativePath.elements, 0);
        if (res.length === 0) {
            return new node_opcua_types_1.BrowsePathResult({ statusCode: node_opcua_status_code_1.StatusCodes.BadNoMatch });
        }
        return new node_opcua_types_1.BrowsePathResult({
            statusCode: node_opcua_status_code_1.StatusCodes.Good,
            targets: res
        });
    }
    // - Extension Object ----------------------------------------------------------------------------------------------
    getExtensionObjectConstructor(dataType) {
        (0, node_opcua_assert_1.assert)(dataType, "expecting a dataType");
        if (dataType instanceof node_opcua_nodeid_1.NodeId) {
            const tmp = this.findNode(dataType);
            /* istanbul ignore next */
            if (!tmp) {
                throw new Error("getExtensionObjectConstructor: cannot resolve dataType " + dataType);
            }
            dataType = tmp;
        }
        /* istanbul ignore next */
        if (!(dataType instanceof ua_data_type_impl_1.UADataTypeImpl)) {
            // may be dataType was the NodeId of the "Binary Encoding" node
            throw new Error("getExtensionObjectConstructor: dataType has unexpected type" + dataType);
        }
        const _dataType = dataType;
        // to do verify that dataType is of type "Structure"
        /* istanbul ignore next */
        if (!_dataType.isSubtypeOf(this.findDataType("Structure"))) {
            debugLog(_dataType.toString());
        }
        (0, node_opcua_assert_1.assert)(_dataType.isSubtypeOf(this.findDataType("Structure")));
        if (!_dataType._extensionObjectConstructor) {
            const dataTypeManager = this.$$extraDataTypeManager;
            _dataType._extensionObjectConstructor = dataTypeManager.getExtensionObjectConstructorFromDataType(_dataType.nodeId);
        }
        (0, node_opcua_assert_1.assert)(_dataType._extensionObjectConstructor, "dataType must have a constructor");
        const Constructor = _dataType._extensionObjectConstructor;
        return Constructor;
    }
    /**
     * @param dataType
     * @param [options]
     * @return the constructed extension object
     *
     *
     * @example
     *
     *             // example 1
     *             var extObj = addressSpace.constructExtensionObject("BuildInfo",{ productName: "PRODUCT_NAME"});
     *
     *             // example 2
     *             serverStatusDataType.nodeClass.should.eql(NodeClass.DataType);
     *             serverStatusDataType.browseName.toString().should.eql("ServerStatusDataType");
     *             var serverStatus  = addressSpace.constructExtensionObject(serverStatusDataType);
     *             serverStatus.should.be.instanceof(ServerStatusDataType);
     */
    constructExtensionObject(dataType, options) {
        const Constructor = this.getExtensionObjectConstructor(dataType);
        return new Constructor(options);
    }
    /**
     * cleanup all resources maintained by this addressSpace.
     */
    dispose() {
        this._namespaceArray.map((namespace) => namespace.dispose());
        AddressSpace.registry.unregister(this);
        /* istanbul ignore next */
        if (this._shutdownTask && this._shutdownTask.length > 0) {
            throw new Error("AddressSpace#dispose : shutdown has not been called");
        }
    }
    /**
     * register a function that will be called when the server will perform its shut down.
     */
    registerShutdownTask(task) {
        this._shutdownTask = this._shutdownTask || [];
        (0, node_opcua_assert_1.assert)(typeof task === "function");
        this._shutdownTask.push(task);
    }
    async shutdown() {
        const performTasks = async (tasks) => {
            // perform registerShutdownTask
            for (const task of tasks) {
                await task.call(this);
            }
        };
        if (this._shutdownTask) {
            const tasks = this._shutdownTask;
            this._shutdownTask = [];
            await performTasks(tasks);
        }
    }
    browseSingleNode(nodeId, browseDescription, context) {
        const browseResult = {
            continuationPoint: undefined,
            references: null,
            statusCode: node_opcua_status_code_1.StatusCodes.Good
        };
        if (!browseDescription || browseDescription.browseDirection === node_opcua_data_model_1.BrowseDirection.Invalid) {
            browseResult.statusCode = node_opcua_status_code_1.StatusCodes.BadBrowseDirectionInvalid;
            return new node_opcua_service_browse_1.BrowseResult(browseResult);
        }
        browseDescription.browseDirection = (0, adjust_browse_direction_1.adjustBrowseDirection)(browseDescription.browseDirection, node_opcua_data_model_1.BrowseDirection.Forward);
        /* istanbul ignore next */
        if (typeof nodeId === "number") {
            throw new Error("Not Implemented");
        }
        if (typeof nodeId === "string") {
            const node = this.findNode(this.resolveNodeId(nodeId));
            if (node) {
                nodeId = node.nodeId;
            }
        }
        // check if referenceTypeId is correct
        if (browseDescription.referenceTypeId instanceof node_opcua_nodeid_1.NodeId) {
            if (browseDescription.referenceTypeId.value === 0) {
                browseDescription.referenceTypeId = null;
            }
            else {
                const rf = this.findNode(browseDescription.referenceTypeId);
                if (!rf || !(rf instanceof ua_reference_type_impl_1.UAReferenceTypeImpl)) {
                    browseResult.statusCode = node_opcua_status_code_1.StatusCodes.BadReferenceTypeIdInvalid;
                    return new node_opcua_service_browse_1.BrowseResult(browseResult);
                }
            }
        }
        const obj = this.findNode(nodeId);
        if (!obj) {
            // Object Not Found
            browseResult.statusCode = node_opcua_status_code_1.StatusCodes.BadNodeIdUnknown;
        }
        else {
            browseResult.statusCode = node_opcua_status_code_1.StatusCodes.Good;
            browseResult.references = obj.browseNode(browseDescription, context);
        }
        return new node_opcua_service_browse_1.BrowseResult(browseResult);
    }
    /**
     * @param folder
     * @private
     */
    _coerceFolder(folder) {
        folder = this._coerceNode(folder);
        /* istanbul ignore next */
        if (folder && !_isFolder(this, folder)) {
            throw new Error("Parent folder must be of FolderType " + folder.typeDefinition.toString());
        }
        return folder;
    }
    /**
     *
     * @param view
     * @param modelChange
     * @private
     */
    _collectModelChange(view, modelChange) {
        this._modelChanges.push(modelChange);
    }
    /**
     *
     * walk up the hierarchy of objects until a view is found
     * objects may belong to multiples views.
     * Note: this method doesn't return the main view => Server object.
     * @param node {BaseNode}
     * @return {BaseNode[]}
     */
    /**
     *
     * @param node
     * @private
     */
    extractRootViews(node) {
        (0, node_opcua_assert_1.assert)(node.nodeClass === node_opcua_data_model_1.NodeClass.Object || node.nodeClass === node_opcua_data_model_1.NodeClass.Variable);
        const visitedMap = {};
        const q = new Dequeue();
        q.push(node);
        const objectsFolder = this.rootFolder.objects;
        (0, node_opcua_assert_1.assert)(objectsFolder instanceof ua_object_impl_1.UAObjectImpl);
        const results = [];
        while (q.length) {
            node = q.shift();
            const references = node.findReferencesEx("HierarchicalReferences", node_opcua_data_model_1.BrowseDirection.Inverse);
            const parentNodes = references.map((r) => reference_impl_1.ReferenceImpl.resolveReferenceNode(this, r));
            for (const parent of parentNodes) {
                if ((0, node_opcua_nodeid_1.sameNodeId)(parent.nodeId, objectsFolder.nodeId)) {
                    continue; // nothing to do
                }
                if (parent.nodeClass === node_opcua_data_model_1.NodeClass.View) {
                    results.push(parent);
                }
                else {
                    const key = parent.nodeId.toString();
                    if (Object.prototype.hasOwnProperty.call(visitedMap, key)) {
                        continue;
                    }
                    visitedMap[key] = parent;
                    q.push(parent);
                }
            }
        }
        return results;
    }
    /**
     *
     * @param func
     * @private
     */
    modelChangeTransaction(func) {
        this._modelChangeTransactionCounter = this._modelChangeTransactionCounter || 0;
        function beginModelChange() {
            this._modelChanges = this._modelChanges || [];
            (0, node_opcua_assert_1.assert)(this._modelChangeTransactionCounter >= 0);
            this._modelChangeTransactionCounter += 1;
        }
        function endModelChange() {
            this._modelChangeTransactionCounter -= 1;
            if (this._modelChangeTransactionCounter === 0) {
                if (this._modelChanges.length === 0) {
                    return; // nothing to do
                }
                // increase version number of participating nodes
                // https://github.com/you-dont-need/You-Dont-Need-Lodash-Underscore
                // const nodeIds = _.uniq(this._modelChanges.map((c: any) => c.affected));
                const nodeIds = [...new Set(this._modelChanges.map((c) => c.affected))];
                const nodes = nodeIds.map((nodeId) => this.findNode(nodeId));
                nodes.forEach(_increase_version_number);
                // raise events
                if (this.rootFolder.objects.server) {
                    const eventTypeNode = this.findEventType("GeneralModelChangeEventType");
                    if (eventTypeNode) {
                        this.rootFolder.objects.server.raiseEvent(eventTypeNode, {
                            // Part 5 - 6.4.32 GeneralModelChangeEventType
                            changes: {
                                dataType: node_opcua_variant_1.DataType.ExtensionObject,
                                arrayType: node_opcua_variant_1.VariantArrayType.Array,
                                value: this._modelChanges
                            }
                        });
                    }
                }
                this._modelChanges = [];
                // _notifyModelChange(this);
            }
        }
        beginModelChange.call(this);
        try {
            func();
        }
        catch (err) {
            errorLog("modelChangeTransaction", err?.message);
            throw err;
        }
        finally {
            endModelChange.call(this);
        }
    }
    /**
     * normalize the ReferenceType field of the Reference Object
     * @param params.referenceType  {String|nodeId}
     * @param params.isForward  {Boolean} default value: true;
     * @return {Object} a new reference object  with the normalized name { referenceType: <value>, isForward: <flag>}
     */
    normalizeReferenceType(params) {
        if (params instanceof reference_impl_1.ReferenceImpl) {
            // a reference has already been normalized
            return params;
        }
        // ----------------------------------------------- handle is forward
        (0, node_opcua_assert_1.assert)(params.isForward === undefined || typeof params.isForward === "boolean");
        params.isForward = (0, node_opcua_utils_1.isNullOrUndefined)(params.isForward) ? true : !!params.isForward;
        // referenceType = Organizes   , isForward = true =>   referenceType = Organizes , isForward = true
        // referenceType = Organizes   , isForward = false =>  referenceType = Organizes , isForward = false
        // referenceType = OrganizedBy , isForward = true =>   referenceType = Organizes , isForward = **false**
        // referenceType = OrganizedBy , isForward = false =>  referenceType = Organizes , isForward =  **true**
        // ----------------------------------------------- handle referenceType
        if (params.referenceType instanceof ua_reference_type_impl_1.UAReferenceTypeImpl) {
            params.referenceType = params.referenceType;
            params.referenceType = params.referenceType.nodeId;
        }
        else if (typeof params.referenceType === "string") {
            const inv = this.findReferenceTypeFromInverseName(params.referenceType);
            if (inv) {
                params.referenceType = inv.nodeId;
                params._referenceType = inv;
                params.isForward = !params.isForward;
            }
            else {
                params.referenceType = (0, node_opcua_nodeid_1.resolveNodeId)(params.referenceType);
                const refType = this.findReferenceType(params.referenceType);
                if (refType) {
                    params._referenceType = refType;
                }
            }
        }
        (0, node_opcua_assert_1.assert)(params.referenceType instanceof node_opcua_nodeid_1.NodeId);
        // ----------- now resolve target NodeId;
        if (params.nodeId instanceof base_node_impl_1.BaseNodeImpl) {
            (0, node_opcua_assert_1.assert)(!Object.prototype.hasOwnProperty.call(params, "node"));
            params.node = params.nodeId;
            params.nodeId = params.node.nodeId;
        }
        else {
            let _nodeId = params.nodeId;
            (0, node_opcua_assert_1.assert)(!!_nodeId, "missing 'nodeId' in reference");
            if (_nodeId && _nodeId.nodeId) {
                _nodeId = _nodeId.nodeId;
            }
            _nodeId = (0, node_opcua_nodeid_1.resolveNodeId)(_nodeId);
            params.nodeId = _nodeId;
        }
        return new reference_impl_1.ReferenceImpl(params);
    }
    normalizeReferenceTypes(references) {
        if (!references || references.length === 0) {
            return [];
        }
        return references.map((el) => this.normalizeReferenceType(el));
    }
    // -- Historical Node  -----------------------------------------------------------------------------------------
    /**
     *
     * @param node
     * @param options
     */
    installHistoricalDataNode(node, options) {
        address_space_historical_data_node_1.AddressSpace_installHistoricalDataNode.call(this, node, options);
    }
    // -- Alarms & Conditions  -----------------------------------------------------------------------------------------
    /**
     *
     */
    installAlarmsAndConditionsService() {
        alarms_and_conditions_1.UAConditionImpl.install_condition_refresh_handle(this);
        alarms_and_conditions_1.UAAcknowledgeableConditionImpl.install_method_handle_on_type(this);
    }
    // -- internal stuff -----------------------------------------------------------------------------------------------
    _coerceNode(node) {
        function hasTypeDefinition(node1) {
            return (node1.nodeClass === node_opcua_data_model_1.NodeClass.Variable ||
                node1.nodeClass === node_opcua_data_model_1.NodeClass.Object ||
                node1.nodeClass === node_opcua_data_model_1.NodeClass.Method);
        }
        // coerce to BaseNode object
        if (node instanceof base_node_impl_1.BaseNodeImpl) {
            return node;
        }
        // it's a node id like
        // coerce parent folder to an object
        const returnValue = this.findNode((0, node_opcua_nodeid_1.resolveNodeId)(node));
        /*
         if (!hasTypeDefinition(node as BaseNode)) {
             node = this.findNode(node.nodeId) || node;
             if (!node || !hasTypeDefinition(node)) {
                 return null;
             }
         }
         */
        return returnValue;
    }
    _coerce_DataType(dataType) {
        if (dataType instanceof node_opcua_nodeid_1.NodeId) {
            // xx assert(self.findDataType(dataType));
            return dataType;
        }
        if (dataType === 0) {
            return node_opcua_nodeid_1.NodeId.nullNodeId;
        }
        return this._coerce_Type(dataType, node_opcua_constants_1.DataTypeIds, "DataTypeIds", AddressSpace.prototype.findDataType);
    }
    _coerceTypeDefinition(typeDefinition) {
        if (typeof typeDefinition === "string") {
            // coerce parent folder to an node
            const typeDefinitionNode = this.findNode(typeDefinition);
            typeDefinition = typeDefinitionNode.nodeId;
        }
        (0, node_opcua_assert_1.assert)(typeDefinition instanceof node_opcua_nodeid_1.NodeId);
        return typeDefinition;
    }
    _coerceType(baseType, topMostBaseType, nodeClass) {
        (0, node_opcua_assert_1.assert)(typeof topMostBaseType === "string");
        const topMostBaseTypeNode = this.findNode(topMostBaseType);
        /* istanbul ignore next */
        if (!topMostBaseTypeNode) {
            throw new Error("Cannot find topMostBaseTypeNode " + topMostBaseType.toString());
        }
        (0, node_opcua_assert_1.assert)(topMostBaseTypeNode instanceof base_node_impl_1.BaseNodeImpl);
        (0, node_opcua_assert_1.assert)(topMostBaseTypeNode.nodeClass === nodeClass);
        if (!baseType) {
            return topMostBaseTypeNode;
        }
        (0, node_opcua_assert_1.assert)(typeof baseType === "string" || baseType instanceof base_node_impl_1.BaseNodeImpl || baseType instanceof node_opcua_nodeid_1.NodeId);
        let baseTypeNode;
        if (baseType instanceof base_node_impl_1.BaseNodeImpl) {
            baseTypeNode = baseType;
        }
        else {
            baseTypeNode = this.findNode(baseType);
        }
        /* istanbul ignore next*/
        if (!baseTypeNode) {
            throw new Error("Cannot find ObjectType or VariableType for " + baseType.toString());
        }
        /* istanbul ignore next */
        if (!baseTypeNode.isSubtypeOf) {
            throw new Error("Cannot find ObjectType or VariableType for " + baseType.toString());
        }
        /* istanbul ignore next */
        if (!baseTypeNode.isSubtypeOf(topMostBaseTypeNode)) {
            throw new Error("wrong type ");
        }
        return baseTypeNode;
    }
    _coerce_VariableTypeIds(dataType) {
        return this._coerce_Type(dataType, node_opcua_constants_1.VariableTypeIds, "VariableTypeIds", AddressSpace.prototype.findVariableType);
    }
    _register(node) {
        (0, node_opcua_assert_1.assert)(node.nodeId instanceof node_opcua_nodeid_1.NodeId);
        const namespace = this.getNamespace(node.nodeId.namespace);
        namespace._register(node);
    }
    deleteNode(nodeOrNodeId) {
        _getNamespace(this, nodeOrNodeId).deleteNode(nodeOrNodeId);
    }
    isEnumeration(dataType) {
        // DataType must be one of Enumeration
        const dataTypeNode = this.findDataType(dataType);
        if (!dataTypeNode) {
            throw new Error(" Cannot find  DataType  " + dataType.toString() + " in standard address Space");
        }
        const enumerationNode = this.findDataType("Enumeration");
        if (!enumerationNode) {
            throw new Error(" Cannot find 'Enumeration' DataType in standard address Space");
        }
        return dataTypeNode.isSubtypeOf(enumerationNode);
    }
    _coerce_Type(dataType, typeMap, typeMapName, finderMethod) {
        if (typeof dataType === "number") {
            return this._coerce_Type((0, node_opcua_nodeid_1.coerceNodeId)(dataType), typeMap, typeMapName, finderMethod);
        }
        if (dataType instanceof base_node_impl_1.BaseNodeImpl) {
            dataType = dataType.nodeId;
        }
        (0, node_opcua_assert_1.assert)(typeMap !== null && typeof typeMap === "object");
        let nodeId;
        if (typeof dataType === "string") {
            if (isNodeIdString(dataType)) {
                const nodeId = this.resolveNodeId(dataType);
                if (nodeId.isEmpty()) {
                    // this is a valid dataType NodeId, but it has no definition in the address space
                    return node_opcua_nodeid_1.NodeId.nullNodeId;
                }
                const node = this.findNode(nodeId);
                if (node) {
                    // if dataType is a nodeId string, we can use it directly
                    return node.nodeId;
                }
                else {
                    warningLog("Expecting valid nodeId of a DataType : ", nodeId.toString());
                    return nodeId;
                }
            }
            const namespace0 = this.getDefaultNamespace();
            // resolve dataType
            nodeId = namespace0.resolveAlias(dataType);
            if (!nodeId) {
                // dataType was not found in the aliases database
                if (typeMap[dataType]) {
                    nodeId = (0, node_opcua_nodeid_1.makeNodeId)(typeMap[dataType], 0);
                    return nodeId;
                }
                else {
                    nodeId = (0, node_opcua_nodeid_1.resolveNodeId)(dataType);
                }
            }
        }
        else if (typeof dataType === "number") {
            nodeId = (0, node_opcua_nodeid_1.makeNodeId)(dataType, 0);
        }
        else {
            nodeId = (0, node_opcua_nodeid_1.resolveNodeId)(dataType);
        }
        /* istanbul ignore next */
        if (nodeId == null || !(nodeId instanceof node_opcua_nodeid_1.NodeId)) {
            throw new Error("Expecting valid nodeId ");
        }
        const el = finderMethod.call(this, nodeId);
        if (!el) {
            // verify that node Id exists in standard type map typeMap
            const find = Object.values(typeMap).filter((a) => a === nodeId.value);
            /* istanbul ignore next */
            if (find.length !== 1) {
                throw new Error(" cannot find " + dataType.toString() + " in typeMap " + typeMapName + " L = " + find.length);
            }
        }
        return nodeId;
    }
    _findReferenceType(refType, namespaceIndex) {
        if (refType instanceof node_opcua_nodeid_1.NodeId) {
            return _find_by_node_id(this, refType, namespaceIndex);
        }
        const [namespace, browseName] = _extract_namespace_and_browse_name_as_string(this, refType, namespaceIndex);
        return namespace.findReferenceType(browseName);
    }
}
exports.AddressSpace = AddressSpace;
function _getNamespace(addressSpace, nodeOrNodId) {
    const nodeId = nodeOrNodId instanceof base_node_impl_1.BaseNodeImpl ? nodeOrNodId.nodeId : nodeOrNodId;
    return addressSpace.getNamespace(nodeId.namespace);
}
function _find_by_node_id(addressSpace, nodeId, namespaceIndex) {
    const obj = addressSpace.findNode(nodeId);
    return obj;
}
/**
 * return true if nodeId is a UAFolder
 * @param addressSpace
 * @param folder
 * @return {Boolean}
 * @private
 */
function _isFolder(addressSpace, folder) {
    const folderType = addressSpace.findObjectType("FolderType");
    (0, node_opcua_assert_1.assert)(folder instanceof base_node_impl_1.BaseNodeImpl);
    (0, node_opcua_assert_1.assert)(folder.typeDefinitionObj);
    return folder.typeDefinitionObj.isSubtypeOf(folderType);
}
function _increase_version_number(node) {
    var uaNodeVersion = node?.getNodeVersion();
    if (uaNodeVersion) {
        let rawValue = uaNodeVersion.readValue().value.value || "";
        let previousValue = parseInt(rawValue || "0", 10);
        if (Number.isNaN(previousValue)) {
            warningLog("NodeVersion was ", rawValue);
            previousValue = 0;
        }
        uaNodeVersion.setValueFromSource({
            dataType: node_opcua_variant_1.DataType.String,
            value: (previousValue + 1).toString()
        });
    }
}
//# sourceMappingURL=address_space.js.map