/**
 * @module node-opcua-address-space
 */
import { ExtraDataTypeManager } from "node-opcua-client-dynamic-extension-object";
import { NodeClass } from "node-opcua-data-model";
import { ExtensionObject } from "node-opcua-extension-object";
import { NodeId, NodeIdLike } from "node-opcua-nodeid";
import { BrowseResult } from "node-opcua-service-browse";
import { BrowseDescription, BrowsePath, BrowsePathResult, ModelChangeStructureDataType } from "node-opcua-types";
import { DataType, VariantT } from "node-opcua-variant";
import { AddReferenceOpts, IEventData, ISessionContext, UAEventType, UAReference, IHistoricalDataNodeOptions, UAVariable, UADataType, UAObjectType, BaseNode, UAMethod, UAVariableType, UAReferenceType, UAObject, UAView, ShutdownTask, RaiseEventData } from "node-opcua-address-space-base";
import { UARootFolder } from "../source/ua_root_folder";
import { ExtensionObjectConstructorFuncWithSchema } from "../source/interfaces/extension_object_constructor";
import { AddressSpacePrivate } from "./address_space_private";
import { isNonEmptyQualifiedName } from "./namespace_impl";
import { NamespacePrivate } from "./namespace_private";
import { ReferenceImpl } from "./reference_impl";
/**
 * returns true if str matches a nodeID, e.g i=123 or ns=...
 */
declare function isNodeIdString(str: unknown): boolean;
/**
 * `AddressSpace` is a collection of UA nodes.
 *
 *     const addressSpace = AddressSpace.create();
 */
export declare class AddressSpace implements AddressSpacePrivate {
    get rootFolder(): UARootFolder;
    static isNonEmptyQualifiedName: typeof isNonEmptyQualifiedName;
    static historizerFactory?: any;
    static create(): AddressSpace;
    private static registry;
    /***
     * @internal
     * @private
     */
    suspendBackReference: boolean;
    isFrugal: boolean;
    historizingNodes: Set<UAVariable>;
    _condition_refresh_in_progress: boolean;
    readonly isNodeIdString: typeof isNodeIdString;
    private readonly _private_namespaceIndex;
    private readonly _namespaceArray;
    private _shutdownTask?;
    private _modelChangeTransactionCounter;
    private _modelChanges;
    constructor();
    /**
     * @internal
     */
    getDataTypeManager(): ExtraDataTypeManager;
    getNamespaceUri(namespaceIndex: number): string;
    /***
     */
    getNamespace(namespaceIndexOrName: string | number): NamespacePrivate;
    /***
     * @return  the  default namespace (standard OPCUA namespace)
     */
    getDefaultNamespace(): NamespacePrivate;
    /***
     *
     * objects instances managed by the server will be created in this namespace.
     *
     * @return  address space own namespace
     */
    getOwnNamespace(): NamespacePrivate;
    /**
     * @return the namespace index of a namespace given by its namespace uri
     *
     */
    getNamespaceIndex(namespaceUri: string): number;
    /**
     *
     * register a new namespace,
     * it is OK to call registerNamespace even if namespace has already been registered;
     * in this case the registerNamespace has no effect and returns the existing namespace.
     *
     * @param namespaceUri {string}
     * @returns {Namespace}
     */
    registerNamespace(namespaceUri: string): NamespacePrivate;
    /***
     * @return {Namespace[]} the namespace array
     */
    getNamespaceArray(): NamespacePrivate[];
    /**
     *
     * @param alias {String} the alias name
     * @param nodeId {NodeId}
     * @internal
     */
    addAlias(alias: string, nodeId: NodeId): void;
    /**
     * find an node by node Id
     * @param nodeId   a nodeId or a string coerce-able to nodeID, representing the object to find.
     * @return {BaseNode|null}
     */
    findNode(nodeId: NodeIdLike): BaseNode | null;
    findMethod(nodeId: NodeId | string): UAMethod | null;
    /**
     * resolved a string or a nodeId to a nodeID
     */
    resolveNodeId(nodeId: NodeIdLike): NodeId;
    /**
     *
     * @param objectType  {String|NodeId|QualifiedName}
     * @param [namespaceIndex=0 {Number}] an optional namespace index
     * @return {UAObjectType|null}
     *
     * @example
     *
     * ```javascript
     *     const objectType = addressSpace.findObjectType("ns=0;i=58");
     *     objectType.browseName.toString().should.eql("BaseObjectType");
     *
     *     const objectType = addressSpace.findObjectType("BaseObjectType");
     *     objectType.browseName.toString().should.eql("BaseObjectType");
     *
     *     const objectType = addressSpace.findObjectType(resolveNodeId("ns=0;i=58"));
     *     objectType.browseName.toString().should.eql("BaseObjectType");
     *
     *     const objectType = addressSpace.findObjectType("CustomObjectType",36);
     *     objectType.nodeId.namespace.should.eql(36);
     *     objectType.browseName.toString().should.eql("BaseObjectType");
     *
     *     const objectType = addressSpace.findObjectType("36:CustomObjectType");
     *     objectType.nodeId.namespace.should.eql(36);
     *     objectType.browseName.toString().should.eql("BaseObjectType");
     * ```
     */
    findObjectType(objectType: NodeIdLike, namespaceIndex?: number): UAObjectType | null;
    /**
     * @param variableType  {String|NodeId}
     * @param [namespaceIndex=0 {Number}] an optional namespace index
     * @return {UAObjectType|null}
     *
     * @example
     *
     * ```javascript
     *     const objectType = addressSpace.findVariableType("ns=0;i=62");
     *     objectType.browseName.toString().should.eql("BaseVariableType");
     *
     *     const objectType = addressSpace.findVariableType("BaseVariableType");
     *     objectType.browseName.toString().should.eql("BaseVariableType");
     *
     *     const objectType = addressSpace.findVariableType(resolveNodeId("ns=0;i=62"));
     *     objectType.browseName.toString().should.eql("BaseVariableType");
     * ```
     */
    findVariableType(variableType: string | NodeId, namespaceIndex?: number): UAVariableType | null;
    /**
     * Find the DataType node from a NodeId or a browseName
     * @param dataType {String|NodeId}
     * @param [namespaceIndex=0 {Number}] an optional namespace index
     * @return {DataType|null}
     *
     *
     * @example
     *
     * ```javascript
     *      const dataDouble = addressSpace.findDataType("Double");
     *      const dataDouble = addressSpace.findDataType(resolveNodeId("ns=0;i=3"));
     * ```
     */
    findDataType(dataType: number | string | NodeId | DataType, namespaceIndex?: number): UADataType | null;
    /**
     *
     * @example
     *
     *     const dataType = addressSpace.findDataType("ns=0;i=12");
     *     addressSpace.findCorrespondingBasicDataType(dataType).should.eql(DataType.String);
     *
     *     const dataType = addressSpace.findDataType("ServerStatusDataType"); // ServerStatus
     *     addressSpace.findCorrespondingBasicDataType(dataType).should.eql(DataType.ExtensionObject);
     *
     */
    findCorrespondingBasicDataType(dataTypeNode: NodeIdLike | UADataType): DataType;
    /**
     * find a ReferenceType by its inverse name.
     * @param inverseName  the inverse name of the ReferenceType to find
     * @deprecated
     */
    findReferenceTypeFromInverseName(inverseName: string): UAReferenceType | null;
    /**
     * @param refType {String|NodeId}
     * @param [namespaceIndex=0 {Number}] an optional namespace index
     * @return {ReferenceType|null}
     *
     * refType could be
     *   - a string representing a nodeid       : e.g.    `i=9004` or ns=1;i=6030
     *   - a string representing a browse name  : e.g     `"HasTypeDefinition"`.
     *     In this case it should be in the alias list.
     *
     */
    findReferenceType(refType: NodeIdLike, namespaceIndex?: number): UAReferenceType | null;
    /**
     * returns the inverse name of the referenceType.
     *
     * @param referenceType {String} : the reference type name
     * @return {String} the name of the inverse reference type.
     *
     * @example
     *
     *    ```javascript
     *    addressSpace.inverseReferenceType("OrganizedBy").should.eql("Organizes");
     *    addressSpace.inverseReferenceType("Organizes").should.eql("OrganizedBy");
     *    ```
     *
     */
    inverseReferenceType(referenceType: string): string;
    /**
     * find an EventType node in the address space
     * @param eventTypeId {String|NodeId|UAObjectType} the eventType to find
     * @param namespaceIndex the namespace index of the event to find
     * @return {UAObjectType|null} the EventType found or null.
     *
     * note:
     *    - the method with throw an exception if a node is found
     *      that is not a BaseEventType or a subtype of it.
     *
     * @example
     *
     *     var evtType = addressSpace.findEventType("AuditEventType");
     *
     */
    findEventType(eventTypeId: NodeIdLike | UAEventType, namespaceIndex?: number): UAEventType | null;
    /**
     * EventId is generated by the Server to uniquely identify a particular Event Notification.
     * @return {Variant}  dataType: "ByteString"
     */
    generateEventId(): VariantT<Buffer, DataType.ByteString>;
    constructEventData(eventTypeId: UAEventType, data: RaiseEventData): IEventData;
    /**
     * browse some path.
     *
     * @param  {BrowsePath} browsePath
     * @return {BrowsePathResult}
     *
     * This service can be used translates one or more browse paths into NodeIds.
     * A browse path is constructed of a starting Node and a RelativePath. The specified starting Node
     * identifies the Node from which the RelativePath is based. The RelativePath contains a sequence of
     * ReferenceTypes and BrowseNames.
     *
     *   |StatusCode                    |                                                            |
     *   |------------------------------|:-----------------------------------------------------------|
     *   |BadNodeIdUnknown              |                                                            |
     *   |BadNodeIdInvalid              |                                                            |
     *   |BadNothingToDo                | - the relative path contains an empty list )               |
     *   |BadBrowseNameInvalid          | - target name is missing in relative path                  |
     *   |UncertainReferenceOutOfServer | - The path element has targets which are in another server.|
     *   |BadTooManyMatches             |                                                            |
     *   |BadQueryTooComplex            |                                                            |
     *   |BadNoMatch                    |                                                            |
     *
     *
     *
     */
    browsePath(browsePath: BrowsePath): BrowsePathResult;
    getExtensionObjectConstructor(dataType: NodeId | UADataType): ExtensionObjectConstructorFuncWithSchema;
    /**
     * @param dataType
     * @param [options]
     * @return the constructed extension object
     *
     *
     * @example
     *
     *             // example 1
     *             var extObj = addressSpace.constructExtensionObject("BuildInfo",{ productName: "PRODUCT_NAME"});
     *
     *             // example 2
     *             serverStatusDataType.nodeClass.should.eql(NodeClass.DataType);
     *             serverStatusDataType.browseName.toString().should.eql("ServerStatusDataType");
     *             var serverStatus  = addressSpace.constructExtensionObject(serverStatusDataType);
     *             serverStatus.should.be.instanceof(ServerStatusDataType);
     */
    constructExtensionObject(dataType: UADataType | NodeId, options?: Record<string, unknown>): ExtensionObject;
    /**
     * cleanup all resources maintained by this addressSpace.
     */
    dispose(): void;
    /**
     * register a function that will be called when the server will perform its shut down.
     */
    registerShutdownTask(task: ShutdownTask): void;
    shutdown(): Promise<void>;
    browseSingleNode(nodeId: NodeIdLike, browseDescription: BrowseDescription, context?: ISessionContext): BrowseResult;
    /**
     * @param folder
     * @private
     */
    _coerceFolder(folder: UAObject): BaseNode | null;
    /**
     *
     * @param view
     * @param modelChange
     * @private
     */
    _collectModelChange(view: UAView | null, modelChange: ModelChangeStructureDataType): void;
    /**
     *
     * walk up the hierarchy of objects until a view is found
     * objects may belong to multiples views.
     * Note: this method doesn't return the main view => Server object.
     * @param node {BaseNode}
     * @return {BaseNode[]}
     */
    /**
     *
     * @param node
     * @private
     */
    extractRootViews(node: UAObject | UAVariable): UAView[];
    /**
     *
     * @param func
     * @private
     */
    modelChangeTransaction(func: () => void): void;
    /**
     * normalize the ReferenceType field of the Reference Object
     * @param params.referenceType  {String|nodeId}
     * @param params.isForward  {Boolean} default value: true;
     * @return {Object} a new reference object  with the normalized name { referenceType: <value>, isForward: <flag>}
     */
    normalizeReferenceType(params: AddReferenceOpts | ReferenceImpl): UAReference;
    normalizeReferenceTypes(references?: AddReferenceOpts[] | ReferenceImpl[] | null): UAReference[];
    /**
     *
     * @param node
     * @param options
     */
    installHistoricalDataNode(node: UAVariable, options?: IHistoricalDataNodeOptions): void;
    /**
     *
     */
    installAlarmsAndConditionsService(): void;
    _coerceNode(node: string | BaseNode | NodeId): BaseNode | null;
    _coerce_DataType(dataType: number | NodeId | string | BaseNode): NodeId;
    _coerceTypeDefinition(typeDefinition: string | NodeId): NodeId;
    _coerceType<T extends BaseNode>(baseType: string | NodeId | BaseNode, topMostBaseType: string, nodeClass: NodeClass): T;
    _coerce_VariableTypeIds(dataType: NodeId | string | BaseNode): NodeId;
    _register(node: BaseNode): void;
    deleteNode(nodeOrNodeId: NodeId | BaseNode): void;
    isEnumeration(dataType: NodeId): boolean;
    private _coerce_Type;
    private _findReferenceType;
}
export {};
