"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PseudoSession = void 0;
exports.innerBrowse = innerBrowse;
exports.innerBrowseNext = innerBrowseNext;
/**
 * @module node-opcua-address-space
 */
const util_1 = require("util");
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_data_value_1 = require("node-opcua-data-value");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_pseudo_session_1 = require("node-opcua-pseudo-session");
const node_opcua_service_browse_1 = require("node-opcua-service-browse");
const node_opcua_service_call_1 = require("node-opcua-service-call");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_basic_types_1 = require("node-opcua-basic-types");
const continuation_point_manager_1 = require("./continuation_points/continuation_point_manager");
const call_helpers_1 = require("./helpers/call_helpers");
const session_context_1 = require("./session_context");
const errorLog = (0, node_opcua_debug_1.make_errorLog)("PseudoSession");
function coerceBrowseDescription(browseDescription) {
    if (typeof browseDescription === "string") {
        return coerceBrowseDescription({
            nodeId: (0, node_opcua_nodeid_1.resolveNodeId)(browseDescription)
        });
    }
    else if (browseDescription instanceof node_opcua_service_browse_1.BrowseDescription) {
        return browseDescription;
    }
    else {
        return new node_opcua_service_browse_1.BrowseDescription(browseDescription);
    }
}
function innerBrowse(engine, nodesToBrowse, callback) {
    engine.browseAll(nodesToBrowse, (err, results) => {
        if (err || !results) {
            return callback(err);
        }
        // handle continuation points
        results = results.map((result, index) => {
            (0, node_opcua_assert_1.assert)(!result.continuationPoint);
            // istanbul ignore next
            if (!engine.continuationPointManager) {
                return new node_opcua_service_browse_1.BrowseResult({ statusCode: node_opcua_status_code_1.StatusCodes.BadNoContinuationPoints });
            }
            if (engine.continuationPointManager.hasReachedMaximum(engine.maxBrowseContinuationPoints)) {
                return new node_opcua_service_browse_1.BrowseResult({ statusCode: node_opcua_status_code_1.StatusCodes.BadNoContinuationPoints });
            }
            const truncatedResult = engine.continuationPointManager.registerReferences(engine.requestedMaxReferencesPerNode, result.references || [], { continuationPoint: null });
            let { statusCode } = truncatedResult;
            const { continuationPoint, values } = truncatedResult;
            (0, node_opcua_assert_1.assert)(statusCode.isGood() || statusCode.equals(node_opcua_status_code_1.StatusCodes.GoodNoData));
            statusCode = result.statusCode;
            return new node_opcua_service_browse_1.BrowseResult({
                statusCode,
                continuationPoint,
                references: values
            });
        });
        callback(null, results);
    });
}
function innerBrowseNext(engine, continuationPoints, releaseContinuationPoints, callback) {
    const results = continuationPoints
        .map((continuationPoint, index) => {
        return engine.continuationPointManager.getNextReferences(0, {
            continuationPoint,
            releaseContinuationPoints
        });
    })
        .map((r) => new node_opcua_service_browse_1.BrowseResult({
        statusCode: r.statusCode,
        continuationPoint: r.continuationPoint,
        references: r.values
    }));
    callback(null, results);
}
const $addressSpace = Symbol("addressSpace");
const $context = Symbol("context");
/**
 * Pseudo session is an helper object that exposes the same async methods
 * than the ClientSession. It can be used on a server address space.
 *
 * Code reused !
 * The primary benefit of this object  is that its makes advanced OPCUA
 * operations that uses browse, translate, read, write etc similar
 * whether we work inside a server or through a client session.
 *
 */
class PseudoSession {
    requestedMaxReferencesPerNode = 0;
    maxBrowseContinuationPoints = 0; // 0=no limits
    _sessionId = new node_opcua_nodeid_1.NodeId(node_opcua_nodeid_1.NodeIdType.GUID, (0, node_opcua_basic_types_1.randomGuid)());
    [$addressSpace];
    continuationPointManager;
    [$context];
    constructor(addressSpace, context) {
        this[$addressSpace] = addressSpace;
        this[$context] = context || session_context_1.SessionContext.defaultContext;
        this.continuationPointManager = new continuation_point_manager_1.ContinuationPointManager();
    }
    getSessionId() {
        return this._sessionId;
    }
    browse(nodesToBrowse, callback) {
        const isArray = Array.isArray(nodesToBrowse);
        if (!isArray) {
            return this.browse([nodesToBrowse], (err, results) => {
                return callback(err, results ? results[0] : undefined);
            });
        }
        const browseAll = (nodesToBrowse, callack) => {
            const results = [];
            for (const browseDescription of nodesToBrowse) {
                browseDescription.referenceTypeId = (0, node_opcua_nodeid_1.resolveNodeId)(browseDescription.referenceTypeId);
                const _browseDescription = coerceBrowseDescription(browseDescription);
                const nodeId = (0, node_opcua_nodeid_1.resolveNodeId)(_browseDescription.nodeId);
                const r = this[$addressSpace].browseSingleNode(nodeId, _browseDescription, this[$context]);
                results.push(r);
            }
            callack(null, results);
        };
        setImmediate(() => {
            innerBrowse({
                browseAll,
                context: this[$context],
                continuationPointManager: this.continuationPointManager,
                requestedMaxReferencesPerNode: this.requestedMaxReferencesPerNode,
                maxBrowseContinuationPoints: this.maxBrowseContinuationPoints
            }, nodesToBrowse, callback);
        });
    }
    read(nodesToRead, callback) {
        const isArray = Array.isArray(nodesToRead);
        if (!isArray) {
            nodesToRead = [nodesToRead];
        }
        const _nodesToRead = nodesToRead;
        const context = this[$context];
        const readV = async (nodeToRead) => {
            const obj = this[$addressSpace].findNode(nodeToRead.nodeId);
            if (!obj) {
                return new node_opcua_data_value_1.DataValue({ statusCode: node_opcua_status_code_1.StatusCodes.BadNodeIdUnknown });
            }
            // refresh the variable value if the attribute to read is the Value attribute
            if (obj.nodeClass === node_opcua_data_model_1.NodeClass.Variable && nodeToRead.attributeId == node_opcua_data_model_1.AttributeIds.Value) {
                return await obj.readValueAsync(context);
            }
            (0, node_opcua_assert_1.assert)(!!nodeToRead.nodeId, "expecting a nodeId");
            (0, node_opcua_assert_1.assert)(!!nodeToRead.attributeId, "expecting a attributeId");
            const attributeId = nodeToRead.attributeId;
            const indexRange = nodeToRead.indexRange;
            const dataEncoding = nodeToRead.dataEncoding;
            const dataValue = obj.readAttribute(context, attributeId, indexRange, dataEncoding);
            return dataValue;
        };
        Promise.all(_nodesToRead.map(async (nodeToRead) => await readV(nodeToRead)))
            .then((dataValues) => callback(null, isArray ? dataValues : dataValues[0]))
            .catch((err) => callback(err));
    }
    browseNext(continuationPoints, releaseContinuationPoints, callback) {
        setImmediate(() => {
            if (continuationPoints instanceof Buffer) {
                return this.browseNext([continuationPoints], releaseContinuationPoints, (err, _results) => {
                    if (err) {
                        return callback(err);
                    }
                    callback(null, _results[0]);
                });
            }
            innerBrowseNext({ continuationPointManager: this.continuationPointManager }, continuationPoints, releaseContinuationPoints, callback);
        });
    }
    call(methodsToCall, callback) {
        const isArray = Array.isArray(methodsToCall);
        if (!isArray) {
            methodsToCall = [methodsToCall];
        }
        Promise.all(methodsToCall.map(async (methodToCall) => {
            const callMethodRequest = new node_opcua_service_call_1.CallMethodRequest(methodToCall);
            try {
                const result = await (0, call_helpers_1.callMethodHelper)(this[$context], this[$addressSpace], callMethodRequest);
                return new node_opcua_service_call_1.CallMethodResult(result);
            }
            catch (err) {
                errorLog("Internal Error = ", err);
                return new node_opcua_service_call_1.CallMethodResult({
                    statusCode: node_opcua_status_code_1.StatusCodes.BadInternalError
                });
            }
        }))
            .then((callMethodResults) => {
            callback(null, isArray ? callMethodResults : callMethodResults[0]);
        })
            .catch((err) => {
            callback(err);
        });
    }
    getArgumentDefinition(methodId, callback) {
        (0, node_opcua_pseudo_session_1.getArgumentDefinitionHelper)(this, methodId)
            .then((result) => {
            callback(null, result);
        })
            .catch((err) => {
            callback(err);
        });
    }
    translateBrowsePath(browsePaths, callback) {
        const isArray = Array.isArray(browsePaths);
        if (!isArray) {
            browsePaths = [browsePaths];
        }
        const browsePathResults = browsePaths.map((browsePath) => {
            return this[$addressSpace].browsePath(browsePath);
        });
        callback(null, isArray ? browsePathResults : browsePathResults[0]);
    }
    write(nodesToWrite, callback) {
        const isArray = nodesToWrite instanceof Array;
        const _nodesToWrite = !isArray ? [nodesToWrite] : nodesToWrite;
        const context = this[$context];
        setImmediate(() => {
            const statusCodesPromises = _nodesToWrite.map((nodeToWrite) => {
                (0, node_opcua_assert_1.assert)(!!nodeToWrite.nodeId, "expecting a nodeId");
                (0, node_opcua_assert_1.assert)(!!nodeToWrite.attributeId, "expecting a attributeId");
                const nodeId = nodeToWrite.nodeId;
                const obj = this[$addressSpace].findNode(nodeId);
                if (!obj) {
                    return node_opcua_status_code_1.StatusCodes.BadNodeIdUnknown;
                }
                try {
                    return (0, util_1.promisify)(obj.writeAttribute).call(obj, context, nodeToWrite);
                }
                catch (err) {
                    return node_opcua_status_code_1.StatusCodes.BadInternalError;
                }
            });
            Promise.all(statusCodesPromises)
                .then((statusCodes) => {
                callback(null, isArray ? statusCodes : statusCodes[0]);
            })
                .catch((err) => {
                callback(err);
            });
        });
    }
}
exports.PseudoSession = PseudoSession;
// tslint:disable:no-var-requires
// tslint:disable:max-line-length
const thenify_ex_1 = require("thenify-ex");
PseudoSession.prototype.read = (0, thenify_ex_1.withCallback)(PseudoSession.prototype.read);
PseudoSession.prototype.write = (0, thenify_ex_1.withCallback)(PseudoSession.prototype.write);
PseudoSession.prototype.browse = (0, thenify_ex_1.withCallback)(PseudoSession.prototype.browse);
PseudoSession.prototype.browseNext = (0, thenify_ex_1.withCallback)(PseudoSession.prototype.browseNext);
PseudoSession.prototype.getArgumentDefinition = (0, thenify_ex_1.withCallback)(PseudoSession.prototype.getArgumentDefinition);
PseudoSession.prototype.call = (0, thenify_ex_1.withCallback)(PseudoSession.prototype.call);
PseudoSession.prototype.translateBrowsePath = (0, thenify_ex_1.withCallback)(PseudoSession.prototype.translateBrowsePath);
//# sourceMappingURL=pseudo_session.js.map