"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeVariantReader = makeVariantReader;
const node_opcua_basic_types_1 = require("node-opcua-basic-types");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_variant_1 = require("node-opcua-variant");
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_debug_1 = require("node-opcua-debug");
const localized_text_parser_1 = require("./localized_text_parser");
const qualified_name_parser_1 = require("./qualified_name_parser");
const extension_object_parser_1 = require("./extension_object_parser");
const nodeid_parser_1 = require("./nodeid_parser");
const debugLog = (0, node_opcua_debug_1.make_debugLog)(__dirname);
function BasicType_parser(dataType, parseFunc) {
    const _parser = {};
    const a = {
        init(name, attrs, parent, endElement) {
            this.value = undefined;
        },
        finish() {
            this.value = parseFunc.call(this, this.text);
        }
    };
    _parser[dataType] = a;
    return _parser;
}
function ListOf(_setValue, dataType, parseFunc) {
    return {
        init() {
            this.listData = [];
        },
        parser: BasicType_parser(dataType, parseFunc),
        finish() {
            _setValue({
                arrayType: node_opcua_variant_1.VariantArrayType.Array,
                dataType: node_opcua_basic_types_1.DataType[dataType],
                value: this.listData
            });
        },
        endElement(element) {
            this.listData.push(this.parser[dataType].value);
        }
    };
}
function installExtensionObjectListInitializationPostTask(postTasks2_AssignedExtensionObjectToDataValue, element) {
    let listExtensionObject = element.listExtensionObject;
    let nodeId = element.parent.parent.obj.nodeId;
    (0, node_opcua_assert_1.assert)(nodeId, "expecting a nodeid");
    const task = async (addressSpace2) => {
        const node = addressSpace2.findNode(nodeId);
        if (!node) {
            debugLog("Cannot find node with nodeId " + nodeId + ". may be the node was marked as deprecated");
        }
        else if (node.nodeClass === node_opcua_data_model_1.NodeClass.Variable) {
            const v = node;
            (0, node_opcua_assert_1.assert)(v.getBasicDataType() === node_opcua_basic_types_1.DataType.ExtensionObject, "expecting an extension object");
            v.bindExtensionObject(listExtensionObject, { createMissingProp: false });
        }
        else if (node.nodeClass === node_opcua_data_model_1.NodeClass.VariableType) {
            // no need to bind a variable type
            const v = node;
            v /*fix me*/.value.value = listExtensionObject;
        }
        listExtensionObject.slice(0);
        listExtensionObject = undefined;
        nodeId = undefined;
    };
    postTasks2_AssignedExtensionObjectToDataValue.push(task);
}
function parser2(_setValue, type, p) {
    return {
        finish() {
            _setValue({
                arrayType: node_opcua_variant_1.VariantArrayType.Scalar,
                dataType: node_opcua_basic_types_1.DataType[type],
                value: p(this.text)
            });
        }
    };
}
const parseUInt64 = (str) => (0, node_opcua_basic_types_1.coerceUInt64)(str);
const parseInt64 = (str) => (0, node_opcua_basic_types_1.coerceInt64)(str);
function makeVariantReader(setValue, setDeferredValue, postExtensionObjectDecoding, translateNodeId) {
    let self;
    const setValue2 = (data) => {
        setValue(self, data);
    };
    const reader = {
        init() {
            /* empty */
            self = this;
        },
        parser: {
            QualifiedName: {
                ...(0, qualified_name_parser_1.makeQualifiedNameParser)(translateNodeId).QualifiedName,
                finish() {
                    setValue2({
                        dataType: node_opcua_basic_types_1.DataType.QualifiedName,
                        value: (0, node_opcua_data_model_1.coerceQualifiedName)({ ...this.qualifiedName })
                    });
                }
            },
            LocalizedText: {
                ...localized_text_parser_1.localizedText_parser.LocalizedText,
                finish() {
                    setValue2({
                        dataType: node_opcua_basic_types_1.DataType.LocalizedText,
                        value: (0, node_opcua_data_model_1.coerceLocalizedText)({ ...this.localizedText })
                    });
                }
            },
            XmlElement: {
                finish() {
                    setValue2({
                        dataType: node_opcua_basic_types_1.DataType.XmlElement,
                        value: this.text
                    });
                }
            },
            String: {
                finish() {
                    setValue2({
                        dataType: node_opcua_basic_types_1.DataType.String,
                        value: this.text
                    });
                }
            },
            Guid: {
                parser: {
                    String: {
                        finish() {
                            const guid = this.text;
                            if (!(0, node_opcua_basic_types_1.isValidGuid)(guid)) {
                                /* ?*/
                            }
                            setValue2({
                                dataType: node_opcua_basic_types_1.DataType.Guid,
                                arrayType: node_opcua_variant_1.VariantArrayType.Scalar,
                                value: this.text
                            });
                        }
                    }
                }
            },
            NodeId: {
                parser: {
                    Identifier: {
                        finish() {
                            const nodeId = this.text;
                            setValue2({
                                dataType: node_opcua_basic_types_1.DataType.NodeId,
                                arrayType: node_opcua_variant_1.VariantArrayType.Scalar,
                                value: translateNodeId((0, node_opcua_nodeid_1.resolveNodeId)(nodeId).toString())
                            });
                        }
                    }
                }
            },
            Boolean: parser2(setValue2, "Boolean", node_opcua_basic_types_1.coerceBoolean),
            Byte: parser2(setValue2, "Byte", parseInt),
            Int16: parser2(setValue2, "Int16", parseInt),
            Int32: parser2(setValue2, "Int32", parseInt),
            Int8: parser2(setValue2, "Int8", parseInt),
            SByte: parser2(setValue2, "SByte", parseInt),
            UInt16: parser2(setValue2, "UInt16", parseInt),
            UInt32: parser2(setValue2, "UInt32", parseInt),
            UInt8: parser2(setValue2, "UInt8", parseInt),
            UInt64: parser2(setValue2, "UInt64", parseUInt64),
            Int64: parser2(setValue2, "Int64", parseInt64),
            ByteString: {
                init() {
                    this.value = null;
                },
                finish() {
                    const base64text = this.text;
                    const byteString = Buffer.from(base64text, "base64");
                    setValue2({
                        arrayType: node_opcua_variant_1.VariantArrayType.Scalar,
                        dataType: node_opcua_basic_types_1.DataType.ByteString,
                        value: byteString
                    });
                }
            },
            Float: {
                finish() {
                    setValue2({
                        dataType: node_opcua_basic_types_1.DataType.Float,
                        value: parseFloat(this.text)
                    });
                }
            },
            Double: {
                finish() {
                    setValue2({
                        dataType: node_opcua_basic_types_1.DataType.Double,
                        value: parseFloat(this.text)
                    });
                }
            },
            ExtensionObject: (0, extension_object_parser_1.makeExtensionObjectInnerParser)(translateNodeId, (extensionObject) => {
                setValue2({
                    dataType: node_opcua_basic_types_1.DataType.ExtensionObject,
                    value: extensionObject
                });
            }, (xmlEncodingNodeId, bodyXML, data) => {
                let _capturedExtensionObject = null;
                // extension object creation will be postponed
                const task0 = (0, extension_object_parser_1.createXMLExtensionObjectDecodingTask)(translateNodeId, xmlEncodingNodeId, bodyXML, (extensionObject) => {
                    _capturedExtensionObject = extensionObject;
                });
                postExtensionObjectDecoding(task0);
                setDeferredValue(self, {
                    dataType: node_opcua_basic_types_1.DataType.ExtensionObject,
                    arrayType: node_opcua_variant_1.VariantArrayType.Scalar,
                    value: null
                }, () => _capturedExtensionObject);
            }),
            ListOfExtensionObject: {
                init() {
                    this.listExtensionObject = [];
                    this.listExtensionObjectXML = [];
                    this.isDeferred = false;
                },
                parser: (0, extension_object_parser_1.makeExtensionObjectParser)(translateNodeId, (extensionOject, self) => {
                    // const self = reader.parser!.ListOfExtensionObject as ListOfExtensionObjectParser;
                    self.listExtensionObject.push(extensionOject);
                }, (xmlEncodingNodeId, bodyXML, self) => {
                    // constœ self = reader.parser!.ListOfExtensionObject as ListOfExtensionObjectParser;
                    self.isDeferred = true;
                    self.listExtensionObjectXML.push({ xmlEncodingNodeId, bodyXML });
                    // extension object creation will be postponed
                    self.listExtensionObject.push(null);
                    const index = self.listExtensionObject.length - 1;
                    (0, node_opcua_assert_1.assert)(index >= 0);
                    let listExtensionObject = self.listExtensionObject;
                    const task0 = (0, extension_object_parser_1.createXMLExtensionObjectDecodingTask)(translateNodeId, xmlEncodingNodeId, bodyXML, (extensionObject) => {
                        listExtensionObject[index] = extensionObject;
                    });
                    postExtensionObjectDecoding(task0);
                }),
                finish() {
                    if (!this.isDeferred) {
                        setValue2({
                            arrayType: node_opcua_variant_1.VariantArrayType.Array,
                            dataType: node_opcua_basic_types_1.DataType.ExtensionObject,
                            value: this.listExtensionObject
                        });
                    }
                    else {
                        // postpone the creation of the extension object
                        const listExtensionObject = this.listExtensionObject;
                        setDeferredValue(self, {
                            arrayType: node_opcua_variant_1.VariantArrayType.Array,
                            dataType: node_opcua_basic_types_1.DataType.ExtensionObject,
                            value: null
                        }, () => listExtensionObject);
                        this.listExtensionObject = [];
                    }
                },
            },
            ListOfLocalizedText: {
                init() {
                    this.listData = [];
                },
                parser: {
                    ...localized_text_parser_1.localizedText_parser
                },
                endElement() {
                    this.listData.push(this.parser.LocalizedText.value);
                },
                finish() {
                    setValue2({
                        arrayType: node_opcua_variant_1.VariantArrayType.Array,
                        dataType: node_opcua_basic_types_1.DataType.LocalizedText,
                        value: this.listData
                    });
                }
            },
            ListOfQualifiedName: {
                init() {
                    this.listData = [];
                },
                parser: (0, qualified_name_parser_1.makeQualifiedNameParser)(translateNodeId),
                endElement() {
                    this.listData.push(this.parser.QualifiedName.value);
                },
                finish() {
                    setValue2({
                        arrayType: node_opcua_variant_1.VariantArrayType.Array,
                        dataType: node_opcua_basic_types_1.DataType.QualifiedName,
                        value: this.listData
                    });
                }
            },
            ListOfNodeId: {
                init() {
                    this.listData = [];
                },
                parser: (0, nodeid_parser_1.makeNodeIdParser)(translateNodeId),
                endElement(elementName) {
                    this.listData.push(this.parser.NodeId.nodeId);
                },
                finish() {
                    setValue2({
                        arrayType: node_opcua_variant_1.VariantArrayType.Array,
                        dataType: node_opcua_basic_types_1.DataType.NodeId,
                        value: this.listData
                    });
                }
            },
            ListOfBoolean: ListOf(setValue2, "Boolean", node_opcua_basic_types_1.coerceBoolean),
            ListOfByte: ListOf(setValue2, "Byte", parseInt),
            ListOfDouble: ListOf(setValue2, "Double", parseFloat),
            ListOfFloat: ListOf(setValue2, "Float", parseFloat),
            ListOfInt32: ListOf(setValue2, "Int32", parseInt),
            ListOfInt16: ListOf(setValue2, "Int16", parseInt),
            ListOfInt8: ListOf(setValue2, "Int8", parseInt),
            ListOfUInt32: ListOf(setValue2, "UInt32", parseInt),
            ListOfUInt16: ListOf(setValue2, "UInt16", parseInt),
            ListOfUInt8: ListOf(setValue2, "UInt8", parseInt),
            ListOfString: ListOf(setValue2, "String", (value) => value),
            ListOfXmlElement: ListOf(setValue2, "XmlElement", (value) => value),
        }
    };
    return reader;
}
//# sourceMappingURL=variant_parser.js.map