/**
 * @module node-opcua-address-space.AlarmsAndConditions
 */
import { EventEmitter } from "events";
import { Variant } from "node-opcua-variant";
import { IEventData, UAVariable, BaseNode } from "node-opcua-address-space-base";
import { UInt16 } from "node-opcua-basic-types";
import { LocalizedText, LocalizedTextLike } from "node-opcua-data-model";
import { NodeId } from "node-opcua-nodeid";
import { StatusCode } from "node-opcua-status-code";
import { TimeZoneDataType } from "node-opcua-types";
import { UtcTime } from "../state_machine/ua_state_machine_type";
import { ISetStateOptions } from "../i_set_state_options";
import { IConditionVariableTypeSetterOptions } from "../i_condition_variable_type_setter_options";
export interface ConditionSnapshot {
    on(eventName: "valueChanged", eventHandler: (node: UAVariable, variant: Variant, options: {
        sourceTimestamp: Date;
    }) => void): this;
}
export interface ConditionSnapshot extends EventEmitter {
    emit(eventName: "valueChanged", node: UAVariable, variant: Variant, options: {
        sourceTimestamp: Date;
    }): boolean;
    emit(eventName: string | symbol): boolean;
    condition: BaseNode;
    eventData: IEventData | null;
    branchId: NodeId | null;
    /**
     */
    getBranchId(): NodeId;
    /**
     */
    getEventId(): Buffer;
    /**
     */
    getRetain(): boolean;
    /**
     */
    setRetain(retainFlag: boolean): void;
    /**
     */
    renewEventId(): void;
    /**
     */
    getEnabledState(): boolean;
    /**
     */
    setEnabledState(value: boolean, options?: ISetStateOptions): void;
    /**
     */
    getEnabledStateAsString(): string;
    /**
     */
    getComment(): LocalizedText;
    /**
     * Set condition comment
     *
     * Comment contains the last comment provided for a certain state (ConditionBranch). It may
     * have been provided by an AddComment Method, some other Method or in some other
     * manner. The initial value of this Variable is null, unless it is provided in some other manner. If
     * a Method provides as an option the ability to set a Comment, then the value of this Variable is
     * reset to null if an optional comment is not provided.
     *
     */
    setComment(txtMessage: LocalizedTextLike, options?: IConditionVariableTypeSetterOptions): void;
    /**
     *
     */
    setMessage(txtMessage: LocalizedTextLike | LocalizedText): void;
    /**
     */
    setClientUserId(userIdentity: string): void;
    /**
     * set the condition quality
     */
    setQuality(quality: StatusCode, options?: IConditionVariableTypeSetterOptions): void;
    /**
     *
     */
    getQuality(): StatusCode;
    setSeverity(severity: UInt16, options?: IConditionVariableTypeSetterOptions): void;
    /**
     */
    getSeverity(): UInt16;
    setLastSeverity(severity: UInt16, options?: IConditionVariableTypeSetterOptions): void;
    /**
     */
    getLastSeverity(): UInt16;
    /**
     * setReceiveTime
     *
     * (as per OPCUA 1.0.3 part 5)
     *
     * ReceiveTime provides the time the OPC UA Server received the Event from the underlying
     * device of another Server.
     *
     * ReceiveTime is analogous to ServerTimestamp defined in Part 4, i.e.
     * in the case where the OPC UA Server gets an Event from another OPC UA Server, each Server
     * applies its own ReceiveTime. That implies that a Client may get the same Event, having the
     * same EventId, from different Servers having different values of the ReceiveTime.
     *
     * The ReceiveTime shall always be returned as value and the Server is not allowed to return a
     * StatusCode for the ReceiveTime indicating an error.
     *
     */
    setReceiveTime(time: UtcTime): void;
    /**
     * (as per OPCUA 1.0.3 part 5)
     * Time provides the time the Event occurred. This value is set as close to the event generator as
     * possible. It often comes from the underlying system or device. Once set, intermediate OPC UA
     * Servers shall not alter the value.
     *
     */
    setTime(time: Date): void;
    /**
     * LocalTime is a structure containing the Offset and the DaylightSavingInOffset flag. The Offset
     * specifies the time difference (in minutes) between the Time Property and the time at the location
     * in which the event was issued. If DaylightSavingInOffset is TRUE, then Standard/Daylight
     * savings time (DST) at the originating location is in effect and Offset includes the DST correction.
     * If FALSE then the Offset does not include DST correction and DST may or may not have been
     * in effect.
     */
    setLocalTime(localTime: TimeZoneDataType): void;
    getSourceName(): LocalizedText;
    /**
     *
     */
    getSourceNode(): NodeId;
    /**
     *
     */
    getEventType(): NodeId;
    getMessage(): LocalizedText;
    isCurrentBranch(): boolean;
    getAckedState(): boolean;
    setAckedState(ackedState: boolean): StatusCode;
    getConfirmedState(): boolean;
    setConfirmedStateIfExists(confirmedState: boolean): void;
    setConfirmedState(confirmedState: boolean): void;
    getSuppressedState(): boolean;
    setSuppressedState(suppressed: boolean): void;
    getActiveState(): boolean;
    setActiveState(newActiveState: boolean): StatusCode;
    setShelvingState(): void;
    toString(): string;
    /**
     * @private
     */
    _constructEventData(): IEventData;
}
