"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ensureObjectIsSecure = ensureObjectIsSecure;
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_types_1 = require("node-opcua-types");
const session_context_1 = require("../session_context");
function isChannelSecure(channel) {
    if (channel.securityMode === node_opcua_types_1.MessageSecurityMode.SignAndEncrypt) {
        return true;
    }
    return false;
}
/**
 * make sure that the given ia node can only be read
 * by Administrator user on a encrypted channel
 * @param node

*/
const restrictedPermissions = [
    { roleId: session_context_1.WellKnownRoles.SecurityAdmin, permissions: node_opcua_data_model_1.allPermissions },
    { roleId: session_context_1.WellKnownRoles.ConfigureAdmin, permissions: node_opcua_data_model_1.allPermissions },
    { roleId: session_context_1.WellKnownRoles.Supervisor, permissions: node_opcua_data_model_1.allPermissions },
    { roleId: session_context_1.WellKnownRoles.Operator, permissions: (0, node_opcua_data_model_1.makePermissionFlag)("Browse") },
    { roleId: session_context_1.WellKnownRoles.Engineer, permissions: (0, node_opcua_data_model_1.makePermissionFlag)("Browse") },
    { roleId: session_context_1.WellKnownRoles.Observer, permissions: (0, node_opcua_data_model_1.makePermissionFlag)("Browse") }
    /*
    { roleId: WellKnownRoles.Anonymous, permissions: makePermissionFlag("Browse") },
    { roleId: WellKnownRoles.AuthenticatedUser, permissions: makePermissionFlag("Browse") },
*/
];
const restrictedAccessFlag = (0, node_opcua_data_model_1.makeAccessRestrictionsFlag)("SigningRequired | EncryptionRequired");
/**
 * this method install the access right restriction on the given node and its children
 * values will only be available to user with role Administrator or supervisor and
 * with a signed and encrypted channel.
 *
 * @param node the node which permissions are to be adjusted
 */
function ensureObjectIsSecure(node) {
    node.setAccessRestrictions(restrictedAccessFlag);
    if (node.nodeClass === node_opcua_data_model_1.NodeClass.Variable) {
        const variable = node;
        variable.setRolePermissions(restrictedPermissions);
    }
    if (node.nodeClass === node_opcua_data_model_1.NodeClass.Method) {
        const method = node;
        method.setRolePermissions(restrictedPermissions);
    }
    if (node.nodeClass === node_opcua_data_model_1.NodeClass.Object) {
        const object = node;
        object.setRolePermissions(restrictedPermissions);
    }
    const children = node.findReferencesExAsObject("Aggregates", node_opcua_data_model_1.BrowseDirection.Forward);
    for (const child of children) {
        ensureObjectIsSecure(child);
    }
}
//# sourceMappingURL=ensure_secure_access.js.map