"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.encode_ArgumentList = encode_ArgumentList;
exports.decode_ArgumentList = decode_ArgumentList;
exports.binaryStoreSize_ArgumentList = binaryStoreSize_ArgumentList;
exports.getMethodDeclaration_ArgumentList = getMethodDeclaration_ArgumentList;
exports.isArgumentValid = isArgumentValid;
exports.verifyArguments_ArgumentList = verifyArguments_ArgumentList;
/* eslint-disable complexity */
/**
 * @module node-opcua-address-space
 */
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_basic_types_1 = require("node-opcua-basic-types");
const node_opcua_binary_stream_1 = require("node-opcua-binary-stream");
const node_opcua_debug_1 = require("node-opcua-debug");
const node_opcua_factory_1 = require("node-opcua-factory");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_variant_1 = require("node-opcua-variant");
const node_opcua_variant_2 = require("node-opcua-variant");
const node_opcua_constants_1 = require("node-opcua-constants");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const debugLog = (0, node_opcua_debug_1.make_debugLog)(__filename);
const warningLog = (0, node_opcua_debug_1.make_warningLog)(__filename);
const doDebug = (0, node_opcua_debug_1.checkDebugFlag)(__filename);
function myfindBuiltInType(dataType) {
    return (0, node_opcua_factory_1.findBuiltInType)(node_opcua_variant_1.DataType[dataType]);
}
function encode_ArgumentList(definition, args, stream) {
    (0, node_opcua_assert_1.assert)(definition.length === args.length);
    (0, node_opcua_assert_1.assert)(Array.isArray(definition));
    (0, node_opcua_assert_1.assert)(Array.isArray(args));
    (0, node_opcua_assert_1.assert)(definition.length === args.length);
    (0, node_opcua_assert_1.assert)(definition.length >= 0);
    // we only encode arguments by following the definition
    for (let i = 0; i < definition.length; i++) {
        const def = definition[i];
        const value = args[i];
        const encodeFunc = myfindBuiltInType(def.dataType).encode;
        // assert((def.valueRank === -1) || (def.valueRank === 0));
        // todo : handle -3 -2
        const isArray = def.valueRank && (def.valueRank === 1 || def.valueRank !== -1);
        if (isArray) {
            (0, node_opcua_basic_types_1.encodeArray)(value, stream, encodeFunc);
        }
        else {
            encodeFunc(value, stream);
        }
    }
}
function decode_ArgumentList(definition, stream) {
    // istanbul ignore next
    if (!Array.isArray(definition)) {
        throw new Error("This BaseDataType cannot be decoded because it has no definition.\n" +
            "Please construct a BaseDataType({definition : [{dataType: DataType.UInt32 }]});");
    }
    const args = [];
    let value;
    for (const def of definition) {
        const decodeFunc = myfindBuiltInType(def.dataType).decode;
        // xx assert(def.valueRank === -1 || def.valueRank==0);
        const isArray = def.valueRank === 1 || def.valueRank === -1;
        if (isArray) {
            value = (0, node_opcua_basic_types_1.decodeArray)(stream, decodeFunc);
        }
        else {
            value = decodeFunc(stream);
        }
        args.push(value);
    }
    return args;
}
function binaryStoreSize_ArgumentList(description, args) {
    (0, node_opcua_assert_1.assert)(Array.isArray(description));
    (0, node_opcua_assert_1.assert)(Array.isArray(args));
    (0, node_opcua_assert_1.assert)(args.length === description.length);
    const stream = new node_opcua_binary_stream_1.BinaryStreamSizeCalculator();
    encode_ArgumentList(description, args, stream);
    return stream.length;
}
function getMethodDeclaration_ArgumentList(addressSpace, objectId, methodId) {
    // find object in address space
    const obj = addressSpace.findNode(objectId);
    if (!obj) {
        // istanbul ignore next
        doDebug && debugLog("cannot find node ", objectId.toString());
        return { statusCode: node_opcua_status_code_1.StatusCodes.BadNodeIdUnknown };
    }
    let objectMethod = obj.getMethodById(methodId);
    if (!objectMethod) {
        // the method doesn't belong to the object, nevertheless
        // the method can be called
        objectMethod = addressSpace.findNode(methodId);
        if (!objectMethod || objectMethod.nodeClass !== node_opcua_data_model_1.NodeClass.Method) {
            warningLog("cannot find method with id", methodId.toString(), "object Id = ", objectId.toString());
            return { statusCode: node_opcua_status_code_1.StatusCodes.BadMethodInvalid };
        }
    }
    const methodDeclarationId = objectMethod.methodDeclarationId;
    const methodDeclaration = addressSpace.findNode(methodDeclarationId);
    if (!methodDeclaration) {
        //  return {statusCode: StatusCodes.BadMethodInvalid};
        return { statusCode: node_opcua_status_code_1.StatusCodes.Good, methodDeclaration: objectMethod };
    }
    // istanbul ignore next
    if (methodDeclaration.nodeClass !== node_opcua_data_model_1.NodeClass.Method) {
        throw new Error("Expecting a Method here");
    }
    return { statusCode: node_opcua_status_code_1.StatusCodes.Good, methodDeclaration };
}
function checkValueRank(argDefinition, arg) {
    const isArray = arg.arrayType === node_opcua_variant_2.VariantArrayType.Array;
    const isMatrix = arg.arrayType === node_opcua_variant_2.VariantArrayType.Matrix;
    if (argDefinition.valueRank > 0) {
        if (argDefinition.valueRank === 1) {
            if (!isArray) {
                return false;
            }
        }
        else {
            if (!isMatrix) {
                return false;
            }
        }
    }
    else if (argDefinition.valueRank === -1) {
        // SCALAR
        if (isArray || isMatrix) {
            return false;
        }
    }
    else if (argDefinition.valueRank === -2) {
        // ANY
    }
    else if (argDefinition.valueRank === -3) {
        // Scalar or OneDim
        if (isMatrix) {
            return false;
        }
    }
    else if (argDefinition.valueRank === 0) {
        // array or matrix
        if (!isArray && !isMatrix) {
            return false;
        }
    }
    return true;
}
/**
 * @private
 */
function isArgumentValid(addressSpace, argDefinition, arg) {
    (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(argDefinition, "dataType"));
    (0, node_opcua_assert_1.assert)(Object.prototype.hasOwnProperty.call(argDefinition, "valueRank"));
    const argDefDataType = addressSpace.findDataType(argDefinition.dataType);
    const argDataType = addressSpace.findDataType((0, node_opcua_nodeid_1.resolveNodeId)(arg.dataType));
    // istanbul ignore next
    if (!argDefDataType) {
        debugLog("dataType ", argDefinition.dataType.toString(), "doesn't exist");
        return false;
    }
    if (argDefinition.valueRank >= 0 && arg.dataType === node_opcua_variant_1.DataType.Null) {
        // this is valid to receive an empty array ith DataType.Null;
        return true;
    }
    if (!checkValueRank(argDefinition, arg)) {
        return false;
    }
    // istanbul ignore next
    if (!argDataType) {
        doDebug && debugLog(" cannot find dataType ", arg.dataType, (0, node_opcua_nodeid_1.resolveNodeId)(arg.dataType));
        doDebug && debugLog(" arg = ", arg.toString());
        doDebug && debugLog(" def =", argDefinition.toString());
        return false;
    }
    // istanbul ignore next
    if (doDebug) {
        doDebug && debugLog(" checking argDefDataType ", argDefDataType.toString());
        doDebug && debugLog(" checking argDataType ", argDataType.toString());
    }
    if (argDataType.nodeId.value === argDefDataType.nodeId.value) {
        return true;
    }
    // check that dataType is of the same type (derived )
    if (argDefDataType.isSubtypeOf(argDataType)) {
        // like argDefDataType IntegerId and argDataType Uint32
        return true;
    }
    if (argDataType.isSubtypeOf(argDefDataType)) {
        // like argDefDataType BaseDataType and argDataType any Type
        return true;
    }
    // special case for Enumeration
    if (arg.dataType === node_opcua_variant_1.DataType.Int32) {
        const enumDataType = addressSpace.findDataType((0, node_opcua_nodeid_1.coerceNodeId)(node_opcua_constants_1.DataTypeIds.Enumeration));
        if (argDefDataType.isSubtypeOf(enumDataType)) {
            return true;
        }
    }
    return false;
}
/**
*/
function verifyArguments_ArgumentList(addressSpace, methodInputArguments, inputArguments) {
    const inputArgumentResults = methodInputArguments.map((methodInputArgument, index) => {
        const argument = inputArguments[index];
        if (!argument) {
            return node_opcua_status_code_1.StatusCodes.BadNoData;
        }
        else if (!isArgumentValid(addressSpace, methodInputArgument, argument)) {
            return node_opcua_status_code_1.StatusCodes.BadTypeMismatch;
        }
        else {
            return node_opcua_status_code_1.StatusCodes.Good;
        }
    });
    if (methodInputArguments.length > inputArguments.length) {
        // istanbul ignore next
        doDebug &&
            debugLog("verifyArguments_ArgumentList " +
                "\n       The client did  specify too many input arguments for the method.  " +
                "\n        expected : " +
                methodInputArguments.length +
                "" +
                "\n        actual   : " +
                inputArguments.length);
        return { inputArgumentResults, statusCode: node_opcua_status_code_1.StatusCodes.BadArgumentsMissing };
    }
    if (methodInputArguments.length < inputArguments.length) {
        // istanbul ignore next
        doDebug &&
            debugLog(" verifyArguments_ArgumentList " +
                "\n        The client did not specify all of the input arguments for the method. " +
                "\n        expected : " +
                methodInputArguments.length +
                "" +
                "\n        actual   : " +
                inputArguments.length);
        return { inputArgumentResults, statusCode: node_opcua_status_code_1.StatusCodes.BadTooManyArguments };
    }
    const hasBadTypeMismatch = inputArgumentResults.includes(node_opcua_status_code_1.StatusCodes.BadTypeMismatch);
    const hasBadOutOfRange = inputArgumentResults.includes(node_opcua_status_code_1.StatusCodes.BadOutOfRange);
    const statusCode = hasBadTypeMismatch || hasBadOutOfRange
        ? hasBadTypeMismatch && !hasBadOutOfRange
            ? node_opcua_status_code_1.StatusCodes.BadTypeMismatch
            : node_opcua_status_code_1.StatusCodes.BadInvalidArgument
        : node_opcua_status_code_1.StatusCodes.Good;
    return {
        inputArgumentResults,
        statusCode
    };
}
//# sourceMappingURL=argument_list.js.map