import { ReferenceDescription } from "node-opcua-types";
import { ContinuationPoint, IContinuationPointManager, IContinuationPointInfo, ContinuationData } from "node-opcua-address-space-base";
import { DataValue } from "node-opcua-data-value";
/**
 * from https://reference.opcfoundation.org/v104/Core/docs/Part4/7.6/
 *
 * A ContinuationPoint is used to pause a Browse, QueryFirst or HistoryRead operation and allow it to be restarted later by calling BrowseNext,
 * QueryNext or HistoryRead.
 * - Operations are paused when the number of results found exceeds the limits set by either the Client or the Server.
 * - The Client specifies the maximum number of results per operation in the request message.
 * - A Server shall not return more than this number of results but it may return fewer results.
 * - The Server allocates a  ContinuationPoint if there are more results to return.
 * - Servers shall support at least one ContinuationPoint per Session.
 * - Servers specify a maximum number of ContinuationPoints per Session in the ServerCapabilities Object defined in OPC 10000-5.
 * - ContinuationPoints remain active until
 *     a/ the Client retrieves the remaining results,
 *     b/ or, the Client releases the ContinuationPoint
 *     c/ or the Session is closed.
 * - A Server shall automatically free ContinuationPoints from prior requests from a Session if they are needed to process a new request
 *   from this Session.
 * - The Server returns a Bad_ContinuationPointInvalid error if a Client tries to use a ContinuationPoint that has been released.
 * - A Client can avoid this situation by completing paused operations before starting new operations.
 *   For Session-less Service invocations, the ContinuationPoints are shared across all Session-less Service invocations from all Clients.
 *   The Server shall support at least the maximum number of ContinuationPoints it would allow for one Session.
 *
 * - Requests will often specify multiple operations that may or may not require a ContinuationPoint.
 * - A Server shall process the operations until it uses the maximum number of continuation points in this response.
 *   Once that happens the Server shall return a Bad_NoContinuationPoints error for any remaining operations. A Client can avoid
 *   this situation by sending requests with a number of operations that do not exceed the maximum number  of ContinuationPoints
 *   per Session defined for the Service in the ServerCapabilities Object defined in OPC 10000-5.
 *   A Client restarts an operation by passing the ContinuationPoint back to the Server. Server should always be able to reuse the ContinuationPoint
 *   provided so Servers shall never return Bad_NoContinuationPoints error when continuing a previously halted operation.
 *   A ContinuationPoint is a subtype of the ByteString data type.
 *
 *
 * for historical access: https://reference.opcfoundation.org/v104/Core/docs/Part11/6.3/
 *
 * The continuationPoint parameter in the HistoryRead Service is used to mark a point from which to continue
 * the read if not all values could be returned in one response. The value is opaque for the Client and is
 * only used to maintain the state information for the Server to continue from. *
 *
 * For HistoricalDataNode requests, a Server may use the timestamp of the last returned data item if the timestamp
 * is unique. This can reduce the need in the Server to store state information for the continuation point.
 * The Client specifies the maximum number of results per operation in the request Message. A Server shall
 * not return more than this number of results but it may return fewer results. The Server allocates a
 * ContinuationPoint if there are more results to return. The Server may return fewer results due to buffer issues
 * or other internal constraints. It may also be required to return a continuationPoint due to HistoryRead
 * parameter constraints. If a request is taking a long time to calculate and is approaching the timeout time, the
 * Server may return partial results with a continuation point. This may be done if the calculation is going to
 * take more time than the Client timeout. In some cases it may take longer than the Client timeout to calculate
 * even one result. Then the Server may return zero results with a continuation point that allows the Server to
 * resume the calculation on the next Client read call. For additional discussions regarding ContinuationPoints
 * and HistoryRead please see the individual extensible HistoryReadDetails parameter in 6.4.
 * If the Client specifies a ContinuationPoint, then the HistoryReadDetails parameter and the TimestampsToReturn
 * parameter are ignored, because it does not make sense to request different parameters when continuing from a
 * previous call. It is permissible to change the dataEncoding parameter with each request.
 * If the Client specifies a ContinuationPoint that is no longer valid, then the Server shall return a
 * Bad_ContinuationPointInvalid error.
 * If the releaseContinuationPoints parameter is set in the request the Server shall not return any data and shall
 * release all ContinuationPoints passed in the request. If the ContinuationPoint for an operation is missing or
 * invalid then the StatusCode for the operation shall be Bad_ContinuationPointInvalid.
 */
export declare class ContinuationPointManager implements IContinuationPointManager {
    private _map;
    constructor();
    /**
     * returns true if the current number of active continuation point has reached the limit
     * specified in maxContinuationPoint
     * @param maxBrowseContinuationPoint
     */
    hasReachedMaximum(maxContinuationPoint: number): boolean;
    registerHistoryReadRaw(numValuesPerNode: number, dataValues: DataValue[], continuationData: ContinuationData): IContinuationPointInfo<DataValue>;
    getNextHistoryReadRaw(numValues: number, continuationData: ContinuationData): IContinuationPointInfo<DataValue>;
    registerReferences(maxElements: number, values: ReferenceDescription[], continuationData: ContinuationData): IContinuationPointInfo<ReferenceDescription>;
    /**
     * - releaseContinuationPoints = TRUE
     *
     *  passed continuationPoints shall be reset to free resources in
     *  the Server. The continuation points are released and the results
     *  and diagnosticInfos arrays are empty.
     *
     * - releaseContinuationPoints = FALSE
     *
     *   passed continuationPoints shall be used to get the next set of
     *   browse information
     */
    getNextReferences(numValues: number, continuationData: ContinuationData): IContinuationPointInfo<ReferenceDescription>;
    clearContinuationPoint(continuationPoint: ContinuationPoint): void;
    clearContinuationPoints(continuationPoints: ContinuationPoint[]): {
        continuationPoint: undefined;
        values: never[];
        statusCode: import("node-opcua-status-code").ConstantStatusCode;
    };
    clear(): void;
    dispose(): void;
    private _register;
    private _getNext;
}
