import React from "react";
import styles from "./IoPorts.css";
import { Portal } from "react-portal";
import { NodeDispatchContext, ConnectionRecalculateContext, StageContext, ContextContext, EditorIdContext } from "../../context";
import Control from "../Control/Control";
import Connection from "../Connection/Connection";
import { PortTypesContext } from "../../context";
import usePrevious from "../../hooks/usePrevious";
import { calculateCurve, getPortRect } from "../../connectionCalculator";
import { STAGE_ID, DRAG_CONNECTION_ID } from "../../constants";
import { NodesActionType } from "../../nodesReducer";
function useTransputs(transputsFn, transputType, nodeId, inputData, connections) {
    const nodesDispatch = React.useContext(NodeDispatchContext);
    const executionContext = React.useContext(ContextContext);
    const transputs = React.useMemo(() => {
        if (Array.isArray(transputsFn))
            return transputsFn;
        return transputsFn(inputData, connections, executionContext);
    }, [transputsFn, inputData, connections, executionContext]);
    const prevTransputs = usePrevious(transputs);
    React.useEffect(() => {
        if (!prevTransputs || Array.isArray(transputsFn))
            return;
        for (const transput of prevTransputs) {
            const current = transputs.find(({ name }) => transput.name === name);
            if (!current) {
                nodesDispatch?.({
                    type: NodesActionType.DESTROY_TRANSPUT,
                    transputType,
                    transput: { nodeId, portName: "" + transput.name }
                });
            }
        }
    }, [
        transputsFn,
        transputs,
        prevTransputs,
        nodesDispatch,
        nodeId,
        transputType
    ]);
    return transputs;
}
const IoPorts = ({ nodeId, inputs = [], outputs = [], connections, inputData, updateNodeConnections }) => {
    const inputTypes = React.useContext(PortTypesContext);
    const triggerRecalculation = React.useContext(ConnectionRecalculateContext);
    const resolvedInputs = useTransputs(inputs, "input", nodeId, inputData, connections);
    const resolvedOutputs = useTransputs(outputs, "output", nodeId, inputData, connections);
    if (!triggerRecalculation || !inputTypes) {
        return null;
    }
    return (React.createElement("div", { className: styles.wrapper, "data-flume-component": "ports" },
        resolvedInputs.length ? (React.createElement("div", { className: styles.inputs, "data-flume-component": "ports-inputs" }, resolvedInputs.map(input => (React.createElement(Input, { ...input, data: inputData[input.name] || {}, isConnected: !!connections.inputs[input.name], triggerRecalculation: triggerRecalculation ?? (() => { }), updateNodeConnections: updateNodeConnections, inputTypes: inputTypes ?? {}, nodeId: nodeId, inputData: inputData, key: input.name }))))) : null,
        !!resolvedOutputs.length && (React.createElement("div", { className: styles.outputs, "data-flume-component": "ports-outputs" }, resolvedOutputs.map(output => (React.createElement(Output, { ...output, triggerRecalculation: triggerRecalculation, inputTypes: inputTypes, nodeId: nodeId, key: output.name })))))));
};
export default IoPorts;
const Input = ({ type, label, name, nodeId, data, controls: localControls, inputTypes, noControls, triggerRecalculation, updateNodeConnections, isConnected, inputData, hidePort }) => {
    const { label: defaultLabel, color, controls: defaultControls = [] } = inputTypes[type] || {};
    const prevConnected = usePrevious(isConnected);
    const controls = localControls || defaultControls;
    React.useEffect(() => {
        if (isConnected !== prevConnected) {
            triggerRecalculation();
        }
    }, [isConnected, prevConnected, triggerRecalculation]);
    return (React.createElement("div", { "data-flume-component": "port-input", className: styles.transput, "data-controlless": isConnected || noControls || !controls.length, onDragStart: e => {
            e.preventDefault();
            e.stopPropagation();
        } },
        !hidePort ? (React.createElement(Port, { type: type, color: color, name: name, nodeId: nodeId, isInput: true, triggerRecalculation: triggerRecalculation })) : null,
        (!controls.length || noControls || isConnected) && (React.createElement("label", { "data-flume-component": "port-label", className: styles.portLabel }, label || defaultLabel)),
        !noControls && !isConnected ? (React.createElement("div", { className: styles.controls }, controls.map(control => (React.createElement(Control, { ...control, nodeId: nodeId, portName: name, triggerRecalculation: triggerRecalculation, updateNodeConnections: updateNodeConnections, inputLabel: label, data: data[control.name], allData: data, key: control.name, inputData: inputData, isMonoControl: controls.length === 1 }))))) : null));
};
const Output = ({ label, name, nodeId, type, inputTypes, triggerRecalculation }) => {
    const { label: defaultLabel, color } = inputTypes[type] || {};
    return (React.createElement("div", { "data-flume-component": "port-output", className: styles.transput, "data-controlless": true, onDragStart: e => {
            e.preventDefault();
            e.stopPropagation();
        } },
        React.createElement("label", { "data-flume-component": "port-label", className: styles.portLabel }, label || defaultLabel),
        React.createElement(Port, { type: type, name: name, color: color, nodeId: nodeId, triggerRecalculation: triggerRecalculation })));
};
const Port = ({ color = "grey", name = "", type, isInput, nodeId, triggerRecalculation }) => {
    const nodesDispatch = React.useContext(NodeDispatchContext);
    const stageState = React.useContext(StageContext) || {
        scale: 1,
        translate: { x: 0, y: 0 }
    };
    const editorId = React.useContext(EditorIdContext);
    const stageId = `${STAGE_ID}${editorId}`;
    const inputTypes = React.useContext(PortTypesContext) ?? {};
    const [isDragging, setIsDragging] = React.useState(false);
    const [dragStartCoordinates, setDragStartCoordinates] = React.useState({
        x: 0,
        y: 0
    });
    const dragStartCoordinatesCache = React.useRef(dragStartCoordinates);
    const port = React.useRef(null);
    const line = React.useRef(null);
    const lineInToPort = React.useRef(null);
    const byScale = (value) => (1 / (stageState?.scale ?? 1)) * value;
    const handleDrag = (e) => {
        const { x, y, width, height } = document
            .getElementById(stageId)
            ?.getBoundingClientRect() ?? { x: 0, y: 0, width: 0, height: 0 };
        if (isInput) {
            const to = {
                x: byScale(e.clientX - x - width / 2) +
                    byScale(stageState?.translate?.x ?? 1),
                y: byScale(e.clientY - y - height / 2) +
                    byScale(stageState?.translate?.y ?? 1)
            };
            lineInToPort.current?.setAttribute("d", calculateCurve(dragStartCoordinatesCache.current, to));
        }
        else {
            const to = {
                x: byScale(e.clientX - x - width / 2) +
                    byScale(stageState?.translate?.x ?? 1),
                y: byScale(e.clientY - y - height / 2) +
                    byScale(stageState?.translate?.y ?? 1)
            };
            line.current?.setAttribute("d", calculateCurve(dragStartCoordinatesCache.current, to));
        }
    };
    const handleDragEnd = (e) => {
        const droppedOnPort = !!e.target?.dataset?.portName;
        if (isInput) {
            const { inputNodeId = "", inputPortName = "", outputNodeId = "", outputPortName = "" } = lineInToPort.current?.dataset ?? {};
            nodesDispatch?.({
                type: NodesActionType.REMOVE_CONNECTION,
                input: { nodeId: inputNodeId, portName: inputPortName },
                output: { nodeId: outputNodeId, portName: outputPortName }
            });
            if (droppedOnPort) {
                const { portName: connectToPortName, nodeId: connectToNodeId, portType: connectToPortType, portTransputType: connectToTransputType } = e.target.dataset;
                if (!connectToPortName || !connectToNodeId || !connectToPortType || !connectToTransputType) {
                    return;
                }
                const isNotSameNode = outputNodeId !== connectToNodeId;
                if (isNotSameNode && connectToTransputType !== "output") {
                    const inputWillAcceptConnection = inputTypes[connectToPortType]?.acceptTypes?.includes(type);
                    if (inputWillAcceptConnection) {
                        nodesDispatch?.({
                            type: NodesActionType.ADD_CONNECTION,
                            input: { nodeId: connectToNodeId, portName: connectToPortName },
                            output: { nodeId: outputNodeId, portName: outputPortName }
                        });
                    }
                }
            }
        }
        else {
            if (droppedOnPort) {
                const { portName: inputPortName, nodeId: inputNodeId, portType: inputNodeType, portTransputType: inputTransputType } = e.target.dataset;
                if (!inputPortName || !inputNodeId || !inputNodeType || !inputTransputType) {
                    return;
                }
                const isNotSameNode = inputNodeId !== nodeId;
                if (isNotSameNode && inputTransputType !== "output") {
                    const inputWillAcceptConnection = inputTypes[inputNodeType]?.acceptTypes?.includes(type);
                    if (inputWillAcceptConnection) {
                        nodesDispatch?.({
                            type: NodesActionType.ADD_CONNECTION,
                            output: { nodeId, portName: name },
                            input: { nodeId: inputNodeId, portName: inputPortName }
                        });
                        triggerRecalculation();
                    }
                }
            }
        }
        setIsDragging(false);
        document.removeEventListener("mouseup", handleDragEnd);
        document.removeEventListener("mousemove", handleDrag);
    };
    const handleDragStart = (e) => {
        e.preventDefault();
        e.stopPropagation();
        const { x: startPortX = 0, y: startPortY = 0, width: startPortWidth = 0 } = port.current?.getBoundingClientRect() || {};
        const { x: stageX = 0, y: stageY = 0, width: stageWidth = 0, height: stageHeight = 0 } = document.getElementById(stageId)?.getBoundingClientRect() || {};
        if (isInput) {
            lineInToPort.current = document.querySelector(`[data-input-node-id="${nodeId}"][data-input-port-name="${name}"]`);
            const portIsConnected = !!lineInToPort.current;
            if (portIsConnected &&
                lineInToPort.current &&
                lineInToPort.current.parentElement) {
                lineInToPort.current.parentElement.style.zIndex = "9999";
                const { x: outputPortX = 0, y: outputPortY = 0, width: outputPortWidth = 0 } = getPortRect(lineInToPort.current.dataset.outputNodeId || "", lineInToPort.current.dataset.outputPortName || "", "output") || {};
                const coordinates = {
                    x: byScale(outputPortX - stageX + outputPortWidth / 2 - stageWidth / 2) + byScale(stageState.translate.x),
                    y: byScale(outputPortY - stageY + outputPortWidth / 2 - stageHeight / 2) + byScale(stageState.translate.y)
                };
                setDragStartCoordinates(coordinates);
                dragStartCoordinatesCache.current = coordinates;
                setIsDragging(true);
                document.addEventListener("mouseup", handleDragEnd);
                document.addEventListener("mousemove", handleDrag);
            }
        }
        else {
            const coordinates = {
                x: byScale(startPortX - stageX + startPortWidth / 2 - stageWidth / 2) +
                    byScale(stageState.translate.x),
                y: byScale(startPortY - stageY + startPortWidth / 2 - stageHeight / 2) +
                    byScale(stageState.translate.y)
            };
            setDragStartCoordinates(coordinates);
            dragStartCoordinatesCache.current = coordinates;
            setIsDragging(true);
            document.addEventListener("mouseup", handleDragEnd);
            document.addEventListener("mousemove", handleDrag);
        }
    };
    return (React.createElement(React.Fragment, null,
        React.createElement("div", { style: { zIndex: 999 }, onMouseDown: handleDragStart, className: styles.port, "data-port-color": color, "data-port-name": name, "data-port-type": type, "data-port-transput-type": isInput ? "input" : "output", "data-node-id": nodeId, "data-flume-component": "port-handle", onDragStart: e => {
                e.preventDefault();
                e.stopPropagation();
            }, ref: port }),
        isDragging && !isInput ? (React.createElement(Portal, { node: document.getElementById(`${DRAG_CONNECTION_ID}${editorId}`) },
            React.createElement(Connection, { from: dragStartCoordinates, to: dragStartCoordinates, lineRef: line }))) : null));
};
//# sourceMappingURL=IoPorts.js.map