import React from "react";
import styles from "./Comment.css";
import Draggable from "../Draggable/Draggable";
import ContextMenu from "../ContextMenu/ContextMenu";
import ColorPicker from "../ColorPicker/ColorPicker";
import { StageContext } from "../../context";
import { Portal } from "react-portal";
import clamp from "lodash/clamp";
import { CommentActionTypes } from "../../commentsReducer";
export default ({ dispatch, id, x, y, width, height, color, text, stageRect, onDragStart, isNew }) => {
    const stageState = React.useContext(StageContext);
    const wrapper = React.useRef(null);
    const textarea = React.useRef(null);
    const [isEditing, setIsEditing] = React.useState(false);
    const [isPickingColor, setIsPickingColor] = React.useState(false);
    const [menuOpen, setMenuOpen] = React.useState(false);
    const [menuCoordinates, setMenuCoordinates] = React.useState({ x: 0, y: 0 });
    const [colorPickerCoordinates, setColorPickerCoordinates] = React.useState({
        x: 0,
        y: 0
    });
    const handleContextMenu = e => {
        e.preventDefault();
        e.stopPropagation();
        setMenuCoordinates({ x: e.clientX, y: e.clientY });
        setMenuOpen(true);
        return false;
    };
    const closeContextMenu = () => setMenuOpen(false);
    const startDrag = (e) => {
        onDragStart();
    };
    const handleDrag = ({ x, y }) => {
        if (wrapper.current) {
            wrapper.current.style.transform = `translate(${x}px,${y}px)`;
        }
    };
    const handleDragEnd = (e, { x, y }) => {
        dispatch({
            type: CommentActionTypes.SET_COMMENT_COORDINATES,
            id,
            x,
            y
        });
    };
    const handleResize = (coordinates) => {
        const width = clamp(coordinates.x - x + 10, 80, 10000);
        const height = clamp(coordinates.y - y + 10, 30, 10000);
        if (wrapper.current) {
            wrapper.current.style.width = `${width}px`;
            wrapper.current.style.height = `${height}px`;
        }
    };
    const handleResizeEnd = (e, coordinates) => {
        const width = clamp(coordinates.x - x + 10, 80, 10000);
        const height = clamp(coordinates.y - y + 10, 30, 10000);
        dispatch({
            type: CommentActionTypes.SET_COMMENT_DIMENSIONS,
            id,
            width,
            height
        });
    };
    const handleMenuOption = (option) => {
        switch (option.value) {
            case "edit":
                startTextEdit();
                break;
            case "color":
                setColorPickerCoordinates(menuCoordinates);
                setIsPickingColor(true);
                console.log(menuCoordinates);
                break;
            case "delete":
                dispatch({
                    type: CommentActionTypes.DELETE_COMMENT,
                    id
                });
                break;
            default:
        }
    };
    const startTextEdit = () => {
        setIsEditing(true);
    };
    const endTextEdit = () => {
        setIsEditing(false);
    };
    const handleTextChange = e => {
        dispatch({
            type: CommentActionTypes.SET_COMMENT_TEXT,
            id,
            text: e.target.value
        });
    };
    const handleColorPicked = (color) => {
        dispatch({
            type: CommentActionTypes.SET_COMMENT_COLOR,
            id,
            color
        });
    };
    const handleRequestClose = React.useCallback(() => setIsPickingColor(false), []);
    React.useEffect(() => {
        if (isNew) {
            setIsEditing(true);
            dispatch({
                type: CommentActionTypes.REMOVE_COMMENT_NEW,
                id
            });
        }
    }, [isNew, dispatch, id]);
    return (React.createElement(Draggable, { innerRef: wrapper, className: styles.wrapper, style: {
            transform: `translate(${x}px,${y}px)`,
            width,
            height,
            zIndex: isEditing ? 999 : ""
        }, stageState: stageState, stageRect: stageRect, onDragStart: startDrag, onDrag: handleDrag, onDragEnd: handleDragEnd, onContextMenu: handleContextMenu, onDoubleClick: startTextEdit, onWheel: e => e.stopPropagation(), "data-color": color, "data-flume-component": "comment" },
        isEditing ? (React.createElement("textarea", { "data-flume-component": "comment-textarea", className: styles.textarea, onChange: handleTextChange, onMouseDown: e => e.stopPropagation(), onBlur: endTextEdit, placeholder: "Text of the comment...", autoFocus: true, value: text, ref: textarea })) : (React.createElement("div", { "data-flume-component": "comment-text", "data-comment": true, className: styles.text }, text)),
        React.createElement(Draggable, { className: styles.resizeThumb, stageState: stageState, stageRect: stageRect, onDrag: handleResize, onDragEnd: handleResizeEnd, "data-flume-component": "comment-resize-handle" }),
        menuOpen ? (React.createElement(Portal, null,
            React.createElement(ContextMenu, { hideFilter: true, label: "Comment Options", x: menuCoordinates.x, y: menuCoordinates.y, options: [
                    {
                        value: "edit",
                        label: "Edit Comment",
                        description: "Edit the text of the comment"
                    },
                    {
                        value: "color",
                        label: "Change Color",
                        description: "Change the color of the comment"
                    },
                    {
                        value: "delete",
                        label: "Delete Comment",
                        description: "Delete the comment"
                    }
                ], onRequestClose: closeContextMenu, onOptionSelected: handleMenuOption }))) : null,
        isPickingColor ? (React.createElement(Portal, null,
            React.createElement(ColorPicker, { x: colorPickerCoordinates.x, y: colorPickerCoordinates.y, onRequestClose: handleRequestClose, onColorPicked: handleColorPicked }))) : null));
};
//# sourceMappingURL=Comment.js.map