"use strict";
/* __LC_ALLOW_ENTRYPOINT_SIDE_EFFECTS__ */
Object.defineProperty(exports, "__esModule", { value: true });
exports.MessagesZodState = exports.MessagesZodMeta = exports.MessagesAnnotation = void 0;
const v3_1 = require("zod/v3");
const annotation_js_1 = require("./annotation.cjs");
const message_js_1 = require("./message.cjs");
const meta_js_1 = require("./zod/meta.cjs");
/**
 * Prebuilt state annotation that combines returned messages.
 * Can handle standard messages and special modifiers like {@link RemoveMessage}
 * instances.
 *
 * Specifically, importing and using the prebuilt MessagesAnnotation like this:
 *
 * @example
 * ```ts
 * import { MessagesAnnotation, StateGraph } from "@langchain/langgraph";
 *
 * const graph = new StateGraph(MessagesAnnotation)
 *   .addNode(...)
 *   ...
 * ```
 *
 * Is equivalent to initializing your state manually like this:
 *
 * @example
 * ```ts
 * import { BaseMessage } from "@langchain/core/messages";
 * import { Annotation, StateGraph, messagesStateReducer } from "@langchain/langgraph";
 *
 * export const StateAnnotation = Annotation.Root({
 *   messages: Annotation<BaseMessage[]>({
 *     reducer: messagesStateReducer,
 *     default: () => [],
 *   }),
 * });
 *
 * const graph = new StateGraph(StateAnnotation)
 *   .addNode(...)
 *   ...
 * ```
 */
exports.MessagesAnnotation = annotation_js_1.Annotation.Root({
    messages: (0, annotation_js_1.Annotation)({
        reducer: message_js_1.messagesStateReducer,
        default: () => [],
    }),
});
/**
 * Prebuilt schema meta for Zod state definition.
 *
 * @example
 * ```ts
 * import { z } from "zod/v4-mini";
 * import { MessagesZodState, StateGraph } from "@langchain/langgraph";
 *
 * const AgentState = z.object({
 *   messages: z.custom<BaseMessage[]>().register(registry, MessagesZodMeta),
 * });
 * ```
 */
exports.MessagesZodMeta = {
    reducer: { fn: message_js_1.messagesStateReducer },
    jsonSchemaExtra: { langgraph_type: "messages" },
    default: () => [],
};
/**
 * Prebuilt state object that uses Zod to combine returned messages.
 * This utility is synonymous with the `MessagesAnnotation` annotation,
 * but uses Zod as the way to express messages state.
 *
 * You can use import and use this prebuilt schema like this:
 *
 * @example
 * ```ts
 * import { MessagesZodState, StateGraph } from "@langchain/langgraph";
 *
 * const graph = new StateGraph(MessagesZodState)
 *   .addNode(...)
 *   ...
 * ```
 *
 * Which is equivalent to initializing the schema object manually like this:
 *
 * @example
 * ```ts
 * import { z } from "zod";
 * import type { BaseMessage, BaseMessageLike } from "@langchain/core/messages";
 * import { StateGraph, messagesStateReducer } from "@langchain/langgraph";
 * import "@langchain/langgraph/zod";
 *
 * const AgentState = z.object({
 *   messages: z
 *     .custom<BaseMessage[]>()
 *     .default(() => [])
 *     .langgraph.reducer(
 *        messagesStateReducer,
 *        z.custom<BaseMessageLike | BaseMessageLike[]>()
 *     ),
 * });
 * const graph = new StateGraph(AgentState)
 *   .addNode(...)
 *   ...
 * ```
 */
exports.MessagesZodState = v3_1.z.object({
    messages: (0, meta_js_1.withLangGraph)(v3_1.z.custom(), exports.MessagesZodMeta),
});
//# sourceMappingURL=messages_annotation.js.map