"use strict";
exports.__esModule = true;
var ajv_1 = require("ajv");
var LOCAL_DEFINITION_REGEX = /^#\/([^\/]+)\/([^\/]+)$/;
var OpenAPIResponseValidator = /** @class */ (function () {
    function OpenAPIResponseValidator(args) {
        var loggingKey = args && args.loggingKey ? args.loggingKey + ': ' : '';
        if (!args) {
            throw new Error("".concat(loggingKey, "missing args argument"));
        }
        if (!args.responses) {
            throw new Error("".concat(loggingKey, "args.responses must be an Object"));
        }
        if (!Object.keys(args.responses).length) {
            throw new Error("".concat(loggingKey, "args.responses must contain at least 1 response object"));
        }
        var errorTransformer = typeof args.errorTransformer === 'function' && args.errorTransformer;
        var v = new ajv_1["default"]({
            useDefaults: true,
            allErrors: true,
            strict: false,
            // @ts-ignore TODO get Ajv updated to account for logger
            logger: false
        });
        this.errorMapper = errorTransformer
            ? makeErrorMapper(errorTransformer)
            : toOpenapiValidationError;
        if (args.customFormats) {
            Object.keys(args.customFormats).forEach(function (format) {
                var func = args.customFormats[format];
                if (typeof func === 'function') {
                    v.addFormat(format, func);
                }
            });
        }
        if (args.externalSchemas) {
            Object.keys(args.externalSchemas).forEach(function (id) {
                v.addSchema(args.externalSchemas[id], id);
            });
        }
        var schemas = getSchemas(args.responses, args.definitions, args.components);
        this.validators = compileValidators(v, schemas);
    }
    OpenAPIResponseValidator.prototype.validateResponse = function (statusCode, response) {
        var validator;
        if (statusCode && statusCode in this.validators) {
            validator = this.validators[statusCode];
        }
        else if (statusCode &&
            statusCode.toString()[0] + 'XX' in this.validators) {
            validator = this.validators[statusCode.toString()[0] + 'XX'];
        }
        else if (this.validators["default"]) {
            validator = this.validators["default"];
        }
        else {
            var message = 'An unknown status code was used and no default was provided.';
            return {
                message: message,
                errors: [
                    {
                        message: message
                    },
                ]
            };
        }
        var isValid = validator({
            response: response === undefined ? null : response
        });
        if (!isValid) {
            return {
                message: 'The response was not valid.',
                errors: validator.errors.map(this.errorMapper)
            };
        }
        return undefined;
    };
    return OpenAPIResponseValidator;
}());
exports["default"] = OpenAPIResponseValidator;
function compileValidators(v, schemas) {
    var validators = {};
    Object.keys(schemas).forEach(function (name) {
        validators[name] = v.compile(transformOpenAPIV3Definitions(schemas[name]));
    });
    return validators;
}
function getSchemas(responses, definitions, components) {
    var schemas = {};
    Object.keys(responses).forEach(function (name) {
        var response = responses[name];
        var schema = response
            ? typeof response.schema === 'object'
                ? response.schema
                : typeof response.content === 'object' &&
                    typeof response.content[Object.keys(response.content)[0]] ===
                        'object' &&
                    typeof response.content[Object.keys(response.content)[0]].schema ===
                        'object'
                    ? response.content[Object.keys(response.content)[0]].schema
                    : { type: 'null' }
            : { type: 'null' };
        schemas[name] = {
            $schema: 'http://json-schema.org/schema#',
            type: 'object',
            properties: {
                response: schema
            },
            definitions: definitions || {},
            components: components || {}
        };
    });
    return schemas;
}
function makeErrorMapper(mapper) {
    return function (ajvError) { return mapper(toOpenapiValidationError(ajvError), ajvError); };
}
function toOpenapiValidationError(error) {
    var validationError = {
        path: "instance".concat(error.instancePath),
        errorCode: "".concat(error.keyword, ".openapi.responseValidation"),
        message: error.message
    };
    validationError.path = validationError.path.replace(/^instance\/(response\/)?/, '');
    return validationError;
}
function recursiveTransformOpenAPIV3Definitions(object) {
    // Transformations //
    // OpenAPIV3 nullable
    if (object.nullable === true) {
        if (object["enum"]) {
            // Enums can not be null with type null
            object.oneOf = [
                { type: 'null' },
                {
                    type: object.type,
                    "enum": object["enum"]
                },
            ];
            delete object.type;
            delete object["enum"];
        }
        else if (object.type) {
            object.type = [object.type, 'null'];
        }
        else if (object.allOf) {
            object.anyOf = [{ allOf: object.allOf }, { type: 'null' }];
            delete object.allOf;
        }
        else if (object.oneOf || object.anyOf) {
            var arr = object.oneOf || object.anyOf;
            arr.push({ type: 'null' });
        }
        delete object.nullable;
    }
    // Remove writeOnly properties from required array
    if (object.properties && object.required) {
        var writeOnlyProps = Object.keys(object.properties).filter(function (key) { return object.properties[key].writeOnly; });
        writeOnlyProps.forEach(function (value) {
            var index = object.required.indexOf(value);
            object.required.splice(index, 1);
        });
    }
    Object.keys(object).forEach(function (attr) {
        if (typeof object[attr] === 'object' && object[attr] !== null) {
            recursiveTransformOpenAPIV3Definitions(object[attr]);
        }
        else if (Array.isArray(object[attr])) {
            object[attr].forEach(function (obj) {
                return recursiveTransformOpenAPIV3Definitions(obj);
            });
        }
    });
}
function transformOpenAPIV3Definitions(schema) {
    if (typeof schema !== 'object') {
        return schema;
    }
    var res = JSON.parse(JSON.stringify(schema));
    recursiveTransformOpenAPIV3Definitions(res);
    return res;
}
//# sourceMappingURL=index.js.map