"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.install_optional_cpu_and_memory_usage_node = install_optional_cpu_and_memory_usage_node;
/* istanbul ignore file */
// tslint:disable:no-console
const os_1 = __importDefault(require("os"));
const util_1 = require("util");
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_constants_1 = require("node-opcua-constants");
const node_opcua_server_1 = require("node-opcua-server");
const node_opcua_status_code_1 = require("node-opcua-status-code");
const node_opcua_variant_1 = require("node-opcua-variant");
const node_opcua_debug_1 = require("node-opcua-debug");
const warningLog = (0, node_opcua_debug_1.make_warningLog)(__filename);
// tslint:disable:no-var-requires
const humanize = require("humanize");
/**

 * @param namespace
 * @param options
 * @param options.browseName
 * @private
 */
function addVariableWithHumanizeText(namespace, options) {
    (0, node_opcua_assert_1.assert)(options.componentOf || options.organizedBy);
    (0, node_opcua_assert_1.assert)(typeof options.description === "string");
    const variable = namespace.addVariable(options);
    // add the xxxAsText property
    namespace.addVariable({
        propertyOf: variable,
        browseName: options.browseName.name.toString() + "AsText",
        dataType: "String",
        description: options.description + " as text",
        minimumSamplingInterval: options.minimumSamplingInterval,
        value: {
            get() {
                const v = options.value.get();
                return new node_opcua_variant_1.Variant({ dataType: node_opcua_variant_1.DataType.String, value: humanize.filesize(v.value) });
            }
        }
    });
}
/**
 *
 * optionally install a CPU Usage and Memory Usage node
 * ( condition : running on linux and require("usage")

 * @param server {OPCUAServer}
 *
 */
function install_optional_cpu_and_memory_usage_node(server) {
    const engine = server.engine;
    (0, node_opcua_assert_1.assert)(engine instanceof node_opcua_server_1.ServerEngine);
    let usage;
    try {
        const usage_module = "usage"; // we use a variable here to prevent error in webpack
        usage = require(usage_module); // a warning will be generated here with webpack as the module name is not a litteral
    }
    catch (err) {
        if (util_1.types.isNativeError(err)) {
            warningLog("err", err.message);
        }
        usage = null;
        // xx return;
    }
    const addressSpace = engine.addressSpace;
    const namespace = addressSpace.getOwnNamespace();
    const vendorServerInfo = addressSpace.findNode(node_opcua_constants_1.ObjectIds.Server_VendorServerInfo);
    let usage_result = { memory: 0, cpu: 100 };
    const pid = typeof process === "object" ? process.pid || 0 : 0;
    if (usage) {
        const options = { keepHistory: true };
        setInterval(() => {
            usage.lookup(pid, options, (err, result) => {
                usage_result = result;
                warningLog("result Used Memory: ", humanize.filesize(result.memory), " CPU ", Math.round(result.cpu), " %");
                if (err) {
                    warningLog("err ", err);
                }
            });
        }, 1000);
        namespace.addVariable({
            componentOf: vendorServerInfo,
            browseName: "CPUUsage",
            dataType: node_opcua_variant_1.DataType.Double,
            description: "Current CPU usage of the server process",
            minimumSamplingInterval: 1000,
            nodeId: "s=CPUUsage",
            value: {
                get: () => {
                    if (!usage_result) {
                        return node_opcua_status_code_1.StatusCodes.BadResourceUnavailable;
                    }
                    return new node_opcua_variant_1.Variant({ dataType: node_opcua_variant_1.DataType.Double, value: Math.round(usage_result.cpu) });
                }
            }
        });
        addVariableWithHumanizeText(namespace, {
            componentOf: vendorServerInfo,
            browseName: "MemoryUsage",
            dataType: node_opcua_variant_1.DataType.UInt32,
            description: "Current memory usage of the server process",
            minimumSamplingInterval: 1000,
            nodeId: "s=MemoryUsage",
            value: {
                get: () => {
                    if (!usage_result) {
                        return node_opcua_status_code_1.StatusCodes.BadResourceUnavailable;
                    }
                    return new node_opcua_variant_1.Variant({ dataType: node_opcua_variant_1.DataType.UInt32, value: usage_result.memory });
                }
            }
        });
    }
    else {
        warningLog("skipping installation of cpu_usage and memory_usage nodes");
    }
    namespace.addVariable({
        componentOf: vendorServerInfo,
        browseName: "PercentageMemoryUsed",
        dataType: node_opcua_variant_1.DataType.Double,
        description: "% of  memory used by the server",
        minimumSamplingInterval: 1000,
        nodeId: "s=PercentageMemoryUsed",
        value: {
            get() {
                const percent_used = Math.round(((os_1.default.totalmem() - os_1.default.freemem()) / os_1.default.totalmem()) * 100);
                return new node_opcua_variant_1.Variant({ dataType: node_opcua_variant_1.DataType.Double, value: percent_used });
            }
        }
    });
    addVariableWithHumanizeText(namespace, {
        componentOf: vendorServerInfo,
        accessLevel: "CurrentRead",
        browseName: "SystemMemoryTotal",
        dataType: node_opcua_variant_1.DataType.UInt64,
        description: "Total Memory usage of the server",
        minimumSamplingInterval: 1000,
        nodeId: "s=SystemMemoryTotal",
        value: {
            get() {
                const memory = os_1.default.totalmem();
                return new node_opcua_variant_1.Variant({ dataType: node_opcua_variant_1.DataType.UInt64, value: memory });
            }
        }
    });
    addVariableWithHumanizeText(namespace, {
        componentOf: vendorServerInfo,
        accessLevel: "CurrentRead",
        browseName: "SystemMemoryFree",
        dataType: "UInt64",
        description: "Free Memory usage of the server in MB",
        minimumSamplingInterval: 1000,
        nodeId: "s=SystemMemoryFree",
        value: {
            get() {
                const memory = os_1.default.freemem();
                return new node_opcua_variant_1.Variant({ dataType: node_opcua_variant_1.DataType.UInt64, value: memory });
            }
        }
    });
    namespace.addVariable({
        componentOf: vendorServerInfo,
        accessLevel: "CurrentRead",
        browseName: "NumberOfCPUs",
        dataType: "UInt32",
        description: "Number of cpus on the server",
        minimumSamplingInterval: 1000,
        nodeId: "s=NumberOfCPUs",
        value: {
            get() {
                return new node_opcua_variant_1.Variant({ dataType: node_opcua_variant_1.DataType.UInt32, value: os_1.default.cpus().length });
            }
        }
    });
    namespace.addVariable({
        componentOf: vendorServerInfo,
        accessLevel: "CurrentRead",
        browseName: "Arch",
        dataType: "String",
        description: "ServerArchitecture",
        minimumSamplingInterval: 1000,
        nodeId: "s=ServerArchitecture",
        value: {
            get() {
                return new node_opcua_variant_1.Variant({ dataType: node_opcua_variant_1.DataType.String, value: os_1.default.type() });
            }
        }
    });
    addVariableWithHumanizeText(namespace, {
        componentOf: vendorServerInfo,
        accessLevel: "CurrentRead",
        browseName: "BytesWritten",
        dataType: "UInt64",
        description: "number of bytes written by the server",
        minimumSamplingInterval: 1000,
        nodeId: "s=BytesWritten",
        value: {
            get() {
                return new node_opcua_variant_1.Variant({ dataType: node_opcua_variant_1.DataType.UInt64, value: server.bytesWritten });
            }
        }
    });
    addVariableWithHumanizeText(namespace, {
        componentOf: vendorServerInfo,
        accessLevel: "CurrentRead",
        browseName: "BytesRead",
        dataType: "UInt64",
        description: "number of bytes read by the server",
        minimumSamplingInterval: 1000,
        nodeId: "s=BytesRead",
        value: {
            get() {
                return new node_opcua_variant_1.Variant({ dataType: node_opcua_variant_1.DataType.UInt64, value: server.bytesRead });
            }
        }
    });
    namespace.addVariable({
        componentOf: vendorServerInfo,
        accessLevel: "CurrentRead",
        browseName: "TransactionsCount",
        dataType: "UInt32",
        description: "total number of transactions performed the server",
        minimumSamplingInterval: 1000,
        nodeId: "s=TransactionsCount",
        value: {
            get() {
                return new node_opcua_variant_1.Variant({ dataType: node_opcua_variant_1.DataType.UInt32, value: server.transactionsCount });
            }
        }
    });
    namespace.addVariable({
        componentOf: vendorServerInfo,
        accessLevel: "CurrentRead",
        browseName: "ConnectionsCount",
        dataType: "String",
        description: "number of active Connections",
        minimumSamplingInterval: 1000,
        nodeId: "s=ConnectionCount",
        value: {
            get() {
                return new node_opcua_variant_1.Variant({ dataType: node_opcua_variant_1.DataType.String, value: humanize.filesize(server.currentChannelCount) });
            }
        }
    });
}
//# sourceMappingURL=vendor_diagnostic_nodes.js.map