"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeadbandType = void 0;
exports.isOutsideDeadbandNone = isOutsideDeadbandNone;
exports.isOutsideDeadbandAbsolute = isOutsideDeadbandAbsolute;
exports.isOutsideDeadbandPercent = isOutsideDeadbandPercent;
/**
 * @module node-opcua-service-subscription
 */
const node_opcua_assert_1 = require("node-opcua-assert");
const node_opcua_variant_1 = require("node-opcua-variant");
var DeadbandType;
(function (DeadbandType) {
    DeadbandType[DeadbandType["None"] = 0] = "None";
    DeadbandType[DeadbandType["Absolute"] = 1] = "Absolute";
    DeadbandType[DeadbandType["Percent"] = 2] = "Percent";
    DeadbandType[DeadbandType["Invalid"] = 4096] = "Invalid";
})(DeadbandType || (exports.DeadbandType = DeadbandType = {}));
/**
 * @returns true if the difference between value1 and value2 is greater than absoluteDeadband
 */
function _isOutsideDeadbandScalar(value1, value2, dataType, absoluteDeadband) {
    let diff;
    if (dataType === node_opcua_variant_1.DataType.UInt64 || dataType === node_opcua_variant_1.DataType.Int64) {
        // istanbul ignore next
        if (!(value1 instanceof Array && value2 instanceof Array)) {
            throw new Error("Invalid");
        }
        const h1 = value1[0]; // high
        const h2 = value2[0];
        if (h1 !== h2) {
            diff = (h1 - h2) * 4294967295;
            if (Math.abs(diff) > absoluteDeadband) {
                return true;
            }
        }
        diff = value1[1] - value2[1];
        (0, node_opcua_assert_1.assert)(typeof diff === "number" && isFinite(diff));
        return Math.abs(diff) > absoluteDeadband;
    }
    // istanbul ignore next
    if (!(typeof value1 === "number" && typeof value2 === "number")) {
        throw new Error("Invalid value in _isOutsideDeadbandScalar > expecting number only but got " + typeof value1 + " " + typeof value2);
    }
    diff = value2 - value1;
    (0, node_opcua_assert_1.assert)(typeof diff === "number" && isFinite(diff));
    return Math.abs(diff) > absoluteDeadband;
}
function isOutsideDeadbandVariant(v1, v2, absoluteDeadBand) {
    (0, node_opcua_assert_1.assert)(isFinite(absoluteDeadBand));
    if (v1.arrayType === node_opcua_variant_1.VariantArrayType.Array || v1.arrayType === node_opcua_variant_1.VariantArrayType.Matrix) {
        // If the Value of the MonitoredItem is an array, then the deadband calculation logic shall be applied to
        // each element of the array. If an element that requires a DataChange is found, then no further
        // deadband checking is necessary and the entire array shall be returned.
        if (v1.dataType !== v2.dataType) {
            return true;
        }
        if (v1.value.length !== v2.value.length) {
            return true;
        }
        const n = v1.value.length;
        let i = 0;
        for (i = 0; i < n; i++) {
            if (_isOutsideDeadbandScalar(v1.value[i], v2.value[i], v1.dataType, absoluteDeadBand)) {
                return true;
            }
        }
        return false;
    }
    else {
        (0, node_opcua_assert_1.assert)(v1.arrayType === node_opcua_variant_1.VariantArrayType.Scalar);
        if (v1.dataType !== v2.dataType) {
            return true;
        }
        return _isOutsideDeadbandScalar(v1.value, v2.value, v1.dataType, absoluteDeadBand);
    }
}
// function isOnEdgeOfRangeScalar(
//     currentValue: NumberType,
//     newValue: NumberType,
//     dataType: DataType,
//     range: PseudoRange,
//     deadbandValue: number
// ): boolean {
//     if (dataType === DataType.UInt64 || dataType === DataType.Int64) {
//         // istanbul ignore next
//         if (!(currentValue instanceof Array && newValue instanceof Array)) {
//             throw new Error("Invalid");
//         }
//         currentValue = currentValue[1];
//         newValue = newValue[1];
//     }
//     if (Array.isArray(newValue)) throw new Error("internal error");
//     if (/*currentValue !== range.low && */ Math.abs(newValue - range.low) < deadbandValue) {
//         return true;
//     }
//     if (/*currentValue !== range.high && */ Math.abs(newValue - range.high) < deadbandValue) {
//         return true;
//     }
//     return false;
// }
// function isOnEdgeOfRange(currentValue: Variant, newValue: Variant, range: PseudoRange, deadbandValue: number): boolean {
//     if (currentValue.arrayType === VariantArrayType.Array) {
//         const n = currentValue.value.length;
//         let i = 0;
//         for (i = 0; i < n; i++) {
//             if (isOnEdgeOfRangeScalar(currentValue.value[i], newValue.value[i], newValue.dataType, range, deadbandValue)) {
//                 return true;
//             }
//         }
//         return false;
//     } else {
//         assert(currentValue.arrayType === VariantArrayType.Scalar);
//         assert(currentValue.dataType === newValue.dataType);
//         return isOnEdgeOfRangeScalar(currentValue.value, newValue.value, currentValue.dataType, range, deadbandValue);
//     }
// }
/**

 * @return true if the element is in deadBand
 */
function isOutsideDeadbandNone(variant1, variant2) {
    // No Deadband calculation should be applied.
    return variant1.value !== variant2.value;
}
/**

 * @return true if the element is in deadBand
 */
function isOutsideDeadbandAbsolute(variant1, variant2, deadbandValue) {
    // No Deadband calculation should be applied.
    return isOutsideDeadbandVariant(variant1, variant2, deadbandValue);
}
/**

 * @return true if the element is outside deadBand
 */
function isOutsideDeadbandPercent(variant1, variant2, deadbandValuePercent, range) {
    // DeadbandType = PercentDeadband
    // For this type of deadband the deadbandValue is defined as the percentage of the EURange. That is,
    // it applies only to AnalogItems with an EURange Property that defines the typical value range for the
    // item. This range shall be multiplied with the deadbandValue and then compared to the actual value change
    // to determine the need for a data change notification. The following pseudo code shows how the deadband
    // is calculated:
    //      DataChange if (absolute value of (last cached value - current value) >
    //                                          (deadbandValue/100.0) * ((high-low) of EURange)))
    //
    // The range of the deadbandValue is from 0.0 to 100.0 Percent.
    (0, node_opcua_assert_1.assert)(deadbandValuePercent >= 0 && deadbandValuePercent <= 100);
    const valueRange = Math.abs(range.high - range.low);
    const deadBandValueAbsolute = (deadbandValuePercent / 100.0) * valueRange;
    // if (isOnEdgeOfRange(variant1, variant2, range, deadBandValueAbsolute)) {
    //     return true;
    // }
    return isOutsideDeadbandAbsolute(variant1, variant2, deadBandValueAbsolute);
}
//# sourceMappingURL=deadband_checker.js.map