"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.constructSimpleBrowsePath = constructSimpleBrowsePath;
exports.constructSelectClause = constructSelectClause;
exports.constructEventFilter = constructEventFilter;
/**
 * @module node-opcua-service-filter
 */
const node_opcua_constants_1 = require("node-opcua-constants");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const imports_1 = require("./imports");
function coerceQualifiedName2(a) {
    if (typeof a === "string")
        return (0, node_opcua_data_model_1.coerceQualifiedName)(a);
    a.namespaceIndex = a.namespaceIndex || 0; // need a namespaceIndex
    return (0, node_opcua_data_model_1.coerceQualifiedName)(a);
}
function constructSimpleBrowsePath(a) {
    if (typeof a === "string") {
        return constructSimpleBrowsePath(a.split("."));
    }
    if (Array.isArray(a))
        return a.map(coerceQualifiedName2);
    return [coerceQualifiedName2(a)];
}
function constructSelectClause(arrayOfNames) {
    if (!Array.isArray(arrayOfNames)) {
        return constructSelectClause([arrayOfNames]);
    }
    // construct browse paths array
    const browsePaths = arrayOfNames.map(constructSimpleBrowsePath);
    // Part 4 page 127:
    // In some cases the same BrowsePath will apply to multiple EventTypes. If the Client specifies the BaseEventType
    // in the SimpleAttributeOperand then the Server shall evaluate the BrowsePath without considering the Type.
    const isBrowsePathForConditionId = (browsePath) => browsePath.length === 1 && browsePath[0] && browsePath[0].namespaceIndex === 0 && browsePath[0].name === "ConditionId";
    // [..]
    // The SimpleAttributeOperand structure allows the Client to specify any Attribute, however, the Server is only
    // required to support the Value Attribute for Variable Nodes and the NodeId Attribute for Object Nodes.
    // That said, profiles defined in Part 7 may make support for additional Attributes mandatory.
    const selectClauses = browsePaths.map((browsePath) => {
        if (isBrowsePathForConditionId(browsePath)) {
            // special case
            //
            // The NodeId of the Condition instance is used as ConditionId. It is not explicitly modelled as a
            // component of the ConditionType. However, it can be requested with the following
            // SimpleAttributeOperand (see Table 10) in the SelectClause of the EventFilter:
            //
            //  SimpleAttributeOperand
            //  Name          Type          Description
            //  typeId        NodeId        NodeId of the ConditionType Node
            //  browsePath[]  QualifiedName empty
            //  attributeId   IntegerId     Id of the NodeId Attribute
            //
            return new imports_1.SimpleAttributeOperand({
                attributeId: node_opcua_data_model_1.AttributeIds.NodeId,
                browsePath: null,
                indexRange: undefined, //  NumericRange
                typeDefinitionId: node_opcua_constants_1.ObjectTypeIds.ConditionType // i=2782
            });
        }
        else
            return new imports_1.SimpleAttributeOperand({
                attributeId: node_opcua_data_model_1.AttributeIds.Value,
                browsePath,
                indexRange: undefined, //  NumericRange
                typeDefinitionId: node_opcua_constants_1.ObjectTypeIds.BaseEventType // i=2041
            });
    });
    return selectClauses;
}
/**
 * helper to construct event filters:
 * construct a simple event filter
 *
 *  "ConditionId" in the arrayOfNames has a special meaning
 *
 * @example
 *
 *     constructEventFilter(["SourceName","Message","ReceiveTime"]);
 *     constructEventFilter(["SourceName","2:MyData" ]);
 *     constructEventFilter(["SourceName" ,["EnabledState","EffectiveDisplayName"] ]);
 *     constructEventFilter(["SourceName" ,"EnabledState.EffectiveDisplayName" ]);
 *     constructEventFilter([ ["SourceName",{namespaceIndex:2 , "MyData"} ]]);
 *
 */
function constructEventFilter(arrayOfNames, whereClause) {
    const selectClauses = constructSelectClause(arrayOfNames);
    if (whereClause instanceof imports_1.ContentFilterElement) {
        whereClause = new imports_1.ContentFilter({ elements: [whereClause] });
    }
    const filter = new imports_1.EventFilter({
        selectClauses,
        whereClause
    });
    return filter;
}
//# sourceMappingURL=tools_event_filter.js.map