"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.checkFilter = checkFilter;
const node_opcua_types_1 = require("node-opcua-types");
const node_opcua_variant_1 = require("node-opcua-variant");
const node_opcua_data_model_1 = require("node-opcua-data-model");
const node_opcua_nodeid_1 = require("node-opcua-nodeid");
const node_opcua_debug_1 = require("node-opcua-debug");
const resolve_operand_1 = require("./resolve_operand");
const warningLog = (0, node_opcua_debug_1.make_warningLog)("Filter");
function _coerceToBoolean(value) {
    if (value instanceof node_opcua_variant_1.Variant) {
        return !!value.value;
    }
    return !!value;
}
function _coerceToCompareable(value) {
    if (value instanceof node_opcua_variant_1.Variant) {
        switch (value.dataType) {
            case node_opcua_variant_1.DataType.String:
                return value.value;
            case node_opcua_variant_1.DataType.Byte:
            case node_opcua_variant_1.DataType.SByte:
            case node_opcua_variant_1.DataType.Int16:
            case node_opcua_variant_1.DataType.Int32:
            case node_opcua_variant_1.DataType.UInt16:
            case node_opcua_variant_1.DataType.UInt32:
                return value.value;
            case node_opcua_variant_1.DataType.Double:
                return value.value;
            case node_opcua_variant_1.DataType.DateTime:
                return value.value.getTime();
            case node_opcua_variant_1.DataType.Guid:
                return value.value.toLowerCase();
            case node_opcua_variant_1.DataType.ByteString:
                return value.value.toString("hex");
            case node_opcua_variant_1.DataType.XmlElement:
                return value.value;
            case node_opcua_variant_1.DataType.LocalizedText:
                return value.value.text;
            case node_opcua_variant_1.DataType.QualifiedName:
                return value.value.name; // not sure about this one
            default:
                return "";
        }
    }
    return "";
}
function _coerceToEqualable(value) {
    return value;
}
function evaluateOperand(filterContext, filter, operand, coerce) {
    if (operand instanceof node_opcua_types_1.AttributeOperand) {
        return coerce((0, resolve_operand_1.resolveOperand)(filterContext, operand));
    }
    else if (operand instanceof node_opcua_types_1.SimpleAttributeOperand) {
        return coerce((0, resolve_operand_1.resolveOperand)(filterContext, operand));
    }
    else if (operand instanceof node_opcua_types_1.LiteralOperand) {
        return coerce(operand.value);
    }
    else if (operand instanceof node_opcua_types_1.ElementOperand) {
        const index = operand.index;
        return coerce(checkFilterAtIndex(filterContext, filter, index));
    }
    // istanbul ignore
    return coerce(null);
}
function checkOfType(filterContext, ofType) {
    // istanbul ignore next
    if (!ofType || !(ofType instanceof node_opcua_types_1.LiteralOperand)) {
        warningLog("checkOfType : unsupported case ! ofType is not a LiteralOperand , ofType = ", ofType?.toString());
        return false;
    }
    // istanbul ignore next
    if (ofType.value.dataType !== node_opcua_variant_1.DataType.NodeId) {
        warningLog("invalid operand type (expecting NodeId); got " + node_opcua_variant_1.DataType[ofType.value.dataType]);
        return false;
    }
    const ofTypeNode = ofType.value.value;
    // istanbul ignore next
    if (!ofTypeNode) {
        return false; // the ofType node is not known, we don't know what to do
    }
    const ofTypeNodeNodeClass = filterContext.getNodeClass(ofTypeNode);
    // istanbul ignore next
    if (ofTypeNodeNodeClass !== node_opcua_data_model_1.NodeClass.ObjectType &&
        ofTypeNodeNodeClass !== node_opcua_data_model_1.NodeClass.VariableType &&
        ofTypeNodeNodeClass !== node_opcua_data_model_1.NodeClass.DataType &&
        ofTypeNodeNodeClass !== node_opcua_data_model_1.NodeClass.ReferenceType) {
        warningLog("operand should be a ObjectType " + ofTypeNode.toString());
        return false;
    }
    if (!filterContext.eventSource || filterContext.eventSource.isEmpty()) {
        return false;
    }
    let sourceTypeDefinition = filterContext.eventSource;
    const sourceNodeClass = filterContext.getNodeClass(filterContext.eventSource);
    if (sourceNodeClass === node_opcua_data_model_1.NodeClass.Object || sourceNodeClass === node_opcua_data_model_1.NodeClass.Variable) {
        sourceTypeDefinition = filterContext.getTypeDefinition(filterContext.eventSource);
        if (!sourceTypeDefinition) {
            return false;
        }
    }
    return filterContext.isSubtypeOf(sourceTypeDefinition, ofTypeNode);
}
function checkNot(filterContext, filter, filteredOperands) {
    const operandA = evaluateOperand(filterContext, filter, filteredOperands[0], _coerceToBoolean);
    return !operandA;
}
function checkOr(filterContext, filter, filteredOperands) {
    const operandA = evaluateOperand(filterContext, filter, filteredOperands[0], _coerceToBoolean);
    const operandB = evaluateOperand(filterContext, filter, filteredOperands[1], _coerceToBoolean);
    return operandA || operandB;
}
/**
 *
 * TRUE if operand[0] and operand[1] are TRUE.
 * The following restrictions apply to the operands:
 *  [0]: Any operand that resolves to a Boolean.
 *
 *   [1]: Any operand that resolves to a Boolean.
 * If any operand cannot be resolved to a Boolean it is considered a NULL. See below for a discussion on the handling of NULL.
 */
function checkAnd(filterContext, filter, filteredOperands) {
    const operandA = evaluateOperand(filterContext, filter, filteredOperands[0], _coerceToBoolean);
    const operandB = evaluateOperand(filterContext, filter, filteredOperands[1], _coerceToBoolean);
    return operandA && operandB;
}
function checkLessThan(filterContext, filter, filteredOperands) {
    const operandA = evaluateOperand(filterContext, filter, filteredOperands[0], _coerceToCompareable);
    const operandB = evaluateOperand(filterContext, filter, filteredOperands[1], _coerceToCompareable);
    return operandA < operandB;
}
function checkLessThanOrEqual(filterContext, filter, filteredOperands) {
    const operandA = evaluateOperand(filterContext, filter, filteredOperands[0], _coerceToCompareable);
    const operandB = evaluateOperand(filterContext, filter, filteredOperands[1], _coerceToCompareable);
    return operandA <= operandB;
}
function checkGreaterThan(filterContext, filter, filteredOperands) {
    const operandA = evaluateOperand(filterContext, filter, filteredOperands[0], _coerceToCompareable);
    const operandB = evaluateOperand(filterContext, filter, filteredOperands[1], _coerceToCompareable);
    return operandA > operandB;
}
function checkGreaterThanOrEqual(filterContext, filter, filteredOperands) {
    const operandA = evaluateOperand(filterContext, filter, filteredOperands[0], _coerceToCompareable);
    const operandB = evaluateOperand(filterContext, filter, filteredOperands[1], _coerceToCompareable);
    return operandA >= operandB;
}
const isVariantEqual = (a, b) => {
    switch (a.dataType) {
        case node_opcua_variant_1.DataType.Null:
            return b.dataType === node_opcua_variant_1.DataType.Null;
        case node_opcua_variant_1.DataType.Boolean:
            return a.value === b.value;
        case node_opcua_variant_1.DataType.Byte:
        case node_opcua_variant_1.DataType.SByte:
        case node_opcua_variant_1.DataType.Int16:
        case node_opcua_variant_1.DataType.Int32:
        case node_opcua_variant_1.DataType.UInt16:
        case node_opcua_variant_1.DataType.UInt32:
            return a.value === b.value;
        case node_opcua_variant_1.DataType.Double:
            return a.value === b.value;
        case node_opcua_variant_1.DataType.String:
            return a.value === b.value;
        case node_opcua_variant_1.DataType.NodeId:
            return (0, node_opcua_nodeid_1.sameNodeId)(a.value, b.value);
        case node_opcua_variant_1.DataType.DateTime:
            return a.value?.getTime() === b.value.getTime();
        case node_opcua_variant_1.DataType.Guid:
            return a.value.toLowerCase() === ("" + (b.value || "")).toLowerCase();
        case node_opcua_variant_1.DataType.ByteString:
            if (b.dataType !== node_opcua_variant_1.DataType.ByteString) {
                return false;
            }
            return a.value.toString("hex") === b.value.toString("hex");
        case node_opcua_variant_1.DataType.XmlElement:
            return a.value === b.value;
        case node_opcua_variant_1.DataType.LocalizedText:
            return a.value?.text === b.value?.text;
        case node_opcua_variant_1.DataType.QualifiedName:
            return a.value?.namespaceIndex === b.value?.namespaceIndex && a.value?.name === b.value?.name;
        case node_opcua_variant_1.DataType.ExtensionObject:
            console.log("isVariantEqual: Not implemented for DataType.ExtensionObject");
        default:
            return false; // not sure how to do
    }
};
function checkEquals(filterContext, filter, filteredOperands) {
    const operandA = evaluateOperand(filterContext, filter, filteredOperands[0], _coerceToEqualable);
    const operandB = evaluateOperand(filterContext, filter, filteredOperands[1], _coerceToEqualable);
    return isVariantEqual(operandA, operandB);
}
/**
 *
 * TRUE if operand[0] is greater or equal to operand[1] and less than or equal to operand[2].
 * The following restrictions apply to the operands:
 *   [0]: Any operand that resolves to an ordered value.
 *   [1]: Any operand that resolves to an ordered value.
 *   [2]: Any operand that resolves to an ordered value.
 * If the operands are of different types, the system shall perform any implicit conversion to match
 * all operands to a common type. If no implicit conversion is available and the operands are of different
 * types, the particular result is FALSE. See the discussion on data type precedence in Table 123
 * for more information how to convert operands of different types.
 */
function checkBetween(filterContext, filter, filteredOperands) {
    const operandA = evaluateOperand(filterContext, filter, filteredOperands[0], _coerceToCompareable);
    const operandLow = evaluateOperand(filterContext, filter, filteredOperands[1], _coerceToCompareable);
    const operandHigh = evaluateOperand(filterContext, filter, filteredOperands[2], _coerceToCompareable);
    return operandA >= operandLow && operandA <= operandHigh;
}
/**
 *
 * InList
 * TRUE if operand[0] is equal to one or more of the remaining operands.
 * The Equals Operator is evaluated for operand[0] and each remaining operand in the
 * list. If any Equals evaluation is TRUE, InList returns TRUE
 x*/
function checkInList(context, filterOperands) {
    const operand0 = filterOperands[0];
    // istanbul ignore next
    if (!(operand0 instanceof node_opcua_types_1.SimpleAttributeOperand)) {
        // unsupported case
        warningLog("FilterOperator.InList operand0 is not a SimpleAttributeOperand " + operand0.constructor.name);
        return false;
    }
    const value = (0, resolve_operand_1.resolveOperand)(context, operand0);
    // istanbul ignore next
    if (value.dataType !== node_opcua_variant_1.DataType.NodeId) {
        return false;
    }
    const nodeId = value.value;
    // istanbul ignore next
    if (!nodeId) {
        return false;
    }
    function _is(nodeId1, operandX) {
        if (operandX.value.dataType !== node_opcua_variant_1.DataType.NodeId) {
            return false;
        }
        const nodeId2 = operandX.value.value;
        const nodeClass = context.getNodeClass(nodeId2);
        if (nodeClass === node_opcua_data_model_1.NodeClass.Unspecified) {
            return false;
        }
        return (0, node_opcua_nodeid_1.sameNodeId)(nodeId1, nodeId2);
    }
    for (let i = 1; i < filterOperands.length; i++) {
        const filterOperand = filterOperands[i];
        if (filterOperand instanceof node_opcua_types_1.LiteralOperand && _is(nodeId, filterOperand)) {
            return true;
        }
    }
    return false;
}
// eslint-disable-next-line complexity
function checkFilterAtIndex(filterContext, filter, index) {
    if (!filter.elements || filter.elements.length === 0) {
        return true;
    }
    const element = filter.elements[index];
    // istanbul ignore next
    if (!element) {
        return true;
    }
    const filterOperands = element.filterOperands || [];
    switch (element.filterOperator) {
        case node_opcua_types_1.FilterOperator.Equals:
            return checkEquals(filterContext, filter, filterOperands);
        case node_opcua_types_1.FilterOperator.LessThan:
            return checkLessThan(filterContext, filter, filterOperands);
        case node_opcua_types_1.FilterOperator.LessThanOrEqual:
            return checkLessThanOrEqual(filterContext, filter, filterOperands);
        case node_opcua_types_1.FilterOperator.GreaterThan:
            return checkGreaterThan(filterContext, filter, filterOperands);
        case node_opcua_types_1.FilterOperator.GreaterThanOrEqual:
            return checkGreaterThanOrEqual(filterContext, filter, filterOperands);
        case node_opcua_types_1.FilterOperator.Between:
            return checkBetween(filterContext, filter, filterOperands);
        case node_opcua_types_1.FilterOperator.And:
            return checkAnd(filterContext, filter, filterOperands);
        case node_opcua_types_1.FilterOperator.Or:
            return checkOr(filterContext, filter, filterOperands);
        case node_opcua_types_1.FilterOperator.Not:
            return checkNot(filterContext, filter, filterOperands);
        case node_opcua_types_1.FilterOperator.OfType:
            return checkOfType(filterContext, element.filterOperands[0]);
        case node_opcua_types_1.FilterOperator.InList:
            return checkInList(filterContext, filterOperands);
        case node_opcua_types_1.FilterOperator.RelatedTo:
        case node_opcua_types_1.FilterOperator.Like:
        case node_opcua_types_1.FilterOperator.BitwiseAnd:
        case node_opcua_types_1.FilterOperator.BitwiseOr:
        case node_opcua_types_1.FilterOperator.Cast:
        case node_opcua_types_1.FilterOperator.InView:
        case node_opcua_types_1.FilterOperator.IsNull:
        default:
            // from Spec  OPC Unified Architecture, Part 4 133 Release 1.04
            //  Any basic FilterOperator in Table 119 may be used in the whereClause, however, only the
            //  OfType_14 FilterOperator from Table 120 is permitted.
            warningLog(`checkFilter: operator ${node_opcua_types_1.FilterOperator[element.filterOperator]} is currently not supported in filter`);
            return false;
    }
}
function checkFilter(filterContext, contentFilter) {
    return checkFilterAtIndex(filterContext, contentFilter, 0);
}
//# sourceMappingURL=check_where_clause.js.map