/**
 * @module node-opcua-server
 */
import { EventEmitter } from "events";
import { AddressSpace, ContinuationPointManager, ISessionBase, UADynamicVariableArray, UASessionDiagnosticsVariable, UASessionSecurityDiagnostics, DTSessionDiagnostics, DTSessionSecurityDiagnostics, IUserManager } from "node-opcua-address-space";
import { ISessionContext } from "node-opcua-address-space-base";
import { SessionDiagnosticsDataType, SessionSecurityDiagnosticsDataType, SubscriptionDiagnosticsDataType } from "node-opcua-common";
import { NodeId } from "node-opcua-nodeid";
import { ObjectRegistry } from "node-opcua-object-registry";
import { StatusCode } from "node-opcua-status-code";
import { WatchDog } from "node-opcua-utils";
import { ISubscriber, IWatchdogData2 } from "node-opcua-utils";
import { IServerSession, IServerSessionBase, ServerSecureChannelLayer } from "node-opcua-secure-channel";
import { ApplicationDescription, UserIdentityToken, CreateSubscriptionRequestOptions, EndpointDescription } from "node-opcua-types";
import { ServerSidePublishEngine } from "./server_publish_engine";
import { Subscription } from "./server_subscription";
import { ServerEngine } from "./server_engine";
export type SessionStatus = "new" | "active" | "screwed" | "disposed" | "closed";
/**
 *
 * A Server session object.
 *
 * **from OPCUA Spec 1.02:**
 *
 * * Sessions are created to be independent of the underlying communications connection. Therefore, if a communication
 *   connection fails, the Session is not immediately affected. The exact mechanism to recover from an underlying
 *   communication connection error depends on the SecureChannel mapping as described in Part 6.
 *
 * * Sessions are terminated by the Server automatically if the Client fails to issue a Service request on the Session
 *   within the timeout period negotiated by the Server in the CreateSession Service response. This protects the Server
 *   against Client failures and against situations where a failed underlying connection cannot be re-established.
 *
 * * Clients shall be prepared to submit requests in a timely manner to prevent the Session from closing automatically.
 *
 * * Clients may explicitly terminate Sessions using the CloseSession Service.
 *
 * * When a Session is terminated, all outstanding requests on the Session are aborted and BadSessionClosed StatusCodes
 *   are returned to the Client. In addition, the Server deletes the entry for the Client from its
 *   SessionDiagnosticsArray Variable and notifies any other Clients who were subscribed to this entry.
 *
 */
export declare class ServerSession extends EventEmitter implements ISubscriber, ISessionBase, IServerSession, IServerSessionBase {
    static registry: ObjectRegistry;
    static maxPublishRequestInQueue: number;
    __status: SessionStatus;
    parent: ServerEngine;
    authenticationToken: NodeId;
    nodeId: NodeId;
    sessionName: string;
    publishEngine: ServerSidePublishEngine;
    sessionObject: any;
    readonly creationDate: Date;
    sessionTimeout: number;
    sessionDiagnostics?: UASessionDiagnosticsVariable<DTSessionDiagnostics>;
    sessionSecurityDiagnostics?: UASessionSecurityDiagnostics<DTSessionSecurityDiagnostics>;
    subscriptionDiagnosticsArray?: UADynamicVariableArray<SubscriptionDiagnosticsDataType>;
    channel?: ServerSecureChannelLayer;
    nonce?: Buffer;
    userIdentityToken?: UserIdentityToken;
    clientDescription?: ApplicationDescription;
    channelId?: number | null;
    continuationPointManager: ContinuationPointManager;
    sessionContext: ISessionContext;
    _watchDog?: WatchDog;
    _watchDogData?: IWatchdogData2;
    keepAlive: () => void;
    private _registeredNodesCounter;
    private _registeredNodes;
    private _registeredNodesInv;
    private _cumulatedSubscriptionCount;
    private _sessionDiagnostics?;
    private _sessionSecurityDiagnostics?;
    private channel_abort_event_handler;
    constructor(parent: ServerEngine, userManager: IUserManager, sessionTimeout: number);
    getSessionId(): NodeId;
    endpoint?: EndpointDescription;
    getEndpointDescription(): EndpointDescription;
    dispose(): void;
    get clientConnectionTime(): Date;
    /**
     * return the number of milisecond since last session transaction occurs from client
     * the first transaction is the creation of the session
     */
    get clientLastContactTime(): number;
    get status(): SessionStatus;
    set status(value: SessionStatus);
    get addressSpace(): AddressSpace | null;
    get currentPublishRequestInQueue(): number;
    updateClientLastContactTime(): void;
    /**
     * required for watch dog
     * @param currentTime {DateTime}
     * @private
     */
    onClientSeen(): void;
    incrementTotalRequestCount(): void;
    incrementRequestTotalCounter(counterName: string): void;
    incrementRequestErrorCounter(counterName: string): void;
    /**
     * returns rootFolder.objects.server.serverDiagnostics.sessionsDiagnosticsSummary.sessionDiagnosticsArray
     */
    getSessionDiagnosticsArray(): UADynamicVariableArray<SessionDiagnosticsDataType>;
    /**
     * returns rootFolder.objects.server.serverDiagnostics.sessionsDiagnosticsSummary.sessionSecurityDiagnosticsArray
     */
    getSessionSecurityDiagnosticsArray(): UADynamicVariableArray<SessionSecurityDiagnosticsDataType>;
    /**
     * number of active subscriptions
     */
    get currentSubscriptionCount(): number;
    /**
     * number of subscriptions ever created since this object is live
     */
    get cumulatedSubscriptionCount(): number;
    /**
     * number of monitored items
     */
    get currentMonitoredItemCount(): number;
    /**
     * retrieve an existing subscription by subscriptionId
     * @param subscriptionId {Number}
     */
    getSubscription(subscriptionId: number): Subscription | null;
    /**
     * @param subscriptionId {Number}
     * @return {StatusCode}
     */
    deleteSubscription(subscriptionId: number): StatusCode;
    /**
     * close a ServerSession, this will also delete the subscriptions if the flag is set.
     *
     * Spec extract:
     *
     * If a Client invokes the CloseSession Service then all Subscriptions associated with the Session are also deleted
     * if the deleteSubscriptions flag is set to TRUE. If a Server terminates a Session for any other reason,
     * Subscriptions associated with the Session, are not deleted. Each Subscription has its own lifetime to protect
     * against data loss in the case of a Session termination. In these cases, the Subscription can be reassigned to
     * another Client before its lifetime expires.
     *
     * @param deleteSubscriptions : should we delete subscription ?
     * @param [reason = "CloseSession"] the reason for closing the session
     *         (shall be "Timeout", "Terminated" or "CloseSession")
     *
     */
    close(deleteSubscriptions: boolean, reason: string): void;
    registerNode(nodeId: NodeId): NodeId;
    unRegisterNode(aliasNodeId: NodeId): void;
    resolveRegisteredNode(aliasNodeId: NodeId): NodeId;
    /**
     * true if the underlying channel has been closed or aborted...
     */
    get aborted(): boolean;
    createSubscription(parameters: CreateSubscriptionRequestOptions): Subscription;
    _attach_channel(channel: ServerSecureChannelLayer): void;
    _detach_channel(): void;
    _exposeSubscriptionDiagnostics(subscription: Subscription): void;
    _unexposeSubscriptionDiagnostics(subscription: Subscription): void;
    /**
     * used as a callback for the Watchdog
     * @private
     */
    watchdogReset(): void;
    private _createSessionObjectInAddressSpace;
    resendMonitoredItemInitialValues(): Promise<void>;
    /**
     *
     * @private
     */
    private _removeSessionObjectFromAddressSpace;
    /**
     *
     * @private
     */
    private _getSubscriptionDiagnosticsArray;
    private assignSubscription;
    private _deleteSubscriptions;
}
