/**
 * @module node-opcua-server
 */
import { EventEmitter } from "events";
import { AddressSpace, ISessionContext, IServerBase } from "node-opcua-address-space";
import { DataValue } from "node-opcua-data-value";
import { ServerDiagnosticsSummaryDataType, ServerState, ServerStatusDataType } from "node-opcua-common";
import { LocalizedTextLike } from "node-opcua-data-model";
import { NodeId } from "node-opcua-nodeid";
import { BrowseResult } from "node-opcua-service-browse";
import { CreateSubscriptionRequestLike } from "node-opcua-client";
import { ObjectRegistry } from "node-opcua-object-registry";
import { TransferResult } from "node-opcua-service-subscription";
import { ApplicationDescription } from "node-opcua-service-endpoints";
import { HistoryReadRequest, HistoryReadResult, HistoryReadValueId } from "node-opcua-service-history";
import { StatusCode } from "node-opcua-status-code";
import { BrowseDescription, BrowsePath, BrowsePathResult, BuildInfo, BuildInfoOptions, WriteValue, ReadValueId, CallMethodResultOptions, ReadRequestOptions, BrowseDescriptionOptions, CallMethodRequest, ApplicationType } from "node-opcua-types";
import { HistoryServerCapabilities, HistoryServerCapabilitiesOptions } from "./history_server_capabilities";
import { ServerCapabilities, ServerCapabilitiesOptions } from "./server_capabilities";
import { ServerSession } from "./server_session";
import { Subscription } from "./server_subscription";
import { OPCUAServerOptions } from "./opcua_server";
import { IAddressSpaceAccessor } from "./i_address_space_accessor";
export declare function setNextSubscriptionId(n: number): void;
export type StringGetter = () => string;
export type StringArrayGetter = () => string[];
export type ApplicationTypeGetter = () => ApplicationType;
export type BooleanGetter = () => boolean;
export interface ServerConfigurationOptions {
    applicationUri?: string | StringGetter;
    applicationType?: ApplicationType | ApplicationTypeGetter;
    hasSecureElement?: boolean | BooleanGetter;
    multicastDnsEnabled?: boolean | BooleanGetter;
    productUri?: string | StringGetter;
    /**
     * The SupportedPrivateKeyFormats specifies the PrivateKey formats supported by the Server.
     * Possible values include “PEM” (see RFC 5958) or “PFX” (see PKCS #12).
     * @default ["PEM"]
     */
    supportedPrivateKeyFormat: string[] | StringArrayGetter;
    /**
     * The ServerCapabilities Property specifies the capabilities from Annex D
     * ( see https://reference.opcfoundation.org/GDS/v104/docs/D)  which the Server supports. The value is
     * the same as the value reported to the LocalDiscoveryServer when the Server calls the RegisterServer2 Service.
     */
    serverCapabilities?: string[] | StringArrayGetter;
}
export interface ServerEngineOptions {
    applicationUri: string | StringGetter;
    buildInfo?: BuildInfoOptions;
    isAuditing?: boolean;
    /**
     * set to true to enable serverDiagnostics
     */
    serverDiagnosticsEnabled?: boolean;
    serverCapabilities?: ServerCapabilitiesOptions;
    historyServerCapabilities?: HistoryServerCapabilitiesOptions;
    serverConfiguration?: ServerConfigurationOptions;
}
export interface CreateSessionOption {
    clientDescription?: ApplicationDescription;
    sessionTimeout?: number;
    server?: IServerBase;
}
export type ClosingReason = "Timeout" | "Terminated" | "CloseSession" | "Forcing";
export type ServerEngineShutdownTask = (this: ServerEngine) => void | Promise<void>;
/**
 *
 */
export declare class ServerEngine extends EventEmitter implements IAddressSpaceAccessor {
    static readonly registry: ObjectRegistry;
    isAuditing: boolean;
    serverDiagnosticsSummary: ServerDiagnosticsSummaryDataType;
    serverDiagnosticsEnabled: boolean;
    serverCapabilities: ServerCapabilities;
    historyServerCapabilities: HistoryServerCapabilities;
    serverConfiguration: ServerConfigurationOptions;
    clientDescription?: ApplicationDescription;
    addressSpace: AddressSpace | null;
    addressSpaceAccessor: IAddressSpaceAccessor | null;
    _internalState: "creating" | "initializing" | "initialized" | "shutdown" | "disposed";
    private _sessions;
    private _closedSessions;
    private _orphanPublishEngine?;
    private _shutdownTasks;
    private _applicationUri;
    private _expectedShutdownTime;
    private _serverStatus;
    private _globalCounter;
    constructor(options?: ServerEngineOptions);
    isStarted(): boolean;
    dispose(): void;
    get startTime(): Date;
    get currentTime(): Date;
    get buildInfo(): BuildInfo;
    /**
     * register a function that will be called when the server will perform its shut down.
     */
    registerShutdownTask(task: ServerEngineShutdownTask): void;
    /**
     */
    shutdown(): Promise<void>;
    /**
     * the number of active sessions
     */
    get currentSessionCount(): number;
    /**
     * the cumulated number of sessions that have been opened since this object exists
     */
    get cumulatedSessionCount(): number;
    /**
     * the number of active subscriptions.
     */
    get currentSubscriptionCount(): number;
    /**
     * the cumulated number of subscriptions that have been created since this object exists
     */
    get cumulatedSubscriptionCount(): number;
    get rejectedSessionCount(): number;
    get rejectedRequestsCount(): number;
    get sessionAbortCount(): number;
    get sessionTimeoutCount(): number;
    get publishingIntervalCount(): number;
    incrementSessionTimeoutCount(): void;
    incrementSessionAbortCount(): void;
    incrementRejectedRequestsCount(): void;
    /**
     * increment rejected session count (also increment rejected requests count)
     */
    incrementRejectedSessionCount(): void;
    incrementSecurityRejectedRequestsCount(): void;
    /**
     * increment rejected session count (also increment rejected requests count)
     */
    incrementSecurityRejectedSessionCount(): void;
    setShutdownTime(date: Date): void;
    setShutdownReason(reason: LocalizedTextLike): void;
    /**
     * @return the approximate number of seconds until the server will be shut down. The
     * value is only relevant once the state changes into SHUTDOWN.
     */
    secondsTillShutdown(): number;
    /**
     * the name of the server
     */
    get serverName(): string;
    /**
     * the server urn
     */
    get serverNameUrn(): string;
    /**
     * the urn of the server namespace
     */
    get serverNamespaceUrn(): string;
    get serverStatus(): ServerStatusDataType;
    setServerState(serverState: ServerState): void;
    getServerDiagnosticsEnabledFlag(): boolean;
    /**
     *
     */
    initialize(options: OPCUAServerOptions, callback: (err?: Error | null) => void): void;
    browseWithAutomaticExpansion(nodesToBrowse: BrowseDescription[], context: ISessionContext): Promise<BrowseResult[]>;
    browse(context: ISessionContext, nodesToBrowse: BrowseDescriptionOptions[]): Promise<BrowseResult[]>;
    read(context: ISessionContext, readRequest: ReadRequestOptions): Promise<DataValue[]>;
    write(context: ISessionContext, nodesToWrite: WriteValue[]): Promise<StatusCode[]>;
    call(context: ISessionContext, methodsToCall: CallMethodRequest[]): Promise<CallMethodResultOptions[]>;
    historyRead(context: ISessionContext, historyReadRequest: HistoryReadRequest): Promise<HistoryReadResult[]>;
    getOldestInactiveSession(): ServerSession | null;
    /**
     * create a new server session object.
     */
    createSession(options?: CreateSessionOption): ServerSession;
    /**
     * @param authenticationToken
     * @param deleteSubscriptions {Boolean} : true if session's subscription shall be deleted
     * @param {String} [reason = "CloseSession"] the reason for closing the session (
     *                 shall be "Timeout", "Terminated" or "CloseSession")
     *
     *
     * what the specs say:
     * -------------------
     *
     * If a Client invokes the CloseSession Service then all Subscriptions associated with the Session are also deleted
     * if the deleteSubscriptions flag is set to TRUE. If a Server terminates a Session for any other reason,
     * Subscriptions associated with the Session, are not deleted. Each Subscription has its own lifetime to protect
     * against data loss in the case of a Session termination. In these cases, the Subscription can be reassigned to
     * another Client before its lifetime expires.
     */
    closeSession(authenticationToken: NodeId, deleteSubscriptions: boolean, reason: ClosingReason): void;
    findSubscription(subscriptionId: number): Subscription | null;
    findOrphanSubscription(subscriptionId: number): Subscription | null;
    deleteOrphanSubscription(subscription: Subscription): StatusCode;
    /**
     * @param session           {ServerSession}  - the new session that will own the subscription
     * @param subscriptionId    {IntegerId}      - the subscription Id to transfer
     * @param sendInitialValues {Boolean}        - true if initial values will be resent.
     * @return                  {TransferResult}
     */
    transferSubscription(session: ServerSession, subscriptionId: number, sendInitialValues: boolean): Promise<TransferResult>;
    /**
     * retrieve a session by its authenticationToken.
     *
     * @param authenticationToken
     * @param activeOnly
     * @return {ServerSession}
     */
    getSession(authenticationToken: NodeId, activeOnly?: boolean): ServerSession | null;
    translateBrowsePaths(browsePaths: BrowsePath[]): Promise<BrowsePathResult[]>;
    translateBrowsePath(browsePath: BrowsePath): Promise<BrowsePathResult>;
    /**
     *
     * performs a call to ```asyncRefresh``` on all variable nodes that provide an async refresh func.
     *
     * @param nodesToRefresh {Array<ReadValueId|HistoryReadValueId>}  an array containing the node to consider
     * Each element of the array shall be of the form { nodeId: <xxx>, attributeIds: <value> }.
     * @param maxAge {number}  the maximum age of the value to be read, in milliseconds.
     * @param callback
     *
     */
    refreshValues(nodesToRefresh: ReadValueId[] | HistoryReadValueId[], maxAge: number, 
    /**
     * @param err
     * @param dataValues an array containing value read
     * The array length matches the number of  nodeIds that are candidate for an
     * async refresh (i.e: nodes that are of type Variable with asyncRefresh func }
     */
    callback: (err: Error | null, dataValues?: DataValue[]) => void): void;
    private _exposeSubscriptionDiagnostics;
    protected _unexposeSubscriptionDiagnostics(subscription: Subscription): void;
    /**
     * create a new subscription
     * @return {Subscription}
     */
    _createSubscriptionOnSession(session: ServerSession, request: CreateSubscriptionRequestLike): Subscription;
    /**
     */
    private __internal_bindMethod;
    private _getServerSubscriptionDiagnosticsArrayNode;
}
