import { EventEmitter } from "events";
import { PseudoVariantBoolean, PseudoVariantByteString, PseudoVariantDateTime, PseudoVariantDuration, PseudoVariantExtensionObject, PseudoVariantExtensionObjectArray, PseudoVariantLocalizedText, PseudoVariantNodeId, PseudoVariantString, RaiseEventData, PseudoVariantStringPredefined, UAEventType } from "node-opcua-address-space";
import { OPCUACertificateManager } from "node-opcua-certificate-manager";
import { Certificate, Nonce } from "node-opcua-crypto/web";
import { NodeId } from "node-opcua-nodeid";
import { ObjectRegistry } from "node-opcua-object-registry";
import { Message, MessageSecurityMode, Request, Response, SecurityPolicy, ServerSecureChannelLayer, SignatureData } from "node-opcua-secure-channel";
import { BrowseNextResponse, BrowseResponse } from "node-opcua-service-browse";
import { CallResponse } from "node-opcua-service-call";
import { HistoryReadResponse } from "node-opcua-service-history";
import { ReadResponse } from "node-opcua-service-read";
import { RegisterNodesResponse, UnregisterNodesResponse } from "node-opcua-service-register-node";
import { UserNameIdentityToken, X509IdentityToken } from "node-opcua-service-session";
import { CreateMonitoredItemsResponse, CreateSubscriptionResponse, DeleteSubscriptionsResponse, ModifyMonitoredItemsResponse, ModifySubscriptionResponse, RepublishResponse, SetPublishingModeResponse, SetTriggeringResponse, TransferSubscriptionsResponse } from "node-opcua-service-subscription";
import { TranslateBrowsePathsToNodeIdsResponse } from "node-opcua-service-translate-browse-path";
import { WriteResponse } from "node-opcua-service-write";
import { StatusCode } from "node-opcua-status-code";
import { ApplicationDescriptionOptions, BuildInfo, EndpointDescription, UserIdentityToken, UserTokenPolicy } from "node-opcua-types";
import { OPCUABaseServer, OPCUABaseServerOptions } from "./base_server";
import { IRegisterServerManager } from "./i_register_server_manager";
import { ServerCapabilitiesOptions } from "./server_capabilities";
import { IServerTransportSettings, OPCUAServerEndPoint } from "./server_end_point";
import { CreateSessionOption, ServerEngine } from "./server_engine";
import { ServerSession } from "./server_session";
import { CreateMonitoredItemHook, DeleteMonitoredItemHook, Subscription } from "./server_subscription";
import { ISocketData } from "./i_socket_data";
import { IChannelData } from "./i_channel_data";
import { UAUserManagerBase, UserManagerOptions } from "./user_manager";
type ResponseClassType = typeof BrowseResponse | typeof BrowseNextResponse | typeof CallResponse | typeof CreateMonitoredItemsResponse | typeof CreateSubscriptionResponse | typeof DeleteSubscriptionsResponse | typeof HistoryReadResponse | typeof ModifyMonitoredItemsResponse | typeof ModifySubscriptionResponse | typeof ReadResponse | typeof RegisterNodesResponse | typeof RepublishResponse | typeof SetPublishingModeResponse | typeof SetTriggeringResponse | typeof TransferSubscriptionsResponse | typeof TranslateBrowsePathsToNodeIdsResponse | typeof UnregisterNodesResponse | typeof WriteResponse;
export declare function filterDiagnosticInfo(returnDiagnostics: number, response: CallResponse): void;
export declare enum RegisterServerMethod {
    HIDDEN = 1,// the server doesn't expose itself to the external world
    MDNS = 2,// the server publish itself to the mDNS Multicast network directly
    LDS = 3
}
export interface OPCUAServerEndpointOptions {
    /**
     * the primary hostname of the endpoint.
     * @default getFullyQualifiedDomainName()
     */
    hostname?: string;
    /**
     * Host IP address or hostname where the TCP server listens for connections.
     * If omitted, defaults to listening on all network interfaces:
     * - Unspecified IPv6 address (::) if IPv6 is available,
     * - Unspecified IPv4 address (0.0.0.0) otherwise.
     * Use this to bind the server to a specific interface or IP.
     */
    host?: string;
    /**
     * the TCP port to listen to.
     * @default 26543
     */
    port?: number;
    /**
     * the possible security policies that the server will expose
     * @default  [SecurityPolicy.None, SecurityPolicy.Basic128Rsa15, SecurityPolicy.Basic256Sha256, SecurityPolicy.Aes128_Sha256_RsaOaep, SecurityPolicy.Aes256_Sha256_RsaPss  ]
     */
    securityPolicies?: SecurityPolicy[];
    /**
     * the possible security mode that the server will expose
     * @default [MessageSecurityMode.None, MessageSecurityMode.Sign, MessageSecurityMode.SignAndEncrypt]
     */
    securityModes?: MessageSecurityMode[];
    /**
     * tells if the server default endpoints should allow anonymous connection.
     * @default true
     */
    allowAnonymous?: boolean;
    /** alternate hostname  or IP to use */
    alternateHostname?: string | string[];
    /**
     *  true, if discovery service on secure channel shall be disabled
     */
    disableDiscovery?: boolean;
}
export interface OPCUAServerOptions extends OPCUABaseServerOptions, OPCUAServerEndpointOptions {
    /**
     * @deprecated
     */
    alternateEndpoints?: OPCUAServerEndpointOptions[];
    endpoints?: OPCUAServerEndpointOptions[];
    /**
     * the server certificate full path filename
     *
     * the certificate should be in PEM format
     */
    certificateFile?: string;
    /**
     * the server private key full path filename
     *
     * This file should contains the private key that has been used to generate
     * the server certificate file.
     *
     * the private key should be in PEM format
     *
     */
    privateKeyFile?: string;
    /**
     * the default secure token life time in ms.
     */
    defaultSecureTokenLifetime?: number;
    /**
     * the HEL/ACK transaction timeout in ms.
     *
     * Use a large value ( i.e 15000 ms) for slow connections or embedded devices.
     * @default 10000
     */
    timeout?: number;
    /**
     * the maximum number of simultaneous sessions allowed.
     * @default 10
     * @deprecated use serverCapabilities: { maxSessions: } instead

     */
    maxAllowedSessionNumber?: number;
    /**
     * the maximum number authorized simultaneous connections per endpoint
     * @default 10
     */
    maxConnectionsPerEndpoint?: number;
    /**
     * the nodeset.xml file(s) to load
     *
     * node-opcua comes with pre-installed node-set files that can be used
     *
     * example:
     *
     * ```javascript
     * import { nodesets } from "node-opcua-nodesets";
     * const server = new OPCUAServer({
     *     nodeset_filename: [
     *         nodesets.standard,
     *         nodesets.di,
     *         nodesets.adi,
     *         nodesets.machinery,
     *     ],
     * });
     * ```
     */
    nodeset_filename?: string[] | string;
    /**
     * the server Info
     *
     * this object contains the value that will populate the
     * Root/ObjectS/Server/ServerInfo OPCUA object in the address space.
     */
    serverInfo?: ApplicationDescriptionOptions;
    buildInfo?: {
        productName?: string;
        productUri?: string | null;
        manufacturerName?: string;
        softwareVersion?: string;
        buildNumber?: string;
        buildDate?: Date;
    };
    /**
     *  an object that implements user authentication methods
     */
    userManager?: UserManagerOptions;
    /** resource Path is a string added at the end of the url such as "/UA/Server" */
    resourcePath?: string;
    /**
     *
     */
    serverCapabilities?: ServerCapabilitiesOptions;
    /**
     * if server shall raise AuditingEvent
     * @default true
     */
    isAuditing?: boolean;
    /**
     * strategy used by the server to declare itself to a discovery server
     *
     * - HIDDEN: the server doesn't expose itself to the external world
     * - MDNS: the server publish itself to the mDNS Multicast network directly
     * - LDS: the server registers itself to the LDS or LDS-ME (Local Discovery Server)
     *
     *  @default    .HIDDEN - by default the server
     *            will not register itself to the local discovery server
     *
     */
    registerServerMethod?: RegisterServerMethod;
    /**
     *
     * @default "opc.tcp://localhost:4840"]
     */
    discoveryServerEndpointUrl?: string;
    /**
     *
     *  supported server capabilities for the Multicast (mDNS)
     *  @default ["NA"]
     *  the possible values are any of node-opcua-discovery.serverCapabilities)
     *
     */
    capabilitiesForMDNS?: string[];
    /**
     * user Certificate Manager
     * this certificate manager holds the X509 certificates used
     * by client that uses X509 certificate token to impersonate a user
     */
    userCertificateManager?: OPCUACertificateManager;
    /**
     * Server Certificate Manager
     *
     * this certificate manager will be used by the server to access
     * and store certificates from the connecting clients
     */
    serverCertificateManager?: OPCUACertificateManager;
    /**
     *
     */
    onCreateMonitoredItem?: CreateMonitoredItemHook;
    onDeleteMonitoredItem?: DeleteMonitoredItemHook;
    /**
     * skipOwnNamespace to true, if you don't want the server to create
     * a dedicated namespace for its own (namespace=1).
     * Use this flag if you intend to load the server own namespace
     * from an external source.
     * @default false
     */
    skipOwnNamespace?: boolean;
    transportSettings?: IServerTransportSettings;
}
export interface OPCUAServer {
    /**
     *
     */
    engine: ServerEngine;
    /**
     *
     */
    registerServerMethod: RegisterServerMethod;
    /**
     *
     */
    discoveryServerEndpointUrl: string;
    /**
     *
     */
    registerServerManager?: IRegisterServerManager;
    /**
     *
     */
    capabilitiesForMDNS: string[];
    /**
     *
     */
    userCertificateManager: OPCUACertificateManager;
}
/**
 *
 */
export declare class OPCUAServer extends OPCUABaseServer {
    static defaultShutdownTimeout: number;
    /**
     * if requestExactEndpointUrl is set to true the server will only accept createSession that have a endpointUrl that strictly matches
     * one of the provided endpoint.
     * This mean that if the server expose a endpoint with url such as opc.tcp://MYHOSTNAME:1234, client will not be able to reach the server
     * with the ip address of the server.
     * requestExactEndpointUrl = true => emulates the Prosys Server behavior
     * requestExactEndpointUrl = false => emulates the Unified Automation behavior.
     */
    static requestExactEndpointUrl: boolean;
    /**
     * total number of bytes written  by the server since startup
     */
    get bytesWritten(): number;
    /**
     * total number of bytes read  by the server since startup
     */
    get bytesRead(): number;
    /**
     * Number of transactions processed by the server since startup
     */
    get transactionsCount(): number;
    /**
     * The server build info
     */
    get buildInfo(): BuildInfo;
    /**
     * the number of connected channel on all existing end points
     */
    get currentChannelCount(): number;
    /**
     * The number of active subscriptions from all sessions
     */
    get currentSubscriptionCount(): number;
    /**
     * the number of session activation requests that have been rejected
     */
    get rejectedSessionCount(): number;
    /**
     * the number of request that have been rejected
     */
    get rejectedRequestsCount(): number;
    /**
     * the number of sessions that have been aborted
     */
    get sessionAbortCount(): number;
    /**
     * the publishing interval count
     */
    get publishingIntervalCount(): number;
    /**
     * the number of sessions currently active
     */
    get currentSessionCount(): number;
    /**
     * true if the server has been initialized
     *
     */
    get initialized(): boolean;
    /**
     * is the server auditing ?
     */
    get isAuditing(): boolean;
    static registry: ObjectRegistry;
    static fallbackSessionName: string;
    /**
     * the maximum number of subscription that can be created per server
     * @deprecated
     */
    static deprecated_MAX_SUBSCRIPTION: number;
    /**
     * the maximum number of concurrent sessions allowed on the server
     */
    get maxAllowedSessionNumber(): number;
    /**
     * the maximum number for concurrent connection per end point
     */
    maxConnectionsPerEndpoint: number;
    /**
     * false if anonymous connection are not allowed
     */
    allowAnonymous: boolean;
    /**
     * the user manager
     */
    userManager: UAUserManagerBase;
    readonly options: OPCUAServerOptions;
    private objectFactory?;
    private _delayInit?;
    constructor(options?: OPCUAServerOptions);
    /**
     * Initialize the server by installing default node set.
     *
     * and instruct the server to listen to its endpoints.
     *
     * ```javascript
     * const server = new OPCUAServer();
     * await server.initialize();
     *
     * // default server namespace is now initialized
     * // it is a good time to create life instance objects
     * const namespace = server.engine.addressSpace.getOwnNamespace();
     * namespace.addObject({
     *     browseName: "SomeObject",
     *     organizedBy: server.engine.addressSpace.rootFolder.objects
     * });
     *
     * // the addressSpace is now complete
     * // let's now start listening to clients
     * await server.start();
     * ```
     */
    initialize(): Promise<void>;
    initialize(done: () => void): void;
    /**
     * Initiate the server by starting all its endpoints
     */
    start(): Promise<void>;
    start(done: () => void): void;
    /**
     * shutdown all server endpoints
     * @param  timeout the timeout (in ms) before the server is actually shutdown
     *
     * @example
     *
     * ```javascript
     *    // shutdown immediately
     *    server.shutdown(function(err) {
     *    });
     * ```
     * ```ts
     *   // in typescript with promises
     *   server.shutdown(10000).then(()=>{
     *      console.log("Server has shutdown");
     *   });
     * ```
     * ```javascript
     *    // shutdown within 10 seconds
     *    server.engine.shutdownReason = coerceLocalizedText("Shutdown for maintenance");
     *    server.shutdown(10000,function(err) {
     *    });
     *   ```
     */
    shutdown(timeout?: number): Promise<void>;
    shutdown(callback: (err?: Error) => void): void;
    shutdown(timeout: number, callback: (err?: Error) => void): void;
    dispose(): void;
    raiseEvent(eventType: "AuditSessionEventType", options: RaiseEventAuditSessionEventData): void;
    raiseEvent(eventType: "AuditCreateSessionEventType", options: RaiseEventAuditCreateSessionEventData): void;
    raiseEvent(eventType: "AuditActivateSessionEventType", options: RaiseEventAuditActivateSessionEventData): void;
    raiseEvent(eventType: "AuditCreateSessionEventType", options: RaiseEventData): void;
    raiseEvent(eventType: "AuditConditionCommentEventType", options: RaiseEventAuditConditionCommentEventData): void;
    raiseEvent(eventType: "AuditUrlMismatchEventType", options: RaiseEventAuditUrlMismatchEventTypeData): void;
    raiseEvent(eventType: "TransitionEventType", options: RaiseEventTransitionEventData): void;
    raiseEvent(eventType: "AuditCertificateInvalidEventType", options: RaiseAuditCertificateInvalidEventData): void;
    raiseEvent(eventType: "AuditCertificateExpiredEventType", options: RaiseAuditCertificateExpiredEventData): void;
    raiseEvent(eventType: "AuditCertificateUntrustedEventType", options: RaiseAuditCertificateUntrustedEventData): void;
    raiseEvent(eventType: "AuditCertificateRevokedEventType", options: RaiseAuditCertificateRevokedEventData): void;
    raiseEvent(eventType: "AuditCertificateMismatchEventType", options: RaiseAuditCertificateMismatchEventData): void;
    /**
     * create and register a new session
     * @private
     */
    protected createSession(options: CreateSessionOption): ServerSession;
    /**
     * retrieve a session by authentication token
     * @private
     */
    getSession(authenticationToken: NodeId, activeOnly?: boolean): ServerSession | null;
    /**
     *
     * @param channel
     * @param clientCertificate
     * @param clientNonce
     * @private
     */
    protected computeServerSignature(channel: ServerSecureChannelLayer, clientCertificate: Certificate, clientNonce: Nonce): SignatureData | undefined;
    /**
     *
     * @param session
     * @param channel
     * @param clientSignature
     */
    protected verifyClientSignature(session: ServerSession, channel: ServerSecureChannelLayer, clientSignature: SignatureData): boolean;
    protected isValidUserNameIdentityToken(channel: ServerSecureChannelLayer, session: ServerSession, userTokenPolicy: UserTokenPolicy, userIdentityToken: UserNameIdentityToken, userTokenSignature: SignatureData, callback: (err: Error | null, statusCode?: StatusCode) => void): void;
    protected isValidX509IdentityToken(channel: ServerSecureChannelLayer, session: ServerSession, userTokenPolicy: UserTokenPolicy, userIdentityToken: X509IdentityToken, userTokenSignature: SignatureData, callback: (err: Error | null, statusCode?: StatusCode) => void): void;
    /**
     * @internal
     */
    protected userNameIdentityTokenAuthenticateUser(channel: ServerSecureChannelLayer, session: ServerSession, userTokenPolicy: UserTokenPolicy, userIdentityToken: UserNameIdentityToken, callback: (err: Error | null, isAuthorized?: boolean) => void): void;
    /**
     * @internal
     */
    protected isValidUserIdentityToken(channel: ServerSecureChannelLayer, session: ServerSession, userIdentityToken: UserIdentityToken, userTokenSignature: SignatureData, endpointDescription: EndpointDescription, callback: (err: Error | null, statusCode?: StatusCode) => void): void;
    /**
     *
     * @internal
     * @param channel
     * @param session
     * @param userIdentityToken
     * @param callback
     * @returns {*}
     */
    protected isUserAuthorized(channel: ServerSecureChannelLayer, session: ServerSession, userIdentityToken: UserIdentityToken, callback: (err: Error | null, isAuthorized?: boolean) => void): void;
    protected makeServerNonce(): Nonce;
    protected _on_CreateSessionRequest(message: Message, channel: ServerSecureChannelLayer): Promise<void>;
    /**
     *
     * @private
     *
     *
     */
    protected _on_ActivateSessionRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected prepare(message: Message, channel: ServerSecureChannelLayer): void;
    /**
     * ensure that action is performed on a valid session object,
     * @param ResponseClass the constructor of the response Class
     * @param message
     * @param channel
     * @param actionToPerform
     * @param actionToPerform.session {ServerSession}
     * @param actionToPerform.sendResponse
     * @param actionToPerform.sendResponse.response
     * @param actionToPerform.sendError
     * @param actionToPerform.sendError.statusCode
     * @param actionToPerform.sendError.diagnostics
     *
     * @private
     */
    protected _apply_on_SessionObject(ResponseClass: ResponseClassType, message: Message, channel: ServerSecureChannelLayer, actionToPerform: (session: ServerSession, sendResponse: (response: Response) => void, sendError: (statusCode: StatusCode) => void) => void | Promise<void>): Promise<void>;
    protected _apply_on_Subscription(ResponseClass: ResponseClassType, message: Message, channel: ServerSecureChannelLayer, actionToPerform: (session: ServerSession, subscription: Subscription, sendResponse: (response: Response) => void, sendError: (statusCode: StatusCode) => void) => Promise<void>): Promise<void>;
    protected _apply_on_SubscriptionIds<T>(ResponseClass: typeof SetPublishingModeResponse | typeof TransferSubscriptionsResponse | typeof DeleteSubscriptionsResponse, message: Message, channel: ServerSecureChannelLayer, actionToPerform: (session: ServerSession, subscriptionId: number) => Promise<T>): void;
    protected _apply_on_Subscriptions(ResponseClass: typeof SetPublishingModeResponse | typeof TransferSubscriptionsResponse | typeof DeleteSubscriptionsResponse, message: Message, channel: ServerSecureChannelLayer, actionToPerform: (session: ServerSession, subscription: Subscription) => Promise<StatusCode>): void;
    private _closeSession;
    /**
     * @param message
     * @param channel
     * @private
     */
    protected _on_CloseSessionRequest(message: Message, channel: ServerSecureChannelLayer): void;
    /**
     * @param message
     * @param channel
     * @private
     */
    protected _on_BrowseRequest(message: Message, channel: ServerSecureChannelLayer): void;
    /**
     */
    protected _on_BrowseNextRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_ReadRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_HistoryReadRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_WriteRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_CreateSubscriptionRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_DeleteSubscriptionsRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_TransferSubscriptionsRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_CreateMonitoredItemsRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_ModifySubscriptionRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_ModifyMonitoredItemsRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_PublishRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_SetPublishingModeRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_DeleteMonitoredItemsRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_SetTriggeringRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected _beforeDeleteSubscription(subscription: Subscription): Promise<void>;
    protected _on_RepublishRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_SetMonitoringModeRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_TranslateBrowsePathsToNodeIdsRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_CallRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_RegisterNodesRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_UnregisterNodesRequest(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_Cancel(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_AddNodes(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_AddReferences(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_DeleteNodes(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_DeleteReferences(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_QueryFirst(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_QueryNext(message: Message, channel: ServerSecureChannelLayer): void;
    protected _on_HistoryUpdate(message: Message, channel: ServerSecureChannelLayer): void;
    private createEndpoint;
    private createEndpointDescriptions;
    initializeCM(): Promise<void>;
}
export interface RaiseEventAuditEventData extends RaiseEventData {
    actionTimeStamp: PseudoVariantDateTime;
    status: PseudoVariantBoolean;
    serverId: PseudoVariantString;
    /**
     * ClientAuditEntryId contains the human-readable AuditEntryId defined in Part 3.
     */
    clientAuditEntryId: PseudoVariantString;
    /**
     * The ClientUserId identifies the user of the client requesting an action. The ClientUserId can be
     * obtained from the UserIdentityToken passed in the ActivateSession call.
     */
    clientUserId: PseudoVariantString;
    sourceName: PseudoVariantString;
}
export interface RaiseEventAuditUpdateMethodEventData extends RaiseEventAuditEventData {
    methodId: PseudoVariantNodeId;
    inputArguments: any;
}
export interface RaiseEventAuditConditionCommentEventData extends RaiseEventAuditUpdateMethodEventData {
    eventId: PseudoVariantByteString;
    comment: PseudoVariantLocalizedText;
}
export interface RaiseEventAuditSessionEventData extends RaiseEventAuditEventData {
    /**
     *  part 5 - 6.4.7 AuditSessionEventType
     */
    sessionId: PseudoVariantNodeId;
}
export interface RaiseEventAuditCreateSessionEventData extends RaiseEventAuditSessionEventData {
    /**
     *  part 5 - 6.4.8 AuditCreateSessionEventType
     *  SecureChannelId shall uniquely identify the SecureChannel.
     *  The application shall use the same identifier in
     *  all AuditEvents related to the Session Service Set (AuditCreateSessionEventType, AuditActivateSessionEventType
     *  and their subtypes) and the SecureChannel Service Set (AuditChannelEventType and its subtype
     */
    secureChannelId: PseudoVariantString;
    revisedSessionTimeout: PseudoVariantDuration;
    clientCertificate: PseudoVariantByteString;
    clientCertificateThumbprint: PseudoVariantString;
}
export interface RaiseEventAuditActivateSessionEventData extends RaiseEventAuditSessionEventData {
    /**
     * part 5 - 6.4.10 AuditActivateSessionEventType
     */
    clientSoftwareCertificates: PseudoVariantExtensionObjectArray;
    /**
     * UserIdentityToken reflects the userIdentityToken parameter of the ActivateSession Service call.
     * For Username/Password tokens the password should NOT be included.
     */
    userIdentityToken: PseudoVariantExtensionObject;
    /**
     * SecureChannelId shall uniquely identify the SecureChannel. The application shall use the same identifier
     * in all AuditEvents related to the Session Service Set (AuditCreateSessionEventType,
     * AuditActivateSessionEventType and their subtypes) and the SecureChannel Service Set
     * (AuditChannelEventType and its subtypes).
     */
    secureChannelId: PseudoVariantString;
}
export interface RaiseEventTransitionEventData extends RaiseEventData {
}
export interface RaiseEventAuditUrlMismatchEventTypeData extends RaiseEventData {
    endpointUrl: PseudoVariantString;
}
/**
 * The SourceName for Events of this type shall be “Security/Certificate”.
 */
export interface RaiseAuditCertificateEventData extends RaiseEventData {
    certificate: PseudoVariantByteString;
    sourceName: PseudoVariantStringPredefined<"Security/Certificate">;
}
/**
 * This EventType inherits all Properties of the AuditCertificateEventType.
 * Either the InvalidHostname or InvalidUri shall be provided.
 */
export interface RaiseAuditCertificateDataMismatchEventData extends RaiseAuditCertificateEventData {
    /**
     * InvalidHostname is the string that represents the host name passed in as part of the URL
     * that is found to be invalid. If the host name was not invalid it can be null.
     */
    invalidHostname: PseudoVariantString;
    invalidUri: PseudoVariantString;
}
export interface RaiseAuditCertificateUntrustedEventData extends RaiseAuditCertificateEventData {
}
/**
 * This EventType inherits all Properties of the AuditCertificateEventType.
 *
 * The SourceName for Events of this type shall be “Security/Certificate”.
 *
 * The Message Variable shall include a description of why the certificate was expired
 * (i.e. time before start or time after end).
 *
 * There are no additional Properties defined for this EventType.
 *
 */
export interface RaiseAuditCertificateExpiredEventData extends RaiseAuditCertificateEventData {
}
/**
 * This EventType inherits all Properties of the AuditCertificateEventType.
 *
 * The SourceName for Events of this type shall be “Security/Certificate”.
 *
 * The Message shall include a description of why the certificate is invalid.
 *
 * There are no additional Properties defined for this EventType.
 */
export interface RaiseAuditCertificateInvalidEventData extends RaiseAuditCertificateEventData {
}
/**
 * This EventType inherits all Properties of the AuditCertificateEventType.
 *
 * The SourceName for Events of this type shall be “Security/Certificate”.
 *
 * The Message Variable shall include a description of why the certificate is not trusted.
 * If a trust chain is involved then the certificate that failed in the trust chain should be described.
 * There are no additional Properties defined for this EventType.
 */
export interface RaiseAuditCertificateUntrustedEventData extends RaiseAuditCertificateEventData {
}
/**
 * This EventType inherits all Properties of the AuditCertificateEventType.
 *
 * The SourceName for Events of this type shall be “Security/Certificate”.
 *
 * The Message Variable shall include a description of why the certificate is revoked
 * (was the revocation list unavailable or was the certificate on the list).
 *
 *  There are no additional Properties defined for this EventType.
 */
export interface RaiseAuditCertificateRevokedEventData extends RaiseAuditCertificateEventData {
    sourceName: PseudoVariantStringPredefined<"Security/Certificate">;
}
/**
 * This EventType inherits all Properties of the AuditCertificateEventType.
 *
 * The SourceName for Events of this type shall be “Security/Certificate”.
 *
 * The Message Variable shall include a description of misuse of the certificate.
 *
 * There are no additional Properties defined for this EventType
 */
export interface RaiseAuditCertificateMismatchEventData extends RaiseAuditCertificateEventData {
}
export interface OPCUAServer {
    /**
     * @internal
     * @param eventType
     * @param options
     */
    raiseEvent(eventType: "AuditSessionEventType", options: RaiseEventAuditSessionEventData): void;
    raiseEvent(eventType: "AuditCreateSessionEventType", options: RaiseEventAuditCreateSessionEventData): void;
    raiseEvent(eventType: "AuditActivateSessionEventType", options: RaiseEventAuditActivateSessionEventData): void;
    raiseEvent(eventType: "AuditCreateSessionEventType", options: RaiseEventData): void;
    raiseEvent(eventType: "AuditConditionCommentEventType", options: RaiseEventAuditConditionCommentEventData): void;
    raiseEvent(eventType: "AuditUrlMismatchEventType", options: RaiseEventAuditUrlMismatchEventTypeData): void;
    raiseEvent(eventType: "TransitionEventType", options: RaiseEventTransitionEventData): void;
    raiseEvent(eventType: "AuditCertificateInvalidEventType", options: RaiseAuditCertificateInvalidEventData): void;
    raiseEvent(eventType: "AuditCertificateExpiredEventType", options: RaiseAuditCertificateExpiredEventData): void;
    raiseEvent(eventType: "AuditCertificateUntrustedEventType", options: RaiseAuditCertificateUntrustedEventData): void;
    raiseEvent(eventType: "AuditCertificateRevokedEventType", options: RaiseAuditCertificateRevokedEventData): void;
    raiseEvent(eventType: "AuditCertificateMismatchEventType", options: RaiseAuditCertificateMismatchEventData): void;
    raiseEvent(eventType: UAEventType, options: RaiseEventData): void;
}
export interface OPCUAServer extends EventEmitter {
    on(event: "create_session", eventHandler: (session: ServerSession) => void): this;
    on(event: "session_activated", eventHandler: (session: ServerSession) => void): this;
    on(event: "session_closed", eventHandler: (session: ServerSession, reason: string) => void): this;
    on(event: "post_initialize", eventHandler: () => void): this;
    /**
     * emitted when the server is trying to registered the LDS
     * but when the connection to the lds has failed
     * serverRegistrationPending is sent when the backoff signal of the
     * connection process is raised
     * @event serverRegistrationPending
     */
    on(event: "serverRegistrationPending", eventHandler: () => void): this;
    /**
     * event raised when server  has been successfully registered on the local discovery server
     * @event serverRegistered
     */
    on(event: "serverRegistered", eventHandler: () => void): this;
    /**
     * event raised when server registration has been successfully renewed on the local discovery server
     * @event serverRegistered
     */
    on(event: "serverRegistrationRenewed", eventHandler: () => void): this;
    /**
     * event raised when server  has been successfully unregistered from the local discovery server
     * @event serverUnregistered
     */
    on(event: "serverUnregistered", eventHandler: () => void): this;
    /**
     * event raised after the server has raised an OPCUA event toward a client
     */
    on(event: "event", eventHandler: (eventData: any) => void): this;
    /**
     * event raised when the server received a request from one of its connected client.
     * useful for trace purpose.
     */
    on(event: "request", eventHandler: (request: Request, channel: ServerSecureChannelLayer) => void): this;
    /**
     * event raised when the server send an response to a request to one of its connected client.
     * useful for trace purpose.
     */
    on(event: "response", eventHandler: (request: Response, channel: ServerSecureChannelLayer) => void): this;
    /**
     * event raised when a new secure channel is opened
     */
    on(event: "newChannel", eventHandler: (channel: ServerSecureChannelLayer, endpoint: OPCUAServerEndPoint) => void): this;
    /**
     * event raised when a new secure channel is closed
     */
    on(event: "closeChannel", eventHandler: (channel: ServerSecureChannelLayer, endpoint: OPCUAServerEndPoint) => void): this;
    /**
     * event raised when the server refused a tcp connection from a client. ( for instance because too any connections)
     */
    on(event: "connectionRefused", eventHandler: (socketData: ISocketData, endpoint: OPCUAServerEndPoint) => void): this;
    /**
     * event raised when a OpenSecureChannel has failed, it could be a invalid certificate or malformed message
     */
    on(event: "openSecureChannelFailure", eventHandler: (socketData: ISocketData, channelData: IChannelData, endpoint: OPCUAServerEndPoint) => void): this;
    on(event: string, eventHandler: (...args: [any?, ...any[]]) => void): this;
}
export {};
